/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "xml.hh"
#include "exception.hh"
#include "application.hh"

void XmlNode::create_node(xmlNodePtr n)
{
	if (n == NULL)
	{
		throw Exception(_("Failed to create node: node was NULL"));
	}
	node = n;
}

XmlNode::XmlNode(const XmlNode& n)
{
	create_node(n.node);
}

XmlNode::XmlNode(xmlNodePtr node)
{
	create_node(node);
}

xmlNodePtr XmlNode::create_child_node(const String& name)
{
	xmlNodePtr child_node = xmlNewChild(node, NULL, (const xmlChar*)name.c_str(), NULL);
	if (child_node == NULL)
	{
		throw Exception(_("Failed to create child node '%s'"), name.c_str());
	}
	return child_node;
}

XmlNode& XmlNode::operator=(xmlNodePtr n)
{
	node = n;
	return *this;
}

xmlNodePtr XmlNode::get_parent() const
{
	return node->parent;
}

String XmlNode::get_attribute_value(const String& name) const
{
	char* value = (char*)xmlGetProp(node, (const xmlChar*)name.c_str());
	if (value == NULL)
	{
		throw Exception(_("Attribute '%s' does not exist"), name.c_str());
	}
	String result = value;
	xmlFree(value);
	return result;
}

int XmlNode::get_int_attribute_value(const String& name) const
{
	String value = get_attribute_value(name);
	return atoi(value.c_str());
}

void XmlNode::set_attribute(const String& name, const String& value)
{	
	xmlSetProp(node,
		(const xmlChar*)name.c_str(),
		(const xmlChar*)value.c_str());
}

void XmlNode::set_attribute(const String& name, int value)
{
	String string_value = Integer::to_string(value);
	set_attribute(name, string_value);
}

void XmlNode::set_attribute(const String& name)
{
	xmlSetProp(node,
		(const xmlChar*)name.c_str(),
		NULL);
}

void XmlNode::unlink()
{
	xmlUnlinkNode(node);
}

xmlNodePtr XmlNode::get_node()
{
	return node;
}

String XmlNode::get_path() const
{
	xmlChar* path = xmlGetNodePath(node);
	String result = (const char*)path;
	xmlFree(path);
	return result;
}

xmlDocPtr XmlNode::get_document()
{
	return node->doc;
}

void XmlNode::set_content(const String& content, gboolean encode)
{
	if (encode == true)
	{
		xmlChar* encoded = xmlEncodeSpecialChars(node->doc, (const xmlChar*)content.c_str());	
		xmlNodeSetContent(node, encoded);
		xmlFree(encoded);
	}
	else
	{
		xmlNodeSetContent(node, (const xmlChar*)content.c_str());
	}
}

XmlDocument::XmlDocument(const XmlDocument& document)
{
	throw Exception("XmlDocument::XmlDocument(const XmlDocument& document): Don't do this");
}

XmlDocument::XmlDocument()
{
	document = xmlNewDoc((const xmlChar*)"1.0");
	url = "";
}

XmlDocument::XmlDocument(const String& url, bool create_if_dne)
{
	document = NULL;
	load(url, create_if_dne);
}

XmlDocument::~XmlDocument()
{
	if (document != NULL)
	{
		xmlFreeDoc(document);
		document = NULL;
	}
}

void XmlDocument::load(const String& url, bool create_if_dne)
{
	if (document != NULL)
	{
		xmlFreeDoc(document);
	}
	
	this->url = url;
	
	int check = xmlCheckFilename(url.c_str());
	if (check == 2)
	{
		throw Exception(_("Failed to open XML file '%s' because it is a directory"), url.c_str());
	}
					
	if (check == 1)
	{
		document = xmlParseFile(url.c_str());
	}
	else
	{
		document = xmlNewDoc((const xmlChar*)"1.0");
	}
	
	if (document == NULL)
	{
		throw Exception(_("Failed to open XML document '%s'"), url.c_str());
	}
	
	xmlCleanupParser();				
}

void XmlDocument::save(const String& to)
{
	url = to;
	xmlSaveFormatFile(to.c_str(), document, 0);
}

void XmlDocument::save()
{	
	save(url);
}
			
xmlNodePtr XmlDocument::get_root_node()
{
	return xmlDocGetRootElement(document);
}

xmlNodePtr XmlDocument::create_root_node(const String& name)
{
	xmlNodePtr node = xmlNewNode(NULL, (const xmlChar*)name.c_str());
	xmlDocSetRootElement(document, node);
	return node;
}
	
XPathResult::XPathResult(const XPathResult& result)
{
	throw Exception("ASSERT XPathResult() cannot be called");
}

XPathResult::XPathResult(xmlXPathObjectPtr result)
{
	if (result == NULL)
	{
		throw Exception(_("Failed to create XPathResult for NULL object"));
	}
	
	this->result = result;
}

XPathResult::~XPathResult()
{
	xmlXPathFreeObject(result);
}

int XPathResult::get_count()
{
	int count = 0;				
	if (result != NULL && !xmlXPathNodeSetIsEmpty(result->nodesetval))
	{
		count = result->nodesetval->nodeNr;
	}
	return count;
}

xmlNodePtr XPathResult::get_result(int index)
{
	return result->nodesetval->nodeTab[index];
}
	
void XPath::create(xmlDocPtr document)
{
	context = xmlXPathNewContext(document);
	if (context == NULL)
	{
		throw Exception(_("Failed to create XPath context"));
	}
}

XPath::XPath(XmlDocument& document)
{
	create(document.get_document());
}

XPath::XPath(XmlNode& node)
{
	create(node.get_document());
}

XPath::XPath(xmlNodePtr node)
{
	create(node->doc);
}

XPath::XPath(xmlDocPtr document)
{
	create(document);
}

XPath::~XPath()
{
	if (context != NULL)
	{
		xmlXPathFreeContext(context);
		context = NULL;
	}
}

xmlXPathObjectPtr XPath::evaluate_expression(const String& expression)
{
	xmlXPathObjectPtr result = NULL;
	result = xmlXPathEvalExpression((const xmlChar*)(expression.c_str()), context);
	if (result == NULL)
	{
		throw Exception(_("Failed to evaluate XPath expression: '%s'"), expression.c_str());
	}
	return result;
}

xmlNodePtr XPath::evaluate_expression_single(const String& expression)
{
	XPathResult result = evaluate_expression(expression);
	if (result.get_count() != 1)
	{
		throw Exception(_("Failed to get a single result for '%s'"), expression.c_str());
	}
	return result.get_result(0);
}

String XmlDocument::to_string(gboolean pretty) const
{
	String result;
	xmlChar* text = NULL;
	int size;
	
	xmlDocDumpFormatMemory(document, &text, &size, pretty ? 1 : 0);
	
	if (text == NULL)
	{
		throw Exception(_("Failed to convert XML tree into text"));
	}
	
	result = (const char*)text;

	xmlFree(text);
	
	return result;
}
