/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "string.hh"
#include "exception.hh"
#include <string.h>

String::String(const String& s)
{
	buffer = g_strdup(s.buffer);
}

String::String(const gchar* s)
{
	if (s == NULL)
	{
		throw Exception("Failed to create a NULL string");
	}
	buffer = g_strdup(s);
}

String::String(const gchar* s, gsize length)
{
	if (s == NULL)
	{
		throw Exception("Failed to create a NULL string");
	}
	buffer = g_strndup(s, length);	
}

String::~String()
{
	g_free(buffer);
	buffer = NULL;
}

String String::format(const char* fmt, ...)
{
	va_list ap;
	
	va_start(ap, fmt);
	gchar* text = g_strdup_vprintf(fmt, ap);
	String result = text;
	g_free(text);
	va_end(ap);
	
	return result;
}

const gchar* String::c_str() const
{
	return buffer;
}

gboolean String::is_empty() const
{
	return buffer[0] == 0;
}

gsize String::get_buffer_length() const
{
	return strlen(buffer);
}

gsize String::get_character_length() const
{
	return g_utf8_strlen(buffer, -1);
}

void String::canon(const gchar *valid_chars, gchar substitutor)
{
	g_strcanon(buffer, valid_chars, substitutor);
}

String String::trim() const
{
	String result(buffer);
	result = g_strstrip(result.buffer);
	return result;
}

void String::assign(const gchar* s)
{
	gchar* temp = buffer;
	buffer = g_strdup(s);
	g_free(temp);
}

void String::assign(const gchar* s, gsize length)
{
	gchar* temp = buffer;
	buffer = g_strndup(s, length);
	g_free(temp);
}

void String::assign(const String& s)
{
	assign(s.buffer);
}

void String::append(const gchar* s, gsize length)
{
	gchar* temp = buffer;
	String str(s, length);
	buffer = g_strconcat(buffer, str.c_str(), NULL);
	g_free(temp);
}

void String::append(const gchar* s)
{
	gchar* temp = buffer;	
	buffer = g_strconcat(buffer, s, NULL);
	g_free(temp);
}

gboolean String::equals(const gchar* s) const
{
	return g_str_equal(buffer, s);
}

gboolean String::equals(const String& s) const
{
	return equals(s.buffer);
}

String& String::operator+=(const gchar* s)
{
	append(s);
	return *this;
}

String& String::operator+=(const String& s)
{
	append(s.buffer);
	return *this;
}

String& String::operator+=(gunichar ch)
{
	gchar outbuf[6];
	gint length = g_unichar_to_utf8(ch, outbuf);
	append(outbuf, length);
	return *this;
}

gint String::compare(const gchar* s) const
{
	return g_strncasecmp(buffer, s, get_buffer_length());
}

String String::lowercase() const
{
	String result;
	const gchar* p = buffer;
	
	gunichar ch = g_utf8_get_char(p);
	while (ch != 0)
	{
		result += g_unichar_tolower(ch);
		p = g_utf8_next_char(p);
		ch = g_utf8_get_char(p);
	}

	return result;
}

const gchar* String::find(const gchar* needle) const
{
	return g_strstr_len(buffer, -1, needle);
}

String& String::operator=(const String& s)
{
	assign(s);
	return *this;
}

String& String::operator=(const gchar* s)
{
	assign(s);
	return *this;
}

String String::escape() const
{
	gchar* escaped = g_strescape(buffer, NULL);
	String result = escaped;
	g_free(escaped);
	
	return result;
}

String String::replace(gchar ch1, gchar ch2) const
{
	String result;
	
	gchar* p = result.buffer;
	while (*p != 0)
	{
		if (*p == ch1)
		{
			*p = ch2;
		}
		p++;
	}
	
	return result;
}

String String::replace(const String& s1, const String& s2) const
{
	gchar** parts = g_strsplit(buffer, s1.buffer, -1);
	gchar* joined = g_strjoinv(s2.buffer, parts);
	String result = joined;
	g_free(joined);
	g_strfreev(parts);
	return result;
}

gboolean String::operator==(const String& s) const
{
	return equals(s);
}

gboolean String::operator!=(const String& s) const
{
	return !equals(s);
}
