//
// expression.cs: Expression representation for the IL tree.
//
// Author:
//   Miguel de Icaza (miguel@ximian.com)
//
// (C) 2001, 2002, 2003 Ximian, Inc.
// (C) 2003, 2004 Novell, Inc.
//
#define USE_OLD

namespace Mono.CSharp {
	using System;
	using System.Collections;
	using System.Reflection;
	using System.Reflection.Emit;
	using System.Text;

	/// <summary>
	///   This is just a helper class, it is generated by Unary, UnaryMutator
	///   when an overloaded method has been found.  It just emits the code for a
	///   static call.
	/// </summary>
	public class StaticCallExpr : ExpressionStatement {
		ArrayList args;
		MethodInfo mi;

		public StaticCallExpr (MethodInfo m, ArrayList a, Location l)
		{
			mi = m;
			args = a;

			type = m.ReturnType;
			eclass = ExprClass.Value;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// We are born fully resolved
			//
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			if (args != null) 
				Invocation.EmitArguments (ec, mi, args);

			ec.ig.Emit (OpCodes.Call, mi);
			return;
		}
		
		static public Expression MakeSimpleCall (EmitContext ec, MethodGroupExpr mg,
							 Expression e, Location loc)
		{
			ArrayList args;
			MethodBase method;
			
			args = new ArrayList (1);
			Argument a = new Argument (e, Argument.AType.Expression);

                        // We need to resolve the arguments before sending them in !
                        if (!a.Resolve (ec, loc))
                                return null;

                        args.Add (a);
			method = Invocation.OverloadResolve (ec, (MethodGroupExpr) mg, args, loc);

			if (method == null)
				return null;

			return new StaticCallExpr ((MethodInfo) method, args, loc);
		}

		public override void EmitStatement (EmitContext ec)
		{
			Emit (ec);
			if (TypeManager.TypeToCoreType (type) != TypeManager.void_type)
				ec.ig.Emit (OpCodes.Pop);
		}
	}

	public class ParenthesizedExpression : Expression
	{
		public Expression Expr;

		public ParenthesizedExpression (Expression expr, Location loc)
		{
			this.Expr = expr;
			this.loc = loc;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			Expr = Expr.Resolve (ec);
			return Expr;
		}

		public override void Emit (EmitContext ec)
		{
			throw new Exception ("Should not happen");
		}
	}
	
	/// <summary>
	///   Unary expressions.  
	/// </summary>
	///
	/// <remarks>
	///   Unary implements unary expressions.   It derives from
	///   ExpressionStatement becuase the pre/post increment/decrement
	///   operators can be used in a statement context.
	/// </remarks>
	public class Unary : Expression {
		public enum Operator : byte {
			UnaryPlus, UnaryNegation, LogicalNot, OnesComplement,
			Indirection, AddressOf,  TOP
		}

		public Operator Oper;
		public Expression Expr;
		
		public Unary (Operator op, Expression expr, Location loc)
		{
			this.Oper = op;
			this.Expr = expr;
			this.loc = loc;
		}

		/// <summary>
		///   Returns a stringified representation of the Operator
		/// </summary>
		static public string OperName (Operator oper)
		{
			switch (oper){
			case Operator.UnaryPlus:
				return "+";
			case Operator.UnaryNegation:
				return "-";
			case Operator.LogicalNot:
				return "!";
			case Operator.OnesComplement:
				return "~";
			case Operator.AddressOf:
				return "&";
			case Operator.Indirection:
				return "*";
			}

			return oper.ToString ();
		}

		public static readonly string [] oper_names;

		static Unary ()
		{
			oper_names = new string [(int)Operator.TOP];

			oper_names [(int) Operator.UnaryPlus] = "op_UnaryPlus";
			oper_names [(int) Operator.UnaryNegation] = "op_UnaryNegation";
			oper_names [(int) Operator.LogicalNot] = "op_LogicalNot";
			oper_names [(int) Operator.OnesComplement] = "op_OnesComplement";
			oper_names [(int) Operator.Indirection] = "op_Indirection";
			oper_names [(int) Operator.AddressOf] = "op_AddressOf";
		}

		void Error23 (Type t)
		{
			Error (
				23, "Operator " + OperName (Oper) +
				" cannot be applied to operand of type `" +
				TypeManager.CSharpName (t) + "'");
		}

		/// <remarks>
		///   The result has been already resolved:
		///
		///   FIXME: a minus constant -128 sbyte cant be turned into a
		///   constant byte.
		/// </remarks>
		static Expression TryReduceNegative (Constant expr)
		{
			Expression e = null;

			if (expr is IntConstant)
				e = new IntConstant (-((IntConstant) expr).Value);
			else if (expr is UIntConstant){
				uint value = ((UIntConstant) expr).Value;

				if (value < 2147483649)
					return new IntConstant (-(int)value);
				else
					e = new LongConstant (-value);
			}
			else if (expr is LongConstant)
				e = new LongConstant (-((LongConstant) expr).Value);
			else if (expr is ULongConstant){
				ulong value = ((ULongConstant) expr).Value;

				if (value < 9223372036854775809)
					return new LongConstant(-(long)value);
			}
			else if (expr is FloatConstant)
				e = new FloatConstant (-((FloatConstant) expr).Value);
			else if (expr is DoubleConstant)
				e = new DoubleConstant (-((DoubleConstant) expr).Value);
			else if (expr is DecimalConstant)
				e = new DecimalConstant (-((DecimalConstant) expr).Value);
			else if (expr is ShortConstant)
				e = new IntConstant (-((ShortConstant) expr).Value);
			else if (expr is UShortConstant)
				e = new IntConstant (-((UShortConstant) expr).Value);
			return e;
		}

		// <summary>
		//   This routine will attempt to simplify the unary expression when the
		//   argument is a constant.  The result is returned in `result' and the
		//   function returns true or false depending on whether a reduction
		//   was performed or not
		// </summary>
		bool Reduce (EmitContext ec, Constant e, out Expression result)
		{
			Type expr_type = e.Type;
			
			switch (Oper){
			case Operator.UnaryPlus:
				result = e;
				return true;
				
			case Operator.UnaryNegation:
				result = TryReduceNegative (e);
				return true;
				
			case Operator.LogicalNot:
				if (expr_type != TypeManager.bool_type) {
					result = null;
					Error23 (expr_type);
					return false;
				}
				
				BoolConstant b = (BoolConstant) e;
				result = new BoolConstant (!(b.Value));
				return true;
				
			case Operator.OnesComplement:
				if (!((expr_type == TypeManager.int32_type) ||
				      (expr_type == TypeManager.uint32_type) ||
				      (expr_type == TypeManager.int64_type) ||
				      (expr_type == TypeManager.uint64_type) ||
				      (expr_type.IsSubclassOf (TypeManager.enum_type)))){

					result = null;
					if (Convert.ImplicitConversionExists (ec, e, TypeManager.int32_type)){
						result = new Cast (new TypeExpression (TypeManager.int32_type, loc), e, loc);
						result = result.Resolve (ec);
					} else if (Convert.ImplicitConversionExists (ec, e, TypeManager.uint32_type)){
						result = new Cast (new TypeExpression (TypeManager.uint32_type, loc), e, loc);
						result = result.Resolve (ec);
					} else if (Convert.ImplicitConversionExists (ec, e, TypeManager.int64_type)){
						result = new Cast (new TypeExpression (TypeManager.int64_type, loc), e, loc);
						result = result.Resolve (ec);
					} else if (Convert.ImplicitConversionExists (ec, e, TypeManager.uint64_type)){
						result = new Cast (new TypeExpression (TypeManager.uint64_type, loc), e, loc);
						result = result.Resolve (ec);
					}

					if (result == null || !(result is Constant)){
						result = null;
						Error23 (expr_type);
						return false;
					}

					expr_type = result.Type;
					e = (Constant) result;
				}

				if (e is EnumConstant){
					EnumConstant enum_constant = (EnumConstant) e;
					Expression reduced;
					
					if (Reduce (ec, enum_constant.Child, out reduced)){
						result = new EnumConstant ((Constant) reduced, enum_constant.Type);
						return true;
					} else {
						result = null;
						return false;
					}
				}

				if (expr_type == TypeManager.int32_type){
					result = new IntConstant (~ ((IntConstant) e).Value);
				} else if (expr_type == TypeManager.uint32_type){
					result = new UIntConstant (~ ((UIntConstant) e).Value);
				} else if (expr_type == TypeManager.int64_type){
					result = new LongConstant (~ ((LongConstant) e).Value);
				} else if (expr_type == TypeManager.uint64_type){
					result = new ULongConstant (~ ((ULongConstant) e).Value);
				} else {
					result = null;
					Error23 (expr_type);
					return false;
				}
				return true;

			case Operator.AddressOf:
				result = this;
				return false;

			case Operator.Indirection:
				result = this;
				return false;
			}
			throw new Exception ("Can not constant fold: " + Oper.ToString());
		}

		Expression ResolveOperator (EmitContext ec)
		{
			Type expr_type = Expr.Type;

			//
			// Step 1: Perform Operator Overload location
			//
			Expression mg;
			string op_name;
			
			op_name = oper_names [(int) Oper];

			mg = MemberLookup (ec, expr_type, op_name, MemberTypes.Method, AllBindingFlags, loc);
			
			if (mg != null) {
				Expression e = StaticCallExpr.MakeSimpleCall (
					ec, (MethodGroupExpr) mg, Expr, loc);

				if (e == null){
					Error23 (expr_type);
					return null;
				}
				
				return e;
			}

			// Only perform numeric promotions on:
			// +, - 

			if (expr_type == null)
				return null;
			
			//
			// Step 2: Default operations on CLI native types.
			//

			// Attempt to use a constant folding operation.
			if (Expr is Constant){
				Expression result;
				
				if (Reduce (ec, (Constant) Expr, out result))
					return result;
			}

			switch (Oper){
			case Operator.LogicalNot:
				if (expr_type != TypeManager.bool_type) {
					Expr = ResolveBoolean (ec, Expr, loc);
					if (Expr == null){
						Error23 (expr_type);
						return null;
					}
				}
				
				type = TypeManager.bool_type;
				return this;

			case Operator.OnesComplement:
				if (!((expr_type == TypeManager.int32_type) ||
				      (expr_type == TypeManager.uint32_type) ||
				      (expr_type == TypeManager.int64_type) ||
				      (expr_type == TypeManager.uint64_type) ||
				      (expr_type.IsSubclassOf (TypeManager.enum_type)))){
					Expression e;

					e = Convert.ImplicitConversion (ec, Expr, TypeManager.int32_type, loc);
					if (e != null){
						type = TypeManager.int32_type;
						return this;
					}
					e = Convert.ImplicitConversion (ec, Expr, TypeManager.uint32_type, loc);
					if (e != null){
						type = TypeManager.uint32_type;
						return this;
					}
					e = Convert.ImplicitConversion (ec, Expr, TypeManager.int64_type, loc);
					if (e != null){
						type = TypeManager.int64_type;
						return this;
					}
					e = Convert.ImplicitConversion (ec, Expr, TypeManager.uint64_type, loc);
					if (e != null){
						type = TypeManager.uint64_type;
						return this;
					}
					Error23 (expr_type);
					return null;
				}
				type = expr_type;
				return this;

			case Operator.AddressOf:
				if (Expr.eclass != ExprClass.Variable){
					Error (211, "Cannot take the address of non-variables");
					return null;
				}
				
				if (!ec.InUnsafe) {
					UnsafeError (loc); 
					return null;
				}
				
				if (!TypeManager.VerifyUnManaged (Expr.Type, loc)){
					return null;
				}

				IVariable variable = Expr as IVariable;
				if (!ec.InFixedInitializer && ((variable == null) || !variable.VerifyFixed (false))) {
					Error (212, "You can only take the address of an unfixed expression inside " +
					       "of a fixed statement initializer");
					return null;
				}
				
				if (ec.InFixedInitializer && ((variable != null) && variable.VerifyFixed (false))) {
					Error (213, "You can not fix an already fixed expression");
					return null;
				}

				// According to the specs, a variable is considered definitely assigned if you take
				// its address.
				if ((variable != null) && (variable.VariableInfo != null))
					variable.VariableInfo.SetAssigned (ec);

				type = TypeManager.GetPointerType (Expr.Type);
				return this;

			case Operator.Indirection:
				if (!ec.InUnsafe){
					UnsafeError (loc);
					return null;
				}
				
				if (!expr_type.IsPointer){
					Error (193, "The * or -> operator can only be applied to pointers");
					return null;
				}
				
				//
				// We create an Indirection expression, because
				// it can implement the IMemoryLocation.
				// 
				return new Indirection (Expr, loc);
			
			case Operator.UnaryPlus:
				//
				// A plus in front of something is just a no-op, so return the child.
				//
				return Expr;

			case Operator.UnaryNegation:
				//
				// Deals with -literals
				// int     operator- (int x)
				// long    operator- (long x)
				// float   operator- (float f)
				// double  operator- (double d)
				// decimal operator- (decimal d)
				//
				Expression expr = null;

				//
				// transform - - expr into expr
				//
				if (Expr is Unary){
					Unary unary = (Unary) Expr;
					
					if (unary.Oper == Operator.UnaryNegation)
						return unary.Expr;
				}

				//
				// perform numeric promotions to int,
				// long, double.
				//
				//
				// The following is inneficient, because we call
				// ImplicitConversion too many times.
				//
				// It is also not clear if we should convert to Float
				// or Double initially.
				//
				if (expr_type == TypeManager.uint32_type){
					//
					// FIXME: handle exception to this rule that
					// permits the int value -2147483648 (-2^31) to
					// bt wrote as a decimal interger literal
					//
					type = TypeManager.int64_type;
					Expr = Convert.ImplicitConversion (ec, Expr, type, loc);
					return this;
				}

				if (expr_type == TypeManager.uint64_type){
					//
					// FIXME: Handle exception of `long value'
					// -92233720368547758087 (-2^63) to be wrote as
					// decimal integer literal.
					//
					Error23 (expr_type);
					return null;
				}

				if (expr_type == TypeManager.float_type){
					type = expr_type;
					return this;
				}
				
				expr = Convert.ImplicitConversion (ec, Expr, TypeManager.int32_type, loc);
				if (expr != null){
					Expr = expr;
					type = expr.Type;
					return this;
				} 

				expr = Convert.ImplicitConversion (ec, Expr, TypeManager.int64_type, loc);
				if (expr != null){
					Expr = expr;
					type = expr.Type;
					return this;
				}

				expr = Convert.ImplicitConversion (ec, Expr, TypeManager.double_type, loc);
				if (expr != null){
					Expr = expr;
					type = expr.Type;
					return this;
				}
				
				Error23 (expr_type);
				return null;
			}

			Error (187, "No such operator '" + OperName (Oper) + "' defined for type '" +
			       TypeManager.CSharpName (expr_type) + "'");
			return null;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			if (Oper == Operator.AddressOf)
				Expr = Expr.ResolveLValue (ec, new EmptyExpression ());
			else
				Expr = Expr.Resolve (ec);
			
			if (Expr == null)
				return null;

			eclass = ExprClass.Value;
			return ResolveOperator (ec);
		}

		public override Expression DoResolveLValue (EmitContext ec, Expression right)
		{
			if (Oper == Operator.Indirection)
				return base.DoResolveLValue (ec, right);

			Error (131, "The left-hand side of an assignment must be a " +
			       "variable, property or indexer");
			return null;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			switch (Oper) {
			case Operator.UnaryPlus:
				throw new Exception ("This should be caught by Resolve");
				
			case Operator.UnaryNegation:
				if (ec.CheckState) {
					ig.Emit (OpCodes.Ldc_I4_0);
					if (type == TypeManager.int64_type)
						ig.Emit (OpCodes.Conv_U8);
					Expr.Emit (ec);
					ig.Emit (OpCodes.Sub_Ovf);
				} else {
					Expr.Emit (ec);
					ig.Emit (OpCodes.Neg);
				}
				
				break;
				
			case Operator.LogicalNot:
				Expr.Emit (ec);
				ig.Emit (OpCodes.Ldc_I4_0);
				ig.Emit (OpCodes.Ceq);
				break;
				
			case Operator.OnesComplement:
				Expr.Emit (ec);
				ig.Emit (OpCodes.Not);
				break;
				
			case Operator.AddressOf:
				((IMemoryLocation)Expr).AddressOf (ec, AddressOp.LoadStore);
				break;
				
			default:
				throw new Exception ("This should not happen: Operator = "
						     + Oper.ToString ());
			}
		}

		public override void EmitBranchable (EmitContext ec, Label target, bool onTrue)
		{
			if (Oper == Operator.LogicalNot)
				Expr.EmitBranchable (ec, target, !onTrue);
			else
				base.EmitBranchable (ec, target, onTrue);
		}

		public override string ToString ()
		{
			return "Unary (" + Oper + ", " + Expr + ")";
		}
		
	}

	//
	// Unary operators are turned into Indirection expressions
	// after semantic analysis (this is so we can take the address
	// of an indirection).
	//
	public class Indirection : Expression, IMemoryLocation, IAssignMethod {
		Expression expr;
		LocalTemporary temporary;
		bool have_temporary;
		
		public Indirection (Expression expr, Location l)
		{
			this.expr = expr;
			this.type = TypeManager.GetElementType (expr.Type);
			eclass = ExprClass.Variable;
			loc = l;
		}

		void LoadExprValue (EmitContext ec)
		{
		}
		
		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			if (temporary != null){
				if (have_temporary) {
					temporary.Emit (ec);
				} else {
					expr.Emit (ec);
					ec.ig.Emit (OpCodes.Dup);
					temporary.Store (ec);
					have_temporary = true;
				}
			} else
				expr.Emit (ec);
			
			LoadFromPtr (ig, Type);
		}

		public void EmitAssign (EmitContext ec, Expression source)
		{
			if (temporary != null){
				if (have_temporary)
					temporary.Emit (ec);
				else {
					expr.Emit (ec);
					ec.ig.Emit (OpCodes.Dup);
					temporary.Store (ec);
					have_temporary = true;
				}
			} else 
				expr.Emit (ec);

			source.Emit (ec);
			StoreFromPtr (ec.ig, type);
		}
		
		public void AddressOf (EmitContext ec, AddressOp Mode)
		{
			if (temporary != null){
				if (have_temporary){
					temporary.Emit (ec);
					return;
				}
				expr.Emit (ec);
				ec.ig.Emit (OpCodes.Dup);
				temporary.Store (ec);
				have_temporary = true;
			} else
				expr.Emit (ec);
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// Born fully resolved
			//
			return this;
		}

		public new void CacheTemporaries (EmitContext ec)
		{
			temporary = new LocalTemporary (ec, expr.Type);
		}

		public override string ToString ()
		{
			return "*(" + expr + ")";
		}
	}
	
	/// <summary>
	///   Unary Mutator expressions (pre and post ++ and --)
	/// </summary>
	///
	/// <remarks>
	///   UnaryMutator implements ++ and -- expressions.   It derives from
	///   ExpressionStatement becuase the pre/post increment/decrement
	///   operators can be used in a statement context.
	///
	/// FIXME: Idea, we could split this up in two classes, one simpler
	/// for the common case, and one with the extra fields for more complex
	/// classes (indexers require temporary access;  overloaded require method)
	///
	/// </remarks>
	public class UnaryMutator : ExpressionStatement {
		[Flags]
		public enum Mode : byte {
			IsIncrement    = 0,
			IsDecrement    = 1,
			IsPre          = 0,
			IsPost         = 2,
			
			PreIncrement   = 0,
			PreDecrement   = IsDecrement,
			PostIncrement  = IsPost,
			PostDecrement  = IsPost | IsDecrement
		}
		
		Mode mode;
		Expression expr;
		LocalTemporary temp_storage;

		//
		// This is expensive for the simplest case.
		//
		Expression method;
			
		public UnaryMutator (Mode m, Expression e, Location l)
		{
			mode = m;
			loc = l;
			expr = e;
		}

		static string OperName (Mode mode)
		{
			return (mode == Mode.PreIncrement || mode == Mode.PostIncrement) ?
				"++" : "--";
		}
		
		void Error23 (Type t)
		{
			Error (
				23, "Operator " + OperName (mode) + 
				" cannot be applied to operand of type `" +
				TypeManager.CSharpName (t) + "'");
		}

		/// <summary>
		///   Returns whether an object of type `t' can be incremented
		///   or decremented with add/sub (ie, basically whether we can
		///   use pre-post incr-decr operations on it, but it is not a
		///   System.Decimal, which we require operator overloading to catch)
		/// </summary>
		static bool IsIncrementableNumber (Type t)
		{
			return (t == TypeManager.sbyte_type) ||
				(t == TypeManager.byte_type) ||
				(t == TypeManager.short_type) ||
				(t == TypeManager.ushort_type) ||
				(t == TypeManager.int32_type) ||
				(t == TypeManager.uint32_type) ||
				(t == TypeManager.int64_type) ||
				(t == TypeManager.uint64_type) ||
				(t == TypeManager.char_type) ||
				(t.IsSubclassOf (TypeManager.enum_type)) ||
				(t == TypeManager.float_type) ||
				(t == TypeManager.double_type) ||
				(t.IsPointer && t != TypeManager.void_ptr_type);
		}

		Expression ResolveOperator (EmitContext ec)
		{
			Type expr_type = expr.Type;

			//
			// Step 1: Perform Operator Overload location
			//
			Expression mg;
			string op_name;
			
			if (mode == Mode.PreIncrement || mode == Mode.PostIncrement)
				op_name = "op_Increment";
			else 
				op_name = "op_Decrement";

			mg = MemberLookup (ec, expr_type, op_name, MemberTypes.Method, AllBindingFlags, loc);

			if (mg == null && expr_type.BaseType != null)
				mg = MemberLookup (ec, expr_type.BaseType, op_name,
						   MemberTypes.Method, AllBindingFlags, loc);
			
			if (mg != null) {
				method = StaticCallExpr.MakeSimpleCall (
					ec, (MethodGroupExpr) mg, expr, loc);

				type = method.Type;
				return this;
			}

			//
			// The operand of the prefix/postfix increment decrement operators
			// should be an expression that is classified as a variable,
			// a property access or an indexer access
			//
			type = expr_type;
			if (expr.eclass == ExprClass.Variable){
				LocalVariableReference var = expr as LocalVariableReference;
				if ((var != null) && var.IsReadOnly)
					Error (1604, "cannot assign to `" + var.Name + "' because it is readonly");
				if (IsIncrementableNumber (expr_type) ||
				    expr_type == TypeManager.decimal_type){
					return this;
				}
			} else if (expr.eclass == ExprClass.IndexerAccess){
				IndexerAccess ia = (IndexerAccess) expr;
				
				temp_storage = new LocalTemporary (ec, expr.Type);
				
				expr = ia.ResolveLValue (ec, temp_storage);
				if (expr == null)
					return null;

				return this;
			} else if (expr.eclass == ExprClass.PropertyAccess){
				PropertyExpr pe = (PropertyExpr) expr;

				if (pe.VerifyAssignable ())
					return this;

				return null;
			} else {
				expr.Error_UnexpectedKind ("variable, indexer or property access");
				return null;
			}

			Error (187, "No such operator '" + OperName (mode) + "' defined for type '" +
			       TypeManager.CSharpName (expr_type) + "'");
			return null;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			expr = expr.Resolve (ec);
			
			if (expr == null)
				return null;

			eclass = ExprClass.Value;
			return ResolveOperator (ec);
		}

		static int PtrTypeSize (Type t)
		{
			return GetTypeSize (TypeManager.GetElementType (t));
		}

		//
		// Loads the proper "1" into the stack based on the type, then it emits the
		// opcode for the operation requested
		//
		void LoadOneAndEmitOp (EmitContext ec, Type t)
		{
			//
			// Measure if getting the typecode and using that is more/less efficient
			// that comparing types.  t.GetTypeCode() is an internal call.
			//
			ILGenerator ig = ec.ig;
						     
			if (t == TypeManager.uint64_type || t == TypeManager.int64_type)
				LongConstant.EmitLong (ig, 1);
			else if (t == TypeManager.double_type)
				ig.Emit (OpCodes.Ldc_R8, 1.0);
			else if (t == TypeManager.float_type)
				ig.Emit (OpCodes.Ldc_R4, 1.0F);
			else if (t.IsPointer){
				int n = PtrTypeSize (t);
				
				if (n == 0)
					ig.Emit (OpCodes.Sizeof, t);
				else
					IntConstant.EmitInt (ig, n);
			} else 
				ig.Emit (OpCodes.Ldc_I4_1);

			//
			// Now emit the operation
			//
			if (ec.CheckState){
				if (t == TypeManager.int32_type ||
				    t == TypeManager.int64_type){
					if ((mode & Mode.IsDecrement) != 0)
						ig.Emit (OpCodes.Sub_Ovf);
					else
						ig.Emit (OpCodes.Add_Ovf);
				} else if (t == TypeManager.uint32_type ||
					   t == TypeManager.uint64_type){
					if ((mode & Mode.IsDecrement) != 0)
						ig.Emit (OpCodes.Sub_Ovf_Un);
					else
						ig.Emit (OpCodes.Add_Ovf_Un);
				} else {
					if ((mode & Mode.IsDecrement) != 0)
						ig.Emit (OpCodes.Sub_Ovf);
					else
						ig.Emit (OpCodes.Add_Ovf);
				}
			} else {
				if ((mode & Mode.IsDecrement) != 0)
					ig.Emit (OpCodes.Sub);
				else
					ig.Emit (OpCodes.Add);
			}

			if (t == TypeManager.sbyte_type){
				if (ec.CheckState)
					ig.Emit (OpCodes.Conv_Ovf_I1);
				else
					ig.Emit (OpCodes.Conv_I1);
			} else if (t == TypeManager.byte_type){
				if (ec.CheckState)
					ig.Emit (OpCodes.Conv_Ovf_U1);
				else
					ig.Emit (OpCodes.Conv_U1);
			} else if (t == TypeManager.short_type){
				if (ec.CheckState)
					ig.Emit (OpCodes.Conv_Ovf_I2);
				else
					ig.Emit (OpCodes.Conv_I2);
			} else if (t == TypeManager.ushort_type || t == TypeManager.char_type){
				if (ec.CheckState)
					ig.Emit (OpCodes.Conv_Ovf_U2);
				else
					ig.Emit (OpCodes.Conv_U2);
			}
			
		}

		static EmptyExpression empty_expr;
		
		void EmitCode (EmitContext ec, bool is_expr)
		{
			ILGenerator ig = ec.ig;
			IAssignMethod ia = (IAssignMethod) expr;
			Type expr_type = expr.Type;

			ia.CacheTemporaries (ec);

			//
			// NOTE: We should probably handle three cases:
			//
			//     * method invocation required.
			//     * direct stack manipulation possible
			//     * the object requires an "instance" field
			//
			if (temp_storage == null){
				//
				// Temporary improvement: if we are dealing with something that does
				// not require complicated instance setup, avoid using a temporary
				//
				// For now: only localvariables when not remapped
				//

				if (method == null &&
				    ((expr is LocalVariableReference) ||(expr is FieldExpr && ((FieldExpr) expr).FieldInfo.IsStatic))){
					if (empty_expr == null)
						empty_expr = new EmptyExpression ();
					
					switch (mode){
					case Mode.PreIncrement:
					case Mode.PreDecrement:
						expr.Emit (ec);
					
						LoadOneAndEmitOp (ec, expr_type);
						if (is_expr)
							ig.Emit (OpCodes.Dup);
						ia.EmitAssign (ec, empty_expr);
						break;
						
					case Mode.PostIncrement:
					case Mode.PostDecrement:
						expr.Emit (ec);
						if (is_expr)
							ig.Emit (OpCodes.Dup);
						
						LoadOneAndEmitOp (ec, expr_type);
						ia.EmitAssign (ec, empty_expr);
						break;
					}
					return;
				}
				temp_storage = new LocalTemporary (ec, expr_type);
			}
			
			switch (mode){
			case Mode.PreIncrement:
			case Mode.PreDecrement:
				if (method == null){
					expr.Emit (ec);
					
					LoadOneAndEmitOp (ec, expr_type);
				} else 
					method.Emit (ec);
				
				temp_storage.Store (ec);
				ia.EmitAssign (ec, temp_storage);
				if (is_expr)
					temp_storage.Emit (ec);
				break;
				
			case Mode.PostIncrement:
			case Mode.PostDecrement:
				if (is_expr)
					expr.Emit (ec);
				
				if (method == null){
					if (!is_expr)
						expr.Emit (ec);
					else
						ig.Emit (OpCodes.Dup);
					
					LoadOneAndEmitOp (ec, expr_type);
				} else {
					method.Emit (ec);
				}
				
				temp_storage.Store (ec);
				ia.EmitAssign (ec, temp_storage);
				break;
			}

			temp_storage.Release (ec);
		}

		public override void Emit (EmitContext ec)
		{
			EmitCode (ec, true);
			
		}
		
		public override void EmitStatement (EmitContext ec)
		{
			EmitCode (ec, false);
		}

	}

	/// <summary>
	///   Base class for the `Is' and `As' classes. 
	/// </summary>
	///
	/// <remarks>
	///   FIXME: Split this in two, and we get to save the `Operator' Oper
	///   size. 
	/// </remarks>
	public abstract class Probe : Expression {
		public readonly Expression ProbeType;
		protected Expression expr;
		protected Type probe_type;
		
		public Probe (Expression expr, Expression probe_type, Location l)
		{
			ProbeType = probe_type;
			loc = l;
			this.expr = expr;
		}

		public Expression Expr {
			get {
				return expr;
			}
		}

		public override Expression DoResolve (EmitContext ec)
		{
			probe_type = ec.DeclSpace.ResolveType (ProbeType, false, loc);

			if (probe_type == null)
				return null;

			CheckObsoleteAttribute (probe_type);

			expr = expr.Resolve (ec);
			if (expr == null)
				return null;
			
			return this;
		}
	}

	/// <summary>
	///   Implementation of the `is' operator.
	/// </summary>
	public class Is : Probe {
		public Is (Expression expr, Expression probe_type, Location l)
			: base (expr, probe_type, l)
		{
		}

		enum Action {
			AlwaysTrue, AlwaysNull, AlwaysFalse, LeaveOnStack, Probe
		}

		Action action;
		
		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			expr.Emit (ec);

			switch (action){
			case Action.AlwaysFalse:
				ig.Emit (OpCodes.Pop);
				IntConstant.EmitInt (ig, 0);
				return;
			case Action.AlwaysTrue:
				ig.Emit (OpCodes.Pop);
				IntConstant.EmitInt (ig, 1);
				return;
			case Action.LeaveOnStack:
				// the `e != null' rule.
				ig.Emit (OpCodes.Ldnull);
				ig.Emit (OpCodes.Ceq);
				ig.Emit (OpCodes.Ldc_I4_0);
				ig.Emit (OpCodes.Ceq);
				return;
			case Action.Probe:
				ig.Emit (OpCodes.Isinst, probe_type);
				ig.Emit (OpCodes.Ldnull);
				ig.Emit (OpCodes.Cgt_Un);
				return;
			}
			throw new Exception ("never reached");
		}
		
		public override void EmitBranchable (EmitContext ec, Label target, bool onTrue)
		{
			ILGenerator ig = ec.ig;

			switch (action){
			case Action.AlwaysFalse:
				if (! onTrue)
					ig.Emit (OpCodes.Br, target);
				
				return;
			case Action.AlwaysTrue:
				if (onTrue)
					ig.Emit (OpCodes.Br, target);
				
				return;
			case Action.LeaveOnStack:
				// the `e != null' rule.
				expr.Emit (ec);
				ig.Emit (onTrue ? OpCodes.Brtrue : OpCodes.Brfalse, target);
				return;
			case Action.Probe:
				expr.Emit (ec);
				ig.Emit (OpCodes.Isinst, probe_type);
				ig.Emit (onTrue ? OpCodes.Brtrue : OpCodes.Brfalse, target);
				return;
			}
			throw new Exception ("never reached");
		}

		public override Expression DoResolve (EmitContext ec)
		{
			Expression e = base.DoResolve (ec);

			if ((e == null) || (expr == null))
				return null;

			Type etype = expr.Type;
			bool warning_always_matches = false;
			bool warning_never_matches = false;

			type = TypeManager.bool_type;
			eclass = ExprClass.Value;

			//
			// First case, if at compile time, there is an implicit conversion
			// then e != null (objects) or true (value types)
			//
			e = Convert.ImplicitConversionStandard (ec, expr, probe_type, loc);
			if (e != null){
				expr = e;
				if (etype.IsValueType)
					action = Action.AlwaysTrue;
				else
					action = Action.LeaveOnStack;

				warning_always_matches = true;
			} else if (Convert.ExplicitReferenceConversionExists (etype, probe_type)){
				//
				// Second case: explicit reference convresion
				//
				if (expr is NullLiteral)
					action = Action.AlwaysFalse;
				else
					action = Action.Probe;
			} else {
				action = Action.AlwaysFalse;
				warning_never_matches = true;
			}
			
			if (RootContext.WarningLevel >= 1){
				if (warning_always_matches)
					Warning (183, "The expression is always of type `" +
						 TypeManager.CSharpName (probe_type) + "'");
				else if (warning_never_matches){
					if (!(probe_type.IsInterface || expr.Type.IsInterface))
						Warning (184,
							 "The expression is never of type `" +
							 TypeManager.CSharpName (probe_type) + "'");
				}
			}

			return this;
		}				
	}

	/// <summary>
	///   Implementation of the `as' operator.
	/// </summary>
	public class As : Probe {
		public As (Expression expr, Expression probe_type, Location l)
			: base (expr, probe_type, l)
		{
		}

		bool do_isinst = false;
		
		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			expr.Emit (ec);

			if (do_isinst)
				ig.Emit (OpCodes.Isinst, probe_type);
		}

		static void Error_CannotConvertType (Type source, Type target, Location loc)
		{
			Report.Error (
				39, loc, "as operator can not convert from `" +
				TypeManager.CSharpName (source) + "' to `" +
				TypeManager.CSharpName (target) + "'");
		}
		
		public override Expression DoResolve (EmitContext ec)
		{
			Expression e = base.DoResolve (ec);

			if (e == null)
				return null;

			type = probe_type;
			eclass = ExprClass.Value;
			Type etype = expr.Type;

			if (TypeManager.IsValueType (probe_type)){
				Report.Error (77, loc, "The as operator should be used with a reference type only (" +
					      TypeManager.CSharpName (probe_type) + " is a value type)");
				return null;
			
			}
			
			e = Convert.ImplicitConversion (ec, expr, probe_type, loc);
			if (e != null){
				expr = e;
				do_isinst = false;
				return this;
			}

			if (Convert.ExplicitReferenceConversionExists (etype, probe_type)){
				do_isinst = true;
				return this;
			}

			Error_CannotConvertType (etype, probe_type, loc);
			return null;
		}				
	}
	
	/// <summary>
	///   This represents a typecast in the source language.
	///
	///   FIXME: Cast expressions have an unusual set of parsing
	///   rules, we need to figure those out.
	/// </summary>
	public class Cast : Expression {
		Expression target_type;
		Expression expr;
			
		public Cast (Expression cast_type, Expression expr, Location loc)
		{
			this.target_type = cast_type;
			this.expr = expr;
			this.loc = loc;
		}

		public Expression TargetType {
			get {
				return target_type;
			}
		}

		public Expression Expr {
			get {
				return expr;
			}
			set {
				expr = value;
			}
		}

		bool CheckRange (EmitContext ec, long value, Type type, long min, long max)
		{
			if (!ec.ConstantCheckState)
				return true;

			if ((value < min) || (value > max)) {
				Error (221, "Constant value `" + value + "' cannot be converted " +
				       "to a `" + TypeManager.CSharpName (type) + "' (use `unchecked' " +
				       "syntax to override)");
				return false;
			}

			return true;
		}

		bool CheckRange (EmitContext ec, ulong value, Type type, ulong max)
		{
			if (!ec.ConstantCheckState)
				return true;

			if (value > max) {
				Error (221, "Constant value `" + value + "' cannot be converted " +
				       "to a `" + TypeManager.CSharpName (type) + "' (use `unchecked' " +
				       "syntax to override)");
				return false;
			}

			return true;
		}

		bool CheckUnsigned (EmitContext ec, long value, Type type)
		{
			if (!ec.ConstantCheckState)
				return true;

			if (value < 0) {
				Error (221, "Constant value `" + value + "' cannot be converted " +
				       "to a `" + TypeManager.CSharpName (type) + "' (use `unchecked' " +
				       "syntax to override)");
				return false;
			}

			return true;
		}

		/// <summary>
		///   Attempts to do a compile-time folding of a constant cast.
		/// </summary>
		Expression TryReduce (EmitContext ec, Type target_type)
		{
			Expression real_expr = expr;
			if (real_expr is EnumConstant)
				real_expr = ((EnumConstant) real_expr).Child;

			if (real_expr is ByteConstant){
				byte v = ((ByteConstant) real_expr).Value;
	
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, SByte.MinValue, SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.short_type)
					return new ShortConstant ((short) v);
				if (target_type == TypeManager.ushort_type)
					return new UShortConstant ((ushort) v);
				if (target_type == TypeManager.int32_type)
					return new IntConstant ((int) v);
				if (target_type == TypeManager.uint32_type)
					return new UIntConstant ((uint) v);
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type)
					return new ULongConstant ((ulong) v);
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type)
					return new CharConstant ((char) v);
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is SByteConstant){
				sbyte v = ((SByteConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.short_type)
					return new ShortConstant ((short) v);
				if (target_type == TypeManager.ushort_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new UShortConstant ((ushort) v);
				} if (target_type == TypeManager.int32_type)
					return new IntConstant ((int) v);
				if (target_type == TypeManager.uint32_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new UIntConstant ((uint) v);
				} if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new ULongConstant ((ulong) v);
				}
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is ShortConstant){
				short v = ((ShortConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type) {
					if (!CheckRange (ec, v, target_type, Byte.MinValue, Byte.MaxValue))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, SByte.MinValue, SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.ushort_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new UShortConstant ((ushort) v);
				}
				if (target_type == TypeManager.int32_type)
					return new IntConstant ((int) v);
				if (target_type == TypeManager.uint32_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new UIntConstant ((uint) v);
				}
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new ULongConstant ((ulong) v);
				}
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckRange (ec, v, target_type, Char.MinValue, Char.MaxValue))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is UShortConstant){
				ushort v = ((UShortConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type) {
					if (!CheckRange (ec, v, target_type, Byte.MinValue, Byte.MaxValue))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, SByte.MinValue, SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.short_type) {
					if (!CheckRange (ec, v, target_type, Int16.MinValue, Int16.MaxValue))
						return null;
					return new ShortConstant ((short) v);
				}
				if (target_type == TypeManager.int32_type)
					return new IntConstant ((int) v);
				if (target_type == TypeManager.uint32_type)
					return new UIntConstant ((uint) v);
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type)
					return new ULongConstant ((ulong) v);
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckRange (ec, v, target_type, Char.MinValue, Char.MaxValue))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is IntConstant){
				int v = ((IntConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type) {
					if (!CheckRange (ec, v, target_type, Byte.MinValue, Byte.MaxValue))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, SByte.MinValue, SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.short_type) {
					if (!CheckRange (ec, v, target_type, Int16.MinValue, Int16.MaxValue))
						return null;
					return new ShortConstant ((short) v);
				}
				if (target_type == TypeManager.ushort_type) {
					if (!CheckRange (ec, v, target_type, UInt16.MinValue, UInt16.MaxValue))
						return null;
					return new UShortConstant ((ushort) v);
				}
				if (target_type == TypeManager.uint32_type) {
					if (!CheckRange (ec, v, target_type, Int32.MinValue, Int32.MaxValue))
						return null;
					return new UIntConstant ((uint) v);
				}
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new ULongConstant ((ulong) v);
				}
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckRange (ec, v, target_type, Char.MinValue, Char.MaxValue))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is UIntConstant){
				uint v = ((UIntConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type) {
					if (!CheckRange (ec, v, target_type, Char.MinValue, Char.MaxValue))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, SByte.MinValue, SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.short_type) {
					if (!CheckRange (ec, v, target_type, Int16.MinValue, Int16.MaxValue))
						return null;
					return new ShortConstant ((short) v);
				}
				if (target_type == TypeManager.ushort_type) {
					if (!CheckRange (ec, v, target_type, UInt16.MinValue, UInt16.MaxValue))
						return null;
					return new UShortConstant ((ushort) v);
				}
				if (target_type == TypeManager.int32_type) {
					if (!CheckRange (ec, v, target_type, Int32.MinValue, Int32.MaxValue))
						return null;
					return new IntConstant ((int) v);
				}
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type)
					return new ULongConstant ((ulong) v);
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckRange (ec, v, target_type, Char.MinValue, Char.MaxValue))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is LongConstant){
				long v = ((LongConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type) {
					if (!CheckRange (ec, v, target_type, Byte.MinValue, Byte.MaxValue))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, SByte.MinValue, SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.short_type) {
					if (!CheckRange (ec, v, target_type, Int16.MinValue, Int16.MaxValue))
						return null;
					return new ShortConstant ((short) v);
				}
				if (target_type == TypeManager.ushort_type) {
					if (!CheckRange (ec, v, target_type, UInt16.MinValue, UInt16.MaxValue))
						return null;
					return new UShortConstant ((ushort) v);
				}
				if (target_type == TypeManager.int32_type) {
					if (!CheckRange (ec, v, target_type, Int32.MinValue, Int32.MaxValue))
						return null;
					return new IntConstant ((int) v);
				}
				if (target_type == TypeManager.uint32_type) {
					if (!CheckRange (ec, v, target_type, UInt32.MinValue, UInt32.MaxValue))
						return null;
					return new UIntConstant ((uint) v);
				}
				if (target_type == TypeManager.uint64_type) {
					if (!CheckUnsigned (ec, v, target_type))
						return null;
					return new ULongConstant ((ulong) v);
				}
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckRange (ec, v, target_type, Char.MinValue, Char.MaxValue))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is ULongConstant){
				ulong v = ((ULongConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type) {
					if (!CheckRange (ec, v, target_type, Byte.MaxValue))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, (ulong) SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.short_type) {
					if (!CheckRange (ec, v, target_type, (ulong) Int16.MaxValue))
						return null;
					return new ShortConstant ((short) v);
				}
				if (target_type == TypeManager.ushort_type) {
					if (!CheckRange (ec, v, target_type, UInt16.MaxValue))
						return null;
					return new UShortConstant ((ushort) v);
				}
				if (target_type == TypeManager.int32_type) {
					if (!CheckRange (ec, v, target_type, Int32.MaxValue))
						return null;
					return new IntConstant ((int) v);
				}
				if (target_type == TypeManager.uint32_type) {
					if (!CheckRange (ec, v, target_type, UInt32.MaxValue))
						return null;
					return new UIntConstant ((uint) v);
				}
				if (target_type == TypeManager.int64_type) {
					if (!CheckRange (ec, v, target_type, (ulong) Int64.MaxValue))
						return null;
					return new LongConstant ((long) v);
				}
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckRange (ec, v, target_type, Char.MaxValue))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is FloatConstant){
				float v = ((FloatConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type)
					return new ByteConstant ((byte) v);
				if (target_type == TypeManager.sbyte_type)
					return new SByteConstant ((sbyte) v);
				if (target_type == TypeManager.short_type)
					return new ShortConstant ((short) v);
				if (target_type == TypeManager.ushort_type)
					return new UShortConstant ((ushort) v);
				if (target_type == TypeManager.int32_type)
					return new IntConstant ((int) v);
				if (target_type == TypeManager.uint32_type)
					return new UIntConstant ((uint) v);
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type)
					return new ULongConstant ((ulong) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type)
					return new CharConstant ((char) v);
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}
			if (real_expr is DoubleConstant){
				double v = ((DoubleConstant) real_expr).Value;
	
				if (target_type == TypeManager.byte_type){
					return new ByteConstant ((byte) v);
				} if (target_type == TypeManager.sbyte_type)
					return new SByteConstant ((sbyte) v);
				if (target_type == TypeManager.short_type)
					return new ShortConstant ((short) v);
				if (target_type == TypeManager.ushort_type)
					return new UShortConstant ((ushort) v);
				if (target_type == TypeManager.int32_type)
					return new IntConstant ((int) v);
				if (target_type == TypeManager.uint32_type)
					return new UIntConstant ((uint) v);
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type)
					return new ULongConstant ((ulong) v);
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.char_type)
					return new CharConstant ((char) v);
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}

			if (real_expr is CharConstant){
				char v = ((CharConstant) real_expr).Value;
				
				if (target_type == TypeManager.byte_type) {
					if (!CheckRange (ec, v, target_type, Byte.MinValue, Byte.MaxValue))
						return null;
					return new ByteConstant ((byte) v);
				}
				if (target_type == TypeManager.sbyte_type) {
					if (!CheckRange (ec, v, target_type, SByte.MinValue, SByte.MaxValue))
						return null;
					return new SByteConstant ((sbyte) v);
				}
				if (target_type == TypeManager.short_type) {
					if (!CheckRange (ec, v, target_type, Int16.MinValue, Int16.MaxValue))
						return null;
					return new ShortConstant ((short) v);
				}
				if (target_type == TypeManager.int32_type)
					return new IntConstant ((int) v);
				if (target_type == TypeManager.uint32_type)
					return new UIntConstant ((uint) v);
				if (target_type == TypeManager.int64_type)
					return new LongConstant ((long) v);
				if (target_type == TypeManager.uint64_type)
					return new ULongConstant ((ulong) v);
				if (target_type == TypeManager.float_type)
					return new FloatConstant ((float) v);
				if (target_type == TypeManager.double_type)
					return new DoubleConstant ((double) v);
				if (target_type == TypeManager.char_type) {
					if (!CheckRange (ec, v, target_type, Char.MinValue, Char.MaxValue))
						return null;
					return new CharConstant ((char) v);
				}
				if (target_type == TypeManager.decimal_type)
					return new DecimalConstant ((decimal) v);
			}

			return null;
		}
		
		public override Expression DoResolve (EmitContext ec)
		{
			expr = expr.Resolve (ec);
			if (expr == null)
				return null;

			type = ec.DeclSpace.ResolveType (target_type, false, Location);
			
			if (type == null)
				return null;

			CheckObsoleteAttribute (type);

			eclass = ExprClass.Value;

			if (expr is Constant){
				Expression e = TryReduce (ec, type);

				if (e != null)
					return e;
			}

			if (type.IsPointer && !ec.InUnsafe) {
				UnsafeError (loc);
				return null;
			}
			expr = Convert.ExplicitConversion (ec, expr, type, loc);
			return expr;
		}

		public override void Emit (EmitContext ec)
		{
			//
			// This one will never happen
			//
			throw new Exception ("Should not happen");
		}
	}

	/// <summary>
	///   Binary operators
	/// </summary>
	public class Binary : Expression {
		public enum Operator : byte {
			Multiply, Division, Modulus,
			Addition, Subtraction,
			LeftShift, RightShift,
			LessThan, GreaterThan, LessThanOrEqual, GreaterThanOrEqual, 
			Equality, Inequality,
			BitwiseAnd,
			ExclusiveOr,
			BitwiseOr,
			LogicalAnd,
			LogicalOr,
			TOP
		}

		Operator oper;
		Expression left, right;

		// This must be kept in sync with Operator!!!
		public static readonly string [] oper_names;
		
		static Binary ()
		{
			oper_names = new string [(int) Operator.TOP];

			oper_names [(int) Operator.Multiply] = "op_Multiply";
			oper_names [(int) Operator.Division] = "op_Division";
			oper_names [(int) Operator.Modulus] = "op_Modulus";
			oper_names [(int) Operator.Addition] = "op_Addition";
			oper_names [(int) Operator.Subtraction] = "op_Subtraction";
			oper_names [(int) Operator.LeftShift] = "op_LeftShift";
			oper_names [(int) Operator.RightShift] = "op_RightShift";
			oper_names [(int) Operator.LessThan] = "op_LessThan";
			oper_names [(int) Operator.GreaterThan] = "op_GreaterThan";
			oper_names [(int) Operator.LessThanOrEqual] = "op_LessThanOrEqual";
			oper_names [(int) Operator.GreaterThanOrEqual] = "op_GreaterThanOrEqual";
			oper_names [(int) Operator.Equality] = "op_Equality";
			oper_names [(int) Operator.Inequality] = "op_Inequality";
			oper_names [(int) Operator.BitwiseAnd] = "op_BitwiseAnd";
			oper_names [(int) Operator.BitwiseOr] = "op_BitwiseOr";
			oper_names [(int) Operator.ExclusiveOr] = "op_ExclusiveOr";
			oper_names [(int) Operator.LogicalOr] = "op_LogicalOr";
			oper_names [(int) Operator.LogicalAnd] = "op_LogicalAnd";
		}

		public Binary (Operator oper, Expression left, Expression right, Location loc)
		{
			this.oper = oper;
			this.left = left;
			this.right = right;
			this.loc = loc;
		}

		public Operator Oper {
			get {
				return oper;
			}
			set {
				oper = value;
			}
		}
		
		public Expression Left {
			get {
				return left;
			}
			set {
				left = value;
			}
		}

		public Expression Right {
			get {
				return right;
			}
			set {
				right = value;
			}
		}


		/// <summary>
		///   Returns a stringified representation of the Operator
		/// </summary>
		static string OperName (Operator oper)
		{
			switch (oper){
			case Operator.Multiply:
				return "*";
			case Operator.Division:
				return "/";
			case Operator.Modulus:
				return "%";
			case Operator.Addition:
				return "+";
			case Operator.Subtraction:
				return "-";
			case Operator.LeftShift:
				return "<<";
			case Operator.RightShift:
				return ">>";
			case Operator.LessThan:
				return "<";
			case Operator.GreaterThan:
				return ">";
			case Operator.LessThanOrEqual:
				return "<=";
			case Operator.GreaterThanOrEqual:
				return ">=";
			case Operator.Equality:
				return "==";
			case Operator.Inequality:
				return "!=";
			case Operator.BitwiseAnd:
				return "&";
			case Operator.BitwiseOr:
				return "|";
			case Operator.ExclusiveOr:
				return "^";
			case Operator.LogicalOr:
				return "||";
			case Operator.LogicalAnd:
				return "&&";
			}

			return oper.ToString ();
		}

		public override string ToString ()
		{
			return "operator " + OperName (oper) + "(" + left.ToString () + ", " +
				right.ToString () + ")";
		}
		
		Expression ForceConversion (EmitContext ec, Expression expr, Type target_type)
		{
			if (expr.Type == target_type)
				return expr;

			return Convert.ImplicitConversion (ec, expr, target_type, loc);
		}

		public static void Error_OperatorAmbiguous (Location loc, Operator oper, Type l, Type r)
		{
			Report.Error (
				34, loc, "Operator `" + OperName (oper) 
				+ "' is ambiguous on operands of type `"
				+ TypeManager.CSharpName (l) + "' "
				+ "and `" + TypeManager.CSharpName (r)
				+ "'");
		}

		bool IsOfType (EmitContext ec, Type l, Type r, Type t, bool check_user_conversions)
		{
			if ((l == t) || (r == t))
				return true;

			if (!check_user_conversions)
				return false;

			if (Convert.ImplicitUserConversionExists (ec, l, t))
				return true;
			else if (Convert.ImplicitUserConversionExists (ec, r, t))
				return true;
			else
				return false;
		}

		//
		// Note that handling the case l == Decimal || r == Decimal
		// is taken care of by the Step 1 Operator Overload resolution.
		//
		// If `check_user_conv' is true, we also check whether a user-defined conversion
		// exists.  Note that we only need to do this if both arguments are of a user-defined
		// type, otherwise ConvertImplict() already finds the user-defined conversion for us,
		// so we don't explicitly check for performance reasons.
 		//
		bool DoNumericPromotions (EmitContext ec, Type l, Type r, bool check_user_conv)
		{
			if (IsOfType (ec, l, r, TypeManager.double_type, check_user_conv)){
				//
				// If either operand is of type double, the other operand is
				// conveted to type double.
				//
				if (r != TypeManager.double_type)
					right = Convert.ImplicitConversion (ec, right, TypeManager.double_type, loc);
				if (l != TypeManager.double_type)
					left = Convert.ImplicitConversion (ec, left, TypeManager.double_type, loc);
				
				type = TypeManager.double_type;
			} else if (IsOfType (ec, l, r, TypeManager.float_type, check_user_conv)){
				//
				// if either operand is of type float, the other operand is
				// converted to type float.
				//
				if (r != TypeManager.double_type)
					right = Convert.ImplicitConversion (ec, right, TypeManager.float_type, loc);
				if (l != TypeManager.double_type)
					left = Convert.ImplicitConversion (ec, left, TypeManager.float_type, loc);
				type = TypeManager.float_type;
			} else if (IsOfType (ec, l, r, TypeManager.uint64_type, check_user_conv)){
				Expression e;
				Type other;
				//
				// If either operand is of type ulong, the other operand is
				// converted to type ulong.  or an error ocurrs if the other
				// operand is of type sbyte, short, int or long
				//
				if (l == TypeManager.uint64_type){
					if (r != TypeManager.uint64_type){
						if (right is IntConstant){
							IntConstant ic = (IntConstant) right;
							
							e = Convert.TryImplicitIntConversion (l, ic);
							if (e != null)
								right = e;
						} else if (right is LongConstant){
							long ll = ((LongConstant) right).Value;

							if (ll >= 0)
								right = new ULongConstant ((ulong) ll);
						} else {
							e = Convert.ImplicitNumericConversion (ec, right, l, loc);
							if (e != null)
								right = e;
						}
					}
					other = right.Type;
				} else {
					if (left is IntConstant){
						e = Convert.TryImplicitIntConversion (r, (IntConstant) left);
						if (e != null)
							left = e;
					} else if (left is LongConstant){
						long ll = ((LongConstant) left).Value;
						
						if (ll > 0)
							left = new ULongConstant ((ulong) ll);
					} else {
						e = Convert.ImplicitNumericConversion (ec, left, r, loc);
						if (e != null)
							left = e;
					}
					other = left.Type;
				}

				if ((other == TypeManager.sbyte_type) ||
				    (other == TypeManager.short_type) ||
				    (other == TypeManager.int32_type) ||
				    (other == TypeManager.int64_type))
					Error_OperatorAmbiguous (loc, oper, l, r);
				else {
					left = ForceConversion (ec, left, TypeManager.uint64_type);
					right = ForceConversion (ec, right, TypeManager.uint64_type);
				}
				type = TypeManager.uint64_type;
			} else if (IsOfType (ec, l, r, TypeManager.int64_type, check_user_conv)){
				//
				// If either operand is of type long, the other operand is converted
				// to type long.
				//
				if (l != TypeManager.int64_type)
					left = Convert.ImplicitConversion (ec, left, TypeManager.int64_type, loc);
				if (r != TypeManager.int64_type)
					right = Convert.ImplicitConversion (ec, right, TypeManager.int64_type, loc);
				
				type = TypeManager.int64_type;
			} else if (IsOfType (ec, l, r, TypeManager.uint32_type, check_user_conv)){
				//
				// If either operand is of type uint, and the other
				// operand is of type sbyte, short or int, othe operands are
				// converted to type long (unless we have an int constant).
				//
				Type other = null;
				
				if (l == TypeManager.uint32_type){
					if (right is IntConstant){
						IntConstant ic = (IntConstant) right;
						int val = ic.Value;
						
						if (val >= 0){
							right = new UIntConstant ((uint) val);
							type = l;
							
							return true;
						}
					}
					other = r;
				} else if (r == TypeManager.uint32_type){
					if (left is IntConstant){
						IntConstant ic = (IntConstant) left;
						int val = ic.Value;
						
						if (val >= 0){
							left = new UIntConstant ((uint) val);
							type = r;
							return true;
						}
					}
					
					other = l;
				}

				if ((other == TypeManager.sbyte_type) ||
				    (other == TypeManager.short_type) ||
				    (other == TypeManager.int32_type)){
					left = ForceConversion (ec, left, TypeManager.int64_type);
					right = ForceConversion (ec, right, TypeManager.int64_type);
					type = TypeManager.int64_type;
				} else {
					//
					// if either operand is of type uint, the other
					// operand is converd to type uint
					//
					left = ForceConversion (ec, left, TypeManager.uint32_type);
					right = ForceConversion (ec, right, TypeManager.uint32_type);
					type = TypeManager.uint32_type;
				} 
			} else if (l == TypeManager.decimal_type || r == TypeManager.decimal_type){
				if (l != TypeManager.decimal_type)
					left = Convert.ImplicitConversion (ec, left, TypeManager.decimal_type, loc);

				if (r != TypeManager.decimal_type)
					right = Convert.ImplicitConversion (ec, right, TypeManager.decimal_type, loc);
				type = TypeManager.decimal_type;
			} else {
				left = ForceConversion (ec, left, TypeManager.int32_type);
				right = ForceConversion (ec, right, TypeManager.int32_type);

				type = TypeManager.int32_type;
			}

			return (left != null) && (right != null);
		}

		static public void Error_OperatorCannotBeApplied (Location loc, string name, Type l, Type r)
		{
			Report.Error (19, loc,
			       "Operator " + name + " cannot be applied to operands of type `" +
			       TypeManager.CSharpName (l) + "' and `" +
			       TypeManager.CSharpName (r) + "'");
		}
		
		void Error_OperatorCannotBeApplied ()
		{
			Error_OperatorCannotBeApplied (loc, OperName (oper), left.Type, right.Type);
		}

		static bool is_unsigned (Type t)
		{
			return (t == TypeManager.uint32_type || t == TypeManager.uint64_type ||
				t == TypeManager.short_type || t == TypeManager.byte_type);
		}

		static bool is_user_defined (Type t)
		{
			if (t.IsSubclassOf (TypeManager.value_type) &&
			    (!TypeManager.IsBuiltinType (t) || t == TypeManager.decimal_type))
				return true;
			else
				return false;
		}

		Expression Make32or64 (EmitContext ec, Expression e)
		{
			Type t= e.Type;
			
			if (t == TypeManager.int32_type || t == TypeManager.uint32_type ||
			    t == TypeManager.int64_type || t == TypeManager.uint64_type)
				return e;
			Expression ee = Convert.ImplicitConversion (ec, e, TypeManager.int32_type, loc);
			if (ee != null)
				return ee;
			ee = Convert.ImplicitConversion (ec, e, TypeManager.uint32_type, loc);
			if (ee != null)
				return ee;
			ee = Convert.ImplicitConversion (ec, e, TypeManager.int64_type, loc);
			if (ee != null)
				return ee;
			ee = Convert.ImplicitConversion (ec, e, TypeManager.uint64_type, loc);
			if (ee != null)
				return ee;
			return null;
		}
					
		Expression CheckShiftArguments (EmitContext ec)
		{
			Expression e;

			e = ForceConversion (ec, right, TypeManager.int32_type);
			if (e == null){
				Error_OperatorCannotBeApplied ();
				return null;
			}
			right = e;

			if (((e = Convert.ImplicitConversion (ec, left, TypeManager.int32_type, loc)) != null) ||
			    ((e = Convert.ImplicitConversion (ec, left, TypeManager.uint32_type, loc)) != null) ||
			    ((e = Convert.ImplicitConversion (ec, left, TypeManager.int64_type, loc)) != null) ||
			    ((e = Convert.ImplicitConversion (ec, left, TypeManager.uint64_type, loc)) != null)){
				left = e;
				type = e.Type;

				if (type == TypeManager.int32_type || type == TypeManager.uint32_type){
					right = new Binary (Binary.Operator.BitwiseAnd, right, new IntLiteral (31), loc);
					right = right.DoResolve (ec);
				} else {
					right = new Binary (Binary.Operator.BitwiseAnd, right, new IntLiteral (63), loc);
					right = right.DoResolve (ec);
				}

				return this;
			}
			Error_OperatorCannotBeApplied ();
			return null;
		}

		Expression ResolveOperator (EmitContext ec)
		{
			Type l = left.Type;
			Type r = right.Type;

			//
			// Special cases: string comapred to null
			//
			if (oper == Operator.Equality || oper == Operator.Inequality){
				if ((l == TypeManager.string_type && (right is NullLiteral)) ||
				    (r == TypeManager.string_type && (left is NullLiteral))){
					Type = TypeManager.bool_type;
					
					return this;
				}
				
				// IntPtr equality
				if (l == TypeManager.intptr_type && r == TypeManager.intptr_type) {
					Type = TypeManager.bool_type;
					
					return this;
				}
			}

			//
			// Do not perform operator overload resolution when both sides are
			// built-in types
			//
			if (!(TypeManager.IsCLRType (l) && TypeManager.IsCLRType (r))){
				//
				// Step 1: Perform Operator Overload location
				//
				Expression left_expr, right_expr;
				
				string op = oper_names [(int) oper];
				
				MethodGroupExpr union;
				left_expr = MemberLookup (ec, l, op, MemberTypes.Method, AllBindingFlags, loc);
				if (r != l){
					right_expr = MemberLookup (
						ec, r, op, MemberTypes.Method, AllBindingFlags, loc);
					union = Invocation.MakeUnionSet (left_expr, right_expr, loc);
				} else
					union = (MethodGroupExpr) left_expr;
				
				if (union != null) {
					ArrayList args = new ArrayList (2);
					args.Add (new Argument (left, Argument.AType.Expression));
					args.Add (new Argument (right, Argument.AType.Expression));
					
					MethodBase method = Invocation.OverloadResolve (ec, union, args, Location.Null);
					if (method != null) {
						MethodInfo mi = (MethodInfo) method;
						
						return new BinaryMethod (mi.ReturnType, method, args);
					}
				}
			}
			
			//
			// Step 0: String concatenation (because overloading will get this wrong)
			//
			if (oper == Operator.Addition){
				//
				// If any of the arguments is a string, cast to string
				//
				
				// Simple constant folding
				if (left is StringConstant && right is StringConstant)
					return new StringConstant (((StringConstant) left).Value + ((StringConstant) right).Value);
				
				if (l == TypeManager.string_type || r == TypeManager.string_type) {

					if (r == TypeManager.void_type || l == TypeManager.void_type) {
						Error_OperatorCannotBeApplied ();
						return null;
					}
					
					// try to fold it in on the left
					if (left is StringConcat) {

						//
						// We have to test here for not-null, since we can be doubly-resolved
						// take care of not appending twice
						//
						if (type == null){
							type = TypeManager.string_type;
							((StringConcat) left).Append (ec, right);
							return left.Resolve (ec);
						} else {
							return left;
						}
					}
					
					// Otherwise, start a new concat expression
					return new StringConcat (ec, loc, left, right).Resolve (ec);
				}

				//
				// Transform a + ( - b) into a - b
				//
				if (right is Unary){
					Unary right_unary = (Unary) right;

					if (right_unary.Oper == Unary.Operator.UnaryNegation){
						oper = Operator.Subtraction;
						right = right_unary.Expr;
						r = right.Type;
					}
				}
			}

			if (oper == Operator.Equality || oper == Operator.Inequality){
				if (l == TypeManager.bool_type || r == TypeManager.bool_type){
					if (r != TypeManager.bool_type || l != TypeManager.bool_type){
						Error_OperatorCannotBeApplied ();
						return null;
					}
					
					type = TypeManager.bool_type;
					return this;
				}

				//
				// operator != (object a, object b)
				// operator == (object a, object b)
				//
				// For this to be used, both arguments have to be reference-types.
				// Read the rationale on the spec (14.9.6)
				//
				// Also, if at compile time we know that the classes do not inherit
				// one from the other, then we catch the error there.
				//
				if (!(l.IsValueType || r.IsValueType)){
					type = TypeManager.bool_type;

					if (l == r)
						return this;
					
					if (l.IsSubclassOf (r) || r.IsSubclassOf (l))
						return this;

					//
					// Also, a standard conversion must exist from either one
					//
					if (!(Convert.ImplicitStandardConversionExists (left, r) ||
					      Convert.ImplicitStandardConversionExists (right, l))){
						Error_OperatorCannotBeApplied ();
						return null;
					}
					//
					// We are going to have to convert to an object to compare
					//
					if (l != TypeManager.object_type)
						left = new EmptyCast (left, TypeManager.object_type);
					if (r != TypeManager.object_type)
						right = new EmptyCast (right, TypeManager.object_type);

					//
					// FIXME: CSC here catches errors cs254 and cs252
					//
					return this;
				}

				//
				// One of them is a valuetype, but the other one is not.
				//
				if (!l.IsValueType || !r.IsValueType) {
					Error_OperatorCannotBeApplied ();
					return null;
				}
			}

			// Only perform numeric promotions on:
			// +, -, *, /, %, &, |, ^, ==, !=, <, >, <=, >=
			//
			if (oper == Operator.Addition || oper == Operator.Subtraction) {
				if (l.IsSubclassOf (TypeManager.delegate_type)){
					if (right.eclass == ExprClass.MethodGroup && RootContext.V2){
						Expression tmp = Convert.ImplicitConversionRequired (ec, right, l, loc);
						if (tmp == null)
							return null;
						right = tmp;
						r = right.Type;
					}
				
					if (r.IsSubclassOf (TypeManager.delegate_type)){
						MethodInfo method;
						ArrayList args = new ArrayList (2);
						
						args = new ArrayList (2);
						args.Add (new Argument (left, Argument.AType.Expression));
						args.Add (new Argument (right, Argument.AType.Expression));
						
						if (oper == Operator.Addition)
							method = TypeManager.delegate_combine_delegate_delegate;
						else
							method = TypeManager.delegate_remove_delegate_delegate;
						
						if (l != r) {
							Error_OperatorCannotBeApplied ();
							return null;
						}
						
						return new BinaryDelegate (l, method, args);
					}
				}
				
				//
				// Pointer arithmetic:
				//
				// T* operator + (T* x, int y);
				// T* operator + (T* x, uint y);
				// T* operator + (T* x, long y);
				// T* operator + (T* x, ulong y);
				//
				// T* operator + (int y,   T* x);
				// T* operator + (uint y,  T *x);
				// T* operator + (long y,  T *x);
				// T* operator + (ulong y, T *x);
				//
				// T* operator - (T* x, int y);
				// T* operator - (T* x, uint y);
				// T* operator - (T* x, long y);
				// T* operator - (T* x, ulong y);
				//
				// long operator - (T* x, T *y)
				//
				if (l.IsPointer){
					if (r.IsPointer && oper == Operator.Subtraction){
						if (r == l)
							return new PointerArithmetic (
								false, left, right, TypeManager.int64_type,
								loc);
					} else {
						Expression t = Make32or64 (ec, right);
						if (t != null)
							return new PointerArithmetic (oper == Operator.Addition, left, t, l, loc);
					}
				} else if (r.IsPointer && oper == Operator.Addition){
					Expression t = Make32or64 (ec, left);
					if (t != null)
						return new PointerArithmetic (true, right, t, r, loc);
				}
			}
			
			//
			// Enumeration operators
			//
			bool lie = TypeManager.IsEnumType (l);
			bool rie = TypeManager.IsEnumType (r);
			if (lie || rie){
				Expression temp;

				// U operator - (E e, E f)
				if (lie && rie){
					if (oper == Operator.Subtraction){
						if (l == r){
							type = TypeManager.EnumToUnderlying (l);
							return this;
						}
						Error_OperatorCannotBeApplied ();
						return null;
					}
				}
					
				//
				// operator + (E e, U x)
				// operator - (E e, U x)
				//
				if (oper == Operator.Addition || oper == Operator.Subtraction){
					Type enum_type = lie ? l : r;
					Type other_type = lie ? r : l;
					Type underlying_type = TypeManager.EnumToUnderlying (enum_type);
					
					if (underlying_type != other_type){
						temp = Convert.ImplicitConversion (ec, lie ? right : left, underlying_type, loc);
						if (temp != null){
							if (lie)
								right = temp;
							else
								left = temp;
							type = enum_type;
							return this;
						}
							
						Error_OperatorCannotBeApplied ();
						return null;
					}

					type = enum_type;
					return this;
				}
				
				if (!rie){
					temp = Convert.ImplicitConversion (ec, right, l, loc);
					if (temp != null)
						right = temp;
					else {
						Error_OperatorCannotBeApplied ();
						return null;
					}
				} if (!lie){
					temp = Convert.ImplicitConversion (ec, left, r, loc);
					if (temp != null){
						left = temp;
						l = r;
					} else {
						Error_OperatorCannotBeApplied ();
						return null;
					}
				}

				if (oper == Operator.Equality || oper == Operator.Inequality ||
				    oper == Operator.LessThanOrEqual || oper == Operator.LessThan ||
				    oper == Operator.GreaterThanOrEqual || oper == Operator.GreaterThan){
					if (left.Type != right.Type){
						Error_OperatorCannotBeApplied ();
						return null;
					}
					type = TypeManager.bool_type;
					return this;
				}

				if (oper == Operator.BitwiseAnd ||
				    oper == Operator.BitwiseOr ||
				    oper == Operator.ExclusiveOr){
					type = l;
					return this;
				}
				Error_OperatorCannotBeApplied ();
				return null;
			}
			
			if (oper == Operator.LeftShift || oper == Operator.RightShift)
				return CheckShiftArguments (ec);

			if (oper == Operator.LogicalOr || oper == Operator.LogicalAnd){
				if (l == TypeManager.bool_type && r == TypeManager.bool_type) {
					type = TypeManager.bool_type;
					return this;
				}

				if (l != r) {
					Error_OperatorCannotBeApplied ();
					return null;
				}

				Expression e = new ConditionalLogicalOperator (
					oper == Operator.LogicalAnd, left, right, l, loc);
				return e.Resolve (ec);
			} 

			//
			// operator & (bool x, bool y)
			// operator | (bool x, bool y)
			// operator ^ (bool x, bool y)
			//
			if (l == TypeManager.bool_type && r == TypeManager.bool_type){
				if (oper == Operator.BitwiseAnd ||
				    oper == Operator.BitwiseOr ||
				    oper == Operator.ExclusiveOr){
					type = l;
					return this;
				}
			}
			
			//
			// Pointer comparison
			//
			if (l.IsPointer && r.IsPointer){
				if (oper == Operator.Equality || oper == Operator.Inequality ||
				    oper == Operator.LessThan || oper == Operator.LessThanOrEqual ||
				    oper == Operator.GreaterThan || oper == Operator.GreaterThanOrEqual){
					type = TypeManager.bool_type;
					return this;
				}
			}
			
			//
			// This will leave left or right set to null if there is an error
			//
			bool check_user_conv = is_user_defined (l) && is_user_defined (r);
			DoNumericPromotions (ec, l, r, check_user_conv);
			if (left == null || right == null){
				Error_OperatorCannotBeApplied (loc, OperName (oper), l, r);
				return null;
			}

			//
			// reload our cached types if required
			//
			l = left.Type;
			r = right.Type;
			
			if (oper == Operator.BitwiseAnd ||
			    oper == Operator.BitwiseOr ||
			    oper == Operator.ExclusiveOr){
				if (l == r){
					if (((l == TypeManager.int32_type) ||
					     (l == TypeManager.uint32_type) ||
					     (l == TypeManager.short_type) ||
					     (l == TypeManager.ushort_type) ||
					     (l == TypeManager.int64_type) ||
					     (l == TypeManager.uint64_type))){
						type = l;
					} else {
						Error_OperatorCannotBeApplied ();
						return null;
					}
				} else {
					Error_OperatorCannotBeApplied ();
					return null;
				}
			}

			if (oper == Operator.Equality ||
			    oper == Operator.Inequality ||
			    oper == Operator.LessThanOrEqual ||
			    oper == Operator.LessThan ||
			    oper == Operator.GreaterThanOrEqual ||
			    oper == Operator.GreaterThan){
				type = TypeManager.bool_type;
			}

			return this;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			if ((oper == Operator.Subtraction) && (left is ParenthesizedExpression)) {
				left = ((ParenthesizedExpression) left).Expr;
				left = left.Resolve (ec, ResolveFlags.VariableOrValue | ResolveFlags.Type);
				if (left == null)
					return null;

				if (left.eclass == ExprClass.Type) {
					Error (75, "Casting a negative value needs to have the value in parentheses.");
					return null;
				}
			} else
				left = left.Resolve (ec);
			right = right.Resolve (ec);

			if (left == null || right == null)
				return null;

			eclass = ExprClass.Value;

			Constant rc = right as Constant;
			Constant lc = left as Constant;

			if (rc != null & lc != null){
				Expression e = ConstantFold.BinaryFold (
					ec, oper, lc, rc, loc);
					if (e != null)
						return e;
			}

			return ResolveOperator (ec);
		}

		/// <remarks>
		///   EmitBranchable is called from Statement.EmitBoolExpression in the
		///   context of a conditional bool expression.  This function will return
		///   false if it is was possible to use EmitBranchable, or true if it was.
		///
		///   The expression's code is generated, and we will generate a branch to `target'
		///   if the resulting expression value is equal to isTrue
		/// </remarks>
		public override void EmitBranchable (EmitContext ec, Label target, bool onTrue)
		{
			ILGenerator ig = ec.ig;

			//
			// This is more complicated than it looks, but its just to avoid
			// duplicated tests: basically, we allow ==, !=, >, <, >= and <=
			// but on top of that we want for == and != to use a special path
			// if we are comparing against null
			//
			if ((oper == Operator.Equality || oper == Operator.Inequality) && (left is Constant || right is Constant)) {
				bool my_on_true = oper == Operator.Inequality ? onTrue : !onTrue;
				
				//
				// put the constant on the rhs, for simplicity
				//
				if (left is Constant) {
					Expression swap = right;
					right = left;
					left = swap;
				}
				
				if (((Constant) right).IsZeroInteger) {
					left.Emit (ec);
					if (my_on_true)
						ig.Emit (OpCodes.Brtrue, target);
					else
						ig.Emit (OpCodes.Brfalse, target);
					
					return;
				} else if (right is BoolConstant) {
					left.Emit (ec);
					if (my_on_true != ((BoolConstant) right).Value)
						ig.Emit (OpCodes.Brtrue, target);
					else
						ig.Emit (OpCodes.Brfalse, target);
					
					return;
				}

			} else if (oper == Operator.LogicalAnd) {

				if (onTrue) {
					Label tests_end = ig.DefineLabel ();
					
					left.EmitBranchable (ec, tests_end, false);
					right.EmitBranchable (ec, target, true);
					ig.MarkLabel (tests_end);					
				} else {
					left.EmitBranchable (ec, target, false);
					right.EmitBranchable (ec, target, false);
				}
				
				return;
				
			} else if (oper == Operator.LogicalOr){
				if (onTrue) {
					left.EmitBranchable (ec, target, true);
					right.EmitBranchable (ec, target, true);
					
				} else {
					Label tests_end = ig.DefineLabel ();
					left.EmitBranchable (ec, tests_end, true);
					right.EmitBranchable (ec, target, false);
					ig.MarkLabel (tests_end);
				}
				
				return;
				
			} else if (!(oper == Operator.LessThan        || oper == Operator.GreaterThan ||
			             oper == Operator.LessThanOrEqual || oper == Operator.GreaterThanOrEqual ||
			             oper == Operator.Equality        || oper == Operator.Inequality)) {
				base.EmitBranchable (ec, target, onTrue);
				return;
			}
			
			left.Emit (ec);
			right.Emit (ec);

			Type t = left.Type;
			bool isUnsigned = is_unsigned (t) || t == TypeManager.double_type || t == TypeManager.float_type;
			
			switch (oper){
			case Operator.Equality:
				if (onTrue)
					ig.Emit (OpCodes.Beq, target);
				else
					ig.Emit (OpCodes.Bne_Un, target);
				break;

			case Operator.Inequality:
				if (onTrue)
					ig.Emit (OpCodes.Bne_Un, target);
				else
					ig.Emit (OpCodes.Beq, target);
				break;

			case Operator.LessThan:
				if (onTrue)
					if (isUnsigned)
						ig.Emit (OpCodes.Blt_Un, target);
					else
						ig.Emit (OpCodes.Blt, target);
				else
					if (isUnsigned)
						ig.Emit (OpCodes.Bge_Un, target);
					else
						ig.Emit (OpCodes.Bge, target);
				break;

			case Operator.GreaterThan:
				if (onTrue)
					if (isUnsigned)
						ig.Emit (OpCodes.Bgt_Un, target);
					else
						ig.Emit (OpCodes.Bgt, target);
				else
					if (isUnsigned)
						ig.Emit (OpCodes.Ble_Un, target);
					else
						ig.Emit (OpCodes.Ble, target);
				break;

			case Operator.LessThanOrEqual:
				if (onTrue)
					if (isUnsigned)
						ig.Emit (OpCodes.Ble_Un, target);
					else
						ig.Emit (OpCodes.Ble, target);
				else
					if (isUnsigned)
						ig.Emit (OpCodes.Bgt_Un, target);
					else
						ig.Emit (OpCodes.Bgt, target);
				break;


			case Operator.GreaterThanOrEqual:
				if (onTrue)
					if (isUnsigned)
						ig.Emit (OpCodes.Bge_Un, target);
					else
						ig.Emit (OpCodes.Bge, target);
				else
					if (isUnsigned)
						ig.Emit (OpCodes.Blt_Un, target);
					else
						ig.Emit (OpCodes.Blt, target);
				break;
			default:
				Console.WriteLine (oper);
				throw new Exception ("what is THAT");
			}
		}
		
		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			Type l = left.Type;
			OpCode opcode;

			//
			// Handle short-circuit operators differently
			// than the rest
			//
			if (oper == Operator.LogicalAnd) {
				Label load_zero = ig.DefineLabel ();
				Label end = ig.DefineLabel ();
								
				left.EmitBranchable (ec, load_zero, false);
				right.Emit (ec);
				ig.Emit (OpCodes.Br, end);
				
				ig.MarkLabel (load_zero);
				ig.Emit (OpCodes.Ldc_I4_0);
				ig.MarkLabel (end);
				return;
			} else if (oper == Operator.LogicalOr) {
				Label load_one = ig.DefineLabel ();
				Label end = ig.DefineLabel ();

				left.EmitBranchable (ec, load_one, true);
				right.Emit (ec);
				ig.Emit (OpCodes.Br, end);
				
				ig.MarkLabel (load_one);
				ig.Emit (OpCodes.Ldc_I4_1);
				ig.MarkLabel (end);
				return;
			}

			left.Emit (ec);
			right.Emit (ec);

			bool isUnsigned = is_unsigned (left.Type);
			
			switch (oper){
			case Operator.Multiply:
				if (ec.CheckState){
					if (l == TypeManager.int32_type || l == TypeManager.int64_type)
						opcode = OpCodes.Mul_Ovf;
					else if (isUnsigned)
						opcode = OpCodes.Mul_Ovf_Un;
					else
						opcode = OpCodes.Mul;
				} else
					opcode = OpCodes.Mul;

				break;

			case Operator.Division:
				if (isUnsigned)
					opcode = OpCodes.Div_Un;
				else
					opcode = OpCodes.Div;
				break;

			case Operator.Modulus:
				if (isUnsigned)
					opcode = OpCodes.Rem_Un;
				else
					opcode = OpCodes.Rem;
				break;

			case Operator.Addition:
				if (ec.CheckState){
					if (l == TypeManager.int32_type || l == TypeManager.int64_type)
						opcode = OpCodes.Add_Ovf;
					else if (isUnsigned)
						opcode = OpCodes.Add_Ovf_Un;
					else
						opcode = OpCodes.Add;
				} else
					opcode = OpCodes.Add;
				break;

			case Operator.Subtraction:
				if (ec.CheckState){
					if (l == TypeManager.int32_type || l == TypeManager.int64_type)
						opcode = OpCodes.Sub_Ovf;
					else if (isUnsigned)
						opcode = OpCodes.Sub_Ovf_Un;
					else
						opcode = OpCodes.Sub;
				} else
					opcode = OpCodes.Sub;
				break;

			case Operator.RightShift:
				if (isUnsigned)
					opcode = OpCodes.Shr_Un;
				else
					opcode = OpCodes.Shr;
				break;
				
			case Operator.LeftShift:
				opcode = OpCodes.Shl;
				break;

			case Operator.Equality:
				opcode = OpCodes.Ceq;
				break;

			case Operator.Inequality:
				ig.Emit (OpCodes.Ceq);
				ig.Emit (OpCodes.Ldc_I4_0);
				
				opcode = OpCodes.Ceq;
				break;

			case Operator.LessThan:
				if (isUnsigned)
					opcode = OpCodes.Clt_Un;
				else
					opcode = OpCodes.Clt;
				break;

			case Operator.GreaterThan:
				if (isUnsigned)
					opcode = OpCodes.Cgt_Un;
				else
					opcode = OpCodes.Cgt;
				break;

			case Operator.LessThanOrEqual:
				Type lt = left.Type;
				
				if (isUnsigned || (lt == TypeManager.double_type || lt == TypeManager.float_type))
					ig.Emit (OpCodes.Cgt_Un);
				else
					ig.Emit (OpCodes.Cgt);
				ig.Emit (OpCodes.Ldc_I4_0);
				
				opcode = OpCodes.Ceq;
				break;

			case Operator.GreaterThanOrEqual:
				Type le = left.Type;
				
				if (isUnsigned || (le == TypeManager.double_type || le == TypeManager.float_type))
					ig.Emit (OpCodes.Clt_Un);
				else
					ig.Emit (OpCodes.Clt);
				
				ig.Emit (OpCodes.Ldc_I4_0);
				
				opcode = OpCodes.Ceq;
				break;

			case Operator.BitwiseOr:
				opcode = OpCodes.Or;
				break;

			case Operator.BitwiseAnd:
				opcode = OpCodes.And;
				break;

			case Operator.ExclusiveOr:
				opcode = OpCodes.Xor;
				break;

			default:
				throw new Exception ("This should not happen: Operator = "
						     + oper.ToString ());
			}

			ig.Emit (opcode);
		}
	}

	//
	// Object created by Binary when the binary operator uses an method instead of being
	// a binary operation that maps to a CIL binary operation.
	//
	public class BinaryMethod : Expression {
		public MethodBase method;
		public ArrayList  Arguments;
		
		public BinaryMethod (Type t, MethodBase m, ArrayList args)
		{
			method = m;
			Arguments = args;
			type = t;
			eclass = ExprClass.Value;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			if (Arguments != null) 
				Invocation.EmitArguments (ec, method, Arguments);
			
			if (method is MethodInfo)
				ig.Emit (OpCodes.Call, (MethodInfo) method);
			else
				ig.Emit (OpCodes.Call, (ConstructorInfo) method);
		}
	}
	
	//
	// Represents the operation a + b [+ c [+ d [+ ...]]], where a is a string
	// b, c, d... may be strings or objects.
	//
	public class StringConcat : Expression {
		ArrayList operands;
		bool invalid = false;
		
		
		public StringConcat (EmitContext ec, Location loc, Expression left, Expression right)
		{
			this.loc = loc;
			type = TypeManager.string_type;
			eclass = ExprClass.Value;
		
			operands = new ArrayList (2);
			Append (ec, left);
			Append (ec, right);
		}
		
		public override Expression DoResolve (EmitContext ec)
		{
			if (invalid)
				return null;
			
			return this;
		}
		
		public void Append (EmitContext ec, Expression operand)
		{
			//
			// Constant folding
			//
			if (operand is StringConstant && operands.Count != 0) {
				StringConstant last_operand = operands [operands.Count - 1] as StringConstant;
				if (last_operand != null) {
					operands [operands.Count - 1] = new StringConstant (last_operand.Value + ((StringConstant) operand).Value);
					return;
				}
			}
			
			//
			// Conversion to object
			//
			if (operand.Type != TypeManager.string_type) {
				Expression no = Convert.ImplicitConversion (ec, operand, TypeManager.object_type, loc);
				
				if (no == null) {
					Binary.Error_OperatorCannotBeApplied (loc, "+", TypeManager.string_type, operand.Type);
					invalid = true;
				}
				operand = no;
			}
			
			operands.Add (operand);
		}

		public override void Emit (EmitContext ec)
		{
			MethodInfo concat_method = null;
			
			//
			// Are we also concating objects?
			//
			bool is_strings_only = true;
			
			//
			// Do conversion to arguments; check for strings only
			//
			for (int i = 0; i < operands.Count; i ++) {
				Expression e = (Expression) operands [i];
				is_strings_only &= e.Type == TypeManager.string_type;
			}
			
			for (int i = 0; i < operands.Count; i ++) {
				Expression e = (Expression) operands [i];
				
				if (! is_strings_only && e.Type == TypeManager.string_type) {
					// need to make sure this is an object, because the EmitParams
					// method might look at the type of this expression, see it is a
					// string and emit a string [] when we want an object [];
					
					e = Convert.ImplicitConversion (ec, e, TypeManager.object_type, loc);
				}
				operands [i] = new Argument (e, Argument.AType.Expression);
			}
			
			//
			// Find the right method
			//
			switch (operands.Count) {
			case 1:
				//
				// This should not be possible, because simple constant folding
				// is taken care of in the Binary code.
				//
				throw new Exception ("how did you get here?");
			
			case 2:
				concat_method = is_strings_only ? 
					TypeManager.string_concat_string_string :
					TypeManager.string_concat_object_object ;
				break;
			case 3:
				concat_method = is_strings_only ? 
					TypeManager.string_concat_string_string_string :
					TypeManager.string_concat_object_object_object ;
				break;
			case 4:
				//
				// There is not a 4 param overlaod for object (the one that there is
				// is actually a varargs methods, and is only in corlib because it was
				// introduced there before.).
				//
				if (!is_strings_only)
					goto default;
				
				concat_method = TypeManager.string_concat_string_string_string_string;
				break;
			default:
				concat_method = is_strings_only ? 
					TypeManager.string_concat_string_dot_dot_dot :
					TypeManager.string_concat_object_dot_dot_dot ;
				break;
			}
			
			Invocation.EmitArguments (ec, concat_method, operands);
			ec.ig.Emit (OpCodes.Call, concat_method);
		}
	}

	//
	// Object created with +/= on delegates
	//
	public class BinaryDelegate : Expression {
		MethodInfo method;
		ArrayList  args;

		public BinaryDelegate (Type t, MethodInfo mi, ArrayList args)
		{
			method = mi;
			this.args = args;
			type = t;
			eclass = ExprClass.Value;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			Invocation.EmitArguments (ec, method, args);
			
			ig.Emit (OpCodes.Call, (MethodInfo) method);
			ig.Emit (OpCodes.Castclass, type);
		}

		public Expression Right {
			get {
				Argument arg = (Argument) args [1];
				return arg.Expr;
			}
		}

		public bool IsAddition {
			get {
				return method == TypeManager.delegate_combine_delegate_delegate;
			}
		}
	}
	
	//
	// User-defined conditional logical operator
	public class ConditionalLogicalOperator : Expression {
		Expression left, right;
		bool is_and;

		public ConditionalLogicalOperator (bool is_and, Expression left, Expression right, Type t, Location loc)
		{
			type = t;
			eclass = ExprClass.Value;
			this.loc = loc;
			this.left = left;
			this.right = right;
			this.is_and = is_and;
		}

		protected void Error19 ()
		{
			Binary.Error_OperatorCannotBeApplied (loc, is_and ? "&&" : "||", type, type);
		}

		protected void Error218 ()
		{
			Error (218, "The type ('" + TypeManager.CSharpName (type) + "') must contain " +
			       "declarations of operator true and operator false");
		}

		Expression op_true, op_false, op;
		LocalTemporary left_temp;

		public override Expression DoResolve (EmitContext ec)
		{
			MethodInfo method;
			Expression operator_group;

			operator_group = MethodLookup (ec, type, is_and ? "op_BitwiseAnd" : "op_BitwiseOr", loc);
			if (operator_group == null) {
				Error19 ();
				return null;
			}

			left_temp = new LocalTemporary (ec, type);

			ArrayList arguments = new ArrayList ();
			arguments.Add (new Argument (left_temp, Argument.AType.Expression));
			arguments.Add (new Argument (right, Argument.AType.Expression));
			method = Invocation.OverloadResolve (ec, (MethodGroupExpr) operator_group, arguments, loc) as MethodInfo;
			if ((method == null) || (method.ReturnType != type)) {
				Error19 ();
				return null;
			}

			op = new StaticCallExpr (method, arguments, loc);

			op_true = GetOperatorTrue (ec, left_temp, loc);
			op_false = GetOperatorFalse (ec, left_temp, loc);
			if ((op_true == null) || (op_false == null)) {
				Error218 ();
				return null;
			}

			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			Label false_target = ig.DefineLabel ();
			Label end_target = ig.DefineLabel ();

			ig.Emit (OpCodes.Nop);

			left.Emit (ec);
			left_temp.Store (ec);

			(is_and ? op_false : op_true).EmitBranchable (ec, false_target, false);
			left_temp.Emit (ec);
			ig.Emit (OpCodes.Br, end_target);
			ig.MarkLabel (false_target);
			op.Emit (ec);
			ig.MarkLabel (end_target);

			ig.Emit (OpCodes.Nop);
		}
	}

	public class PointerArithmetic : Expression {
		Expression left, right;
		bool is_add;

		//
		// We assume that `l' is always a pointer
		//
		public PointerArithmetic (bool is_addition, Expression l, Expression r, Type t, Location loc)
		{
			type = t;
			eclass = ExprClass.Variable;
			this.loc = loc;
			left = l;
			right = r;
			is_add = is_addition;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// We are born fully resolved
			//
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			Type op_type = left.Type;
			ILGenerator ig = ec.ig;
			Type element = TypeManager.GetElementType (op_type);
			int size = GetTypeSize (element);
			Type rtype = right.Type;
			
			if (rtype.IsPointer){
				//
				// handle (pointer - pointer)
				//
				left.Emit (ec);
				right.Emit (ec);
				ig.Emit (OpCodes.Sub);

				if (size != 1){
					if (size == 0)
						ig.Emit (OpCodes.Sizeof, element);
					else 
						IntLiteral.EmitInt (ig, size);
					ig.Emit (OpCodes.Div);
				}
				ig.Emit (OpCodes.Conv_I8);
			} else {
				//
				// handle + and - on (pointer op int)
				//
				left.Emit (ec);
				ig.Emit (OpCodes.Conv_I);
				right.Emit (ec);
				if (size != 1){
					if (size == 0)
						ig.Emit (OpCodes.Sizeof, element);
					else 
						IntLiteral.EmitInt (ig, size);
					if (rtype == TypeManager.int64_type)
						ig.Emit (OpCodes.Conv_I8);
					else if (rtype == TypeManager.uint64_type)
						ig.Emit (OpCodes.Conv_U8);
					ig.Emit (OpCodes.Mul);
					ig.Emit (OpCodes.Conv_I);
				}
				if (is_add)
					ig.Emit (OpCodes.Add);
				else
					ig.Emit (OpCodes.Sub);
			}
		}
	}
	
	/// <summary>
	///   Implements the ternary conditional operator (?:)
	/// </summary>
	public class Conditional : Expression {
		Expression expr, trueExpr, falseExpr;
		
		public Conditional (Expression expr, Expression trueExpr, Expression falseExpr, Location l)
		{
			this.expr = expr;
			this.trueExpr = trueExpr;
			this.falseExpr = falseExpr;
			this.loc = l;
		}

		public Expression Expr {
			get {
				return expr;
			}
		}

		public Expression TrueExpr {
			get {
				return trueExpr;
			}
		}

		public Expression FalseExpr {
			get {
				return falseExpr;
			}
		}

		public override Expression DoResolve (EmitContext ec)
		{
			expr = expr.Resolve (ec);

			if (expr == null)
				return null;
			
			if (expr.Type != TypeManager.bool_type){
				expr = Expression.ResolveBoolean (
					ec, expr, loc);
				
				if (expr == null)
					return null;
			}
			
			trueExpr = trueExpr.Resolve (ec);
			falseExpr = falseExpr.Resolve (ec);

			if (trueExpr == null || falseExpr == null)
				return null;

			eclass = ExprClass.Value;
			if (trueExpr.Type == falseExpr.Type)
				type = trueExpr.Type;
			else {
				Expression conv;
				Type true_type = trueExpr.Type;
				Type false_type = falseExpr.Type;

				if (trueExpr is NullLiteral){
					type = false_type;
					return this;
				} else if (falseExpr is NullLiteral){
					type = true_type;
					return this;
				}
				
				//
				// First, if an implicit conversion exists from trueExpr
				// to falseExpr, then the result type is of type falseExpr.Type
				//
				conv = Convert.ImplicitConversion (ec, trueExpr, false_type, loc);
				if (conv != null){
					//
					// Check if both can convert implicitl to each other's type
					//
					if (Convert.ImplicitConversion (ec, falseExpr, true_type, loc) != null){
						Error (172,
						       "Can not compute type of conditional expression " +
						       "as `" + TypeManager.CSharpName (trueExpr.Type) +
						       "' and `" + TypeManager.CSharpName (falseExpr.Type) +
						       "' convert implicitly to each other");
						return null;
					}
					type = false_type;
					trueExpr = conv;
				} else if ((conv = Convert.ImplicitConversion(ec, falseExpr, true_type,loc))!= null){
					type = true_type;
					falseExpr = conv;
				} else {
					Error (173, "The type of the conditional expression can " +
					       "not be computed because there is no implicit conversion" +
					       " from `" + TypeManager.CSharpName (trueExpr.Type) + "'" +
					       " and `" + TypeManager.CSharpName (falseExpr.Type) + "'");
					return null;
				}
			}

			if (expr is BoolConstant){
				BoolConstant bc = (BoolConstant) expr;

				if (bc.Value)
					return trueExpr;
				else
					return falseExpr;
			}

			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			Label false_target = ig.DefineLabel ();
			Label end_target = ig.DefineLabel ();

			expr.EmitBranchable (ec, false_target, false);
			trueExpr.Emit (ec);
			ig.Emit (OpCodes.Br, end_target);
			ig.MarkLabel (false_target);
			falseExpr.Emit (ec);
			ig.MarkLabel (end_target);
		}

	}

	/// <summary>
	///   Local variables
	/// </summary>
	public class LocalVariableReference : Expression, IAssignMethod, IMemoryLocation, IVariable {
		public readonly string Name;
		public readonly Block Block;
		LocalInfo local_info;
		bool is_readonly;
		
		public LocalVariableReference (Block block, string name, Location l)
		{
			Block = block;
			Name = name;
			loc = l;
			eclass = ExprClass.Variable;
		}

		// Setting `is_readonly' to false will allow you to create a writable
		// reference to a read-only variable.  This is used by foreach and using.
		public LocalVariableReference (Block block, string name, Location l,
					       LocalInfo local_info, bool is_readonly)
			: this (block, name, l)
		{
			this.local_info = local_info;
			this.is_readonly = is_readonly;
		}

		public VariableInfo VariableInfo {
			get { return local_info.VariableInfo; }
		}

		public bool IsReadOnly {
			get {
				return is_readonly;
			}
		}

		protected void DoResolveBase (EmitContext ec)
		{
			if (local_info == null) {
				local_info = Block.GetLocalInfo (Name);
				is_readonly = local_info.ReadOnly;
			}

			type = local_info.VariableType;
#if false
			if (ec.InAnonymousMethod)
				Block.LiftVariable (local_info);
#endif
		}

		protected Expression DoResolve (EmitContext ec, bool is_lvalue)
		{
			Expression e = Block.GetConstantExpression (Name);
			if (e != null) {
				local_info.Used = true;
				eclass = ExprClass.Value;
				return e.Resolve (ec);
			}

			VariableInfo variable_info = local_info.VariableInfo; 
			if ((variable_info != null) && !variable_info.IsAssigned (ec, loc))
				return null;

			if (!is_lvalue)
				local_info.Used = true;

			if (local_info.LocalBuilder == null)
				return ec.RemapLocal (local_info);
			
			return this;
		}
		
		public override Expression DoResolve (EmitContext ec)
		{
			DoResolveBase (ec);

			return DoResolve (ec, false);
		}

		override public Expression DoResolveLValue (EmitContext ec, Expression right_side)
		{
			DoResolveBase (ec);

			VariableInfo variable_info = local_info.VariableInfo; 
			if (variable_info != null)
				variable_info.SetAssigned (ec);

			Expression e = DoResolve (ec, true);

			if (e == null)
				return null;

			if (is_readonly){
				Error (1604, "cannot assign to `" + Name + "' because it is readonly");
				return null;
			}

			CheckObsoleteAttribute (e.Type);

			if (local_info.LocalBuilder == null)
				return ec.RemapLocalLValue (local_info, right_side);
			
			return this;
		}

		public bool VerifyFixed (bool is_expression)
		{
			return !is_expression || local_info.IsFixed;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			ig.Emit (OpCodes.Ldloc, local_info.LocalBuilder);
		}
		
		public void EmitAssign (EmitContext ec, Expression source)
		{
			ILGenerator ig = ec.ig;

			source.Emit (ec);
			ig.Emit (OpCodes.Stloc, local_info.LocalBuilder);
		}
		
		public void AddressOf (EmitContext ec, AddressOp mode)
		{
			ILGenerator ig = ec.ig;
			
			ig.Emit (OpCodes.Ldloca, local_info.LocalBuilder);
		}

		public override string ToString ()
		{
			return String.Format ("{0} ({1}:{2})", GetType (), Name, loc);
		}
	}

	/// <summary>
	///   This represents a reference to a parameter in the intermediate
	///   representation.
	/// </summary>
	public class ParameterReference : Expression, IAssignMethod, IMemoryLocation, IVariable {
		Parameters pars;
		String name;
		int idx;
		Block block;
		VariableInfo vi;
		public Parameter.Modifier mod;
		public bool is_ref, is_out;
		
		public ParameterReference (Parameters pars, Block block, int idx, string name, Location loc)
		{
			this.pars = pars;
			this.block = block;
			this.idx  = idx;
			this.name = name;
			this.loc = loc;
			eclass = ExprClass.Variable;
		}

		public VariableInfo VariableInfo {
			get { return vi; }
		}

		public bool VerifyFixed (bool is_expression)
		{
			return !is_expression || TypeManager.IsValueType (type);
		}

		public bool IsAssigned (EmitContext ec, Location loc)
		{
			if (!ec.DoFlowAnalysis || !is_out ||
			    ec.CurrentBranching.IsAssigned (vi))
				return true;

			Report.Error (165, loc,
				      "Use of unassigned parameter `" + name + "'");
			return false;
		}

		public bool IsFieldAssigned (EmitContext ec, string field_name, Location loc)
		{
			if (!ec.DoFlowAnalysis || !is_out ||
			    ec.CurrentBranching.IsFieldAssigned (vi, field_name))
				return true;

			Report.Error (170, loc,
				      "Use of possibly unassigned field `" + field_name + "'");
			return false;
		}

		public void SetAssigned (EmitContext ec)
		{
			if (is_out && ec.DoFlowAnalysis)
				ec.CurrentBranching.SetAssigned (vi);
		}

		public void SetFieldAssigned (EmitContext ec, string field_name)
		{
			if (is_out && ec.DoFlowAnalysis)
				ec.CurrentBranching.SetFieldAssigned (vi, field_name);
		}

		protected void DoResolveBase (EmitContext ec)
		{
			type = pars.GetParameterInfo (ec.DeclSpace, idx, out mod);
			is_ref = (mod & Parameter.Modifier.ISBYREF) != 0;
			is_out = (mod & Parameter.Modifier.OUT) != 0;
			eclass = ExprClass.Variable;

			if (is_out)
				vi = block.ParameterMap [idx];
		}

		//
		// Notice that for ref/out parameters, the type exposed is not the
		// same type exposed externally.
		//
		// for "ref int a":
		//   externally we expose "int&"
		//   here we expose       "int".
		//
		// We record this in "is_ref".  This means that the type system can treat
		// the type as it is expected, but when we generate the code, we generate
		// the alternate kind of code.
		//
		public override Expression DoResolve (EmitContext ec)
		{
			DoResolveBase (ec);

			if (is_out && ec.DoFlowAnalysis && !IsAssigned (ec, loc))
				return null;

			if (ec.RemapToProxy)
				return ec.RemapParameter (idx);
			
			return this;
		}

		override public Expression DoResolveLValue (EmitContext ec, Expression right_side)
		{
			DoResolveBase (ec);

			SetAssigned (ec);

			if (ec.RemapToProxy)
				return ec.RemapParameterLValue (idx, right_side);
			
			return this;
		}

		static public void EmitLdArg (ILGenerator ig, int x)
		{
			if (x <= 255){
				switch (x){
				case 0: ig.Emit (OpCodes.Ldarg_0); break;
				case 1: ig.Emit (OpCodes.Ldarg_1); break;
				case 2: ig.Emit (OpCodes.Ldarg_2); break;
				case 3: ig.Emit (OpCodes.Ldarg_3); break;
				default: ig.Emit (OpCodes.Ldarg_S, (byte) x); break;
				}
			} else
				ig.Emit (OpCodes.Ldarg, x);
		}
		
		//
		// This method is used by parameters that are references, that are
		// being passed as references:  we only want to pass the pointer (that
		// is already stored in the parameter, not the address of the pointer,
		// and not the value of the variable).
		//
		public void EmitLoad (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			int arg_idx = idx;

			if (!ec.IsStatic)
				arg_idx++;

			EmitLdArg (ig, arg_idx);
		}
		
		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			int arg_idx = idx;

			if (!ec.IsStatic)
				arg_idx++;

			EmitLdArg (ig, arg_idx);

			if (!is_ref)
				return;

			//
			// If we are a reference, we loaded on the stack a pointer
			// Now lets load the real value
			//
			LoadFromPtr (ig, type);
		}

		public void EmitAssign (EmitContext ec, Expression source)
		{
			ILGenerator ig = ec.ig;
			
			int arg_idx = idx;

			if (!ec.IsStatic)
				arg_idx++;

			if (is_ref)
				EmitLdArg (ig, arg_idx);
			
			source.Emit (ec);

			if (is_ref)
				StoreFromPtr (ig, type);
			else {
				if (arg_idx <= 255)
					ig.Emit (OpCodes.Starg_S, (byte) arg_idx);
				else
					ig.Emit (OpCodes.Starg, arg_idx);
			}
		}

		public void AddressOf (EmitContext ec, AddressOp mode)
		{
			int arg_idx = idx;

			if (!ec.IsStatic)
				arg_idx++;

			if (is_ref){
				if (arg_idx <= 255)
					ec.ig.Emit (OpCodes.Ldarg_S, (byte) arg_idx);
				else
					ec.ig.Emit (OpCodes.Ldarg, arg_idx);
			} else {
				if (arg_idx <= 255)
					ec.ig.Emit (OpCodes.Ldarga_S, (byte) arg_idx);
				else
					ec.ig.Emit (OpCodes.Ldarga, arg_idx);
			}
		}

	}
	
	/// <summary>
	///   Used for arguments to New(), Invocation()
	/// </summary>
	public class Argument {
		public enum AType : byte {
			Expression,
			Ref,
			Out,
			ArgList
		};

		public readonly AType ArgType;
		public Expression Expr;
		
		public Argument (Expression expr, AType type)
		{
			this.Expr = expr;
			this.ArgType = type;
		}

		public Type Type {
			get {
				if (ArgType == AType.Ref || ArgType == AType.Out)
					return TypeManager.GetReferenceType (Expr.Type);
				else
					return Expr.Type;
			}
		}

		public Parameter.Modifier GetParameterModifier ()
		{
			switch (ArgType) {
			case AType.Out:
				return Parameter.Modifier.OUT | Parameter.Modifier.ISBYREF;

			case AType.Ref:
				return Parameter.Modifier.REF | Parameter.Modifier.ISBYREF;

			default:
				return Parameter.Modifier.NONE;
			}
		}

	        public static string FullDesc (Argument a)
		{
			if (a.ArgType == AType.ArgList)
				return "__arglist";

			return (a.ArgType == AType.Ref ? "ref " :
				(a.ArgType == AType.Out ? "out " : "")) +
				TypeManager.CSharpName (a.Expr.Type);
		}

		public bool ResolveMethodGroup (EmitContext ec, Location loc)
		{
			// FIXME: csc doesn't report any error if you try to use `ref' or
			//        `out' in a delegate creation expression.
			Expr = Expr.Resolve (ec, ResolveFlags.VariableOrValue | ResolveFlags.MethodGroup);
			if (Expr == null)
				return false;

			return true;
		}
		
		public bool Resolve (EmitContext ec, Location loc)
		{
			if (ArgType == AType.Ref) {
				Expr = Expr.Resolve (ec);
				if (Expr == null)
					return false;

				Expr = Expr.ResolveLValue (ec, Expr);
			} else if (ArgType == AType.Out)
				Expr = Expr.ResolveLValue (ec, new EmptyExpression ());
			else
				Expr = Expr.Resolve (ec);

			if (Expr == null)
				return false;

			if (ArgType == AType.Expression)
				return true;
			else {
				//
				// Catch errors where fields of a MarshalByRefObject are passed as ref or out
				// This is only allowed for `this'
				//
				FieldExpr fe = Expr as FieldExpr;
				if (fe != null && !fe.IsStatic){
					Expression instance = fe.InstanceExpression;

					if (instance.GetType () != typeof (This)){
						if (fe.InstanceExpression.Type.IsSubclassOf (TypeManager.mbr_type)){
							Report.Error (197, loc,
								      "Can not pass a type that derives from MarshalByRefObject with out or ref");
							return false;
						}
					}
				}
			}

			if (Expr.eclass != ExprClass.Variable){
				//
				// We just probe to match the CSC output
				//
				if (Expr.eclass == ExprClass.PropertyAccess ||
				    Expr.eclass == ExprClass.IndexerAccess){
					Report.Error (
						206, loc,
						"A property or indexer can not be passed as an out or ref " +
						"parameter");
				} else {
					Report.Error (
						1510, loc,
						"An lvalue is required as an argument to out or ref");
				}
				return false;
			}
				
			return true;
		}

		public void Emit (EmitContext ec)
		{
			//
			// Ref and Out parameters need to have their addresses taken.
			//
			// ParameterReferences might already be references, so we want
			// to pass just the value
			//
			if (ArgType == AType.Ref || ArgType == AType.Out){
				AddressOp mode = AddressOp.Store;

				if (ArgType == AType.Ref)
					mode |= AddressOp.Load;
				
				if (Expr is ParameterReference){
					ParameterReference pr = (ParameterReference) Expr;

					if (pr.is_ref)
						pr.EmitLoad (ec);
					else {
						
						pr.AddressOf (ec, mode);
					}
				} else {
					((IMemoryLocation)Expr).AddressOf (ec, mode);
				}
			} else
				Expr.Emit (ec);
		}
	}

	/// <summary>
	///   Invocation of methods or delegates.
	/// </summary>
	public class Invocation : ExpressionStatement {
		public readonly ArrayList Arguments;

		Expression expr;
		MethodBase method = null;
		bool is_base;
		
		static Hashtable method_parameter_cache;

		static Invocation ()
		{
			method_parameter_cache = new PtrHashtable ();
		}
			
		//
		// arguments is an ArrayList, but we do not want to typecast,
		// as it might be null.
		//
		// FIXME: only allow expr to be a method invocation or a
		// delegate invocation (7.5.5)
		//
		public Invocation (Expression expr, ArrayList arguments, Location l)
		{
			this.expr = expr;
			Arguments = arguments;
			loc = l;
		}

		public Expression Expr {
			get {
				return expr;
			}
		}

		/// <summary>
		///   Returns the Parameters (a ParameterData interface) for the
		///   Method `mb'
		/// </summary>
		public static ParameterData GetParameterData (MethodBase mb)
		{
			object pd = method_parameter_cache [mb];
			object ip;
			
			if (pd != null)
				return (ParameterData) pd;

			
			ip = TypeManager.LookupParametersByBuilder (mb);
			if (ip != null){
				method_parameter_cache [mb] = ip;

				return (ParameterData) ip;
			} else {
				ReflectionParameters rp = new ReflectionParameters (mb);
				method_parameter_cache [mb] = rp;

				return (ParameterData) rp;
			}
		}

		/// <summary>
		///   Determines "better conversion" as specified in 7.4.2.3
		///
                ///    Returns : 1 if a->p is better
		///              0 if a->q or neither is better 
		/// </summary>
		static int BetterConversion (EmitContext ec, Argument a, Type p, Type q, Location loc)
		{
			Type argument_type = a.Type;
			Expression argument_expr = a.Expr;

			if (argument_type == null)
				throw new Exception ("Expression of type " + a.Expr +
                                                     " does not resolve its type");

			//
			// This is a special case since csc behaves this way.
			//
			if (argument_expr is NullLiteral &&
                            p == TypeManager.string_type &&
                            q == TypeManager.object_type)
				return 1;
			else if (argument_expr is NullLiteral &&
                                 p == TypeManager.object_type &&
                                 q == TypeManager.string_type)
				return 0;
			
                        //
                        // csc behaves this way so we emulate it. Basically, if the argument
                        // is null and one of the types to compare is 'object' and the other
                        // is a reference type, we prefer the other.
                        //
                        // I can't find this anywhere in the spec but we can interpret this
                        // to mean that null can be of any type you wish in such a context
                        //
                        if (p != null && q != null) {
                                if (argument_expr is NullLiteral &&
                                    !p.IsValueType &&
                                    q == TypeManager.object_type)
                                        return 1;
                                else if (argument_expr is NullLiteral &&
                                         !q.IsValueType &&
                                         p == TypeManager.object_type)
                                        return 0;
                        }
                                
			if (p == q)
				return 0;
			
			if (argument_type == p)
				return 1;

			if (argument_type == q)
				return 0;

			if (q == null) {
				Expression tmp = Convert.ImplicitConversion (ec, argument_expr, p, loc);
				
				if (tmp != null)
					return 1;
				else
					return 0;
			}

			Expression p_tmp = new EmptyExpression (p);
			Expression q_tmp = new EmptyExpression (q);
			
			if (Convert.ImplicitConversionExists (ec, p_tmp, q) == true &&
			    Convert.ImplicitConversionExists (ec, q_tmp, p) == false)
				return 1;

			if (p == TypeManager.sbyte_type)
				if (q == TypeManager.byte_type || q == TypeManager.ushort_type ||
				    q == TypeManager.uint32_type || q == TypeManager.uint64_type)
					return 1;

			if (p == TypeManager.short_type)
				if (q == TypeManager.ushort_type || q == TypeManager.uint32_type ||
				    q == TypeManager.uint64_type)
					return 1;

			if (p == TypeManager.int32_type)
				if (q == TypeManager.uint32_type || q == TypeManager.uint64_type)
					return 1;

			if (p == TypeManager.int64_type)
				if (q == TypeManager.uint64_type)
					return 1;

			return 0;
		}
		
		/// <summary>
		///   Determines "Better function" between candidate
                ///   and the current best match
		/// </summary>
		/// <remarks>
		///    Returns an integer indicating :
		///     0 if candidate ain't better
		///     1 if candidate is better than the current best match
		/// </remarks>
		static int BetterFunction (EmitContext ec, ArrayList args,
					   MethodBase candidate, bool candidate_params,
                                           MethodBase best, bool best_params,
					   Location loc)
		{
			ParameterData candidate_pd = GetParameterData (candidate);
			ParameterData best_pd;
			int argument_count;
		
			if (args == null)
				argument_count = 0;
			else
				argument_count = args.Count;

			int cand_count = candidate_pd.Count;
			
			//
			// If there is no best method, than this one
			// is better, however, if we already found a
			// best method, we cant tell. This happens
			// if we have:
			// 
			//	interface IFoo {
			//		void DoIt ();
			//	}
			//	
			//	interface IBar {
			//		void DoIt ();
			//	}
			//	
			//	interface IFooBar : IFoo, IBar {}
			//
			// We cant tell if IFoo.DoIt is better than IBar.DoIt
			//
			// However, we have to consider that
			// Trim (); is better than Trim (params char[] chars);
                        //
			if (cand_count == 0 && argument_count == 0)
				return best == null || best_params ? 1 : 0;

			if ((candidate_pd.ParameterModifier (cand_count - 1) != Parameter.Modifier.PARAMS) &&
			    (candidate_pd.ParameterModifier (cand_count - 1) != Parameter.Modifier.ARGLIST))
				if (cand_count != argument_count)
					return 0;

			if (best == null) {
				int x = 0;

				if (argument_count == 0 && cand_count == 1 &&
				    candidate_pd.ParameterModifier (cand_count - 1) == Parameter.Modifier.PARAMS)
					return 1;
				
				for (int j = 0; j < argument_count; ++j) {

					Argument a = (Argument) args [j];
					Type t = candidate_pd.ParameterType (j);

					if (candidate_pd.ParameterModifier (j) == Parameter.Modifier.PARAMS)
						if (candidate_params)
							t = TypeManager.GetElementType (t);

					x = BetterConversion (ec, a, t, null, loc);
					
					if (x <= 0)
						break;
				}

				if (x > 0)
					return 1;
				else
					return 0;
			}

			best_pd = GetParameterData (best);

			int rating1 = 0, rating2 = 0;
			
			for (int j = 0; j < argument_count; ++j) {
				int x, y;
				
				Argument a = (Argument) args [j];

				Type ct = candidate_pd.ParameterType (j);
				Type bt = best_pd.ParameterType (j);

				if (candidate_pd.ParameterModifier (j) == Parameter.Modifier.PARAMS)
					if (candidate_params)
						ct = TypeManager.GetElementType (ct);

				if (best_pd.ParameterModifier (j) == Parameter.Modifier.PARAMS)
					if (best_params)
						bt = TypeManager.GetElementType (bt);

				x = BetterConversion (ec, a, ct, bt, loc);
				y = BetterConversion (ec, a, bt, ct, loc);

				if (x < y)
					return 0;
				
				rating1 += x;
				rating2 += y;
			}

                        //
                        // If a method (in the normal form) with the
                        // same signature as the expanded form of the
                        // current best params method already exists,
                        // the expanded form is not applicable so we
                        // force it to select the candidate
                        //
                        if (!candidate_params && best_params && cand_count == argument_count)
                                return 1;

			if (rating1 > rating2)
				return 1;
			else
				return 0;
		}

		public static string FullMethodDesc (MethodBase mb)
		{
			string ret_type = "";

                        if (mb == null)
                                return "";

			if (mb is MethodInfo)
				ret_type = TypeManager.CSharpName (((MethodInfo) mb).ReturnType);
			
			StringBuilder sb = new StringBuilder (ret_type);
			sb.Append (" ");
			sb.Append (mb.ReflectedType.ToString ());
			sb.Append (".");
			sb.Append (mb.Name);
			
			ParameterData pd = GetParameterData (mb);

			int count = pd.Count;
			sb.Append (" (");
			
			for (int i = count; i > 0; ) {
				i--;

				sb.Append (pd.ParameterDesc (count - i - 1));
				if (i != 0)
					sb.Append (", ");
			}
			
			sb.Append (")");
			return sb.ToString ();
		}

		public static MethodGroupExpr MakeUnionSet (Expression mg1, Expression mg2, Location loc)
		{
			MemberInfo [] miset;
			MethodGroupExpr union;

			if (mg1 == null) {
				if (mg2 == null)
					return null;
				return (MethodGroupExpr) mg2;
			} else {
				if (mg2 == null)
					return (MethodGroupExpr) mg1;
			}
			
			MethodGroupExpr left_set = null, right_set = null;
			int length1 = 0, length2 = 0;
			
			left_set = (MethodGroupExpr) mg1;
			length1 = left_set.Methods.Length;
			
			right_set = (MethodGroupExpr) mg2;
			length2 = right_set.Methods.Length;
			
			ArrayList common = new ArrayList ();

			foreach (MethodBase r in right_set.Methods){
				if (TypeManager.ArrayContainsMethod (left_set.Methods, r))
					common.Add (r);
			}

			miset = new MemberInfo [length1 + length2 - common.Count];
			left_set.Methods.CopyTo (miset, 0);
			
			int k = length1;

			foreach (MethodBase r in right_set.Methods) {
				if (!common.Contains (r))
					miset [k++] = r;
			}

			union = new MethodGroupExpr (miset, loc);
			
			return union;
		}

		/// <summary>
		///   Determines if the candidate method, if a params method, is applicable
		///   in its expanded form to the given set of arguments
		/// </summary>
		static bool IsParamsMethodApplicable (EmitContext ec, ArrayList arguments,
						      MethodBase candidate, bool do_varargs)
		{
			int arg_count;
			
			if (arguments == null)
				arg_count = 0;
			else
				arg_count = arguments.Count;
			
			ParameterData pd = GetParameterData (candidate);

			int pd_count = pd.Count;
			if (pd_count == 0)
				return false;

			int count = pd_count - 1;
			if (do_varargs) {
				if (pd.ParameterModifier (count) != Parameter.Modifier.ARGLIST)
					return false;
				if (pd_count != arg_count)
					return false;
			} else {
				if (pd.ParameterModifier (count) != Parameter.Modifier.PARAMS)
					return false;
			}
			
			if (count > arg_count)
				return false;
			
			if (pd_count == 1 && arg_count == 0)
				return true;

			//
			// If we have come this far, the case which
			// remains is when the number of parameters is
			// less than or equal to the argument count.
			//
			for (int i = 0; i < count; ++i) {

				Argument a = (Argument) arguments [i];

				Parameter.Modifier a_mod = a.GetParameterModifier () &
					~(Parameter.Modifier.OUT | Parameter.Modifier.REF);
				Parameter.Modifier p_mod = pd.ParameterModifier (i) &
					~(Parameter.Modifier.OUT | Parameter.Modifier.REF);

				if (a_mod == p_mod) {

					if (a_mod == Parameter.Modifier.NONE)
						if (!Convert.ImplicitConversionExists (ec,
                                                                                       a.Expr,
                                                                                       pd.ParameterType (i)))
							return false;
										
					if ((a_mod & Parameter.Modifier.ISBYREF) != 0) {
						Type pt = pd.ParameterType (i);

						if (!pt.IsByRef)
							pt = TypeManager.GetReferenceType (pt);
						
						if (pt != a.Type)
							return false;
					}
				} else
					return false;
				
			}

			if (do_varargs) {
				Argument a = (Argument) arguments [count];
				if (!(a.Expr is Arglist))
					return false;

				return true;
			}

			Type element_type = TypeManager.GetElementType (pd.ParameterType (pd_count - 1));

			for (int i = pd_count - 1; i < arg_count; i++) {
				Argument a = (Argument) arguments [i];

				if (!Convert.ImplicitConversionExists (ec, a.Expr, element_type))
					return false;
			}
			
			return true;
		}

		/// <summary>
		///   Determines if the candidate method is applicable (section 14.4.2.1)
		///   to the given set of arguments
		/// </summary>
		static bool IsApplicable (EmitContext ec, ArrayList arguments, MethodBase candidate)
		{
			int arg_count;

			if (arguments == null)
				arg_count = 0;
			else
				arg_count = arguments.Count;


			ParameterData pd = GetParameterData (candidate);

			if (arg_count != pd.Count)
				return false;

			for (int i = arg_count; i > 0; ) {
				i--;

				Argument a = (Argument) arguments [i];

				Parameter.Modifier a_mod = a.GetParameterModifier () &
					~(Parameter.Modifier.OUT | Parameter.Modifier.REF);
				Parameter.Modifier p_mod = pd.ParameterModifier (i) &
					~(Parameter.Modifier.OUT | Parameter.Modifier.REF);


				if (a_mod == p_mod ||
				    (a_mod == Parameter.Modifier.NONE && p_mod == Parameter.Modifier.PARAMS)) {
					if (a_mod == Parameter.Modifier.NONE) {
                                                if (!Convert.ImplicitConversionExists (ec,
                                                                                       a.Expr,
                                                                                       pd.ParameterType (i)))
							return false;
                                        }
					
					if ((a_mod & Parameter.Modifier.ISBYREF) != 0) {
						Type pt = pd.ParameterType (i);

						if (!pt.IsByRef)
							pt = TypeManager.GetReferenceType (pt);
                                                
						if (pt != a.Type)
							return false;
					}
				} else
					return false;
			}

			return true;
		}
		
		/// <summary>
		///   Find the Applicable Function Members (7.4.2.1)
		///
		///   me: Method Group expression with the members to select.
		///       it might contain constructors or methods (or anything
		///       that maps to a method).
		///
		///   Arguments: ArrayList containing resolved Argument objects.
		///
		///   loc: The location if we want an error to be reported, or a Null
		///        location for "probing" purposes.
		///
		///   Returns: The MethodBase (either a ConstructorInfo or a MethodInfo)
		///            that is the best match of me on Arguments.
		///
		/// </summary>
		public static MethodBase OverloadResolve (EmitContext ec, MethodGroupExpr me,
							  ArrayList Arguments, Location loc)
		{
			MethodBase method = null;
			Type applicable_type = null;
			int argument_count;
			ArrayList candidates = new ArrayList ();

                        //
                        // Used to keep a map between the candidate
                        // and whether it is being considered in its
                        // normal or expanded form
                        //
                        // false is normal form, true is expanded form
                        //
                        Hashtable candidate_to_form = null;


                        //
                        // First we construct the set of applicable methods
                        //
                        // We start at the top of the type hierarchy and
                        // go down to find applicable methods
                        //
                        applicable_type = me.DeclaringType;
                        
                        if (me.Name == "Invoke" && TypeManager.IsDelegateType (applicable_type)) {
                                Error_InvokeOnDelegate (loc);
                                return null;
                        }

                        bool found_applicable = false;

			foreach (MethodBase candidate in me.Methods){
                                Type decl_type = candidate.DeclaringType;

                                //
                                // If we have already found an applicable method
                                // we eliminate all base types (Section 14.5.5.1)
                                //
                                if (decl_type != applicable_type &&
                                    (applicable_type.IsSubclassOf (decl_type) ||
                                     TypeManager.ImplementsInterface (applicable_type, decl_type)) &&
                                    found_applicable)
                                                continue;


				// Check if candidate is applicable (section 14.4.2.1)
				if (IsApplicable (ec, Arguments, candidate)) {
                                        // Candidate is applicable in normal form
                                        candidates.Add (candidate);
                                        applicable_type = candidate.DeclaringType;
                                        found_applicable = true;
                                } else if (IsParamsMethodApplicable (ec, Arguments, candidate, false)) {
					if (candidate_to_form == null)
						candidate_to_form = new PtrHashtable ();
					
					// Candidate is applicable in expanded form
					candidates.Add (candidate);
					applicable_type = candidate.DeclaringType;
					found_applicable = true; 
					candidate_to_form [candidate] = candidate;
                                } else if (IsParamsMethodApplicable (ec, Arguments, candidate, true)) {
					if (candidate_to_form == null)
						candidate_to_form = new PtrHashtable ();
					
					// Candidate is applicable in expanded form
					candidates.Add (candidate);
					applicable_type = candidate.DeclaringType;
					found_applicable = true; 
					candidate_to_form [candidate] = candidate;
				}
			}
                        

                        //
                        // Now we actually find the best method
                        //
			int candidate_top = candidates.Count;
			for (int ix = 0; ix < candidate_top; ix++){
				MethodBase candidate = (MethodBase) candidates [ix];

                                bool cand_params = candidate_to_form != null && candidate_to_form.Contains (candidate);
                                bool method_params = false;

                                if (method != null)
                                        method_params = candidate_to_form != null && candidate_to_form.Contains (method);
                                
                                int x = BetterFunction (ec, Arguments,
                                                        candidate, cand_params,
                                                        method, method_params,
                                                        loc);
                                
                                if (x == 0)
                                        continue;
                                
                                method = candidate;
                        }

			if (Arguments == null)
				argument_count = 0;
			else
				argument_count = Arguments.Count;
			

			if (method == null) {
				//
				// Okay so we have failed to find anything so we
				// return by providing info about the closest match
				//
				for (int i = 0; i < me.Methods.Length; ++i) {

					MethodBase c = (MethodBase) me.Methods [i];
					ParameterData pd = GetParameterData (c);

					if (pd.Count != argument_count)
						continue;

					VerifyArgumentsCompat (ec, Arguments, argument_count, c, false,
							       null, loc);
                                        break;
				}

                                if (!Location.IsNull (loc)) {
                                        string report_name = me.Name;
                                        if (report_name == ".ctor")
                                                report_name = me.DeclaringType.ToString ();
                                        
                                        Error_WrongNumArguments (loc, report_name, argument_count);
                                }
                                
				return null;
			}

			//
			// Now check that there are no ambiguities i.e the selected method
			// should be better than all the others
			//
                        bool best_params = candidate_to_form != null && candidate_to_form.Contains (method);

			for (int ix = 0; ix < candidate_top; ix++){
				MethodBase candidate = (MethodBase) candidates [ix];

                                if (candidate == method)
                                        continue;
                                               
				//
				// If a normal method is applicable in
				// the sense that it has the same
				// number of arguments, then the
				// expanded params method is never
				// applicable so we debar the params
				// method.
				//
                                // if ((IsParamsMethodApplicable (ec, Arguments, candidate) &&
//                                      IsApplicable (ec, Arguments, method)))
//                                         continue;
                                
                                bool cand_params = candidate_to_form != null && candidate_to_form.Contains (candidate);
				int x = BetterFunction (ec, Arguments,
                                                        method, best_params,
                                                        candidate, cand_params,
							loc);

				if (x != 1) {
 					Report.Error (
 						121, loc,
 						"Ambiguous call when selecting function due to implicit casts");
					return null;
 				}
			}

			//
			// And now check if the arguments are all
			// compatible, perform conversions if
			// necessary etc. and return if everything is
			// all right
			//
                        if (!VerifyArgumentsCompat (ec, Arguments, argument_count, method,
                                                    best_params, null, loc))
				return null;

			return method;
		}

                static void Error_WrongNumArguments (Location loc, String name, int arg_count)
                {
                        Report.Error (1501, loc,
                                      "No overload for method `" + name + "' takes `" +
                                      arg_count + "' arguments");
                }

                static void Error_InvokeOnDelegate (Location loc)
                {
                        Report.Error (1533, loc,
                                      "Invoke cannot be called directly on a delegate");
                }
                        
		static void Error_InvalidArguments (Location loc, int idx, MethodBase method,
                                                    Type delegate_type, string arg_sig, string par_desc)
		{
			if (delegate_type == null) 
				Report.Error (1502, loc,
					      "The best overloaded match for method '" +
					      FullMethodDesc (method) +
					      "' has some invalid arguments");
			else
				Report.Error (1594, loc,
					      "Delegate '" + delegate_type.ToString () +
					      "' has some invalid arguments.");
			Report.Error (1503, loc,
				      String.Format ("Argument {0}: Cannot convert from '{1}' to '{2}'",
						     idx, arg_sig, par_desc));
		}
		
		public static bool VerifyArgumentsCompat (EmitContext ec, ArrayList Arguments,
							  int argument_count,
							  MethodBase method, 
							  bool chose_params_expanded,
							  Type delegate_type,
							  Location loc)
		{
			ParameterData pd = GetParameterData (method);
			int pd_count = pd.Count;

			for (int j = 0; j < argument_count; j++) {
				Argument a = (Argument) Arguments [j];
				Expression a_expr = a.Expr;
				Type parameter_type = pd.ParameterType (j);
				Parameter.Modifier pm = pd.ParameterModifier (j);
				
				if (pm == Parameter.Modifier.PARAMS){
					if ((pm & ~Parameter.Modifier.PARAMS) != a.GetParameterModifier ()) {
						if (!Location.IsNull (loc))
							Error_InvalidArguments (
								loc, j, method, delegate_type,
								Argument.FullDesc (a), pd.ParameterDesc (j));
						return false;
					}

					if (chose_params_expanded)
						parameter_type = TypeManager.GetElementType (parameter_type);
				} else if (pm == Parameter.Modifier.ARGLIST){
					continue;
				} else {
					//
					// Check modifiers
					//
					if (pd.ParameterModifier (j) != a.GetParameterModifier ()){
						if (!Location.IsNull (loc))
							Error_InvalidArguments (
								loc, j, method, delegate_type,
								Argument.FullDesc (a), pd.ParameterDesc (j));
						return false;
					}
				}

				//
				// Check Type
				//
				if (a.Type != parameter_type){
					Expression conv;
					
					conv = Convert.ImplicitConversion (ec, a_expr, parameter_type, loc);

					if (conv == null) {
						if (!Location.IsNull (loc)) 
							Error_InvalidArguments (
								loc, j, method, delegate_type,
								Argument.FullDesc (a), pd.ParameterDesc (j));
						return false;
					}
					
					//
					// Update the argument with the implicit conversion
					//
					if (a_expr != conv)
						a.Expr = conv;
				}

				Parameter.Modifier a_mod = a.GetParameterModifier () &
					~(Parameter.Modifier.OUT | Parameter.Modifier.REF);
				Parameter.Modifier p_mod = pd.ParameterModifier (j) &
					~(Parameter.Modifier.OUT | Parameter.Modifier.REF);
				
				if (a_mod != p_mod &&
				    pd.ParameterModifier (pd_count - 1) != Parameter.Modifier.PARAMS) {
					if (!Location.IsNull (loc)) {
						Report.Error (1502, loc,
						       "The best overloaded match for method '" + FullMethodDesc (method)+
						       "' has some invalid arguments");
						Report.Error (1503, loc,
						       "Argument " + (j+1) +
						       ": Cannot convert from '" + Argument.FullDesc (a) 
						       + "' to '" + pd.ParameterDesc (j) + "'");
					}
					
					return false;
				}
			}

			return true;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// First, resolve the expression that is used to
			// trigger the invocation
			//
			if (expr is BaseAccess)
				is_base = true;

			expr = expr.Resolve (ec, ResolveFlags.VariableOrValue | ResolveFlags.MethodGroup);
			if (expr == null)
				return null;

			if (!(expr is MethodGroupExpr)) {
				Type expr_type = expr.Type;

				if (expr_type != null){
					bool IsDelegate = TypeManager.IsDelegateType (expr_type);
					if (IsDelegate)
						return (new DelegateInvocation (
							this.expr, Arguments, loc)).Resolve (ec);
				}
			}

			if (!(expr is MethodGroupExpr)){
				expr.Error_UnexpectedKind (ResolveFlags.MethodGroup);
				return null;
			}

			//
			// Next, evaluate all the expressions in the argument list
			//
			if (Arguments != null){
				foreach (Argument a in Arguments){
					if (!a.Resolve (ec, loc))
						return null;
				}
			}

			MethodGroupExpr mg = (MethodGroupExpr) expr;
			method = OverloadResolve (ec, mg, Arguments, loc);

			if (method == null){
				Error (-6,
				       "Could not find any applicable function for this argument list");
				return null;
			}
			
			MethodInfo mi = method as MethodInfo;
			if (mi != null) {
				type = TypeManager.TypeToCoreType (mi.ReturnType);
				if (!mi.IsStatic && !mg.IsExplicitImpl && (mg.InstanceExpression == null)) {
					SimpleName.Error_ObjectRefRequired (ec, loc, mi.Name);
					return null;
				}

				Expression iexpr = mg.InstanceExpression;
				if (mi.IsStatic && (iexpr != null) && !(iexpr is This)) {
					if (mg.IdenticalTypeName)
						mg.InstanceExpression = null;
					else {
						MemberAccess.error176 (loc, mi.Name);
						return null;
					}
				}
			}

			if (type.IsPointer){
				if (!ec.InUnsafe){
					UnsafeError (loc);
					return null;
				}
			}
			
			//
			// Only base will allow this invocation to happen.
			//
			if (is_base && method.IsAbstract){
				Report.Error (205, loc, "Cannot call an abstract base member: " +
					      FullMethodDesc (method));
				return null;
			}

			if ((method.Attributes & MethodAttributes.SpecialName) != 0){
				if (TypeManager.IsSpecialMethod (method))
					Report.Error (571, loc, method.Name + ": can not call operator or accessor");
			}

			eclass = ExprClass.Value;
			return this;
		}

		// <summary>
		//   Emits the list of arguments as an array
		// </summary>
		static void EmitParams (EmitContext ec, int idx, ArrayList arguments)
		{
			ILGenerator ig = ec.ig;
			int count = arguments.Count - idx;
			Argument a = (Argument) arguments [idx];
			Type t = a.Expr.Type;
			
			IntConstant.EmitInt (ig, count);
			ig.Emit (OpCodes.Newarr, TypeManager.TypeToCoreType (t));

			int top = arguments.Count;
			for (int j = idx; j < top; j++){
				a = (Argument) arguments [j];
				
				ig.Emit (OpCodes.Dup);
				IntConstant.EmitInt (ig, j - idx);

				bool is_stobj;
				OpCode op = ArrayAccess.GetStoreOpcode (t, out is_stobj);
				if (is_stobj)
					ig.Emit (OpCodes.Ldelema, t);

				a.Emit (ec);

				if (is_stobj)
					ig.Emit (OpCodes.Stobj, t);
				else
					ig.Emit (op);
			}
		}
		
		/// <summary>
		///   Emits a list of resolved Arguments that are in the arguments
		///   ArrayList.
		/// 
		///   The MethodBase argument might be null if the
		///   emission of the arguments is known not to contain
		///   a `params' field (for example in constructors or other routines
		///   that keep their arguments in this structure)
		/// </summary>
		public static void EmitArguments (EmitContext ec, MethodBase mb, ArrayList arguments)
		{
			ParameterData pd;
			if (mb != null)
				pd = GetParameterData (mb);
			else
				pd = null;

			//
			// If we are calling a params method with no arguments, special case it
			//
			if (arguments == null){
				if (pd != null && pd.Count > 0 &&
				    pd.ParameterModifier (0) == Parameter.Modifier.PARAMS){
					ILGenerator ig = ec.ig;

					IntConstant.EmitInt (ig, 0);
					ig.Emit (OpCodes.Newarr, TypeManager.GetElementType (pd.ParameterType (0)));
				}

				return;
			}

			int top = arguments.Count;

			for (int i = 0; i < top; i++){
				Argument a = (Argument) arguments [i];

				if (pd != null){
					if (pd.ParameterModifier (i) == Parameter.Modifier.PARAMS){
						//
						// Special case if we are passing the same data as the
						// params argument, do not put it in an array.
						//
						if (pd.ParameterType (i) == a.Type)
							a.Emit (ec);
						else
							EmitParams (ec, i, arguments);
						return;
					}
				}
					    
				a.Emit (ec);
			}

			if (pd != null && pd.Count > top &&
			    pd.ParameterModifier (top) == Parameter.Modifier.PARAMS){
				ILGenerator ig = ec.ig;

				IntConstant.EmitInt (ig, 0);
				ig.Emit (OpCodes.Newarr, TypeManager.GetElementType (pd.ParameterType (top)));
			}
		}

		static Type[] GetVarargsTypes (EmitContext ec, MethodBase mb,
					       ArrayList arguments)
		{
			ParameterData pd = GetParameterData (mb);

			if (arguments == null)
				return new Type [0];

			Argument a = (Argument) arguments [pd.Count - 1];
			Arglist list = (Arglist) a.Expr;

			return list.ArgumentTypes;
		}

		/// <summary>
		/// This checks the ConditionalAttribute on the method 
		/// </summary>
		static bool IsMethodExcluded (MethodBase method, EmitContext ec)
		{
			if (method.IsConstructor)
				return false;

			IMethodData md = TypeManager.GetMethod (method);
			if (md != null)
				return md.IsExcluded (ec);

			// For some methods (generated by delegate class) GetMethod returns null
			// because they are not included in builder_to_method table
			if (method.DeclaringType is TypeBuilder)
				return false;

			return AttributeTester.IsConditionalMethodExcluded (method);
		}

		/// <remarks>
		///   is_base tells whether we want to force the use of the `call'
		///   opcode instead of using callvirt.  Call is required to call
		///   a specific method, while callvirt will always use the most
		///   recent method in the vtable.
		///
		///   is_static tells whether this is an invocation on a static method
		///
		///   instance_expr is an expression that represents the instance
		///   it must be non-null if is_static is false.
		///
		///   method is the method to invoke.
		///
		///   Arguments is the list of arguments to pass to the method or constructor.
		/// </remarks>
		public static void EmitCall (EmitContext ec, bool is_base,
					     bool is_static, Expression instance_expr,
					     MethodBase method, ArrayList Arguments, Location loc)
		{
			ILGenerator ig = ec.ig;
			bool struct_call = false;
			bool this_call = false;

			Type decl_type = method.DeclaringType;

			if (!RootContext.StdLib) {
				// Replace any calls to the system's System.Array type with calls to
				// the newly created one.
				if (method == TypeManager.system_int_array_get_length)
					method = TypeManager.int_array_get_length;
				else if (method == TypeManager.system_int_array_get_rank)
					method = TypeManager.int_array_get_rank;
				else if (method == TypeManager.system_object_array_clone)
					method = TypeManager.object_array_clone;
				else if (method == TypeManager.system_int_array_get_length_int)
					method = TypeManager.int_array_get_length_int;
				else if (method == TypeManager.system_int_array_get_lower_bound_int)
					method = TypeManager.int_array_get_lower_bound_int;
				else if (method == TypeManager.system_int_array_get_upper_bound_int)
					method = TypeManager.int_array_get_upper_bound_int;
				else if (method == TypeManager.system_void_array_copyto_array_int)
					method = TypeManager.void_array_copyto_array_int;
			}

			//
			// This checks ObsoleteAttribute on the method and on the declaring type
			//
			ObsoleteAttribute oa = AttributeTester.GetMethodObsoleteAttribute (method);
			if (oa != null)
				AttributeTester.Report_ObsoleteMessage (oa, TypeManager.CSharpSignature (method), loc);


			oa = AttributeTester.GetObsoleteAttribute (method.DeclaringType);
			if (oa != null) {
				AttributeTester.Report_ObsoleteMessage (oa, method.DeclaringType.FullName, loc);
			}

			if (IsMethodExcluded (method, ec))
                return; 
			
			if (!is_static){
				if (decl_type.IsValueType)
					struct_call = true;
				//
				// If this is ourselves, push "this"
				//
				if (instance_expr == null) {
					this_call = true;
					ig.Emit (OpCodes.Ldarg_0);
				} else {
					//
					// Push the instance expression
					//
					if (instance_expr.Type.IsValueType){
						//
						// Special case: calls to a function declared in a 
						// reference-type with a value-type argument need
						// to have their value boxed.  

						struct_call = true;
						if (decl_type.IsValueType){
							//
							// If the expression implements IMemoryLocation, then
							// we can optimize and use AddressOf on the
							// return.
							//
							// If not we have to use some temporary storage for
							// it.
							if (instance_expr is IMemoryLocation){
								((IMemoryLocation)instance_expr).
									AddressOf (ec, AddressOp.LoadStore);
							}
							else {
								Type t = instance_expr.Type;
								
								instance_expr.Emit (ec);
								LocalBuilder temp = ig.DeclareLocal (t);
								ig.Emit (OpCodes.Stloc, temp);
								ig.Emit (OpCodes.Ldloca, temp);
							}
						} else {
							instance_expr.Emit (ec);
							ig.Emit (OpCodes.Box, instance_expr.Type);
						} 
					} else
						instance_expr.Emit (ec);
				}
			}

			EmitArguments (ec, method, Arguments);

			OpCode call_op;
			if (is_static || struct_call || is_base || (this_call && !method.IsVirtual))
				call_op = OpCodes.Call;
			else
				call_op = OpCodes.Callvirt;

			if ((method.CallingConvention & CallingConventions.VarArgs) != 0) {
				Type[] varargs_types = GetVarargsTypes (ec, method, Arguments);
				ig.EmitCall (call_op, (MethodInfo) method, varargs_types);
				return;
			}

			//
			// If you have:
			// this.DoFoo ();
			// and DoFoo is not virtual, you can omit the callvirt,
			// because you don't need the null checking behavior.
			//
			if (method is MethodInfo)
				ig.Emit (call_op, (MethodInfo) method);
			else
				ig.Emit (call_op, (ConstructorInfo) method);
		}
		
		public override void Emit (EmitContext ec)
		{
			MethodGroupExpr mg = (MethodGroupExpr) this.expr;

			EmitCall (ec, is_base, method.IsStatic, mg.InstanceExpression, method, Arguments, loc);
		}
		
		public override void EmitStatement (EmitContext ec)
		{
			Emit (ec);

			// 
			// Pop the return value if there is one
			//
			if (method is MethodInfo){
				Type ret = ((MethodInfo)method).ReturnType;
				if (TypeManager.TypeToCoreType (ret) != TypeManager.void_type)
					ec.ig.Emit (OpCodes.Pop);
			}
		}
	}

	public class InvocationOrCast : ExpressionStatement
	{
		Expression expr;
		Expression argument;

		public InvocationOrCast (Expression expr, Expression argument, Location loc)
		{
			this.expr = expr;
			this.argument = argument;
			this.loc = loc;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// First try to resolve it as a cast.
			//
			type = ec.DeclSpace.ResolveType (expr, true, loc);
			if (type != null) {
				Cast cast = new Cast (new TypeExpression (type, loc), argument, loc);
				return cast.Resolve (ec);
			}

			//
			// This can either be a type or a delegate invocation.
			// Let's just resolve it and see what we'll get.
			//
			expr = expr.Resolve (ec, ResolveFlags.Type | ResolveFlags.VariableOrValue);
			if (expr == null)
				return null;

			//
			// Ok, so it's a Cast.
			//
			if (expr.eclass == ExprClass.Type) {
				Cast cast = new Cast (new TypeExpression (expr.Type, loc), argument, loc);
				return cast.Resolve (ec);
			}

			//
			// It's a delegate invocation.
			//
			if (!TypeManager.IsDelegateType (expr.Type)) {
				Error (149, "Method name expected");
				return null;
			}

			ArrayList args = new ArrayList ();
			args.Add (new Argument (argument, Argument.AType.Expression));
			DelegateInvocation invocation = new DelegateInvocation (expr, args, loc);
			return invocation.Resolve (ec);
		}

		void error201 ()
		{
			Error (201, "Only assignment, call, increment, decrement and new object " +
			       "expressions can be used as a statement");
		}

		public override ExpressionStatement ResolveStatement (EmitContext ec)
		{
			//
			// First try to resolve it as a cast.
			//
			type = ec.DeclSpace.ResolveType (expr, true, loc);
			if (type != null) {
				error201 ();
				return null;
			}

			//
			// This can either be a type or a delegate invocation.
			// Let's just resolve it and see what we'll get.
			//
			expr = expr.Resolve (ec, ResolveFlags.Type | ResolveFlags.VariableOrValue);
			if ((expr == null) || (expr.eclass == ExprClass.Type)) {
				error201 ();
				return null;
			}

			//
			// It's a delegate invocation.
			//
			if (!TypeManager.IsDelegateType (expr.Type)) {
				Error (149, "Method name expected");
				return null;
			}

			ArrayList args = new ArrayList ();
			args.Add (new Argument (argument, Argument.AType.Expression));
			DelegateInvocation invocation = new DelegateInvocation (expr, args, loc);
			return invocation.ResolveStatement (ec);
		}

		public override void Emit (EmitContext ec)
		{
			throw new Exception ("Cannot happen");
		}

		public override void EmitStatement (EmitContext ec)
		{
			throw new Exception ("Cannot happen");
		}
	}

	//
	// This class is used to "disable" the code generation for the
	// temporary variable when initializing value types.
	//
	class EmptyAddressOf : EmptyExpression, IMemoryLocation {
		public void AddressOf (EmitContext ec, AddressOp Mode)
		{
			// nothing
		}
	}
	
	/// <summary>
	///    Implements the new expression 
	/// </summary>
	public class New : ExpressionStatement, IMemoryLocation {
		public readonly ArrayList Arguments;

		//
		// During bootstrap, it contains the RequestedType,
		// but if `type' is not null, it *might* contain a NewDelegate
		// (because of field multi-initialization)
		//
		public Expression RequestedType;

		MethodBase method = null;

		//
		// If set, the new expression is for a value_target, and
		// we will not leave anything on the stack.
		//
		Expression value_target;
		bool value_target_set = false;
		
		public New (Expression requested_type, ArrayList arguments, Location l)
		{
			RequestedType = requested_type;
			Arguments = arguments;
			loc = l;
		}

		public bool SetValueTypeVariable (Expression value)
		{
			value_target = value;
			value_target_set = true;
			if (!(value_target is IMemoryLocation)){
				Error_UnexpectedKind ("variable");
				return false;
			}
			return true;
		}

		//
		// This function is used to disable the following code sequence for
		// value type initialization:
		//
		// AddressOf (temporary)
		// Construct/Init
		// LoadTemporary
		//
		// Instead the provide will have provided us with the address on the
		// stack to store the results.
		//
		static Expression MyEmptyExpression;
		
		public void DisableTemporaryValueType ()
		{
			if (MyEmptyExpression == null)
				MyEmptyExpression = new EmptyAddressOf ();

			//
			// To enable this, look into:
			// test-34 and test-89 and self bootstrapping.
			//
			// For instance, we can avoid a copy by using `newobj'
			// instead of Call + Push-temp on value types.
//			value_target = MyEmptyExpression;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// The New DoResolve might be called twice when initializing field
			// expressions (see EmitFieldInitializers, the call to
			// GetInitializerExpression will perform a resolve on the expression,
			// and later the assign will trigger another resolution
			//
			// This leads to bugs (#37014)
			//
			if (type != null){
				if (RequestedType is NewDelegate)
					return RequestedType;
				return this;
			}
			
			type = ec.DeclSpace.ResolveType (RequestedType, false, loc);
			
			if (type == null)
				return null;
			
			CheckObsoleteAttribute (type);

			bool IsDelegate = TypeManager.IsDelegateType (type);
			
			if (IsDelegate){
				RequestedType = (new NewDelegate (type, Arguments, loc)).Resolve (ec);
				if (RequestedType != null)
					if (!(RequestedType is NewDelegate))
						throw new Exception ("NewDelegate.Resolve returned a non NewDelegate: " + RequestedType.GetType ());
				return RequestedType;
			}

			if (type.IsInterface || type.IsAbstract){
				Error (144, "It is not possible to create instances of interfaces or abstract classes");
				return null;
			}
			
			bool is_struct = type.IsValueType;
			eclass = ExprClass.Value;

			//
			// SRE returns a match for .ctor () on structs (the object constructor), 
			// so we have to manually ignore it.
			//
			if (is_struct && Arguments == null)
				return this;
			
			Expression ml;
			// For member-lookup, treat 'new Foo (bar)' as call to 'foo.ctor (bar)', where 'foo' is of type 'Foo'.
			ml = MemberLookupFinal (ec, type, type, ".ctor",
						MemberTypes.Constructor,
						AllBindingFlags | BindingFlags.DeclaredOnly, loc);

			if (ml == null)
				return null;
			
			if (! (ml is MethodGroupExpr)){
				if (!is_struct){
					ml.Error_UnexpectedKind ("method group");
					return null;
				}
			}

			if (ml != null) {
				if (Arguments != null){
					foreach (Argument a in Arguments){
						if (!a.Resolve (ec, loc))
							return null;
					}
				}

				method = Invocation.OverloadResolve (ec, (MethodGroupExpr) ml, Arguments, loc);
				
			}

			if (method == null) { 
                                if (!is_struct || Arguments.Count > 0) {
        				Error (1501, String.Format (
					    "New invocation: Can not find a constructor in `{0}' for this argument list",
					    TypeManager.CSharpName (type)));
					return null;
                                }
			}

			return this;
		}

		//
		// This DoEmit can be invoked in two contexts:
		//    * As a mechanism that will leave a value on the stack (new object)
		//    * As one that wont (init struct)
		//
		// You can control whether a value is required on the stack by passing
		// need_value_on_stack.  The code *might* leave a value on the stack
		// so it must be popped manually
		//
		// If we are dealing with a ValueType, we have a few
		// situations to deal with:
		//
		//    * The target is a ValueType, and we have been provided
		//      the instance (this is easy, we are being assigned).
		//
		//    * The target of New is being passed as an argument,
		//      to a boxing operation or a function that takes a
		//      ValueType.
		//
		//      In this case, we need to create a temporary variable
		//      that is the argument of New.
		//
		// Returns whether a value is left on the stack
		//
		bool DoEmit (EmitContext ec, bool need_value_on_stack)
		{
			bool is_value_type = type.IsValueType;
			ILGenerator ig = ec.ig;

			if (is_value_type){
				IMemoryLocation ml;

				// Allow DoEmit() to be called multiple times.
				// We need to create a new LocalTemporary each time since
				// you can't share LocalBuilders among ILGeneators.
				if (!value_target_set)
					value_target = new LocalTemporary (ec, type);

				ml = (IMemoryLocation) value_target;
				ml.AddressOf (ec, AddressOp.Store);
			}

			if (method != null)
				Invocation.EmitArguments (ec, method, Arguments);

			if (is_value_type){
				if (method == null)
					ig.Emit (OpCodes.Initobj, type);
				else 
					ig.Emit (OpCodes.Call, (ConstructorInfo) method);
                                if (need_value_on_stack){
                                        value_target.Emit (ec);
                                        return true;
                                }
                                return false;
			} else {
				ig.Emit (OpCodes.Newobj, (ConstructorInfo) method);
				return true;
			}
		}

		public override void Emit (EmitContext ec)
		{
			DoEmit (ec, true);
		}
		
		public override void EmitStatement (EmitContext ec)
		{
			if (DoEmit (ec, false))
				ec.ig.Emit (OpCodes.Pop);
		}

		public void AddressOf (EmitContext ec, AddressOp Mode)
		{
			if (!type.IsValueType){
				//
				// We throw an exception.  So far, I believe we only need to support
				// value types:
				// foreach (int j in new StructType ())
				// see bug 42390
				//
				throw new Exception ("AddressOf should not be used for classes");
			}

			if (!value_target_set)
				value_target = new LocalTemporary (ec, type);
					
			IMemoryLocation ml = (IMemoryLocation) value_target;
			ml.AddressOf (ec, AddressOp.Store);
			if (method != null)
				Invocation.EmitArguments (ec, method, Arguments);

			if (method == null)
				ec.ig.Emit (OpCodes.Initobj, type);
			else 
				ec.ig.Emit (OpCodes.Call, (ConstructorInfo) method);
			
			((IMemoryLocation) value_target).AddressOf (ec, Mode);
		}
	}

	/// <summary>
	///   14.5.10.2: Represents an array creation expression.
	/// </summary>
	///
	/// <remarks>
	///   There are two possible scenarios here: one is an array creation
	///   expression that specifies the dimensions and optionally the
	///   initialization data and the other which does not need dimensions
	///   specified but where initialization data is mandatory.
	/// </remarks>
	public class ArrayCreation : Expression {
		Expression requested_base_type;
		ArrayList initializers;

		//
		// The list of Argument types.
		// This is used to construct the `newarray' or constructor signature
		//
		ArrayList arguments;

		//
		// Method used to create the array object.
		//
		MethodBase new_method = null;
		
		Type array_element_type;
		Type underlying_type;
		bool is_one_dimensional = false;
		bool is_builtin_type = false;
		bool expect_initializers = false;
		int num_arguments = 0;
		int dimensions = 0;
		string rank;

		ArrayList array_data;

		Hashtable bounds;

		//
		// The number of array initializers that we can handle
		// via the InitializeArray method - through EmitStaticInitializers
		//
		int num_automatic_initializers;

		const int max_automatic_initializers = 6;
		
		public ArrayCreation (Expression requested_base_type, ArrayList exprs, string rank, ArrayList initializers, Location l)
		{
			this.requested_base_type = requested_base_type;
			this.initializers = initializers;
			this.rank = rank;
			loc = l;

			arguments = new ArrayList ();

			foreach (Expression e in exprs) {
				arguments.Add (new Argument (e, Argument.AType.Expression));
				num_arguments++;
			}
		}

		public ArrayCreation (Expression requested_base_type, string rank, ArrayList initializers, Location l)
		{
			this.requested_base_type = requested_base_type;
			this.initializers = initializers;
			this.rank = rank;
			loc = l;

			//this.rank = rank.Substring (0, rank.LastIndexOf ('['));
			//
			//string tmp = rank.Substring (rank.LastIndexOf ('['));
			//
			//dimensions = tmp.Length - 1;
			expect_initializers = true;
		}

		public Expression FormArrayType (Expression base_type, int idx_count, string rank)
		{
			StringBuilder sb = new StringBuilder (rank);
			
			sb.Append ("[");
			for (int i = 1; i < idx_count; i++)
				sb.Append (",");
			
			sb.Append ("]");

			return new ComposedCast (base_type, sb.ToString (), loc);
		}

		void Error_IncorrectArrayInitializer ()
		{
			Error (178, "Incorrectly structured array initializer");
		}
		
		public bool CheckIndices (EmitContext ec, ArrayList probe, int idx, bool specified_dims)
		{
			if (specified_dims) { 
				Argument a = (Argument) arguments [idx];
				
				if (!a.Resolve (ec, loc))
					return false;
				
				if (!(a.Expr is Constant)) {
					Error (150, "A constant value is expected");
					return false;
				}
				
				int value = (int) ((Constant) a.Expr).GetValue ();
				
				if (value != probe.Count) {
					Error_IncorrectArrayInitializer ();
					return false;
				}
				
				bounds [idx] = value;
			}

			int child_bounds = -1;
			foreach (object o in probe) {
				if (o is ArrayList) {
					int current_bounds = ((ArrayList) o).Count;
					
					if (child_bounds == -1) 
						child_bounds = current_bounds;

					else if (child_bounds != current_bounds){
						Error_IncorrectArrayInitializer ();
						return false;
					}
					if (specified_dims && (idx + 1 >= arguments.Count)){
						Error (623, "Array initializers can only be used in a variable or field initializer, try using the new expression");
						return false;
					}
					
					bool ret = CheckIndices (ec, (ArrayList) o, idx + 1, specified_dims);
					if (!ret)
						return false;
				} else {
					if (child_bounds != -1){
						Error_IncorrectArrayInitializer ();
						return false;
					}
					
					Expression tmp = (Expression) o;
					tmp = tmp.Resolve (ec);
					if (tmp == null)
						continue;

					// Console.WriteLine ("I got: " + tmp);
					// Handle initialization from vars, fields etc.

					Expression conv = Convert.ImplicitConversionRequired (
						ec, tmp, underlying_type, loc);
					
					if (conv == null) 
						return false;
					
					if (conv is StringConstant || conv is DecimalConstant || conv is NullCast) {
						// These are subclasses of Constant that can appear as elements of an
						// array that cannot be statically initialized (with num_automatic_initializers
						// > max_automatic_initializers), so num_automatic_initializers should be left as zero.
						array_data.Add (conv);
					} else if (conv is Constant) {
						// These are the types of Constant that can appear in arrays that can be
						// statically allocated.
						array_data.Add (conv);
						num_automatic_initializers++;
					} else
						array_data.Add (conv);
				}
			}

			return true;
		}
		
		public void UpdateIndices (EmitContext ec)
		{
			int i = 0;
			for (ArrayList probe = initializers; probe != null;) {
				if (probe.Count > 0 && probe [0] is ArrayList) {
					Expression e = new IntConstant (probe.Count);
					arguments.Add (new Argument (e, Argument.AType.Expression));

					bounds [i++] =  probe.Count;
					
					probe = (ArrayList) probe [0];
					
				} else {
					Expression e = new IntConstant (probe.Count);
					arguments.Add (new Argument (e, Argument.AType.Expression));

					bounds [i++] = probe.Count;
					probe = null;
				}
			}

		}
		
		public bool ValidateInitializers (EmitContext ec, Type array_type)
		{
			if (initializers == null) {
				if (expect_initializers)
					return false;
				else
					return true;
			}
			
			if (underlying_type == null)
				return false;
			
			//
			// We use this to store all the date values in the order in which we
			// will need to store them in the byte blob later
			//
			array_data = new ArrayList ();
			bounds = new Hashtable ();
			
			bool ret;

			if (arguments != null) {
				ret = CheckIndices (ec, initializers, 0, true);
				return ret;
			} else {
				arguments = new ArrayList ();

				ret = CheckIndices (ec, initializers, 0, false);
				
				if (!ret)
					return false;
				
				UpdateIndices (ec);
				
				if (arguments.Count != dimensions) {
					Error_IncorrectArrayInitializer ();
					return false;
				}

				return ret;
			}
		}

		void Error_NegativeArrayIndex ()
		{
			Error (284, "Can not create array with a negative size");
		}
		
		//
		// Converts `source' to an int, uint, long or ulong.
		//
		Expression ExpressionToArrayArgument (EmitContext ec, Expression source)
		{
			Expression target;
			
			bool old_checked = ec.CheckState;
			ec.CheckState = true;
			
			target = Convert.ImplicitConversion (ec, source, TypeManager.int32_type, loc);
			if (target == null){
				target = Convert.ImplicitConversion (ec, source, TypeManager.uint32_type, loc);
				if (target == null){
					target = Convert.ImplicitConversion (ec, source, TypeManager.int64_type, loc);
					if (target == null){
						target = Convert.ImplicitConversion (ec, source, TypeManager.uint64_type, loc);
						if (target == null)
							Convert.Error_CannotImplicitConversion (loc, source.Type, TypeManager.int32_type);
					}
				}
			} 
			ec.CheckState = old_checked;

			//
			// Only positive constants are allowed at compile time
			//
			if (target is Constant){
				if (target is IntConstant){
					if (((IntConstant) target).Value < 0){
						Error_NegativeArrayIndex ();
						return null;
					}
				}

				if (target is LongConstant){
					if (((LongConstant) target).Value < 0){
						Error_NegativeArrayIndex ();
						return null;
					}
				}
				
			}

			return target;
		}

		//
		// Creates the type of the array
		//
		bool LookupType (EmitContext ec)
		{
			StringBuilder array_qualifier = new StringBuilder (rank);

			//
			// `In the first form allocates an array instace of the type that results
			// from deleting each of the individual expression from the expression list'
			//
			if (num_arguments > 0) {
				array_qualifier.Append ("[");
				for (int i = num_arguments-1; i > 0; i--)
					array_qualifier.Append (",");
				array_qualifier.Append ("]");				
			}

			//
			// Lookup the type
			//
			Expression array_type_expr;
			array_type_expr = new ComposedCast (requested_base_type, array_qualifier.ToString (), loc);
			type = ec.DeclSpace.ResolveType (array_type_expr, false, loc);

			if (type == null)
				return false;

			underlying_type = type;
			if (underlying_type.IsArray)
				underlying_type = TypeManager.GetElementType (underlying_type);
			dimensions = type.GetArrayRank ();

			return true;
		}
		
		public override Expression DoResolve (EmitContext ec)
		{
			int arg_count;

			if (!LookupType (ec))
				return null;
			
			//
			// First step is to validate the initializers and fill
			// in any missing bits
			//
			if (!ValidateInitializers (ec, type))
				return null;

			if (arguments == null)
				arg_count = 0;
			else {
				arg_count = arguments.Count;
				foreach (Argument a in arguments){
					if (!a.Resolve (ec, loc))
						return null;

					Expression real_arg = ExpressionToArrayArgument (ec, a.Expr, loc);
					if (real_arg == null)
						return null;

					a.Expr = real_arg;
				}
			}
			
			array_element_type = TypeManager.GetElementType (type);

			if (arg_count == 1) {
				is_one_dimensional = true;
				eclass = ExprClass.Value;
				return this;
			}

			is_builtin_type = TypeManager.IsBuiltinType (type);

			if (is_builtin_type) {
				Expression ml;
				
				ml = MemberLookup (ec, type, ".ctor", MemberTypes.Constructor,
						   AllBindingFlags, loc);
				
				if (!(ml is MethodGroupExpr)) {
					ml.Error_UnexpectedKind ("method group");
					return null;
				}
				
				if (ml == null) {
					Error (-6, "New invocation: Can not find a constructor for " +
						      "this argument list");
					return null;
				}
				
				new_method = Invocation.OverloadResolve (ec, (MethodGroupExpr) ml, arguments, loc);

				if (new_method == null) {
					Error (-6, "New invocation: Can not find a constructor for " +
						      "this argument list");
					return null;
				}
				
				eclass = ExprClass.Value;
				return this;
			} else {
				ModuleBuilder mb = CodeGen.Module.Builder;
				ArrayList args = new ArrayList ();
				
				if (arguments != null) {
					for (int i = 0; i < arg_count; i++)
						args.Add (TypeManager.int32_type);
				}
				
				Type [] arg_types = null;

				if (args.Count > 0)
					arg_types = new Type [args.Count];
				
				args.CopyTo (arg_types, 0);
				
				new_method = mb.GetArrayMethod (type, ".ctor", CallingConventions.HasThis, null,
							    arg_types);

				if (new_method == null) {
					Error (-6, "New invocation: Can not find a constructor for " +
						      "this argument list");
					return null;
				}
				
				eclass = ExprClass.Value;
				return this;
			}
		}

		public static byte [] MakeByteBlob (ArrayList array_data, Type underlying_type, Location loc)
		{
			int factor;
			byte [] data;
			byte [] element;
			int count = array_data.Count;

			if (underlying_type.IsEnum)
				underlying_type = TypeManager.EnumToUnderlying (underlying_type);
			
			factor = GetTypeSize (underlying_type);
			if (factor == 0)
				throw new Exception ("unrecognized type in MakeByteBlob: " + underlying_type);

			data = new byte [(count * factor + 4) & ~3];
			int idx = 0;
			
			for (int i = 0; i < count; ++i) {
				object v = array_data [i];

				if (v is EnumConstant)
					v = ((EnumConstant) v).Child;
				
				if (v is Constant && !(v is StringConstant))
					v = ((Constant) v).GetValue ();
				else {
					idx += factor;
					continue;
				}
				
				if (underlying_type == TypeManager.int64_type){
					if (!(v is Expression)){
						long val = (long) v;
						
						for (int j = 0; j < factor; ++j) {
							data [idx + j] = (byte) (val & 0xFF);
							val = (val >> 8);
						}
					}
				} else if (underlying_type == TypeManager.uint64_type){
					if (!(v is Expression)){
						ulong val = (ulong) v;

						for (int j = 0; j < factor; ++j) {
							data [idx + j] = (byte) (val & 0xFF);
							val = (val >> 8);
						}
					}
				} else if (underlying_type == TypeManager.float_type) {
					if (!(v is Expression)){
						element = BitConverter.GetBytes ((float) v);
							
						for (int j = 0; j < factor; ++j)
							data [idx + j] = element [j];
					}
				} else if (underlying_type == TypeManager.double_type) {
					if (!(v is Expression)){
						element = BitConverter.GetBytes ((double) v);

						for (int j = 0; j < factor; ++j)
							data [idx + j] = element [j];
					}
				} else if (underlying_type == TypeManager.char_type){
					if (!(v is Expression)){
						int val = (int) ((char) v);
						
						data [idx] = (byte) (val & 0xff);
						data [idx+1] = (byte) (val >> 8);
					}
				} else if (underlying_type == TypeManager.short_type){
					if (!(v is Expression)){
						int val = (int) ((short) v);
					
						data [idx] = (byte) (val & 0xff);
						data [idx+1] = (byte) (val >> 8);
					}
				} else if (underlying_type == TypeManager.ushort_type){
					if (!(v is Expression)){
						int val = (int) ((ushort) v);
					
						data [idx] = (byte) (val & 0xff);
						data [idx+1] = (byte) (val >> 8);
					}
				} else if (underlying_type == TypeManager.int32_type) {
					if (!(v is Expression)){
						int val = (int) v;
					
						data [idx]   = (byte) (val & 0xff);
						data [idx+1] = (byte) ((val >> 8) & 0xff);
						data [idx+2] = (byte) ((val >> 16) & 0xff);
						data [idx+3] = (byte) (val >> 24);
					}
				} else if (underlying_type == TypeManager.uint32_type) {
					if (!(v is Expression)){
						uint val = (uint) v;
					
						data [idx]   = (byte) (val & 0xff);
						data [idx+1] = (byte) ((val >> 8) & 0xff);
						data [idx+2] = (byte) ((val >> 16) & 0xff);
						data [idx+3] = (byte) (val >> 24);
					}
				} else if (underlying_type == TypeManager.sbyte_type) {
					if (!(v is Expression)){
						sbyte val = (sbyte) v;
						data [idx] = (byte) val;
					}
				} else if (underlying_type == TypeManager.byte_type) {
					if (!(v is Expression)){
						byte val = (byte) v;
						data [idx] = (byte) val;
					}
				} else if (underlying_type == TypeManager.bool_type) {
					if (!(v is Expression)){
						bool val = (bool) v;
						data [idx] = (byte) (val ? 1 : 0);
					}
				} else if (underlying_type == TypeManager.decimal_type){
					if (!(v is Expression)){
						int [] bits = Decimal.GetBits ((decimal) v);
						int p = idx;

						// FIXME: For some reason, this doesn't work on the MS runtime.
						int [] nbits = new int [4];
						nbits [0] = bits [3];
						nbits [1] = bits [2];
						nbits [2] = bits [0];
						nbits [3] = bits [1];
						
						for (int j = 0; j < 4; j++){
							data [p++] = (byte) (nbits [j] & 0xff);
							data [p++] = (byte) ((nbits [j] >> 8) & 0xff);
							data [p++] = (byte) ((nbits [j] >> 16) & 0xff);
							data [p++] = (byte) (nbits [j] >> 24);
						}
					}
				} else
					throw new Exception ("Unrecognized type in MakeByteBlob: " + underlying_type);

                                idx += factor;
			}

			return data;
		}

		//
		// Emits the initializers for the array
		//
		void EmitStaticInitializers (EmitContext ec)
		{
			//
			// First, the static data
			//
			FieldBuilder fb;
			ILGenerator ig = ec.ig;
			
			byte [] data = MakeByteBlob (array_data, underlying_type, loc);

			fb = RootContext.MakeStaticData (data);

			ig.Emit (OpCodes.Dup);
			ig.Emit (OpCodes.Ldtoken, fb);
			ig.Emit (OpCodes.Call,
				 TypeManager.void_initializearray_array_fieldhandle);
		}

		//
		// Emits pieces of the array that can not be computed at compile
		// time (variables and string locations).
		//
		// This always expect the top value on the stack to be the array
		//
		void EmitDynamicInitializers (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			int dims = bounds.Count;
			int [] current_pos = new int [dims];
			int top = array_data.Count;

			MethodInfo set = null;

			if (dims != 1){
				Type [] args;
				ModuleBuilder mb = null;
				mb = CodeGen.Module.Builder;
				args = new Type [dims + 1];

				int j;
				for (j = 0; j < dims; j++)
					args [j] = TypeManager.int32_type;

				args [j] = array_element_type;
				
				set = mb.GetArrayMethod (
					type, "Set",
					CallingConventions.HasThis | CallingConventions.Standard,
					TypeManager.void_type, args);
			}
			
			for (int i = 0; i < top; i++){

				Expression e = null;

				if (array_data [i] is Expression)
					e = (Expression) array_data [i];

				if (e != null) {
					//
					// Basically we do this for string literals and
					// other non-literal expressions
					//
					if (e is EnumConstant){
						e = ((EnumConstant) e).Child;
					}
					
					if (e is StringConstant || e is DecimalConstant || !(e is Constant) ||
					    num_automatic_initializers <= max_automatic_initializers) {
						Type etype = e.Type;
						
						ig.Emit (OpCodes.Dup);

						for (int idx = 0; idx < dims; idx++) 
							IntConstant.EmitInt (ig, current_pos [idx]);

						//
						// If we are dealing with a struct, get the
						// address of it, so we can store it.
						//
						if ((dims == 1) && 
						    etype.IsSubclassOf (TypeManager.value_type) &&
						    (!TypeManager.IsBuiltinOrEnum (etype) ||
						     etype == TypeManager.decimal_type)) {
							if (e is New){
								New n = (New) e;

								//
								// Let new know that we are providing
								// the address where to store the results
								//
								n.DisableTemporaryValueType ();
							}

							ig.Emit (OpCodes.Ldelema, etype);
						}

						e.Emit (ec);

						if (dims == 1)
							ArrayAccess.EmitStoreOpcode (ig, array_element_type);
						else 
							ig.Emit (OpCodes.Call, set);

					}
				}
				
				//
				// Advance counter
				//
				for (int j = dims - 1; j >= 0; j--){
					current_pos [j]++;
					if (current_pos [j] < (int) bounds [j])
						break;
					current_pos [j] = 0;
				}
			}
		}

		void EmitArrayArguments (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			foreach (Argument a in arguments) {
				Type atype = a.Type;
				a.Emit (ec);

				if (atype == TypeManager.uint64_type)
					ig.Emit (OpCodes.Conv_Ovf_U4);
				else if (atype == TypeManager.int64_type)
					ig.Emit (OpCodes.Conv_Ovf_I4);
			}
		}
		
		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			EmitArrayArguments (ec);
			if (is_one_dimensional)
				ig.Emit (OpCodes.Newarr, array_element_type);
			else {
				if (is_builtin_type) 
					ig.Emit (OpCodes.Newobj, (ConstructorInfo) new_method);
				else 
					ig.Emit (OpCodes.Newobj, (MethodInfo) new_method);
			}
			
			if (initializers != null){
				//
				// FIXME: Set this variable correctly.
				// 
				bool dynamic_initializers = true;

				// This will never be true for array types that cannot be statically
				// initialized. num_automatic_initializers will always be zero.  See
				// CheckIndices.
				if (num_automatic_initializers > max_automatic_initializers)
					EmitStaticInitializers (ec);
				
				if (dynamic_initializers)
					EmitDynamicInitializers (ec);
			}
		}

		public object EncodeAsAttribute ()
		{
			if (!is_one_dimensional){
				Report.Error (-211, Location, "attribute can not encode multi-dimensional arrays");
				return null;
			}

			if (array_data == null){
				Report.Error (-212, Location, "array should be initialized when passing it to an attribute");
				return null;
			}
			
			object [] ret = new object [array_data.Count];
			int i = 0;
			foreach (Expression e in array_data){
				object v;
				
				if (e is NullLiteral)
					v = null;
				else {
					if (!Attribute.GetAttributeArgumentExpression (e, Location, out v))
						return null;
				}
				ret [i++] = v;
			}
			return ret;
		}

		public Expression TurnIntoConstant ()
		{
			//
			// Should use something like the above attribute thing.
			// It should return a subclass of Constant that just returns
			// the computed value of the array
			//
			throw new Exception ("Does not support yet Turning array into a Constant");
		}
	}
	
	/// <summary>
	///   Represents the `this' construct
	/// </summary>
	public class This : Expression, IAssignMethod, IMemoryLocation, IVariable {

		Block block;
		VariableInfo variable_info;
		
		public This (Block block, Location loc)
		{
			this.loc = loc;
			this.block = block;
		}

		public This (Location loc)
		{
			this.loc = loc;
		}

		public VariableInfo VariableInfo {
			get { return variable_info; }
		}

		public bool VerifyFixed (bool is_expression)
		{
			if ((variable_info == null) || (variable_info.LocalInfo == null))
				return false;
			else
				return variable_info.LocalInfo.IsFixed;
		}

		public bool ResolveBase (EmitContext ec)
		{
			eclass = ExprClass.Variable;
			type = ec.ContainerType;

			if (ec.IsStatic) {
				Error (26, "Keyword this not valid in static code");
				return false;
			}

			if ((block != null) && (block.ThisVariable != null))
				variable_info = block.ThisVariable.VariableInfo;

			return true;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			if (!ResolveBase (ec))
				return null;

			if ((variable_info != null) && !variable_info.IsAssigned (ec)) {
				Error (188, "The this object cannot be used before all " +
				       "of its fields are assigned to");
				variable_info.SetAssigned (ec);
				return this;
			}

			if (ec.IsFieldInitializer) {
				Error (27, "Keyword `this' can't be used outside a constructor, " +
				       "a method or a property.");
				return null;
			}

			return this;
		}

		override public Expression DoResolveLValue (EmitContext ec, Expression right_side)
		{
			if (!ResolveBase (ec))
				return null;

			if (variable_info != null)
				variable_info.SetAssigned (ec);
			
			if (ec.TypeContainer is Class){
				Error (1604, "Cannot assign to `this'");
				return null;
			}

			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			ec.EmitThis ();
			if (ec.TypeContainer is Struct)
				ig.Emit (OpCodes.Ldobj, type);
		}

		public void EmitAssign (EmitContext ec, Expression source)
		{
			ILGenerator ig = ec.ig;
			
			if (ec.TypeContainer is Struct){
				ec.EmitThis ();
				source.Emit (ec);
				ig.Emit (OpCodes.Stobj, type);
			} else {
				source.Emit (ec);
				ig.Emit (OpCodes.Starg, 0);
			}
		}

		public void AddressOf (EmitContext ec, AddressOp mode)
		{
			ec.EmitThis ();

			// FIMXE
			// FIGURE OUT WHY LDARG_S does not work
			//
			// consider: struct X { int val; int P { set { val = value; }}}
			//
			// Yes, this looks very bad. Look at `NOTAS' for
			// an explanation.
			// ec.ig.Emit (OpCodes.Ldarga_S, (byte) 0);
		}
	}

	/// <summary>
	///   Represents the `__arglist' construct
	/// </summary>
	public class ArglistAccess : Expression
	{
		public ArglistAccess (Location loc)
		{
			this.loc = loc;
		}

		public bool ResolveBase (EmitContext ec)
		{
			eclass = ExprClass.Variable;
			type = TypeManager.runtime_argument_handle_type;
			return true;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			if (!ResolveBase (ec))
				return null;

			if (ec.IsFieldInitializer || !ec.CurrentBlock.HasVarargs) {
				Error (190, "The __arglist construct is valid only within " +
				       "a variable argument method.");
				return null;
			}

			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ec.ig.Emit (OpCodes.Arglist);
		}
	}

	/// <summary>
	///   Represents the `__arglist (....)' construct
	/// </summary>
	public class Arglist : Expression
	{
		public readonly Argument[] Arguments;

		public Arglist (Argument[] args, Location l)
		{
			Arguments = args;
			loc = l;
		}

		public Type[] ArgumentTypes {
			get {
				Type[] retval = new Type [Arguments.Length];
				for (int i = 0; i < Arguments.Length; i++)
					retval [i] = Arguments [i].Type;
				return retval;
			}
		}

		public override Expression DoResolve (EmitContext ec)
		{
			eclass = ExprClass.Variable;
			type = TypeManager.runtime_argument_handle_type;

			foreach (Argument arg in Arguments) {
				if (!arg.Resolve (ec, loc))
					return null;
			}

			return this;
		}

		public override void Emit (EmitContext ec)
		{
			foreach (Argument arg in Arguments)
				arg.Emit (ec);
		}
	}

	//
	// This produces the value that renders an instance, used by the iterators code
	//
	public class ProxyInstance : Expression, IMemoryLocation  {
		public override Expression DoResolve (EmitContext ec)
		{
			eclass = ExprClass.Variable;
			type = ec.ContainerType;
			return this;
		}
		
		public override void Emit (EmitContext ec)
		{
			ec.ig.Emit (OpCodes.Ldarg_0);

		}
		
		public void AddressOf (EmitContext ec, AddressOp mode)
		{
			ec.ig.Emit (OpCodes.Ldarg_0);
		}
	}

	/// <summary>
	///   Implements the typeof operator
	/// </summary>
	public class TypeOf : Expression {
		public readonly Expression QueriedType;
		protected Type typearg;
		
		public TypeOf (Expression queried_type, Location l)
		{
			QueriedType = queried_type;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			typearg = ec.DeclSpace.ResolveType (QueriedType, false, loc);

			if (typearg == null)
				return null;

			if (typearg == TypeManager.void_type) {
				Error (673, "System.Void cannot be used from C# - " +
				       "use typeof (void) to get the void type object");
				return null;
			}

			CheckObsoleteAttribute (typearg);

			type = TypeManager.type_type;
			eclass = ExprClass.Type;
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ec.ig.Emit (OpCodes.Ldtoken, typearg);
			ec.ig.Emit (OpCodes.Call, TypeManager.system_type_get_type_from_handle);
		}

		public Type TypeArg { 
			get { return typearg; }
		}
	}

	/// <summary>
	///   Implements the `typeof (void)' operator
	/// </summary>
	public class TypeOfVoid : TypeOf {
		public TypeOfVoid (Location l) : base (null, l)
		{
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			type = TypeManager.type_type;
			typearg = TypeManager.void_type;
			eclass = ExprClass.Type;
			return this;
		}
	}

	/// <summary>
	///   Implements the sizeof expression
	/// </summary>
	public class SizeOf : Expression {
		public readonly Expression QueriedType;
		Type type_queried;
		
		public SizeOf (Expression queried_type, Location l)
		{
			this.QueriedType = queried_type;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			if (!ec.InUnsafe) {
				Report.Error (
					233, loc, "Sizeof may only be used in an unsafe context " +
					"(consider using System.Runtime.InteropServices.Marshal.Sizeof");
				return null;
			}
				
			type_queried = ec.DeclSpace.ResolveType (QueriedType, false, loc);
			if (type_queried == null)
				return null;

			CheckObsoleteAttribute (type_queried);

			if (!TypeManager.IsUnmanagedType (type_queried)){
				Report.Error (208, loc, "Cannot take the size of an unmanaged type (" + TypeManager.CSharpName (type_queried) + ")");
				return null;
			}
			
			type = TypeManager.int32_type;
			eclass = ExprClass.Value;
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			int size = GetTypeSize (type_queried);

			if (size == 0)
				ec.ig.Emit (OpCodes.Sizeof, type_queried);
			else
				IntConstant.EmitInt (ec.ig, size);
		}
	}

	/// <summary>
	///   Implements the member access expression
	/// </summary>
	public class MemberAccess : Expression {
		public readonly string Identifier;
		Expression expr;
		
		public MemberAccess (Expression expr, string id, Location l)
		{
			this.expr = expr;
			Identifier = id;
			loc = l;
		}

		public Expression Expr {
			get {
				return expr;
			}
		}

		public static void error176 (Location loc, string name)
		{
			Report.Error (176, loc, "Static member `" +
				      name + "' cannot be accessed " +
				      "with an instance reference, qualify with a " +
				      "type name instead");
		}

		public static bool IdenticalNameAndTypeName (EmitContext ec, Expression left_original, Expression left, Location loc)
		{
			SimpleName sn = left_original as SimpleName;
			if (sn == null || left == null || left.Type.Name != sn.Name)
				return false;

			return RootContext.LookupType (ec.DeclSpace, sn.Name, true, loc) != null;
		}
		
		public static Expression ResolveMemberAccess (EmitContext ec, Expression member_lookup,
							      Expression left, Location loc,
							      Expression left_original)
		{
			bool left_is_type, left_is_explicit;

			// If `left' is null, then we're called from SimpleNameResolve and this is
			// a member in the currently defining class.
			if (left == null) {
				left_is_type = ec.IsStatic || ec.IsFieldInitializer;
				left_is_explicit = false;

				// Implicitly default to `this' unless we're static.
				if (!ec.IsStatic && !ec.IsFieldInitializer && !ec.InEnumContext)
					left = ec.GetThis (loc);
			} else {
				left_is_type = left is TypeExpr;
				left_is_explicit = true;
			}

			if (member_lookup is FieldExpr){
				FieldExpr fe = (FieldExpr) member_lookup;
				FieldInfo fi = fe.FieldInfo;
				Type decl_type = fi.DeclaringType;

				if (fi is FieldBuilder) {
					Const c = TypeManager.LookupConstant ((FieldBuilder) fi);
					
					if (c != null) {
						object o;
						if (!c.LookupConstantValue (out o))
							return null;

						object real_value = ((Constant) c.Expr).GetValue ();

						return Constantify (real_value, fi.FieldType);
					}
				}

				if (fi.IsLiteral) {
					Type t = fi.FieldType;
					
					object o;

					if (fi is FieldBuilder)
						o = TypeManager.GetValue ((FieldBuilder) fi);
					else
						o = fi.GetValue (fi);
					
					if (decl_type.IsSubclassOf (TypeManager.enum_type)) {
						if (left_is_explicit && !left_is_type &&
						    !IdenticalNameAndTypeName (ec, left_original, member_lookup, loc)) {
							error176 (loc, fe.FieldInfo.Name);
							return null;
						}					
						
						Expression enum_member = MemberLookup (
							ec, decl_type, "value__", MemberTypes.Field,
							AllBindingFlags, loc); 

						Enum en = TypeManager.LookupEnum (decl_type);

						Constant c;
						if (en != null)
							c = Constantify (o, en.UnderlyingType);
						else 
							c = Constantify (o, enum_member.Type);
						
						return new EnumConstant (c, decl_type);
					}
					
					Expression exp = Constantify (o, t);

					if (left_is_explicit && !left_is_type) {
						error176 (loc, fe.FieldInfo.Name);
						return null;
					}
					
					return exp;
				}

				if (fi.FieldType.IsPointer && !ec.InUnsafe){
					UnsafeError (loc);
					return null;
				}
			}

			if (member_lookup is EventExpr) {
				EventExpr ee = (EventExpr) member_lookup;
				
				//
				// If the event is local to this class, we transform ourselves into
				// a FieldExpr
				//

				if (ee.EventInfo.DeclaringType == ec.ContainerType ||
				    TypeManager.IsNestedChildOf(ec.ContainerType, ee.EventInfo.DeclaringType)) {
					MemberInfo mi = GetFieldFromEvent (ee);

					if (mi == null) {
						//
						// If this happens, then we have an event with its own
						// accessors and private field etc so there's no need
						// to transform ourselves.
						//
						ee.InstanceExpression = left;
						return ee;
					}

					Expression ml = ExprClassFromMemberInfo (ec, mi, loc);
					
					if (ml == null) {
						Report.Error (-200, loc, "Internal error!!");
						return null;
					}

					if (!left_is_explicit)
						left = null;

					ee.InstanceExpression = left;

					return ResolveMemberAccess (ec, ml, left, loc, left_original);
				}
			}

			if (member_lookup is IMemberExpr) {
				IMemberExpr me = (IMemberExpr) member_lookup;
				MethodGroupExpr mg = me as MethodGroupExpr;

				if (left_is_type){
					if ((mg != null) && left_is_explicit && left.Type.IsInterface)
						mg.IsExplicitImpl = left_is_explicit;

					if (!me.IsStatic){
						if ((ec.IsFieldInitializer || ec.IsStatic) &&
						    IdenticalNameAndTypeName (ec, left_original, member_lookup, loc))
							return member_lookup;
						
						SimpleName.Error_ObjectRefRequired (ec, loc, me.Name);
						return null;
					}

				} else {
					if (!me.IsInstance) {
						if (IdenticalNameAndTypeName (ec, left_original, left, loc))
							return member_lookup;

						if (left_is_explicit) {
							error176 (loc, me.Name);
							return null;
						}
					}			

					//
					// Since we can not check for instance objects in SimpleName,
					// becaue of the rule that allows types and variables to share
					// the name (as long as they can be de-ambiguated later, see 
					// IdenticalNameAndTypeName), we have to check whether left 
					// is an instance variable in a static context
					//
					// However, if the left-hand value is explicitly given, then
					// it is already our instance expression, so we aren't in
					// static context.
					//

					if (ec.IsStatic && !left_is_explicit && left is IMemberExpr){
						IMemberExpr mexp = (IMemberExpr) left;

						if (!mexp.IsStatic){
							SimpleName.Error_ObjectRefRequired (ec, loc, mexp.Name);
							return null;
						}
					}

					if ((mg != null) && IdenticalNameAndTypeName (ec, left_original, left, loc))
						mg.IdenticalTypeName = true;

					me.InstanceExpression = left;
				}

				return member_lookup;
			}

			Console.WriteLine ("Left is: " + left);
			Report.Error (-100, loc, "Support for [" + member_lookup + "] is not present yet");
			Environment.Exit (1);
			return null;
		}
		
		public Expression DoResolve (EmitContext ec, Expression right_side, ResolveFlags flags)
		{
			if (type != null)
				throw new Exception ();

			//
			// Resolve the expression with flow analysis turned off, we'll do the definite
			// assignment checks later.  This is because we don't know yet what the expression
			// will resolve to - it may resolve to a FieldExpr and in this case we must do the
			// definite assignment check on the actual field and not on the whole struct.
			//

			Expression original = expr;
			expr = expr.Resolve (ec, flags | ResolveFlags.Intermediate | ResolveFlags.DisableFlowAnalysis);
			if (expr == null)
				return null;

			if (expr is SimpleName){
				SimpleName child_expr = (SimpleName) expr;

				Expression new_expr = new SimpleName (child_expr.Name, Identifier, loc);

				return new_expr.Resolve (ec, flags);
			}
					
			//
			// TODO: I mailed Ravi about this, and apparently we can get rid
			// of this and put it in the right place.
			// 
			// Handle enums here when they are in transit.
			// Note that we cannot afford to hit MemberLookup in this case because
			// it will fail to find any members at all
			//

			Type expr_type = expr.Type;
			if (expr is TypeExpr){
				if (!ec.DeclSpace.CheckAccessLevel (expr_type)){
					Report.Error_T (122, loc, expr_type);
					return null;
				}

				if (expr_type == TypeManager.enum_type || expr_type.IsSubclassOf (TypeManager.enum_type)){
					Enum en = TypeManager.LookupEnum (expr_type);

					if (en != null) {
						object value = en.LookupEnumValue (ec, Identifier, loc);
						
						if (value != null){
							ObsoleteAttribute oa = en.GetObsoleteAttribute (ec, Identifier);
							if (oa != null) {
								AttributeTester.Report_ObsoleteMessage (oa, en.GetSignatureForError (), Location);
							}

							Constant c = Constantify (value, en.UnderlyingType);
							return new EnumConstant (c, expr_type);
						}
					} else {
						CheckObsoleteAttribute (expr_type);

						FieldInfo fi = expr_type.GetField (Identifier);
						if (fi != null) {
							ObsoleteAttribute oa = AttributeTester.GetMemberObsoleteAttribute (fi);
							if (oa != null)
								AttributeTester.Report_ObsoleteMessage (oa, TypeManager.GetFullNameSignature (fi), Location);
						}
					}
				}
			}
			
			if (expr_type.IsPointer){
				Error (23, "The `.' operator can not be applied to pointer operands (" +
				       TypeManager.CSharpName (expr_type) + ")");
				return null;
			}

			Expression member_lookup;
			member_lookup = MemberLookupFinal (ec, expr_type, expr_type, Identifier, loc);
			if (member_lookup == null)
				return null;

			if (member_lookup is TypeExpr) {
				if (!(expr is TypeExpr) && !(expr is SimpleName)) {
					Error (572, "Can't reference type `" + Identifier + "' through an expression; try `" +
					       member_lookup.Type + "' instead");
					return null;
				}

				return member_lookup;
			}
			
			member_lookup = ResolveMemberAccess (ec, member_lookup, expr, loc, original);
			if (member_lookup == null)
				return null;

			// The following DoResolve/DoResolveLValue will do the definite assignment
			// check.

			if (right_side != null)
				member_lookup = member_lookup.DoResolveLValue (ec, right_side);
			else
				member_lookup = member_lookup.DoResolve (ec);

			return member_lookup;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			return DoResolve (ec, null, ResolveFlags.VariableOrValue |
					  ResolveFlags.SimpleName | ResolveFlags.Type);
		}

		public override Expression DoResolveLValue (EmitContext ec, Expression right_side)
		{
			return DoResolve (ec, right_side, ResolveFlags.VariableOrValue |
					  ResolveFlags.SimpleName | ResolveFlags.Type);
		}

		public override Expression ResolveAsTypeStep (EmitContext ec)
		{
			string fname = null;
			MemberAccess full_expr = this;
			while (full_expr != null) {
				if (fname != null)
					fname = String.Concat (full_expr.Identifier, ".", fname);
				else
					fname = full_expr.Identifier;

				if (full_expr.Expr is SimpleName) {
					string full_name = String.Concat (((SimpleName) full_expr.Expr).Name, ".", fname);
					Type fully_qualified = ec.DeclSpace.FindType (loc, full_name);
					if (fully_qualified != null)
						return new TypeExpression (fully_qualified, loc);
				}

				full_expr = full_expr.Expr as MemberAccess;
			}

			Expression new_expr = expr.ResolveAsTypeStep (ec);

			if (new_expr == null)
				return null;

			if (new_expr is SimpleName){
				SimpleName child_expr = (SimpleName) new_expr;
				
				new_expr = new SimpleName (child_expr.Name, Identifier, loc);

				return new_expr.ResolveAsTypeStep (ec);
			}

			Type expr_type = new_expr.Type;
		      
			if (expr_type.IsPointer){
				Error (23, "The `.' operator can not be applied to pointer operands (" +
				       TypeManager.CSharpName (expr_type) + ")");
				return null;
			}
			
			Expression member_lookup;
			member_lookup = MemberLookupFinal (ec, expr_type, expr_type, Identifier, loc);
			if (member_lookup == null)
				return null;

			if (member_lookup is TypeExpr){
				member_lookup.Resolve (ec, ResolveFlags.Type);
				return member_lookup;
			} 

			return null;			
		}

		public override void Emit (EmitContext ec)
		{
			throw new Exception ("Should not happen");
		}

		public override string ToString ()
		{
			return expr + "." + Identifier;
		}
	}

	/// <summary>
	///   Implements checked expressions
	/// </summary>
	public class CheckedExpr : Expression {

		public Expression Expr;

		public CheckedExpr (Expression e, Location l)
		{
			Expr = e;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			bool last_check = ec.CheckState;
			bool last_const_check = ec.ConstantCheckState;

			ec.CheckState = true;
			ec.ConstantCheckState = true;
			Expr = Expr.Resolve (ec);
			ec.CheckState = last_check;
			ec.ConstantCheckState = last_const_check;
			
			if (Expr == null)
				return null;

			if (Expr is Constant)
				return Expr;
			
			eclass = Expr.eclass;
			type = Expr.Type;
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			bool last_check = ec.CheckState;
			bool last_const_check = ec.ConstantCheckState;
			
			ec.CheckState = true;
			ec.ConstantCheckState = true;
			Expr.Emit (ec);
			ec.CheckState = last_check;
			ec.ConstantCheckState = last_const_check;
		}
		
	}

	/// <summary>
	///   Implements the unchecked expression
	/// </summary>
	public class UnCheckedExpr : Expression {

		public Expression Expr;

		public UnCheckedExpr (Expression e, Location l)
		{
			Expr = e;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			bool last_check = ec.CheckState;
			bool last_const_check = ec.ConstantCheckState;

			ec.CheckState = false;
			ec.ConstantCheckState = false;
			Expr = Expr.Resolve (ec);
			ec.CheckState = last_check;
			ec.ConstantCheckState = last_const_check;

			if (Expr == null)
				return null;

			if (Expr is Constant)
				return Expr;
			
			eclass = Expr.eclass;
			type = Expr.Type;
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			bool last_check = ec.CheckState;
			bool last_const_check = ec.ConstantCheckState;
			
			ec.CheckState = false;
			ec.ConstantCheckState = false;
			Expr.Emit (ec);
			ec.CheckState = last_check;
			ec.ConstantCheckState = last_const_check;
		}
		
	}

	/// <summary>
	///   An Element Access expression.
	///
	///   During semantic analysis these are transformed into 
	///   IndexerAccess, ArrayAccess or a PointerArithmetic.
	/// </summary>
	public class ElementAccess : Expression {
		public ArrayList  Arguments;
		public Expression Expr;
		
		public ElementAccess (Expression e, ArrayList e_list, Location l)
		{
			Expr = e;

			loc  = l;
			
			if (e_list == null)
				return;
			
			Arguments = new ArrayList ();
			foreach (Expression tmp in e_list)
				Arguments.Add (new Argument (tmp, Argument.AType.Expression));
			
		}

		bool CommonResolve (EmitContext ec)
		{
			Expr = Expr.Resolve (ec);

			if (Expr == null) 
				return false;

			if (Arguments == null)
				return false;

			foreach (Argument a in Arguments){
				if (!a.Resolve (ec, loc))
					return false;
			}

			return true;
		}

		Expression MakePointerAccess ()
		{
			Type t = Expr.Type;

			if (t == TypeManager.void_ptr_type){
				Error (242, "The array index operation is not valid for void pointers");
				return null;
			}
			if (Arguments.Count != 1){
				Error (196, "A pointer must be indexed by a single value");
				return null;
			}
			Expression p;

			p = new PointerArithmetic (true, Expr, ((Argument)Arguments [0]).Expr, t, loc);
			return new Indirection (p, loc);
		}
		
		public override Expression DoResolve (EmitContext ec)
		{
			if (!CommonResolve (ec))
				return null;

			//
			// We perform some simple tests, and then to "split" the emit and store
			// code we create an instance of a different class, and return that.
			//
			// I am experimenting with this pattern.
			//
			Type t = Expr.Type;

			if (t == TypeManager.array_type){
				Report.Error (21, loc, "Cannot use indexer on System.Array");
				return null;
			}
			
			if (t.IsArray)
				return (new ArrayAccess (this, loc)).Resolve (ec);
			else if (t.IsPointer)
				return MakePointerAccess ();
			else
				return (new IndexerAccess (this, loc)).Resolve (ec);
		}

		public override Expression DoResolveLValue (EmitContext ec, Expression right_side)
		{
			if (!CommonResolve (ec))
				return null;

			Type t = Expr.Type;
			if (t.IsArray)
				return (new ArrayAccess (this, loc)).ResolveLValue (ec, right_side);
			else if (t.IsPointer)
				return MakePointerAccess ();
			else
				return (new IndexerAccess (this, loc)).ResolveLValue (ec, right_side);
		}
		
		public override void Emit (EmitContext ec)
		{
			throw new Exception ("Should never be reached");
		}
	}

	/// <summary>
	///   Implements array access 
	/// </summary>
	public class ArrayAccess : Expression, IAssignMethod, IMemoryLocation {
		//
		// Points to our "data" repository
		//
		ElementAccess ea;

		LocalTemporary [] cached_locations;
		
		public ArrayAccess (ElementAccess ea_data, Location l)
		{
			ea = ea_data;
			eclass = ExprClass.Variable;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
#if false
			ExprClass eclass = ea.Expr.eclass;

			// As long as the type is valid
			if (!(eclass == ExprClass.Variable || eclass == ExprClass.PropertyAccess ||
			      eclass == ExprClass.Value)) {
				ea.Expr.Error_UnexpectedKind ("variable or value");
				return null;
			}
#endif

			Type t = ea.Expr.Type;
			if (t.GetArrayRank () != ea.Arguments.Count){
				ea.Error (22,
					  "Incorrect number of indexes for array " +
					  " expected: " + t.GetArrayRank () + " got: " +
					  ea.Arguments.Count);
				return null;
			}

                        type = TypeManager.GetElementType (t);
                        if (type.IsPointer && !ec.InUnsafe){
				UnsafeError (ea.Location);
				return null;
			}

			foreach (Argument a in ea.Arguments){
				Type argtype = a.Type;

				if (argtype == TypeManager.int32_type ||
				    argtype == TypeManager.uint32_type ||
				    argtype == TypeManager.int64_type ||
				    argtype == TypeManager.uint64_type)
					continue;

				//
				// Mhm.  This is strage, because the Argument.Type is not the same as
				// Argument.Expr.Type: the value changes depending on the ref/out setting.
				//
				// Wonder if I will run into trouble for this.
				//
				a.Expr = ExpressionToArrayArgument (ec, a.Expr, ea.Location);
				if (a.Expr == null)
					return null;
			}
			
			eclass = ExprClass.Variable;

			return this;
		}

		/// <summary>
		///    Emits the right opcode to load an object of Type `t'
		///    from an array of T
		/// </summary>
		static public void EmitLoadOpcode (ILGenerator ig, Type type)
		{
			if (type == TypeManager.byte_type || type == TypeManager.bool_type)
				ig.Emit (OpCodes.Ldelem_U1);
			else if (type == TypeManager.sbyte_type)
				ig.Emit (OpCodes.Ldelem_I1);
			else if (type == TypeManager.short_type)
				ig.Emit (OpCodes.Ldelem_I2);
			else if (type == TypeManager.ushort_type || type == TypeManager.char_type)
				ig.Emit (OpCodes.Ldelem_U2);
			else if (type == TypeManager.int32_type)
				ig.Emit (OpCodes.Ldelem_I4);
			else if (type == TypeManager.uint32_type)
				ig.Emit (OpCodes.Ldelem_U4);
			else if (type == TypeManager.uint64_type)
				ig.Emit (OpCodes.Ldelem_I8);
			else if (type == TypeManager.int64_type)
				ig.Emit (OpCodes.Ldelem_I8);
			else if (type == TypeManager.float_type)
				ig.Emit (OpCodes.Ldelem_R4);
			else if (type == TypeManager.double_type)
				ig.Emit (OpCodes.Ldelem_R8);
			else if (type == TypeManager.intptr_type)
				ig.Emit (OpCodes.Ldelem_I);
			else if (TypeManager.IsEnumType (type)){
				EmitLoadOpcode (ig, TypeManager.EnumToUnderlying (type));
			} else if (type.IsValueType){
				ig.Emit (OpCodes.Ldelema, type);
				ig.Emit (OpCodes.Ldobj, type);
			} else 
				ig.Emit (OpCodes.Ldelem_Ref);
		}

		/// <summary>
		///    Emits the right opcode to store an object of Type `t'
		///    from an array of T.  
		/// </summary>
		static public void EmitStoreOpcode (ILGenerator ig, Type t)
		{
			bool is_stobj;
			OpCode op = GetStoreOpcode (t, out is_stobj);
			if (is_stobj)
				ig.Emit (OpCodes.Stobj, t);
			else
				ig.Emit (op);
		}

		/// <summary>
		///    Returns the right opcode to store an object of Type `t'
		///    from an array of T.  
		/// </summary>
		static public OpCode GetStoreOpcode (Type t, out bool is_stobj)
		{
			//Console.WriteLine (new System.Diagnostics.StackTrace ());
			is_stobj = false;
			t = TypeManager.TypeToCoreType (t);
			if (TypeManager.IsEnumType (t))
				t = TypeManager.EnumToUnderlying (t);
			if (t == TypeManager.byte_type || t == TypeManager.sbyte_type ||
			    t == TypeManager.bool_type)
				return OpCodes.Stelem_I1;
			else if (t == TypeManager.short_type || t == TypeManager.ushort_type ||
				 t == TypeManager.char_type)
				return OpCodes.Stelem_I2;
			else if (t == TypeManager.int32_type || t == TypeManager.uint32_type)
				return OpCodes.Stelem_I4;
			else if (t == TypeManager.int64_type || t == TypeManager.uint64_type)
				return OpCodes.Stelem_I8;
			else if (t == TypeManager.float_type)
				return OpCodes.Stelem_R4;
			else if (t == TypeManager.double_type)
				return OpCodes.Stelem_R8;
			else if (t == TypeManager.intptr_type) {
                                is_stobj = true;
                                return OpCodes.Stobj;
			} else if (t.IsValueType) {
				is_stobj = true;
				return OpCodes.Stobj;
			} else
				return OpCodes.Stelem_Ref;
		}

		MethodInfo FetchGetMethod ()
		{
			ModuleBuilder mb = CodeGen.Module.Builder;
			int arg_count = ea.Arguments.Count;
			Type [] args = new Type [arg_count];
			MethodInfo get;
			
			for (int i = 0; i < arg_count; i++){
				//args [i++] = a.Type;
				args [i] = TypeManager.int32_type;
			}
			
			get = mb.GetArrayMethod (
				ea.Expr.Type, "Get",
				CallingConventions.HasThis |
				CallingConventions.Standard,
				type, args);
			return get;
		}
				

		MethodInfo FetchAddressMethod ()
		{
			ModuleBuilder mb = CodeGen.Module.Builder;
			int arg_count = ea.Arguments.Count;
			Type [] args = new Type [arg_count];
			MethodInfo address;
			Type ret_type;
			
			ret_type = TypeManager.GetReferenceType (type);
			
			for (int i = 0; i < arg_count; i++){
				//args [i++] = a.Type;
				args [i] = TypeManager.int32_type;
			}
			
			address = mb.GetArrayMethod (
				ea.Expr.Type, "Address",
				CallingConventions.HasThis |
				CallingConventions.Standard,
				ret_type, args);

			return address;
		}

		//
		// Load the array arguments into the stack.
		//
		// If we have been requested to cache the values (cached_locations array
		// initialized), then load the arguments the first time and store them
		// in locals.  otherwise load from local variables.
		//
		void LoadArrayAndArguments (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			if (cached_locations == null){
				ea.Expr.Emit (ec);
				foreach (Argument a in ea.Arguments){
					Type argtype = a.Expr.Type;
					
					a.Expr.Emit (ec);
					
					if (argtype == TypeManager.int64_type)
						ig.Emit (OpCodes.Conv_Ovf_I);
					else if (argtype == TypeManager.uint64_type)
						ig.Emit (OpCodes.Conv_Ovf_I_Un);
				}
				return;
			}

			if (cached_locations [0] == null){
				cached_locations [0] = new LocalTemporary (ec, ea.Expr.Type);
				ea.Expr.Emit (ec);
				ig.Emit (OpCodes.Dup);
				cached_locations [0].Store (ec);
				
				int j = 1;
				
				foreach (Argument a in ea.Arguments){
					Type argtype = a.Expr.Type;
					
					cached_locations [j] = new LocalTemporary (ec, TypeManager.intptr_type /* a.Expr.Type */);
					a.Expr.Emit (ec);
					if (argtype == TypeManager.int64_type)
						ig.Emit (OpCodes.Conv_Ovf_I);
					else if (argtype == TypeManager.uint64_type)
						ig.Emit (OpCodes.Conv_Ovf_I_Un);

					ig.Emit (OpCodes.Dup);
					cached_locations [j].Store (ec);
					j++;
				}
				return;
			}

			foreach (LocalTemporary lt in cached_locations)
				lt.Emit (ec);
		}

		public new void CacheTemporaries (EmitContext ec)
		{
			cached_locations = new LocalTemporary [ea.Arguments.Count + 1];
		}
		
		public override void Emit (EmitContext ec)
		{
			int rank = ea.Expr.Type.GetArrayRank ();
			ILGenerator ig = ec.ig;

			LoadArrayAndArguments (ec);
			
			if (rank == 1)
				EmitLoadOpcode (ig, type);
			else {
				MethodInfo method;
				
				method = FetchGetMethod ();
				ig.Emit (OpCodes.Call, method);
			}
		}

		public void EmitAssign (EmitContext ec, Expression source)
		{
			int rank = ea.Expr.Type.GetArrayRank ();
			ILGenerator ig = ec.ig;
			Type t = source.Type;

			LoadArrayAndArguments (ec);

			//
			// The stobj opcode used by value types will need
			// an address on the stack, not really an array/array
			// pair
			//
			if (rank == 1){
				if (t == TypeManager.enum_type || t == TypeManager.decimal_type ||
				    (t.IsSubclassOf (TypeManager.value_type) && !TypeManager.IsEnumType (t) && !TypeManager.IsBuiltinType (t)))
					ig.Emit (OpCodes.Ldelema, t);
			}
			
			source.Emit (ec);

			if (rank == 1)
				EmitStoreOpcode (ig, t);
			else {
				ModuleBuilder mb = CodeGen.Module.Builder;
				int arg_count = ea.Arguments.Count;
				Type [] args = new Type [arg_count + 1];
				MethodInfo set;
				
				for (int i = 0; i < arg_count; i++){
					//args [i++] = a.Type;
					args [i] = TypeManager.int32_type;
				}

				args [arg_count] = type;
				
				set = mb.GetArrayMethod (
					ea.Expr.Type, "Set",
					CallingConventions.HasThis |
					CallingConventions.Standard,
					TypeManager.void_type, args);
				
				ig.Emit (OpCodes.Call, set);
			}
		}

		public void AddressOf (EmitContext ec, AddressOp mode)
		{
			int rank = ea.Expr.Type.GetArrayRank ();
			ILGenerator ig = ec.ig;

			LoadArrayAndArguments (ec);

			if (rank == 1){
				ig.Emit (OpCodes.Ldelema, type);
			} else {
				MethodInfo address = FetchAddressMethod ();
				ig.Emit (OpCodes.Call, address);
			}
		}
	}

	
	class Indexers {
		public ArrayList Properties;
		static Hashtable map;

		public struct Indexer {
			public readonly Type Type;
			public readonly MethodInfo Getter, Setter;

			public Indexer (Type type, MethodInfo get, MethodInfo set)
			{
				this.Type = type;
				this.Getter = get;
				this.Setter = set;
			}
		}

		static Indexers ()
		{
			map = new Hashtable ();
		}

		Indexers ()
		{
			Properties = new ArrayList ();
		}
				
		void Append (MemberInfo [] mi)
		{
			foreach (PropertyInfo property in mi){
				MethodInfo get, set;
				
				get = property.GetGetMethod (true);
				set = property.GetSetMethod (true);
				Properties.Add (new Indexer (property.PropertyType, get, set));
			}
		}

		static private MemberInfo [] GetIndexersForTypeOrInterface (Type caller_type, Type lookup_type)
		{
			string p_name = TypeManager.IndexerPropertyName (lookup_type);

			MemberInfo [] mi = TypeManager.MemberLookup (
				caller_type, caller_type, lookup_type, MemberTypes.Property,
				BindingFlags.Public | BindingFlags.Instance |
				BindingFlags.DeclaredOnly, p_name, null);

			if (mi == null || mi.Length == 0)
				return null;

			return mi;
		}
		
		static public Indexers GetIndexersForType (Type caller_type, Type lookup_type, Location loc) 
		{
			Indexers ix = (Indexers) map [lookup_type];

			if (ix != null)
				return ix;

			Type copy = lookup_type;
			while (copy != TypeManager.object_type && copy != null){
				MemberInfo [] mi = GetIndexersForTypeOrInterface (caller_type, copy);

				if (mi != null){
					if (ix == null)
						ix = new Indexers ();

					ix.Append (mi);
				}
					
				copy = copy.BaseType;
			}

			if (!lookup_type.IsInterface)
				return ix;

			TypeExpr [] ifaces = TypeManager.GetInterfaces (lookup_type);
			if (ifaces != null) {
				foreach (TypeExpr iface in ifaces) {
					Type itype = iface.Type;
					MemberInfo [] mi = GetIndexersForTypeOrInterface (caller_type, itype);
					if (mi != null){
						if (ix == null)
							ix = new Indexers ();
					
						ix.Append (mi);
					}
				}
			}

			return ix;
		}
	}

	/// <summary>
	///   Expressions that represent an indexer call.
	/// </summary>
	public class IndexerAccess : Expression, IAssignMethod {
		//
		// Points to our "data" repository
		//
		MethodInfo get, set;
		ArrayList set_arguments;
		bool is_base_indexer;

		protected Type indexer_type;
		protected Type current_type;
		protected Expression instance_expr;
		protected ArrayList arguments;
		
		public IndexerAccess (ElementAccess ea, Location loc)
			: this (ea.Expr, false, loc)
		{
			this.arguments = ea.Arguments;
		}

		protected IndexerAccess (Expression instance_expr, bool is_base_indexer,
					 Location loc)
		{
			this.instance_expr = instance_expr;
			this.is_base_indexer = is_base_indexer;
			this.eclass = ExprClass.Value;
			this.loc = loc;
		}

		protected virtual bool CommonResolve (EmitContext ec)
		{
			indexer_type = instance_expr.Type;
			current_type = ec.ContainerType;

			return true;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			ArrayList AllGetters = new ArrayList();
			if (!CommonResolve (ec))
				return null;

			//
			// Step 1: Query for all `Item' *properties*.  Notice
			// that the actual methods are pointed from here.
			//
			// This is a group of properties, piles of them.  

			bool found_any = false, found_any_getters = false;
			Type lookup_type = indexer_type;

			Indexers ilist;
			ilist = Indexers.GetIndexersForType (current_type, lookup_type, loc);
			if (ilist != null) {
				found_any = true;
				if (ilist.Properties != null) {
					foreach (Indexers.Indexer ix in ilist.Properties) {
						if (ix.Getter != null)
							AllGetters.Add(ix.Getter);
					}
				}
			}

			if (AllGetters.Count > 0) {
				found_any_getters = true;
				get = (MethodInfo) Invocation.OverloadResolve (
					ec, new MethodGroupExpr (AllGetters, loc), arguments, loc);
			}

			if (!found_any) {
				Report.Error (21, loc,
					      "Type `" + TypeManager.CSharpName (indexer_type) +
					      "' does not have any indexers defined");
				return null;
			}

			if (!found_any_getters) {
				Error (154, "indexer can not be used in this context, because " +
				       "it lacks a `get' accessor");
				return null;
			}

			if (get == null) {
				Error (1501, "No Overload for method `this' takes `" +
				       arguments.Count + "' arguments");
				return null;
			}

			//
			// Only base will allow this invocation to happen.
			//
			if (get.IsAbstract && this is BaseIndexerAccess){
				Report.Error (205, loc, "Cannot call an abstract base indexer: " + Invocation.FullMethodDesc (get));
				return null;
			}

			type = get.ReturnType;
			if (type.IsPointer && !ec.InUnsafe){
				UnsafeError (loc);
				return null;
			}
			
			eclass = ExprClass.IndexerAccess;
			return this;
		}

		public override Expression DoResolveLValue (EmitContext ec, Expression right_side)
		{
			ArrayList AllSetters = new ArrayList();
			if (!CommonResolve (ec))
				return null;

			bool found_any = false, found_any_setters = false;

			Indexers ilist = Indexers.GetIndexersForType (current_type, indexer_type, loc);
			if (ilist != null) {
				found_any = true;
				if (ilist.Properties != null) {
					foreach (Indexers.Indexer ix in ilist.Properties) {
						if (ix.Setter != null)
							AllSetters.Add(ix.Setter);
					}
				}
			}
			if (AllSetters.Count > 0) {
				found_any_setters = true;
				set_arguments = (ArrayList) arguments.Clone ();
				set_arguments.Add (new Argument (right_side, Argument.AType.Expression));
				set = (MethodInfo) Invocation.OverloadResolve (
					ec, new MethodGroupExpr (AllSetters, loc),
					set_arguments, loc);
			}

			if (!found_any) {
				Report.Error (21, loc,
					      "Type `" + TypeManager.CSharpName (indexer_type) +
					      "' does not have any indexers defined");
				return null;
			}

			if (!found_any_setters) {
				Error (154, "indexer can not be used in this context, because " +
				       "it lacks a `set' accessor");
				return null;
			}

			if (set == null) {
				Error (1501, "No Overload for method `this' takes `" +
				       arguments.Count + "' arguments");
				return null;
			}

			//
			// Only base will allow this invocation to happen.
			//
			if (set.IsAbstract && this is BaseIndexerAccess){
				Report.Error (205, loc, "Cannot call an abstract base indexer: " + Invocation.FullMethodDesc (set));
				return null;
			}

			//
			// Now look for the actual match in the list of indexers to set our "return" type
			//
			type = TypeManager.void_type;	// default value
			foreach (Indexers.Indexer ix in ilist.Properties){
				if (ix.Setter == set){
					type = ix.Type;
					break;
				}
			}
			
			eclass = ExprClass.IndexerAccess;
			return this;
		}
		
		public override void Emit (EmitContext ec)
		{
			Invocation.EmitCall (ec, is_base_indexer, false, instance_expr, get, arguments, loc);
		}

		//
		// source is ignored, because we already have a copy of it from the
		// LValue resolution and we have already constructed a pre-cached
		// version of the arguments (ea.set_arguments);
		//
		public void EmitAssign (EmitContext ec, Expression source)
		{
			Invocation.EmitCall (ec, is_base_indexer, false, instance_expr, set, set_arguments, loc);
		}
	}

	/// <summary>
	///   The base operator for method names
	/// </summary>
	public class BaseAccess : Expression {
		string member;
		
		public BaseAccess (string member, Location l)
		{
			this.member = member;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			Expression c = CommonResolve (ec);

			if (c == null)
				return null;

			//
			// MethodGroups use this opportunity to flag an error on lacking ()
			//
			if (!(c is MethodGroupExpr))
				return c.Resolve (ec);
			return c;
		}

		public override Expression DoResolveLValue (EmitContext ec, Expression right_side)
		{
			Expression c = CommonResolve (ec);

			if (c == null)
				return null;

			//
			// MethodGroups use this opportunity to flag an error on lacking ()
			//
			if (! (c is MethodGroupExpr))
				return c.DoResolveLValue (ec, right_side);

			return c;
		}

		Expression CommonResolve (EmitContext ec)
		{
			Expression member_lookup;
			Type current_type = ec.ContainerType;
			Type base_type = current_type.BaseType;
			Expression e;

			if (ec.IsStatic){
				Error (1511, "Keyword base is not allowed in static method");
				return null;
			}

			if (ec.IsFieldInitializer){
				Error (1512, "Keyword base is not available in the current context");
				return null;
			}
			
			member_lookup = MemberLookup (ec, ec.ContainerType, null, base_type, member,
						      AllMemberTypes, AllBindingFlags, loc);
			if (member_lookup == null) {
				MemberLookupFailed (ec, base_type, base_type, member, null, loc);
				return null;
			}

			Expression left;
			
			if (ec.IsStatic)
				left = new TypeExpression (base_type, loc);
			else
				left = ec.GetThis (loc);
			
			e = MemberAccess.ResolveMemberAccess (ec, member_lookup, left, loc, null);

			if (e is PropertyExpr){
				PropertyExpr pe = (PropertyExpr) e;

				pe.IsBase = true;
			}
			
			if (e is MethodGroupExpr)
				((MethodGroupExpr) e).IsBase = true;

			return e;
		}

		public override void Emit (EmitContext ec)
		{
			throw new Exception ("Should never be called"); 
		}
	}

	/// <summary>
	///   The base indexer operator
	/// </summary>
	public class BaseIndexerAccess : IndexerAccess {
		public BaseIndexerAccess (ArrayList args, Location loc)
			: base (null, true, loc)
		{
			arguments = new ArrayList ();
			foreach (Expression tmp in args)
				arguments.Add (new Argument (tmp, Argument.AType.Expression));
		}

		protected override bool CommonResolve (EmitContext ec)
		{
			instance_expr = ec.GetThis (loc);

			current_type = ec.ContainerType.BaseType;
			indexer_type = current_type;

			foreach (Argument a in arguments){
				if (!a.Resolve (ec, loc))
					return false;
			}

			return true;
		}
	}
	
	/// <summary>
	///   This class exists solely to pass the Type around and to be a dummy
	///   that can be passed to the conversion functions (this is used by
	///   foreach implementation to typecast the object return value from
	///   get_Current into the proper type.  All code has been generated and
	///   we only care about the side effect conversions to be performed
	///
	///   This is also now used as a placeholder where a no-action expression
	///   is needed (the `New' class).
	/// </summary>
	public class EmptyExpression : Expression {
		public EmptyExpression ()
		{
			type = TypeManager.object_type;
			eclass = ExprClass.Value;
			loc = Location.Null;
		}

		public EmptyExpression (Type t)
		{
			type = t;
			eclass = ExprClass.Value;
			loc = Location.Null;
		}
		
		public override Expression DoResolve (EmitContext ec)
		{
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			// nothing, as we only exist to not do anything.
		}

		//
		// This is just because we might want to reuse this bad boy
		// instead of creating gazillions of EmptyExpressions.
		// (CanImplicitConversion uses it)
		//
		public void SetType (Type t)
		{
			type = t;
		}
	}

	public class UserCast : Expression {
		MethodBase method;
		Expression source;
		
		public UserCast (MethodInfo method, Expression source, Location l)
		{
			this.method = method;
			this.source = source;
			type = method.ReturnType;
			eclass = ExprClass.Value;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// We are born fully resolved
			//
			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			source.Emit (ec);
			
			if (method is MethodInfo)
				ig.Emit (OpCodes.Call, (MethodInfo) method);
			else
				ig.Emit (OpCodes.Call, (ConstructorInfo) method);

		}
	}

	// <summary>
	//   This class is used to "construct" the type during a typecast
	//   operation.  Since the Type.GetType class in .NET can parse
	//   the type specification, we just use this to construct the type
	//   one bit at a time.
	// </summary>
	public class ComposedCast : TypeExpr {
		Expression left;
		string dim;
		
		public ComposedCast (Expression left, string dim, Location l)
		{
			this.left = left;
			this.dim = dim;
			loc = l;
		}

		public override TypeExpr DoResolveAsTypeStep (EmitContext ec)
		{
			Type ltype = ec.DeclSpace.ResolveType (left, false, loc);
			if (ltype == null)
				return null;

			if ((ltype == TypeManager.void_type) && (dim != "*")) {
				Report.Error (1547, Location,
					      "Keyword 'void' cannot be used in this context");
				return null;
			}

			//
			// ltype.Fullname is already fully qualified, so we can skip
			// a lot of probes, and go directly to TypeManager.LookupType
			//
			string cname = ltype.FullName + dim;
			type = TypeManager.LookupTypeDirect (cname);
			if (type == null){
				//
				// For arrays of enumerations we are having a problem
				// with the direct lookup.  Need to investigate.
				//
				// For now, fall back to the full lookup in that case.
				//
				type = RootContext.LookupType (
					ec.DeclSpace, cname, false, loc);

				if (type == null)
					return null;
			}

			if (!ec.ResolvingTypeTree){
				//
				// If the above flag is set, this is being invoked from the ResolveType function.
				// Upper layers take care of the type validity in this context.
				//
			if (!ec.InUnsafe && type.IsPointer){
				UnsafeError (loc);
				return null;
			}
			}
			
			eclass = ExprClass.Type;
			return this;
		}

		public override string Name {
			get {
				return left + dim;
			}
		}
	}

	//
	// This class is used to represent the address of an array, used
	// only by the Fixed statement, this is like the C "&a [0]" construct.
	//
	public class ArrayPtr : Expression {
		Expression array;
		
		public ArrayPtr (Expression array, Location l)
		{
			Type array_type = TypeManager.GetElementType (array.Type);

			this.array = array;

			type = TypeManager.GetPointerType (array_type);
			eclass = ExprClass.Value;
			loc = l;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			
			array.Emit (ec);
			IntLiteral.EmitInt (ig, 0);
			ig.Emit (OpCodes.Ldelema, TypeManager.GetElementType (array.Type));
		}

		public override Expression DoResolve (EmitContext ec)
		{
			//
			// We are born fully resolved
			//
			return this;
		}
	}

	//
	// Used by the fixed statement
	//
	public class StringPtr : Expression {
		LocalBuilder b;
		
		public StringPtr (LocalBuilder b, Location l)
		{
			this.b = b;
			eclass = ExprClass.Value;
			type = TypeManager.char_ptr_type;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			// This should never be invoked, we are born in fully
			// initialized state.

			return this;
		}

		public override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			ig.Emit (OpCodes.Ldloc, b);
			ig.Emit (OpCodes.Conv_I);
			ig.Emit (OpCodes.Call, TypeManager.int_get_offset_to_string_data);
			ig.Emit (OpCodes.Add);
		}
	}
	
	//
	// Implements the `stackalloc' keyword
	//
	public class StackAlloc : Expression {
		Type otype;
		Expression t;
		Expression count;
		
		public StackAlloc (Expression type, Expression count, Location l)
		{
			t = type;
			this.count = count;
			loc = l;
		}

		public override Expression DoResolve (EmitContext ec)
		{
			count = count.Resolve (ec);
			if (count == null)
				return null;
			
			if (count.Type != TypeManager.int32_type){
				count = Convert.ImplicitConversionRequired (ec, count, TypeManager.int32_type, loc);
				if (count == null)
					return null;
			}

			Constant c = count as Constant;
			// TODO: because we don't have property IsNegative
			if (c != null && c.ConvertToUInt () == null) {
                                // "Cannot use a negative size with stackalloc"
				Report.Error_T (247, loc);
				return null;
			}

			if (ec.CurrentBranching.InCatch () ||
			    ec.CurrentBranching.InFinally (true)) {
				Error (255,
				       "stackalloc can not be used in a catch or finally block");
				return null;
			}

			otype = ec.DeclSpace.ResolveType (t, false, loc);

			if (otype == null)
				return null;

			if (!TypeManager.VerifyUnManaged (otype, loc))
				return null;

			type = TypeManager.GetPointerType (otype);
			eclass = ExprClass.Value;

			return this;
		}

		public override void Emit (EmitContext ec)
		{
			int size = GetTypeSize (otype);
			ILGenerator ig = ec.ig;
				
			if (size == 0)
				ig.Emit (OpCodes.Sizeof, otype);
			else
				IntConstant.EmitInt (ig, size);
			count.Emit (ec);
			ig.Emit (OpCodes.Mul);
			ig.Emit (OpCodes.Localloc);
		}
	}
}
