// -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
//
// System.Xml.XmlChar.cs
//
// Author:
//   Jason Diamond (jason@injektilo.org)
//
// (C) 2001 Jason Diamond  http://injektilo.org/
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

namespace System.Xml
{
	// Now, "XmlChar" and "XmlConstructs" are made as equivalent, so
	// I dicided to rename XmlConstruts class as "XmlChar" and use it
	// for default build.
	// However, this class will be used for the future compact framework 
	// (XmlConstruts class uses not a little memory).
	internal class XmlCharCompact
	{
		public static readonly char [] WhitespaceChars = new char [] {' ', '\n', '\t', '\r'};

		public static bool IsWhitespace (int ch)
		{
			return ch == 0x20 || ch == 0x9 || ch == 0xD || ch == 0xA;
		}

		public static bool IsWhitespace (string str)
		{
			for (int i = 0; i < str.Length; i++)
				if (!IsWhitespace (str [i])) return false;
				
			return true;
		}

		public static bool IsFirstNameChar (int ch)
		{
			bool result = false;

			if (ch >= 0 && ch <= 0xFFFF)
			{
				result = (nameBitmap[(firstNamePages[ch >> 8] << 3) + ((ch & 0xFF) >> 5)] & (1 << (ch & 0x1F))) != 0;
			}

			return result;
		}

		public static bool IsValid (int ch)
		{
			return !IsInvalid (ch);
		}

		public static bool IsInvalid (int ch)
		{
			switch (ch) {
			case 9:
			case 10:
			case 13:
				return false;
			}
			if (ch < 32)
				return true;
			if (ch < 0xD800)
				return false;
			if (ch < 0xE000)
				return true;
			if (ch < 0xFFFE)
				return false;
			if (ch < 0x10000)
				return true;
			if (ch < 0x110000)
				return false;
			else
				return true;
		}

		public static bool IsNameChar (int ch)
		{
			bool result = false;

			if (ch >= 0 && ch <= 0xFFFF)
			{
				result = (nameBitmap[(namePages[ch >> 8] << 3) + ((ch & 0xFF) >> 5)] & (1 << (ch & 0x1F))) != 0;
			}

			return result;
		}

		public static bool IsNCNameChar (int ch)
		{
			bool result = false;

			if (ch >= 0 && ch <= 0xFFFF && ch != ':')
			{
				result = (nameBitmap[(namePages[ch >> 8] << 3) + ((ch & 0xFF) >> 5)] & (1 << (ch & 0x1F))) != 0;
			}

			return result;
		}

		public static bool IsName (string str)
		{
			if (str.Length == 0)
				return false;
			if (!IsFirstNameChar (str [0]))
				return false;
			for (int i = 1; i < str.Length; i++)
				if (!IsNameChar (str [i]))
					return false;
			return true;
		}

		public static bool IsNCName (string str)
		{
			if (str.Length == 0)
				return false;
			if (!IsFirstNameChar (str [0]))
				return false;
			for (int i = 0; i < str.Length; i++)
				if (!IsNCNameChar (str [i]))
					return false;
			return true;
		}

		public static bool IsNmToken (string str)
		{
			if (str.Length == 0)
				return false;
			for (int i = 0; i < str.Length; i++)
				if (!IsNameChar (str [i]))
					return false;
			return true;
		}

		public static bool IsPubidChar (int ch)
		{
			return (IsWhitespace(ch) && ch != '\t') | ('a' <= ch && ch <= 'z') | ('A' <= ch && ch <= 'Z') | ('0' <= ch && ch <= '9') | "-'()+,./:=?;!*#@$_%".IndexOf((char)ch) >= 0;
		}

		public static bool IsPubid (string str)
		{
			for (int i = 0; i < str.Length; i++)
				if (!IsPubidChar (str [i]))
					return false;
			return true;
		}

		// encodings (copied from XmlConstructs.cs)

		/// <summary>
		/// Returns true if the encoding name is a valid IANA encoding.
		/// This method does not verify that there is a decoder available
		/// for this encoding, only that the characters are valid for an
		/// IANA encoding name.
		/// </summary>
		/// <param name="ianaEncoding">The encoding to check.</param>
		/// <returns></returns>
		public static bool IsValidIANAEncoding (String ianaEncoding) 
		{
			if (ianaEncoding != null) 
			{
				int length = ianaEncoding.Length;
				if (length > 0) 
				{
					char c = ianaEncoding[0];
					if ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z')) 
					{
						for (int i = 1; i < length; i++) 
						{
							c = ianaEncoding[i];
							if ((c < 'A' || c > 'Z') && (c < 'a' || c > 'z') &&
								(c < '0' || c > '9') && c != '.' && c != '_' &&
								c != '-') 
							{
								return false;
							}
						}
						return true;
					}
				}
			}
			return false;
		}

		public static int GetPredefinedEntity (string name)
		{
			switch (name) {
			case "amp":
				return '&';
			case "lt":
				return '<';
			case "gt":
				return '>';
			case "quot":
				return '"';
			case "apos":
				return '\'';
			default:
				return -1;
			}
		}

		static readonly byte [] firstNamePages =
		{
			0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x00,
			0x00, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
			0x10, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x13,
			0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x15, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x17,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x18,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		};

		static readonly byte [] namePages =
		{
			0x19, 0x03, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x00,
			0x00, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
			0x10, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x13,
			0x26, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x27, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x17,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x18,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		};

		static readonly uint [] nameBitmap =
		{
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF,
			0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF,
			0x00000000, 0x04000000, 0x87FFFFFE, 0x07FFFFFE,
			0x00000000, 0x00000000, 0xFF7FFFFF, 0xFF7FFFFF,
			0xFFFFFFFF, 0x7FF3FFFF, 0xFFFFFDFE, 0x7FFFFFFF,
			0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFE00F, 0xFC31FFFF,
			0x00FFFFFF, 0x00000000, 0xFFFF0000, 0xFFFFFFFF,
			0xFFFFFFFF, 0xF80001FF, 0x00000003, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0xFFFFD740, 0xFFFFFFFB, 0x547F7FFF, 0x000FFFFD,
			0xFFFFDFFE, 0xFFFFFFFF, 0xDFFEFFFF, 0xFFFFFFFF,
			0xFFFF0003, 0xFFFFFFFF, 0xFFFF199F, 0x033FCFFF,
			0x00000000, 0xFFFE0000, 0x027FFFFF, 0xFFFFFFFE,
			0x0000007F, 0x00000000, 0xFFFF0000, 0x000707FF,
			0x00000000, 0x07FFFFFE, 0x000007FE, 0xFFFE0000,
			0xFFFFFFFF, 0x7CFFFFFF, 0x002F7FFF, 0x00000060,
			0xFFFFFFE0, 0x23FFFFFF, 0xFF000000, 0x00000003,
			0xFFF99FE0, 0x03C5FDFF, 0xB0000000, 0x00030003,
			0xFFF987E0, 0x036DFDFF, 0x5E000000, 0x001C0000,
			0xFFFBAFE0, 0x23EDFDFF, 0x00000000, 0x00000001,
			0xFFF99FE0, 0x23CDFDFF, 0xB0000000, 0x00000003,
			0xD63DC7E0, 0x03BFC718, 0x00000000, 0x00000000,
			0xFFFDDFE0, 0x03EFFDFF, 0x00000000, 0x00000003,
			0xFFFDDFE0, 0x03EFFDFF, 0x40000000, 0x00000003,
			0xFFFDDFE0, 0x03FFFDFF, 0x00000000, 0x00000003,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0xFFFFFFFE, 0x000D7FFF, 0x0000003F, 0x00000000,
			0xFEF02596, 0x200D6CAE, 0x0000001F, 0x00000000,
			0x00000000, 0x00000000, 0xFFFFFEFF, 0x000003FF,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0xFFFFFFFF, 0xFFFF003F, 0x007FFFFF,
			0x0007DAED, 0x50000000, 0x82315001, 0x002C62AB,
			0x40000000, 0xF580C900, 0x00000007, 0x02010800,
			0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF,
			0x0FFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x03FFFFFF,
			0x3F3FFFFF, 0xFFFFFFFF, 0xAAFF3F3F, 0x3FFFFFFF,
			0xFFFFFFFF, 0x5FDFFFFF, 0x0FCF1FDC, 0x1FDC1FFF,
			0x00000000, 0x00004C40, 0x00000000, 0x00000000,
			0x00000007, 0x00000000, 0x00000000, 0x00000000,
			0x00000080, 0x000003FE, 0xFFFFFFFE, 0xFFFFFFFF,
			0x001FFFFF, 0xFFFFFFFE, 0xFFFFFFFF, 0x07FFFFFF,
			0xFFFFFFE0, 0x00001FFF, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF,
			0xFFFFFFFF, 0x0000003F, 0x00000000, 0x00000000,
			0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF,
			0xFFFFFFFF, 0x0000000F, 0x00000000, 0x00000000,
			0x00000000, 0x07FF6000, 0x87FFFFFE, 0x07FFFFFE,
			0x00000000, 0x00800000, 0xFF7FFFFF, 0xFF7FFFFF,
			0x00FFFFFF, 0x00000000, 0xFFFF0000, 0xFFFFFFFF,
			0xFFFFFFFF, 0xF80001FF, 0x00030003, 0x00000000,
			0xFFFFFFFF, 0xFFFFFFFF, 0x0000003F, 0x00000003,
			0xFFFFD7C0, 0xFFFFFFFB, 0x547F7FFF, 0x000FFFFD,
			0xFFFFDFFE, 0xFFFFFFFF, 0xDFFEFFFF, 0xFFFFFFFF,
			0xFFFF007B, 0xFFFFFFFF, 0xFFFF199F, 0x033FCFFF,
			0x00000000, 0xFFFE0000, 0x027FFFFF, 0xFFFFFFFE,
			0xFFFE007F, 0xBBFFFFFB, 0xFFFF0016, 0x000707FF,
			0x00000000, 0x07FFFFFE, 0x0007FFFF, 0xFFFF03FF,
			0xFFFFFFFF, 0x7CFFFFFF, 0xFFEF7FFF, 0x03FF3DFF,
			0xFFFFFFEE, 0xF3FFFFFF, 0xFF1E3FFF, 0x0000FFCF,
			0xFFF99FEE, 0xD3C5FDFF, 0xB080399F, 0x0003FFCF,
			0xFFF987E4, 0xD36DFDFF, 0x5E003987, 0x001FFFC0,
			0xFFFBAFEE, 0xF3EDFDFF, 0x00003BBF, 0x0000FFC1,
			0xFFF99FEE, 0xF3CDFDFF, 0xB0C0398F, 0x0000FFC3,
			0xD63DC7EC, 0xC3BFC718, 0x00803DC7, 0x0000FF80,
			0xFFFDDFEE, 0xC3EFFDFF, 0x00603DDF, 0x0000FFC3,
			0xFFFDDFEC, 0xC3EFFDFF, 0x40603DDF, 0x0000FFC3,
			0xFFFDDFEC, 0xC3FFFDFF, 0x00803DCF, 0x0000FFC3,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0xFFFFFFFE, 0x07FF7FFF, 0x03FF7FFF, 0x00000000,
			0xFEF02596, 0x3BFF6CAE, 0x03FF3F5F, 0x00000000,
			0x03000000, 0xC2A003FF, 0xFFFFFEFF, 0xFFFE03FF,
			0xFEBF0FDF, 0x02FE3FFF, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x1FFF0000, 0x00000002,
			0x000000A0, 0x003EFFFE, 0xFFFFFFFE, 0xFFFFFFFF,
			0x661FFFFF, 0xFFFFFFFE, 0xFFFFFFFF, 0x77FFFFFF
		};
	}
}
