/*
 * output.c
 *
 * Copyright 1998, 1999 Michael Elizabeth Chastain, <mailto:mec@shout.net>.
 * Licensed under the Gnu Public License, Version 2.
 *
 * Write to the output files.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "mconfig.h"


/*
 * An output block keeps track of output state in the scb function.
 */
typedef struct {
	stretch_type    stretch_config_out;
	stretch_type    stretch_autoconf_h;
} output_type;

static void
output_co_piece(output_type *output, piece_type piece)
{
	stretch_append(&output->stretch_config_out, piece.ptr, piece.len);
}

static void
output_co_string(output_type *output, const char *str)
{
	stretch_append(&output->stretch_config_out, str, strlen(str));
}

static void
output_co_symbol(output_type *output, const symbol_type *sym)
{
	stretch_append(&output->stretch_config_out,
			sym->name.ptr, sym->name.len);
}

static void
output_ac_piece(output_type *output, piece_type piece)
{
	stretch_append(&output->stretch_autoconf_h, piece.ptr, piece.len);
}

static void
output_ac_string(output_type *output, const char *str)
{
	stretch_append(&output->stretch_autoconf_h, str, strlen(str));
}

static void
output_ac_symbol(output_type *output, const symbol_type *sym)
{
	stretch_append(&output->stretch_autoconf_h,
			sym->name.ptr, sym->name.len);
}


/*
 * Process a statement to write its variable out to a file.
 *
 * I generate all the output in memory first in order to separate the output
 * generation code from a bunch of delicate file-handling code.
 */
static void 
scb_output(const statement_type * statement, void *param_output)
{
	output_type *output = (output_type *)param_output;

	/*
         * Construct output.
         */

	switch (statement->verb) {
	case verb_MENU:
	case verb_mainmenu_name:
	case verb_text:
	case verb_unset:
		break;

	case verb_comment:
		output_co_string(output, "\n#\n# ");
		output_co_piece(output, statement->sb_prompt->value);
		output_co_string(output, "\n#\n");

		output_ac_string(output, "\n/*\n * ");
		output_ac_piece(output, statement->sb_prompt->value);
		output_ac_string(output, "\n */\n");
		break;

	case verb_ask_bool:
	case verb_def_bool:
	case verb_dep_bool:
	case verb_dep_mbool:
	case verb_ask_tristate:
	case verb_def_tristate:
	case verb_dep_tristate:
	    {
		const symbol_type *symbol = statement->sb_symbol;
		piece_type value = symbol->value;

		if (value.len == 1 && value.ptr[0] == 'y') {
			output_co_symbol(output, symbol);
			output_co_string(output, "=y\n");

			output_ac_string(output, "#define ");
			output_ac_symbol(output, symbol);
			output_ac_string(output, " 1\n");
		} else if (value.len == 1 && value.ptr[0] == 'm') {
			output_co_symbol(output, symbol);
			output_co_string(output, "=m\n");

			output_ac_string(output, "#undef  ");
			output_ac_symbol(output, symbol);
			output_ac_string(output, "\n");

			output_ac_string(output, "#define ");
			output_ac_symbol(output, symbol);
			output_ac_string(output, "_MODULE 1\n");
		} else {
			output_co_string(output, "# ");
			output_co_symbol(output, symbol);
			output_co_string(output, " is not set\n");

			output_ac_string(output, "#undef  ");
			output_ac_symbol(output, symbol);
			output_ac_string(output, "\n");
		}

		break;
	    }

	case verb_ask_hex:
	case verb_def_hex:
	case verb_dep_hex:
	case verb_ask_int:
	case verb_def_int:
	case verb_dep_int:
	case verb_ask_string:
	case verb_def_string:
	case verb_dep_string:
	    {
		const symbol_type *symbol = statement->sb_symbol;
		piece_type value = symbol->value;
		const char *str_open = "";
		const char *str_close = "";

		switch (statement->verb) {
		default:
			error_enum_bogus();
			break;

		case verb_ask_hex:
		case verb_def_hex:
		case verb_dep_hex:
			str_open = "0x";
			str_close = "";
			break;

		case verb_ask_int:
		case verb_def_int:
		case verb_dep_int:
			if (value.len > 0 && value.ptr[0] == '-') {
				str_open = "(";
				str_close = ")";
			}
			break;

		case verb_ask_string:
		case verb_def_string:
		case verb_dep_string:
			str_open = "\"";
			str_close = "\"";
			break;
		}

		output_co_symbol(output, symbol);
		output_co_string(output, "=");
		output_co_piece(output, value);
		output_co_string(output, "\n");

		output_ac_string(output, "#define ");
		output_ac_symbol(output, symbol);
		output_ac_string(output, " ");
		output_ac_string(output, str_open);
		output_ac_piece(output, value);
		output_ac_string(output, str_close);
		output_ac_string(output, "\n");

		break;
	    }

	case verb_nchoice:
	    {
		const symbol_type *symbol;

		for (symbol = statement->sn_choice_list->symbol_first;
		     symbol != NULL; symbol = symbol->next)
		{
			piece_type value = symbol->value;

			if (value.len == 1 && value.ptr[0] == 'y') {
				output_co_symbol(output, symbol);
				output_co_string(output, "=y\n");

				output_ac_string(output, "#define ");
				output_ac_symbol(output, symbol);
				output_ac_string(output, " 1\n");
			} else {
				output_co_string(output, "# ");
				output_co_symbol(output, symbol);
				output_co_string(output, " is not set\n");

				output_ac_string(output, "#undef  ");
				output_ac_symbol(output, symbol);
				output_ac_string(output, "\n");
			}
		}

		break;
	    }

	default:
	case verb_NULL:
	case verb_IF:
		error_enum_bogus();
		break;

	}
}



/*
 * Generate both output files in memory.
 */
static void 
output_generate(const block_type * block, output_type * output)
{
	/*
         * Allocate memory for the stretches.
         */
	stretch_ctor(&output->stretch_config_out, 32 * 1024);
	stretch_ctor(&output->stretch_autoconf_h, 32 * 1024);

	/*
         * Generate header for $file-target-config-out.
         */
	output_co_string(output, "#\n# This file was generated by ");
	output_co_string(output, argument.cmdname);
	output_co_string(output, " ");
	output_co_string(output, argument.version);
	output_co_string(output, ".\n# If you edit this file, you must run ");
	output_co_string(output, argument.cmdname);
	output_co_string(output, " again.\n#\n# Command line: ");
	output_co_string(output, argument.cmdline);
	output_co_string(output, "\n# Title: ");
	output_co_string(output, argument.title);

	if (argument.mode == mode_random || argument.mode == mode_mutate) {
		char            buf[64];
		sprintf(buf, "%u", argument.useed);
		output_co_string(output, "\n# Seed: ");
		output_co_string(output, buf);
	}
	output_co_string(output, "\n#\n\n");

	/*
         * Generate header for $file-target-autoconf-h.
         */
	output_ac_string(output, "/*\n * This file was generated by ");
	output_ac_string(output, argument.cmdname);
	output_ac_string(output, " ");
	output_ac_string(output, argument.version);
	output_ac_string(output, ".\n * Do not edit this file.\n");
	output_ac_string(output, " * \n * Command line: ");
	output_ac_string(output, argument.cmdline);
	output_ac_string(output, "\n * Title: ");
	output_ac_string(output, argument.title);

	if (argument.mode == mode_random || argument.mode == mode_mutate) {
		char            buf[64];
		sprintf(buf, "%u", argument.useed);
		output_ac_string(output, "\n * Seed: ");
		output_ac_string(output, buf);
	}
	output_ac_string(output, "\n */\n\n#define AUTOCONF_INCLUDED\n\n");

	/*
         * Generate the bodies.
         * I got the idea of block-walking from the old Menuconfig.
         */
	block_walk(block, 1, scb_output, output);
}


/*
 * Write output files from a block.
 *
 * Return value: 0 for success, -1 for error.  On error, the second
 * parameter is set to a string that describes the error.
 */

#define error2(n1, n2)						\
do {								\
	if (error_string != NULL)				\
		*error_string =					\
		    format_system_error(errno, (n1), (n2));	\
	retval = -1;						\
	goto label_cleanup;					\
} while (0)

#define error(n1) error2(n1, NULL)

int 
block_output(const block_type *block, const char **error_string)
{
	output_type     output;
	struct stat     stat_buf;
	int             fd_out = -1;
	int             retval = 0;
	char           *fco_new = NULL;
	char           *fco_old = NULL;
	stretch_type    stretch_cwd;
	char           *sl_current = NULL;
	char           *sl_source = NULL;
	char           *sl_target = NULL;

	stretch_ctor(&stretch_cwd, 1024);

	if (error_string != NULL)
		*error_string = NULL;

	output_generate(block, &output);

	/*
         * Write out $file-target-config-out.
         * In bash, this would look like:
         *
         *   output > .config.new
         *   rm -f .config.old
         *   [ -f .config ] && mv .config .config.old
         *   mv .config.new .config
         *
         * FIXME: only do "mv .config .config.old" on the first write.
         */
	if (argument.fco != NULL) {
		int index;

		fco_new = check_malloc(strlen(argument.fco) + 5);
		sprintf(fco_new, "%s.new", argument.fco);

		fco_old = check_malloc(strlen(argument.fco) + 5);
		sprintf(fco_old, "%s.old", argument.fco);

		if (argument.no_backup) {
			fd_out = open(argument.fco,
					O_WRONLY|O_CREAT|O_TRUNC, 0666);
			if (fd_out < 0)
				error(argument.fco);
		} else {
			fd_out = open(fco_new,
					O_WRONLY|O_CREAT|O_TRUNC, 0666);
			if (fd_out < 0)
				error(fco_new);
		}

		for (index = 0; index < output.stretch_config_out.len;) {
			const int count = write(fd_out,
					output.stretch_config_out.buf + index,
					output.stretch_config_out.len - index);

			if (count < 0)
				error(argument.no_backup ?
						argument.fco : fco_new);
			if (count == 0)
				error_internal(
				    "block_write_files: write returned zero");

			index += count;
		}

		if (close(fd_out) != 0) {
			fd_out = -1;
			error(argument.no_backup ? argument.fco : fco_new);
		}
		fd_out = -1;

		if (!argument.no_backup) {
			unlink(fco_old);

			if (stat(argument.fco, &stat_buf) == 0) {
				if (rename(argument.fco, fco_old) != 0)
					error2(argument.fco, fco_old);
			}
			if (rename(fco_new, argument.fco) != 0)
				error2(fco_new, argument.fco);
		}
	}

	/*
         * Write out $file-target-autoconf-h.
         */
	if (argument.fac != NULL) {
		int index;

		fd_out = open(argument.fac, O_WRONLY|O_CREAT|O_TRUNC, 0666);
		if (fd_out < 0)
			error(argument.fac);

		for (index = 0; index < output.stretch_autoconf_h.len;) {
			const int count = write(fd_out,
					output.stretch_autoconf_h.buf + index,
					output.stretch_autoconf_h.len - index);

			if (count < 0)
				error(argument.fac);
			if (count == 0)
				error_internal(
				    "block_write_files: write returned zero");

			index += count;
		}

		if (close(fd_out) != 0) {
			fd_out = -1;
			error(argument.fac);
		}
		fd_out = -1;
	}
	/*
         * Create the include/asm link.
         */
	if (!argument.no_asm_symlink) {
		int len_source =
		    strlen(argument.ds) + 32 + strlen(argument.arch);

		if (argument.ds[0] != '/') {
			for (;;) {
				if (getcwd(stretch_cwd.buf,
				    stretch_cwd.size) != NULL)
					break;
				if (errno != ERANGE)
					error(".");
				stretch_cwd.buf =
				    check_realloc(stretch_cwd.buf,
						  (stretch_cwd.size *= 2) *
						  sizeof(*stretch_cwd.buf));
			}
			stretch_cwd.len = strlen(stretch_cwd.buf);
			len_source += stretch_cwd.len;
		}
		sl_current = check_malloc(len_source);
		sl_source = check_malloc(len_source);
		sl_target = check_malloc(strlen(argument.dt) + 16);

		if (strcmp(argument.ds, argument.dt) == 0) {
			sprintf(sl_source, "asm-%s", argument.arch);
		} else if (argument.ds[0] == '/') {
			sprintf(sl_source, "%sinclude/asm-%s",
				argument.ds, argument.arch);
		} else {
			sprintf(sl_source, "%s/%sinclude/asm-%s",
				stretch_cwd.buf, argument.ds, argument.arch);
		}

		sprintf(sl_target, "%sinclude/asm", argument.dt);

		if (readlink(sl_target, sl_current, len_source) == len_source
		    && memcmp(sl_current, sl_source, len_source) == 0) {
			/* no action, the link is fine */
		} else {
			unlink(sl_target);
			if (symlink(sl_source, sl_target) != 0)
				error2(sl_target, sl_source);
		}
	}

	/*
         * Free at last!
         */
label_cleanup:
	if (fd_out >= 0)
		close(fd_out);

	stretch_dtor(&stretch_cwd);

	if (sl_current != NULL)
		free(sl_current);
	if (sl_target != NULL)
		free(sl_target);
	if (sl_source != NULL)
		free(sl_source);
	if (fco_old != NULL)
		free(fco_old);
	if (fco_new != NULL)
		free(fco_new);

	if (retval == 0)
		retval = output.stretch_config_out.len;

	stretch_dtor(&output.stretch_config_out);
	stretch_dtor(&output.stretch_autoconf_h);

	return retval;
}
