""" An window in the workarea. """


# Enthought library imports.
from enthought.pyface.api import Window
from enthought.pyface.action.api import StatusBarManager, ToolBarManager
from enthought.traits.api import Any, Bool, Delegate, Instance


class WorkareaWindow(Window):
    """ An window in the workarea. """

    # The parent workbench window.
    workbench_window = Any

    # The (optional) tool bar for the window.
    tool_bar_manager = Instance(ToolBarManager)

    # The (optional) status bar manager.
    status_bar_manager = Delegate('workbench_window')

    # Should the window be an mdi window?
    is_mdi = Bool(True)

    # Should the window always float above the main window?
    is_float = Bool(True)
    
    ###########################################################################
    # Protected 'Window' interface.
    ###########################################################################

    def _create_control(self, parent):
        """ Create the toolkit-specific control that represents the window.

        Here ask the workbench window to provide us with a frame.  Since it
        will ultimately manage the window, it gets to control its construction.
        For example, if the workbench window is operating in MDI mode, it will
        return a properly parented MDI child window.  In SDI mode, it would
        return an SDI window.
        """

        workbench_window = self._get_workbench_window(self.workbench_window)

        workarea_window = workbench_window.create_editor(
            self.title, self.tool_bar_manager, self.is_mdi, float=self.is_float
        )

        if not self.is_mdi:
            self.status_bar_manager = manager = StatusBarManager()
            status_bar = manager.create_status_bar(workarea_window)
            workarea_window.SetStatusBar(status_bar)
            
        return workarea_window

    ###########################################################################
    # Editor interface.
    ###########################################################################

    def get_auxillary_views(self):
        """ Get additional views that should be displayed with the editor. """

        return []
    
    ###########################################################################
    # Private interface.
    ###########################################################################

    def _get_workbench_window(self, workbench_window):
        """ Get the appropriate workbench window. """

        if workbench_window is None:
            from enthought.envisage.ui import UIPlugin
            workbench_window = UIPlugin.instance.active_window

        self.workbench_window = workbench_window

        return workbench_window

    #### Trait event handlers #################################################

    #### Static ####

    def _opened_changed(self):
        """ Called when the editor is opened. """

        # Notify the active workbench window that a new editor has opened.
        workbench_window = self._get_workbench_window(self.workbench_window)
        workbench_window.editor_opened(self)

        return
        
    def _size_changed(self, size):
        """ Called when the size is changed. """
        
        if self.control is not None:
            position = self.control.GetPositionTuple()
            self.control.SetSize(size)

            # WX FIXME HACK HACK HACK
            # WX (in 2.4.12 at least) gets confused about the position of the 
            # client area windows.
            # UPDATE: wx 2.6.1 does NOT fix this problem.
            client_position = self.workbench_window._wx_offset
            self.control.SetPosition((position[0] + client_position[0], 
                                      position[1] + client_position[1]))
        return
        


#### EOF ######################################################################
