""" Abstract base class for all workbench preference pages. """


# Enthought library imports.
from enthought.envisage import get_application
from enthought.pyface.preference.api import PreferencePage

# Local imports.
from preference_helper import PreferenceHelper, FontPreferenceHelper


class WorkbenchPreferencePage(PreferencePage):
    """ Abstract base class for all workbench preference pages. """

    # All preference helpers.
    PREFERENCE_HELPERS = [
        FontPreferenceHelper
    ]

    # Mapping of trait names to the preference instance that contains the
    # value for the trait.
    _preference_map = {}


    ###########################################################################
    # 'PreferencePage' interface.
    ###########################################################################

    def __init__(self, help_id="", *args, **kw):
        """ Construct with an empty list of preference containers. """
        self._preference_map = {}
        self._preferences = []
        self.help_id = help_id
        super(WorkbenchPreferencePage, self).__init__(*args, **kw)


    def create_control(self, parent):
        """ Creates the toolkit-specific control for the page. """

        # Initialize the page with the preferences.
        self._initialize_preferences()

        # The control is just the page's trait sheet!
        ui = self.edit_traits(parent=parent, kind='subpanel')

        return ui.control


    def restore_defaults(self):
        """ Restore the default preference values. """

        for preference_set in self._preferences:
            for pref_name, pref_value in preference_set.defaults.items():
                preference_set.set(pref_name, preference_set.get_default(pref_name))

        return


    def show_help_topic(self):
        """ Show the help topic for this preference page. """
        try: 
            hp = get_application().get_service('enthought.help.IHelp')
            hp.library.show_topic(self.help_id)
        except SystemError:
            pass # Do nothing if no Help service
        return


    ###########################################################################
    # Protected 'WorkbenchPreferencePage' interface.
    ###########################################################################

    def _get_preferences(self):
        """ Returns the preferences that this page is editing.

            This may be a single instance or a list of instances.
            The instances are expected to conform to the API of
            enthought.envisage.core.preferences.Preferences
        """

        raise NotImplementedError


    def _get_preferences_as_list(self):
        """ Returns a list of the preferences that this page is editing.

            This method simply converts the return type of _get_preferences
            so that code needing preferences can expect a constant return value.
        """

        preferences = self._get_preferences()
        if not isinstance(preferences, list):
            preferences = [preferences]
        return preferences


    ###########################################################################
    # Private interface.
    ###########################################################################

    def _initialize_preferences(self):
        """ Initializes the page with the specified preferences. """

        # Clear out any previous history of what preference instances
        # contained what traits.
        self._preference_map = {}

        # The preferences object that we are editing.
        self._preferences = self._get_preferences_as_list()

        # Get the preference value for each of this editor's traits.
        trait_dict = self.get() # Get current traits and values
        for trait_name in trait_dict:
            # fixme: Ignore traits defined on the base 'HasTraits' class.
            # And instance variables on this class
            if not (trait_name.startswith('trait_') or \
                   trait_name.startswith("_preference")):
                # Use the first preferences instance to define a value.
                for p in self._preferences:
                    if p.contains(trait_name):
                        value = p.get(trait_name)
                        self._preference_map[trait_name] = p
                        break
                else:
                    # If the trait is not in the preferences list, use its
                    # current value.
                    value = trait_dict[trait_name]
                if isinstance(value, PreferenceHelper):
                    value = value.get_editor_value(value)

                setattr(self, trait_name, value)

        # Even thought we are listening to our own trait changes, we use a
        # dynamic handler so that we don't get called on the above calls to
        # 'setattr'.
        self.on_trait_change(self._on_anytrait_changed)

        return


    ###########################################################################
    # Trait event handlers.
    ###########################################################################

    def _on_anytrait_changed(self, object, trait_name, old, new):
        """ Called when a trait has been changed. """

        for helper in self.PREFERENCE_HELPERS:
            if helper.handles(new):
                new = helper.get_preference_value(new)
                break

        self._preference_map[trait_name].set(trait_name, new)

        return


#### EOF ######################################################################
