#-------------------------------------------------------------------------------
#
#  Reusable Envisage related trait definitions
#
#  Written by: David C. Morrill
#
#  Date: 06/15/2006
#
#  (c) Copyright 2006 by Enthought, Inc.
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------
"""
Reusable trait definitions that are related to Envisage.
"""
from enthought.traits.api \
    import Trait, Property, TraitHandler, TraitFactory, TraitError, Undefined

from enthought.traits.trait_base \
    import class_of
    
#-------------------------------------------------------------------------------
#  'UOL' trait definition:  
#-------------------------------------------------------------------------------
    
def _get_uol ( object, name ):
    name   = '_' + name
    result = getattr( object, name, None )
    if result is not Undefined:
        return result
    try:
        from enthought.envisage import get_application
        result = get_application().lookup_application_object( 
                     getattr( object, name + '_' ) )
    except:
        raise TraitError, ("Could not resolve the UOL: '%s'" % 
                           getattr( object, name + '_' ))
    setattr( object, name, result )
    return result
        
def _set_uol ( object, name, value ):
    name   = '_' + name
    if isinstance( value, basestring ):
        setattr( object, name + '_', value )
        setattr( object, name, Undefined )
    else:
        setattr( object, name, value )
        
UOL = TraitFactory( Property( _get_uol, _set_uol ) )

#-------------------------------------------------------------------------------
#  'TraitUOL' class:
#-------------------------------------------------------------------------------

class TraitUOL ( TraitHandler ):
    """Checks whether the value assigned to a trait attribute is a valid Envisage
       UOL (Universal Object Locator) and maps it to its corresponding object.
    """

    def __init__ ( self, klass = None, allow_none = False ):
        self.klass      = klass
        self.allow_none = allow_none

    def _coerce ( self, value ):        
        if isinstance( value, basestring ):
            try:
                from enthought.envisage import get_application
                return get_application().lookup_application_object( value )
            except:
                import traceback
                traceback.print_exc()
                raise TraitError
                
        return value

    def validate ( self, object, name, value ):
        try:
            value = self._coerce( value )
        except:
            self.error( object, name, self.repr( value ) )

        if value is None:
            if self.allow_none:
                return value
        elif (self.klass is None) or isinstance( value, self.klass ):
            return value

        self.error( object, name, self.repr( value ) )

    def info ( self ):
        klass = self.klass
        if klass is None:
            phrase = 'any object'
        else:
            phrase = class_of( klass.__name__ )
        if self.allow_none:
            phrase += ' or None'
        return ('%s or an Envisage UOL (Universal Object Locator) which '
                'resolves to %s') % ( phrase, phrase )

#-------------------------------------------------------------------------------
#  'MappedTraitUOL' class:
#-------------------------------------------------------------------------------

class MappedTraitUOL ( TraitUOL ):
    """Checks whether the value assigned to a trait attribute is a valid Envisage
       UOL (Universal Object Locator) and maps it to its corresponding object,
       for mapped traits.
    """
    is_mapped = True

    def validate ( self, object, name, value ):
        try:
            instance = self._coerce( value )
        except:
            self.error( object, name, self.repr( value ) )

        if instance is None:
            if self.allow_none:
                return value
        elif (self.klass is None) or isinstance( instance, self.klass ):
            return value

        self.error( object, name, self.repr( value ) )

    def mapped_value ( self, value ):
        if isinstance( value, basestring ):
            try:
                from enthought.envisage import get_application
                return get_application().lookup_application_object( value )
            except:
                raise TraitError, 'Unmappable'

        return value

    def post_setattr ( self, object, name, value ):
        setattr( object, name + '_', self.mapped_value( value ) )

#-------------------------------------------------------------------------------
#  'InstanceUOL' trait
#-------------------------------------------------------------------------------

def InstanceUOL ( default = '', klass = None, allow_none = False,
                  mapped = False, **metadata ):
    """ Returns a trait whose assigned value must be an Envisage UOL (Universal
    Object Locator).
    """
    if not isinstance( default, basestring ) and (klass is None):
        default, klass = '', default
    if mapped:
        handler = MappedTraitUOL( klass, allow_none )
    else:
        handler = TraitUOL( klass, allow_none )
    return Trait( default, handler, **metadata )

InstanceUOL = TraitFactory( InstanceUOL )
    
#-------------------------------------------------------------------------------
#  'ExtensionPoints' trait definition:  
#-------------------------------------------------------------------------------
    
def _get_extension_points ( object, name ):
    name   = '_' + name
    result = getattr( object, name, None )
    if result is not Undefined:
        return result
    try:
        from enthought.envisage import get_application
        application = get_application()
        klass       = application.import_symbol( getattr( object, name + '_' ) )
        result      = application.load_extensions( klass )
    except:
        raise TraitError, ("Could not resolve the extension points for: '%s'" % 
                           getattr( object, name + '_' ))
    setattr( object, name, result )
    return result
        
def _set_extension_points ( object, name, value ):
    name = '_' + name
    setattr( object, name + '_', value )
    setattr( object, name, Undefined )
        
ExtensionPoints = TraitFactory( Property( _get_extension_points,
                                          _set_extension_points ) )
    
#-------------------------------------------------------------------------------
#  'Import' trait definition:  
#-------------------------------------------------------------------------------
    
def _get_import ( object, name ):
    name   = '_' + name
    result = getattr( object, name, None )
    if result is not Undefined:
        return result
    try:
        from enthought.envisage import get_application
        application = get_application()
        result      = application.import_symbol( getattr( object, name + '_' ) )
    except:
        raise TraitError, ("Could not import: '%s'" % 
                           getattr( object, name + '_' ))
    setattr( object, name, result )
    return result
        
Import = TraitFactory( Property( _get_import, _set_extension_points ) )

