""" A node monitor for lists. """


# Enthought library imports.
from enthought.naming.api import Binding
from enthought.naming.adapter import ListContextAdapter
from enthought.pyface.tree.api import NodeMonitor
from enthought.traits.api import Any, Str


class ListNodeMonitor(NodeMonitor):
    """ A monitor for lists. """

    #### 'ListNodeMonitor' interface ##########################################

    # fixme: These are for trait lists only, not regular Python lists. We
    # should put these in a 'TraitListNode' monitor derived from this one and
    # make sure we have a separate node type and resource type etc.
    #
    # The name of the trait that contains the list!
    list_trait = Str

    # The name of the trait on the *items* of the list that contains the 'name'
    # used in creating bindings.
    item_name_trait = Str('name')
    
    ###########################################################################
    # 'NodeMonitor' interface.
    ###########################################################################

    def start(self):
        """ Start listening to changes to the node. """

        trait_list = self._get_trait_list()

        # For trait lists ONLY.
        #
        # Get the object that contains the list.
        if hasattr(trait_list, 'object'):
            self._setup_trait_change_handlers(trait_list)
            
        return
    
    def stop(self):
        """ Stop listening to changes to the node. """

        trait_list = self._get_trait_list()

        # For trait lists ONLY.
        #
        # Get the object that contains the list.
        if hasattr(trait_list, 'object'):
            self._setup_trait_change_handlers(trait_list, remove=True)

        return

    ###########################################################################
    # Protected 'ListNodeMonitor' interface.
    ###########################################################################

    def _get_trait_list(self):
        """ Returns the trait list that we are monitoring. """

        return self.node.obj

    def _get_context(self):
        """ Returns the context used to create bindings. """

        if self.node.context is not None:
            environment = self.node.context.environment

        else:
            from enthought.naming.context import ENVIRONMENT
            environment = ENVIRONMENT.copy()
            
        # The parent context is the list (via an adapter of course).
        context = ListContextAdapter(
            adaptee     = self._get_trait_list(),
            environment = environment
        )

        return context
    
    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_bindings(self, items):
        """ Creates a list of bindings for a list of items. """

        # The context tfor the bindings is the list (via an adapter of course).
        context = self._get_context()
        
        bindings = [
            Binding(
                name    = getattr(item, self.item_name_trait),
                obj     = item,
                context = context
            )

            for item in items
        ]

        return bindings

    def _setup_trait_change_handlers(self, trait_list, remove=False):
        """ Adds or removes trait change handlers to/from a trait list. """

        obj = trait_list.object()
        if obj is not None:
            obj.on_trait_change(
                self._on_list_changed, self.node.obj.name, remove=remove
            )
            
            obj.on_trait_change(
                self._on_items_changed, self.node.obj.name_items, remove=remove
            )

        return

    #### Trait change handlers ################################################
    
    def _on_list_changed(self, obj, trait_name, old, new):
        """ Called when the entire list has changed. """

        if len(old) > 0:
            self.fire_nodes_removed(self._create_bindings(old))

        if len(new) > 0:
            self.fire_nodes_inserted(self._create_bindings(new))

        return

    def _on_items_changed(self, event):
        """ Called when items in the list have changed. """

        if len(event.removed) > 0:
            self.fire_nodes_removed(self._create_bindings(event.removed))

        if len(event.added) > 0:
            self.fire_nodes_inserted(self._create_bindings(event.added))
            
        return
    
#### EOF ######################################################################
