""" Displays the new file wizard. """


# Standard library imports.
from os.path import join, splitext

# Enthought library imports.
from enthought.io import File
from enthought.pyface.api import GUI, OK

# Plugin imports.
from enthought.envisage.project import CookieManager
from enthought.envisage.project.wizard.new_file_wizard import FileFactory
from enthought.envisage.project.wizard.new_file_wizard import NewFileWizard

# Local imports.
from new_resource_action import NewResourceAction
from open_cookie import OpenCookie


class NewFileAction(NewResourceAction):
    """ Displays the new file wizard. """

    ###########################################################################
    # 'Action' interface.
    ###########################################################################

    def perform(self, event):
        """ Performs the action. """

        parent_folder = self._get_default_folder()
        if parent_folder is not None and hasattr(event, 'widget'):
            # Make sure that the folder that the resource is to be created in
            # is expanded.
            event.widget.expand(parent_folder)
            
            # The context to create the folder in.
            context = parent_folder.obj

            # Generate a name that is unique within the context.
            default_name = self._get_default_name('New Script.py', context)

            # Create the folder.
            context.bind(default_name, File(join(context.path, default_name)))

            # Lookup the binding.
            binding = parent_folder.obj.lookup_binding(default_name)

            # Start a label edit.
            GUI.invoke_later(
                event.widget.edit_label, *(binding, self._on_label_edited)
            )

        else:
            # The factory used to create the file.
            factory = FileFactory(parent_folder=self._get_default_folder())

            # Open the wizard.
            wizard = NewFileWizard(parent=self.window.control, factory=factory)
            if wizard.open() == OK:
                # Create the resource.
                wizard.create_resource()
            
        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _get_default_name(self, prefix, context):
        """ Returns a name that is not already bound in a context. """

        names = context.list_names('')
        name, ext  = splitext(prefix)
        
        id = 2
        while name + '.py' in names:
            name = '%s (%d)' % (prefix, id)
            id += 1
            
        return name + '.py'

    def _on_label_edited(self, tree, node, label):
        """ Called when the label edit is complete. """

        cookie_manager = CookieManager()
        
        cookie = cookie_manager.get_cookie(OpenCookie, node.obj)
        cookie.open(self.window, node)
        
        return
        
#### EOF ######################################################################
