
# major package imports
import logging
import os
import wx
from zipfile import is_zipfile, ZipFile

# Enthought library imports
from enthought.envisage import get_application
from enthought.envisage.core.core_plugin import CorePlugin
from enthought.envisage.core.core_plugin_definition import UserAddedPlugin
from enthought.envisage.ui.preference import WorkbenchPreferencePage
from enthought.traits.api import Button, HasTraits
from enthought.traits.ui.api import Group, Item, TableEditor, View
from enthought.traits.ui.table_column import ObjectColumn
from enthought.util.wx.dialog import error


# Setup a logger for this module.
logger = logging.getLogger(__name__)


def browse_for_new_plugin(parent):

    result = UserAddedPlugin()
    mask = 'Python files (*.py)|*.py|Zip files (*.zip)|*.zip|All files|*.*'

    while True:
        dlg = wx.FileDialog(parent, "Select a Plug-in", '',
                            '', mask, wx.OPEN)

        if dlg.ShowModal() == wx.ID_OK:
            result.path = dlg.GetPath()
            # if the path is a zip file, then check to make sure we can find
            # the plugin definition
            if is_zipfile(result.path):
                zipname, ext = os.path.splitext(os.path.basename(result.path))
                z = ZipFile(result.path)
                try:
                    # the plugin definition file must live at the top of
                    # the zip archive
                    z.getinfo(zipname + '_plugin_definition.py')
                    result.path = result.path + '/' + zipname + \
                                  '_plugin_definition.py'
                except:
                    # pop up an error dialog telling the user the problem
                    # and then go back to the file dialog
                    logger.error('Unable to find file ' + zipname + \
                                 '_plugin_definition.py in zip archive ' + \
                                 result.path)
                    error(parent,
                          'Unable to find file ' + zipname + \
                          '_plugin_definition.py in zip archive ' + \
                          result.path,
                          'Invalid plugin definition')
                    z.close()
                    continue

                # if we get to this point, then we have a valid zip file, so
                # we close it and then break out of the loop
                z.close()
                break
            else:
                # if it's not a zip file then we assume everything is okay
                # and return
                break
        else:
            result = None
            break

    return result

plugin_columns = [ ObjectColumn(name = 'enabled', editable = True),
                   ObjectColumn(name = 'path', width = 0.8, editable = False),
                 ]

# a dummy class used to provide a button to the user starting any added plugins
class RefreshPluginsActor(HasTraits):

    start = Button(label = 'Start Plug-ins')

    def _start_changed(self):

        # notify the core plugin to start any additional plugins
        app = get_application()
        core = app.service_registry.get_service(CorePlugin.IADDITIONS_MANAGER)
        core._load_and_start_additional_plugins()
        return


class AdditionsPreferencePage(WorkbenchPreferencePage):
    """ A preference page for the Additions plugin. """

    ###########################################################################
    # Protected 'WorkbenchPreferencePage' interface.
    ###########################################################################

    def create_control(self, parent):
        """ Creates the toolkit-specific control for the page. """

        # define the plugin editor
        plugin_editor = TableEditor(columns = plugin_columns,
                                    configurable = False,
                                    edit_view = '',
                                    orientation = 'vertical',
                                    row_factory = browse_for_new_plugin,
                                    row_factory_args = (parent,),
                                    show_column_labels = False,
                                    deletable = True,
                                    sortable = False,
                                    )

        # The control is just the page's trait sheet!
        app = get_application()
        core = app.service_registry.get_service(CorePlugin.IADDITIONS_MANAGER)
        actor = RefreshPluginsActor()
        context = { 'object': core,
                    'actor': actor,
                  }
        view = View(Group(Item(name = 'user_added_plugins',
                               editor = plugin_editor,
                               ),
                          Item(name = 'start',
                               object = 'actor',
                               ),
                          show_labels = False,
                          show_border = True,
                          label = 'Available Plug-ins',
                          ),
                    )

        ui = view.ui(parent = parent, kind = 'subpanel', context = context)

        return ui.control

#### EOF ######################################################################


