""" Builds tool bars from action, group and menu extensions. """


# Enthought library imports.
from enthought.envisage import action
from enthought.envisage.workbench import Window
from enthought.pyface.action.api import Group
from enthought.traits.api import Instance

# Local imports.
from workbench_action_factory import WorkbenchActionFactory


class ToolBarBuilder(action.MenuBuilder):
    """ Builds tool bars from action, group and menu extensions. """

    #### 'MenuBuilder' interface ##############################################

    # The workbench window that we are building tool bars for.
    window = Instance(Window)

    ###########################################################################
    # Protected 'action.MenuBuilder' interface.
    ###########################################################################

    def _create_action(self, extension):
        """ Creates an action implementation from an action extension. """

        factory = WorkbenchActionFactory()

        return factory.create_action(self.window, extension)

    def _create_group(self, extension):
        """ Creates a group implementation from a group extension. """

        if len(extension.class_name) > 0:
            klass = self._import_class(extension)

        else:
            klass = Group

        return klass(id=extension.id, separator=extension.separator)

    def _create_menu_manager(self, extension):
        """ Creates a menu manager implementation from a menu extension. """

        raise NotImplementedError

    ###########################################################################
    # Private interface.
    ###########################################################################

    # fixme: This is such a common thing to do in Envisage that we should
    # put it on the Application.
    def _import_class(self, extension):
        """ Imports an extension implementation class. """

        application = self.window.application

        # The plugin definition that the extension was defined in.
        plugin_definition = extension._action_set_._definition_

        # We are about to actually import the menu manager class, so make
        # sure that the plugin that contributed it has been started.
        application.start_plugin(plugin_definition.id)

        # Import the implementation class
        return application.import_symbol(extension.class_name)

#### EOF ######################################################################
