/*!
  -----------------------------------------------------------------------------

  module: vls27_dataextract.cpp

  -----------------------------------------------------------------------------

  responsible:  SteffenS and d033893

  special area: SAP DB LOADER

  description:  Implementation of functions for dataextract

  version:      7.5.

  -----------------------------------------------------------------------------

  copyright:    (c) 1999-2004 SAP AG-2003

  -----------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 1999-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
*/

#include <time.h>

#include "gls00.h"
#include "hsp77.h"
#include "hls04.h"
#include "hls041.h"
#include "hls05.h"
#include "hls07.h"
#include "hls10.h"
#include "hls11.h"
#include "hls13.h"
#include "hls15.h"
#include "hls20.h"
#include "hls25.h"
#include "hls27_dataextract.h"
#include "hls27_DEExec.h"
#include "hls31_odbc_Result.h"
#include "hls31_catalogextract.h"
#include "hls12.h"
#include "hls18.h"
#include "hls30.h"
#include "hls98msg.h"
#include "hls99.h"


/*
  -----------------------------------------------------------------------------
  Chapter: Local functions declaration
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------

  function: ls27_ExtractCatalog

  -----------------------------------------------------------------------------

  description:


  arguments:    pDBInfo     [IN]   - structure with all information on the running
                                     session and the db connected to.
                pDatExrCmd  [IN]   - structure with information for de supplied by the user
                ErrText     [OUT]  - error text

  returnvalue:

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_ExtractCatalog(tls00_DBInfo      *pDBInfo,
                    tls00_DatExtrCmd  *pDatExtrCmd,
                    tsp00_Addr         ErrText);


/*
  -----------------------------------------------------------------------------

  function:     ls27_InitDatExtrRec  - local function

  -----------------------------------------------------------------------------

  description:  initialize structure used for execution of table extract


  arguments:    pDBInfo      [IN]   - structure with all information on the running
                                      session and the db connected to.
                pDatExtrCmd  [IN]   - structure with information for te supplied by the user
                pDatExtrRec  [IN]   - dataextract record structure that must be initialized.

  returnvalue:  errOK_els00, errNoMoreMemory_els00
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_InitDatExtrRec(tls00_DBInfo      *pDBInfo,
                    tls00_DatExtrCmd  *pDatExtrCmd,
                    tls00_DextractRec *&pDatExtrRec,
                    tsp00_Addr         ErrText );

/*
  -----------------------------------------------------------------------------
  function: ls27_FreeDatExtrRec
  -----------------------------------------------------------------------------

  description:  destroys the tableextract structure built in ls27_InitDatExtrRec


  arguments:    pDatExtrCmd  [IN]   - structure with information supplied by the user
                pDatExtrRec  [IN]   - dataextract record structure

  returnvalue:  No
  -----------------------------------------------------------------------------
*/
static void
ls27_FreeDatExtrRec(tls00_DatExtrCmd* pDatExtrCmd, tls00_DextractRec* pDatExtrRec);


/*
  -----------------------------------------------------------------------------
  function:     ls27_InitStreams  - local function
  -----------------------------------------------------------------------------

  description:


  arguments:    pDBInfo      [IN]   - structure with all information on the running
                                      session and the db connected to.
                pDatExtrCmd  [IN]   - structure with information for te supplied by the user
                pDatExtrRec  [IN]   - dataextract record structure that must be initialized.

  returnvalue:

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_InitStreams( tls00_DBInfo      *pDBInfo,
                  tls00_DatExtrCmd  *pDatExtrCmd,
                  tls00_DextractRec *pDatExtrRec,
                  tsp00_Addr         ErrText);


/*
  -----------------------------------------------------------------------------
  function:     ls27_SwitchStreams - local function
  -----------------------------------------------------------------------------

  description:


  arguments:    pDBInfo      [IN]   - structure with all information on the running
                                      session and the db connected to.
                pDatExtrCmd  [IN]   - structure with information for te supplied by the user
                pDatExtrRec  [IN]   - dataextract record structure that must be initialized.

  returnvalue:

  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_SwitchStreams( tls00_DBInfo      *pDBInfo,
                    tls00_DatExtrCmd  *pDatExtrCmd,
                    tls00_DextractRec *pDatExtrRec,
                    tsp00_Addr        ErrText);

/*
  -----------------------------------------------------------------------------
  function:     ls27_CloseStreams       - local function
  -----------------------------------------------------------------------------
  description:  Closes open streams

  arguments:    pDatExtrCmd  [IN]   - structure with information supplied by the user
                pDatExtrRec  [IN]   - dataextract record structure

  returnvalue:  None
  -----------------------------------------------------------------------------
*/
static void
ls27_CloseStreams(tls00_DatExtrCmd* pDatExtrCmd, tls00_DextractRec* pDatExtrRec);


/*
  -----------------------------------------------------------------------------
  function:     ls27_FreeStreams       - local function
  -----------------------------------------------------------------------------
  description:

  arguments:    pDatExtrCmd  [IN]   - structure with information supplied by the user
                pDatExtrRec  [IN]   - dataextract record structure

  returnvalue:

  -----------------------------------------------------------------------------
*/
static void
ls27_FreeStreams(tls00_DatExtrCmd* pDatExtrCmd, tls00_DextractRec* pDatExtrRec);


/*
  -----------------------------------------------------------------------------
  EndChapter  : Local functions declaration
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  BeginChapter: Local functions implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  function:     ls27_ExtractCatalog
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_ExtractCatalog(tls00_DBInfo      *pDBInfo,
                    tls00_DatExtrCmd  *pDatExtrCmd,
                    tsp00_Addr         ErrText)
{
    ROUTINE_DBG_MEO00 ("ls27ExtractCatalog");

    tsp00_Int4 rc     = errOK_els00;

    if ( pDatExtrCmd->dexRestore_ls00 < resDataUpdt_els00      &&
         pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 > ptTable_els00 )
    {
        pDatExtrCmd->dexCmdFile_ls00.fsAppend_ls00 = true;

        // append ddl stmts
        Tools_DynamicUTF8String  catalogextract;
        Tools_DynamicUTF8String  outfile;
        outfile.Erase();

        catalogextract = "CATALOGEXTRACT ";

        if (ptAll_els00 == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00)
            catalogextract += " ALL ";
        if (ptUser_els00 == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00)
            catalogextract += " USER ";
        outfile.ConvertFromASCII_Latin1( pDatExtrCmd->dexCmdFile_ls00.fsFileName_ls00,
                                         pDatExtrCmd->dexCmdFile_ls00.fsFileName_ls00 +
                                 _tcslen(pDatExtrCmd->dexCmdFile_ls00.fsFileName_ls00));

        catalogextract += " OUTSTREAM ";
        catalogextract += "'";
        catalogextract += outfile;
        catalogextract += "'";

        rc = ls11ParseDispatchLoadCommand (catalogextract, pDBInfo, ErrText);

        // append load       stmts
        // append extracted  data
    }
    return rc;
}
// ls27_ExtractCatalog()


/*
  -----------------------------------------------------------------------------
  function:     ls27_InitDatExtrRec  - local function
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_InitDatExtrRec( tls00_DBInfo      *pDBInfo,
                     tls00_DatExtrCmd  *pDatExtrCmd,
                     tls00_DextractRec *&pDatExtrRec,
                     tsp00_Addr        ErrText )
{
    ROUTINE_DBG_MEO00 ("ls27_InitDatExtrRec");

    tsp00_Int4 rc     = errOK_els00;

    pDatExtrRec = new tls00_DextractRec;
    if (NULL == pDatExtrRec )
        rc = errNoMoreMemory_els98;
    else
    {
        memset(pDatExtrRec, 0, sizeof(tls00_DextractRec));

        //create and initialize mapping file and outfile
        if (pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 > ptTable_els00)
        {
            pDatExtrRec->derTaskSwitched_ls00 = true;

            pDatExtrRec->derPartSpec_ls00 = pDatExtrCmd->dexPartSpec_ls00;
            pDatExtrRec->derRestore_ls00  = pDatExtrCmd->dexRestore_ls00;

            // repman sessioninfo
            pDatExtrRec->derDefStruct_ls00.pdUserDefault      = pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp();
            pDatExtrRec->derDefStruct_ls00.pdUserGroupDefault = pDBInfo->dbiSession_ls00.siUserGroup_ls00.asCharp();
            memset(pDatExtrRec->derDefStruct_ls00.pdOracleUid_ls00, BLANK_LS00, sizeof(tsp00_Number));
            pDatExtrRec->derSwapKind_ls00                     = pDBInfo->dbiOwnSwap_ls00;
            // rejected = siMaxAdmissibleErrors - derLRejected_ls00
            pDatExtrRec->derLRejected_ls00           = pDBInfo->dbiSession_ls00.siMaxAdmissibleErrors;

            pDatExtrRec->lNameCount_ls00             = (pDBInfo->dbiPktSize_ls00 + 4*1024)/(sizeof(tsp00_KnlIdentifierc) * 2);
            pDatExtrRec->derStartPos_ls00            = 0;

            pDatExtrRec->derMediumSize_ls00          = MAX_MEDIUM_SIZE_IN_BYTES_LS27;    // 1 GB in Byte so far !
            pDatExtrRec->derMediumLongIntBlocks_ls00 = 0;

        }// all|user
        else
        {
            pDatExtrRec->derPartSpec_ls00 = pDatExtrCmd->dexPartSpec_ls00;

            // repman sessioninfo
            pDatExtrRec->derDefStruct_ls00.pdUserDefault      = pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp();
            pDatExtrRec->derDefStruct_ls00.pdUserGroupDefault = pDBInfo->dbiSession_ls00.siUserGroup_ls00.asCharp();
            memset(pDatExtrRec->derDefStruct_ls00.pdOracleUid_ls00, BLANK_LS00, sizeof(tsp00_Number));
            pDatExtrRec->derSwapKind_ls00                     = pDBInfo->dbiOwnSwap_ls00;
            // rejected = siMaxAdmissibleErrors - derLRejected_ls00
            pDatExtrRec->derLRejected_ls00           = pDBInfo->dbiSession_ls00.siMaxAdmissibleErrors;

            pDatExtrRec->lNameCount_ls00             = 1;
            pDatExtrRec->derpStartPos_ls00[0]        = 0;

            pDatExtrRec->derMediumSize_ls00          = MAX_MEDIUM_SIZE_IN_BYTES_LS27;    // 1 GB in Byte so far !
            pDatExtrRec->derMediumLongIntBlocks_ls00 = 0;

            // table
            pDatExtrRec->derTabEmpty_ls00 = false;
            memset(&pDatExtrRec->derDataSource_ls00, 0, sizeof(tls00_DataSource));

            pDatExtrRec->derTableInfo_ls00.tdKernelSwap_ls00.becomes(sw_dummy);
            pDatExtrRec->derFillValue_ls00           = ls07GetFillByte(dcha);

             // pattern structure in case of external data file format COMPRESSED
            if ( NULL != pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00 &&
                 true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompress_ls00)
            {
                pDatExtrRec->pderPattern_ls00 = new tls00_Pattern;
                memset(pDatExtrRec->pderPattern_ls00, 0, sizeof(tls00_Pattern));

                rc = ls18ConstructPattern(pDatExtrRec->pderPattern_ls00,
                                          pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoSeperat_ls00,
                                          pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoDelimit_ls00,
                                          pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCodeType_ls00,
                                          ErrText);
            }
        }   // end else of if (pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 > ptTable_els00)

        if (errOK_els00 == rc)
             rc = ls27_InitStreams( pDBInfo, pDatExtrCmd, pDatExtrRec, ErrText);
    }

    return rc;
}
// ls27_InitDatExtrRec()


/*
  -----------------------------------------------------------------------------
  function: ls27_FreeDatExtrRec
  -----------------------------------------------------------------------------
*/
static void
ls27_FreeDatExtrRec(tls00_DatExtrCmd* pDatExtrCmd, tls00_DextractRec* pDatExtrRec)
{
    ROUTINE_DBG_MEO00 ("ls27_FreeDatExtrRec");

    if ( NULL != pDatExtrRec )
    {
        if ( dtypeTape_ls00 == pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00 &&
             ptTable_els00  == pDatExtrRec->derPartSpec_ls00.psPartType_ls00 )
        {
            ls27_CloseStreams(pDatExtrCmd, pDatExtrRec);
            pDatExtrCmd->dexDatFile_ls00.fsBackupId_ls00.bBackupIdName_ls00  = REINTERPRET_CAST(SAPDB_Char *,(CONST_CAST (unsigned char* ,pDatExtrRec->derDataFile_ls00->vfBackupId_ls00.StrPtr())));
        }
        if (dtypeTape_ls00 != pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00)
            ls27_CloseStreams(pDatExtrCmd, pDatExtrRec);

        ls27_FreeStreams(pDatExtrCmd, pDatExtrRec);

        delete pDatExtrRec;
        pDatExtrRec = NULL;
    }   // if ( NULL != pDatExtrRec )
}
// ls27_FreeDatExtrRec()

/*
  -----------------------------------------------------------------------------
  EndChapter: Private functions implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
  BeginChapter: Public functions implementation
  -----------------------------------------------------------------------------
*/

/*
  -----------------------------------------------------------------------------
   function:     ls27ExecDataExtract
  -----------------------------------------------------------------------------
*/
int ls27ExecDataExtract(tls00_DatExtrCmd *pDatExtrCmd, tls00_DBInfo *pDBInfo, tsp00_Addr ErrText)
{
    ROUTINE_DBG_MEO00 ("ls27ExecDataExtract");

    RepServMessages *pMsgInstance = RepServMessages::Instance();

    tsp00_Longint rc = errOK_els00;

    //*
    //*     Before doing anything else we check external encoding and
    //*     database encoding.
    //*     Not supported is loading UNICODE data into ASCII databases.
    //*
    rc = ls05AreEncodingsCompatible(pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCodeType_ls00,
                                    ls00IsDBUnicode(pDBInfo) ? ctUCS2_els00 : ctAscii_els00,
                                    false);
    if (errOK_els00 != rc)
    {
        RepServMessages::Instance()->ls98Msg(layDataExtract_els98, ErrText, errIncompatibleEncodings_els98);

        return STATIC_CAST(tsp00_Int4, rc);
    }
    //*
    //*   check if user is sysdba
    //*
    if (ptAll_els00 == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00)
    {
        rc = ls15CheckUserMode(pDBInfo,
                               pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp(),
                               dbuSysdba_els00,
                               ErrText);
        if (errOK_els00 != rc)
            return STATIC_CAST(tsp00_Int4, rc);
    }
    //*
    //*   extract ddl statements in command file
    //*
    if (errOK_els00 == rc                                      &&
        pDatExtrCmd->dexRestore_ls00 != resDataloadNoDDL_els00 &&
        pDatExtrCmd->dexRestore_ls00 != resUnknown_els00           )
        rc = ls27_ExtractCatalog(pDBInfo, pDatExtrCmd, ErrText);

    if (errOK_els00 == rc)
    {
        //*
        //*  extract data all|user|table records, formatted, compressed
        //*
        MetaDataDef       *pMetaDataDef = new MetaDataDef( pDBInfo );
        tls00_DextractRec *pDatExtrRec  = NULL;
        StudioOAL_String   transformationpackage;

        rc = ls27_InitDatExtrRec(pDBInfo, pDatExtrCmd, pDatExtrRec, ErrText);

        if (errOK_els00 == rc)
        {
            //*
            //*  configure transformationpackage
            //*
            transformationpackage = "LOAD TRANSFORMATIONPACKAGE";

            rc = ls27LoadTransformationPackage( pMetaDataDef,pDatExtrCmd,pDatExtrRec,ErrText);

            if ( errOK_els00 == rc )
            {
                //*
                //*  execute transformationpackage
                //*
                transformationpackage = "EXECUTE PACKAGE TO TRANSFORM DATA";
                if ( ptTable_els00  < pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00  )
                {
                     TransformationDef * pTransformationDef = pMetaDataDef->getTransformationModel();
                     pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgCommandExecuting_els98,transformationpackage.StrPtr());
                     pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgNumTabToTransform_els98,pTransformationDef->m_nTableCount);
                }
                //*
                //*  update  transformationpackage
                //*
                if ( ptAll_els00   == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 ||
                     ptUser_els00  == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00    )
                     rc = ls27ExtractPackageTables( pMetaDataDef, pDatExtrRec, ErrText );
                if ( ptTable_els00 == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00    )
                     rc = ls27ExtractTableRows( pMetaDataDef, pDatExtrCmd, pDatExtrRec, ErrText );
                //*
                //*  unload  transformationpackage
                //*
                transformationpackage = "UNLOAD TRANSFORMATIONPACKAGE ";
                transformationpackage += "x'";
                transformationpackage += pDBInfo->dbiPackageid_ls00;
                transformationpackage += "'";
            }
            if ( errOK_els00 == rc )
                 rc = ls27UnloadTransformationPackage(pMetaDataDef,pDatExtrCmd, pDatExtrRec, ErrText);
        }
        if ( NULL != pMetaDataDef)
        {
            delete pMetaDataDef;
            pMetaDataDef = NULL;
        }
        if ( NULL != pDatExtrRec )
        {
            ls27_FreeDatExtrRec(pDatExtrCmd, pDatExtrRec);
            pDatExtrRec = NULL;
        }
    }

    return STATIC_CAST(tsp00_Int4, rc);
}
//  ls27ExecDataExtract()

/*
  -----------------------------------------------------------------------------
  function:     ls27ExtractPackageTables
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls27ExtractPackageTables( MetaDataDef       *&pMetaDataDef,
                          tls00_DextractRec *pDatExtrRec,
                          tsp00_Addr         ErrText)
{
    ROUTINE_DBG_MEO00 ("ls27ExtractPackageTables");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    tsp00_Int4       rc           = errOK_els00;

    TransformationDef* pTransformationDef = pMetaDataDef->getTransformationModel();
    tls00_DBInfo     * pDBInfo            = pMetaDataDef->getConnection();
    tls00_DatExtrCmd * pDatExtrCmd        = NULL;

    StudioOAL_String  select;
    _TCHAR            szselect[2048];
    szselect[0] = 0;

    sp77sprintf(szselect, 2048, SEL_MAP_TABLE_ALL_76_LS00, pDBInfo->dbiPackageid_ls00.StrPtr());
    select.ConvertFromASCII_Latin1(szselect, szselect + _tcslen(szselect));

    if ( !pTransformationDef->executeSQL( select ) )
    {
        SQLRETURN           ReturnCode = 0;
        StudioOAL_String    NativeErrorAsString;
        ReturnCode          = pTransformationDef->getReturnCode();
        NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

        pMsgInstance->ls98PrintMessage(layDataExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());
        rc = ReturnCode;
    }

    StudioOAL_String sessionUser;
    StudioOAL_String currentUser;

    if (errOK_els00 == rc)
    {
        ls05ConvertToUTF8(sessionUser,REINTERPRET_CAST(unsigned char*, pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp()), _tcslen(pDBInfo->dbiSession_ls00.siUserName_ls00.asCharp()), ctAscii_els00, ErrText);
        currentUser = sessionUser;
    }

    StudioOAL_String record[17];

    /*-------------------------------------------------------------------------------*
     * Read/Write the catalog definition of the specified object from/to resources   *
     *-------------------------------------------------------------------------------*/
    while ( errOK_els00 == rc &&  pTransformationDef->initTransformationStep( record )  )
    {
        pTransformationDef->freeTransformation     ( srFctDatExtr_els00 );

        rc = pTransformationDef->initTransformation();

        if (errOK_els00 == rc)
        {
            pDatExtrRec->derTableSwitched_ls00 = true;

            pDatExtrCmd = REINTERPRET_CAST(tls00_DatExtrCmd*, pTransformationDef->m_pCommand->cmdScnDat_ls00);

            rc = pTransformationDef->switchUser( currentUser, pTransformationDef->m_owner, ErrText);

            if (errOK_els00 == rc)
            {
                rc = ls27ExtractTableRows( pMetaDataDef,pDatExtrCmd, pDatExtrRec, ErrText);

                if (errOK_els00 == rc)
                {
                    ++pDatExtrRec->derNumTabExtracted_ls00;
                    ++pDatExtrRec->derNumTabInFile_ls00;
                }
                else
                {
                    if ( (errInternal_els98 != rc) && (errCancelRequest_els98 != rc) )
                    {
                         ++pDatExtrRec->derNumTabExtractFailed_ls00;

                        if (pDatExtrRec->derNumTabExtractFailed_ls00 <= pDBInfo->dbiSession_ls00.siMaxAdmissibleErrors)
                        {
                            ErrText[0] = 0;
                            rc = errOK_els00;
                        }
                    }
                }

            }// pTransformationDef->switchUser
        }
        currentUser = pTransformationDef->m_owner ;

    }// while pTransformationDef->initTransformationStep

    if (errOK_els00 == rc)
    {
        pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgNumTabTransformed_els98,
                                       pDatExtrRec->derNumTabExtracted_ls00 -  pDatExtrRec->derNumTabExcluded_ls00 );
    }
    //*
    //* switch user back to session user
    //*
    rc = pTransformationDef->switchUser( currentUser, sessionUser, ErrText);

    if ( pDatExtrCmd != NULL  &&  dtypeTape_ls00 == pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00 )
    {
        ls27_CloseStreams(pDatExtrCmd, pDatExtrRec);
        pDatExtrCmd->dexDatFile_ls00.fsBackupId_ls00.bBackupIdName_ls00 = REINTERPRET_CAST(SAPDB_Char *,(CONST_CAST (unsigned char* ,pDatExtrRec->derDataFile_ls00->vfBackupId_ls00.StrPtr())));
        pDatExtrRec->derTaskSwitched_ls00 = true;

        rc = ls27UpdateTransformationPackage( pMetaDataDef,pDatExtrCmd,pDatExtrRec,ErrText);
    }

    pTransformationDef->freeTransformation( srFctDatExtr_els00 );

    return rc;
}
// ls27ExtractPackageTables()


/*
  -----------------------------------------------------------------------------
  function:     ls27LoadTransformationPackage
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls27LoadTransformationPackage ( MetaDataDef       *&pMetaDataDef,
                                tls00_DatExtrCmd  *pDatExtrCmd,
                                tls00_DextractRec *pDatExtrRec,
                                tsp00_Addr         ErrText)
{
    ROUTINE_DBG_MEO00 ("ls27LoadTransformationPackage");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    SAPDB_Long rc = errOK_els00;

    if ( NULL != pMetaDataDef )
    {
        tls00_DBInfo  *pDBInfo = pMetaDataDef->getConnection();

        pDBInfo->dbiSession_ls00.siInstallOn_ls00 = true;

        if ( pDatExtrCmd->dexRestart_ls00.rstRestart_ls00  == SAPDB_TRUE || pDatExtrCmd->dexInternal_ls00 == SAPDB_TRUE   )
            pMetaDataDef->setTransMode(tmRestart_els31);
        else
            pMetaDataDef->setTransMode(tmLoadTable_els31);

        pMetaDataDef->setPartSpec(pDatExtrCmd->dexPartSpec_ls00);

        //*
        //*  packageid, taskid, stepid, owner, tablename, tabletype
        //*
        if ( pMetaDataDef->initResultSets(pDBInfo) )
        {
            TransformationDef * pTransformationDef = pMetaDataDef->getTransformationModel();

            if ( pDatExtrCmd->dexRestart_ls00.rstRestart_ls00  == SAPDB_FALSE )
            {
                //*
                //* update transformation package from sapdb loader
                //*
                Tools_DynamicUTF8String transformationpackage;
                transformationpackage = "CONFIGURE  TRANSFORMATIONPACKAGE";
                if ( ptTable_els00 < pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 )
                     pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgCommandExecuting_els98, transformationpackage.StrPtr());
                //*
                //*  storagedevice, servernode, serverdb, username, codetype, codepage, transformation, part, streamformat, exclude
                //*
                rc = ls27UpdateTransformationPackage( pMetaDataDef,pDatExtrCmd,pDatExtrRec,ErrText);
                //*
                //*  exclude
                //*
                if (  errOK_els00 == rc && NULL != pDatExtrCmd->dexConfigurationFile_ls00.fsFileName_ls00 )
                {
                    Tools_DynamicUTF8String  transformation;
                    _TCHAR sztransformation[4096];
                    sztransformation[0] = 0;
                    sp77sprintf(sztransformation, 4096, UPDATE_CONFIGURATION_TABLE_LS00, pDBInfo->dbiPackageid_ls00.StrPtr(), pDatExtrCmd->dexConfigurationFile_ls00.fsFileName_ls00 );
                    transformation.ConvertFromASCII_Latin1(sztransformation,sztransformation + _tcslen(sztransformation));

                    rc = ls11ParseDispatchLoadCommand (transformation, pDBInfo, ErrText);

                    transformation =  " UPDATE ";
                    transformation += " SYSDBA.TRANSFORMATIONMODEL ";
                    transformation += " SET EXCLUDE = TRUE WHERE EXCLUDE IS NULL ";
                    pTransformationDef->executeSQL( transformation );

                    transformation  = " SELECT COUNT (*) FROM ";
                    transformation += " SYSDBA.TRANSFORMATIONMODEL ";
                    transformation += " WHERE CMD NOT LIKE 'GRANT *' ";
                    transformation += " AND TABLETYPE = 'TABLE' AND EXCLUDE = TRUE ";
                    StudioOAL_String  record[1];

                    if ( errOK_els00 == rc && pTransformationDef->executeSQL( transformation ) )
                    {
                         pTransformationDef->readResult( record ) ;
                         pMsgInstance->ls98PrintMessage(layDataExtract_els98, msgNumTabExcluded_els98, record[0].StrPtr() );
                    }
                    else
                        rc = errOK_els00;

                }
            }
            if ( errOK_els00 == rc )
                 rc = ls15Commit(pDBInfo, ErrText);

            pDBInfo->dbiSession_ls00.siInstallOn_ls00 = false;

        }//if ( pMetaDataDef->initResultSets(pDBInfo) )
        else
        {
            rc = pMetaDataDef->getConnection()->dbiOdbcConnection.GetErr().GetErrId();
            return STATIC_CAST(tsp00_Int4, rc);
        }
    }

    return STATIC_CAST(tsp00_Int4, rc);

}// ls27LoadTransformationPackage

/*
  -----------------------------------------------------------------------------
  function:     ls27UpdateTransformationPackage
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls27UpdateTransformationPackage( MetaDataDef       *&pMetaDataDef,
                                 tls00_DatExtrCmd  *pDatExtrCmd,
                                 tls00_DextractRec *pDatExtrRec,
                                 tsp00_Addr        ErrText)
{
    ROUTINE_DBG_MEO00 ("ls27UpdateTransformationPackage");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    tsp00_Int4 rc = errOK_els00;

    // dataload    table records|compressed|formatted
    // dataextract table records|compressed|formatted

    if ( pDatExtrRec->derPartSpec_ls00.psPartType_ls00 > ptTable_els00)
    {
         TransformationDef * pTransformationDef = pMetaDataDef->getTransformationModel();
         tls00_DBInfo      * pDBInfo             = pMetaDataDef->getConnection();

         SAPDB_Bool       Error       = SAPDB_TRUE;
         SAPDB_Long       NativeError= 0;
         SAPDB_Int4       ErrorPos   = 0;
         SQLRETURN        ReturnCode = 0;
         StudioOAL_String ErrorText;
         StudioOAL_String NativeErrorAsString;

         //*
         //* performance
         //*
         extern tls00_Performance performance;
         clock_t        start, finish;

         //*
         //*  storagedevice, servernode, serverdb, username, codetype, codepage, transformation, part, streamformat
         //*
         if ( pDatExtrRec->derTaskSwitched_ls00 == true  )
         {
              pDatExtrRec->derTaskSwitched_ls00 =  false;

              StudioTD_String  servernode;
              StudioTD_String  serverdb;
              StudioTD_String  codetype;
              StudioTD_String  codepage;
              StudioTD_String  username;
              StudioTD_String  datastreamebid;
              StudioOAL_String parameter[13];
              SAPDB_Char   destBeg[1000] ;
              SAPDB_UInt4  destAt  = 1000;
              // stepid
              parameter[0]  = " + ";
              // servernode
              parameter[1]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetServerNode( REINTERPRET_CAST(unsigned char*, destBeg ), destAt);
              servernode.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[1] += servernode;
              parameter[1] += "'";
              // serverdb
              parameter[2]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetServerDb ( REINTERPRET_CAST(unsigned char*,destBeg ),destAt);
              serverdb.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[2] += serverdb;
              parameter[2] += "'";
              // username
              parameter[3]  = "'";
              memset(destBeg,0,sizeof(destBeg));
              pMetaDataDef->getConnection()->dbiOdbcConnection.GetUser( REINTERPRET_CAST(unsigned char*,destBeg ),destAt);
              username.ConvertFromASCII_Latin1( destBeg, destBeg+strlen(destBeg));
              parameter[3] += username;
              parameter[3] += "'";
              // codepage
              if (ctAscii_els00 == pDBInfo->dbiSession_ls00.siCodeType_ls00)
              {
                  parameter[4]  = "'";
                  parameter[4] += pDBInfo->dbiSession_ls00.siCodePage_ls00.name.asCharp();
                  parameter[4] += "'";
              }
              // storagedevice
              parameter[5]  =  pszBackupToolValue_ls00[pDBInfo->dbiSession_ls00.siBackupTool_ls00];
              // transformation
              parameter[6]  = "'EXTRACT'";
              // part
              if (ptAll_els00 == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'ALL'";
              if (ptUser_els00 == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'USER'";
              if (ptTable_els00 == pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00)
                  parameter[7]  = "'TABLE'";
              // datastreamtype
              parameter[8]  = pszDeviceTypeValue_ls00[pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00];
              // datastreamebid
              if ( NULL == pDatExtrCmd->dexDatFile_ls00.fsBackupId_ls00.bBackupIdName_ls00 )
                  parameter[9] = datastreamebid.AssignRaw(REINTERPRET_CAST(unsigned char*, pDBInfo->dbiSession_ls00.siNullSpec_ls00.asCharp()),
                                                                                   _tcslen(pDBInfo->dbiSession_ls00.siNullSpec_ls00.asCharp()));
              else
                  parameter[9] = datastreamebid.AssignRaw(REINTERPRET_CAST(unsigned char*, pDatExtrCmd->dexDatFile_ls00.fsBackupId_ls00.bBackupIdName_ls00),
                                                                                   _tcslen(pDatExtrCmd->dexDatFile_ls00.fsBackupId_ls00.bBackupIdName_ls00));
              // datastreamformat
              if  ( true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoPage_ls00 )
                    parameter[10]   = "'PAGES'";
              if  ( true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompact_ls00 )
                    parameter[10]  =  "'RECORDS'";
              // codetype
              parameter[11] = pszCodeTypeValue_ls00[pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCodeType_ls00];
              // transactionsize
              _TCHAR _transactionsize [256];
              _transactionsize        [0] = 0;
              _itot(pDBInfo->dbiSession_ls00.siTASize_ls00,_transactionsize,10);
              parameter[12].ConvertFromASCII_Latin1(_transactionsize,_transactionsize + _tcslen(_transactionsize));

              if ( !pTransformationDef->updateDatTask(parameter))
              {
                    Error               = pTransformationDef->isError();
                    NativeError         = pTransformationDef->getNativeError();
                    ErrorPos            = pTransformationDef->getErrorPos();
                    ReturnCode          = pTransformationDef->getReturnCode();
                    ErrorText           = pTransformationDef->getErrorText();
                    NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                    pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

              }
         }//if ( pDatExtrRec->derTaskSwitched_ls00 == true  )

         else //if ( pDatExtrRec->derTaskSwitched_ls00 == false  )
         {
             StudioOAL_String parameter[5];

             _TCHAR RECORD_EXTRACTED[256];
             _TCHAR HEADER_BLOCK    [256];
             _TCHAR HEADER_POS      [256];
             _TCHAR DATA_BLOCK      [256];
             _TCHAR DATA_POS        [256];
             _TCHAR TRAILER_BLOCK    [256];
             _TCHAR TRAILER_POS      [256];
             RECORD_EXTRACTED[0] = 0;
             HEADER_BLOCK    [0] = 0;
             HEADER_POS      [0] = 0;
             DATA_BLOCK      [0] = 0;
             DATA_POS        [0] = 0;
             TRAILER_BLOCK   [0] = 0;
             TRAILER_POS     [0] = 0;

             parameter[3].ConvertFromASCII_Latin1( pDatExtrCmd->dexPartSpec_ls00.psTabOwnr_ls00.asCharp(),
                                                   pDatExtrCmd->dexPartSpec_ls00.psTabOwnr_ls00.asCharp()+
                                                   pDatExtrCmd->dexPartSpec_ls00.psTabOwnr_ls00.length ()  );

             parameter[4].ConvertFromASCII_Latin1( pDatExtrCmd->dexPartSpec_ls00.psTabName_ls00.asCharp(),
                                                   pDatExtrCmd->dexPartSpec_ls00.psTabName_ls00.asCharp()+
                                                   pDatExtrCmd->dexPartSpec_ls00.psTabName_ls00.length ()  );

             if ( pDatExtrRec->derTableSwitched_ls00 == true  )
             {
                   // <EXCLUDE>
                  if ( pDatExtrCmd->dexExclude_ls00 == true )
                       pDatExtrRec->derNumTabExcluded_ls00 ++;

                  // TIMESTAMP BEGIN
                  pDatExtrRec->derTableSwitched_ls00 =  false;

                  if ( !pTransformationDef->updateBegin( parameter[3], parameter[4] ) )
                  {
                       Error               = pTransformationDef->isError();
                       NativeError         = pTransformationDef->getNativeError();
                       ErrorPos            = pTransformationDef->getErrorPos();
                       ReturnCode          = pTransformationDef->getReturnCode();
                       ErrorText           = pTransformationDef->getErrorText();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                       return ReturnCode;
                  }
                  // RECORDS
                  if ( !pTransformationDef->updateRecords( parameter[3], parameter[4] ) )
                  {
                       Error               = pTransformationDef->isError();
                       NativeError         = pTransformationDef->getNativeError();
                       ErrorPos            = pTransformationDef->getErrorPos();
                       ReturnCode          = pTransformationDef->getReturnCode();
                       ErrorText           = pTransformationDef->getErrorText();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                       return ReturnCode;
                  }
                  // <DATASTREAMNAME>
                  parameter[0]  = "'";
                  parameter[0] += pDatExtrRec->derDataFile_ls00->vfName_ls00;
                  parameter[0] += "'";
                  parameter[1]  = parameter[3];
                  parameter[2]  = parameter[4];

                   if ( !pTransformationDef->updateDatStream( parameter ) )
                   {
                       Error               = pTransformationDef->isError();
                       NativeError         = pTransformationDef->getNativeError();
                       ErrorPos            = pTransformationDef->getErrorPos();
                       ReturnCode          = pTransformationDef->getReturnCode();
                       ErrorText           = pTransformationDef->getErrorText();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                       return ReturnCode;
                   }
                   rc = ls15RSCommit(pDBInfo, ErrText);

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   // ROWCOUNT
                   parameter[0] = "0";
                  // <HEADER_BLOCK>:<HEADER_POS>
                  _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derDataFile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00),
                        HEADER_BLOCK, 10);
                  _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derDataFile_ls00->vfFilePos_ls00.lnuRemainder_ls00),
                        HEADER_POS, 10);

                   parameter[1].ConvertFromASCII_Latin1(HEADER_BLOCK,HEADER_BLOCK + _tcslen(HEADER_BLOCK));
                   parameter[2].ConvertFromASCII_Latin1(HEADER_POS  ,HEADER_POS   + _tcslen(HEADER_POS  ));

                   if ( !pTransformationDef->updateHeader( parameter ) )
                   {
                       Error               = pTransformationDef->isError();
                       NativeError         = pTransformationDef->getNativeError();
                       ErrorPos            = pTransformationDef->getErrorPos();
                       ReturnCode          = pTransformationDef->getReturnCode();
                       ErrorText           = pTransformationDef->getErrorText();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                       return ReturnCode;
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   pDatExtrRec->derHeaderFilePos_ls00 = pDatExtrRec->derDataFile_ls00->vfFilePos_ls00;

             }// if ( pDatExtrRec->derTableSwitched_ls00 == true  )

             // RECORD_EXTRACTED
             pDatExtrRec->derLastCommLine_ls00  = pDatExtrRec->derExtractedRows_ls00;
             _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derLoadedRows_ls00 + pDatExtrRec->derLastCommLine_ls00), RECORD_EXTRACTED, 10);
             // <DATA_BLOCK>:<DATA_POS>
             _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derDataFile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00), DATA_BLOCK, 10);
             _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derDataFile_ls00->vfFilePos_ls00.lnuRemainder_ls00), DATA_POS, 10);

             parameter[0].ConvertFromASCII_Latin1(RECORD_EXTRACTED,RECORD_EXTRACTED + _tcslen(RECORD_EXTRACTED));
             parameter[1].ConvertFromASCII_Latin1(DATA_BLOCK      ,DATA_BLOCK       + _tcslen(DATA_BLOCK));
             parameter[2].ConvertFromASCII_Latin1(DATA_POS        ,DATA_POS         + _tcslen(DATA_POS));

             start = clock();

             // EXTRACT
             if ( !pTransformationDef->updateExtrData( parameter ) )
             {
                   Error               = pTransformationDef->isError();
                   NativeError         = pTransformationDef->getNativeError();
                   ErrorPos            = pTransformationDef->getErrorPos();
                   ReturnCode          = pTransformationDef->getReturnCode();
                   ErrorText           = pTransformationDef->getErrorText();
                   NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                   pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                   return ReturnCode;
             }

             parameter[0].Erase();
             parameter[1].Erase();
             parameter[2].Erase();

             finish = clock();

             performance.perLog_ls00 += (double)(finish - start)/CLOCKS_PER_SEC;

             if (  pDatExtrRec->derTabEmpty_ls00      == true ||
                   pDatExtrRec->derTabExtracted_ls00  == true ||
                   pDatExtrCmd->dexExclude_ls00       == true    )
             {
                   // RECORD_EXTRACTED
                   pDatExtrRec->derLastCommLine_ls00  = pDatExtrRec->derExtractedRows_ls00;
                   _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derLoadedRows_ls00 + pDatExtrRec->derLastCommLine_ls00), RECORD_EXTRACTED, 10);
                   // <DATA_BLOCK>:<DATA_POS>
                   _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derDataFile_ls00->vfFilePos_ls00.lnu2GB_Blocks_ls00), DATA_BLOCK, 10);
                   _itot(STATIC_CAST(tsp00_Int4, pDatExtrRec->derDataFile_ls00->vfFilePos_ls00.lnuRemainder_ls00), DATA_POS, 10);

                   parameter[0].ConvertFromASCII_Latin1(RECORD_EXTRACTED,RECORD_EXTRACTED + _tcslen(RECORD_EXTRACTED));
                   parameter[1].ConvertFromASCII_Latin1(DATA_BLOCK      ,DATA_BLOCK       + _tcslen(DATA_BLOCK));
                   parameter[2].ConvertFromASCII_Latin1(DATA_POS        ,DATA_POS         + _tcslen(DATA_POS));

                   if ( !pTransformationDef->updateExtrData( parameter ) )
                   {
                         Error               = pTransformationDef->isError();
                         NativeError         = pTransformationDef->getNativeError();
                         ErrorPos            = pTransformationDef->getErrorPos();
                         ReturnCode          = pTransformationDef->getReturnCode();
                         ErrorText           = pTransformationDef->getErrorText();
                         NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                         pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                         return ReturnCode;
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   // <TRAILER_BLOCK>:<TRAILER_POS>
                   _itot(STATIC_CAST(tsp00_Int4, pTransformationDef->m_trailerpos.lnu2GB_Blocks_ls00),TRAILER_BLOCK,10);
                   _itot(STATIC_CAST(tsp00_Int4, pTransformationDef->m_trailerpos.lnuRemainder_ls00) ,TRAILER_POS,10);

                   parameter[0] = "0";
                   parameter[1].ConvertFromASCII_Latin1(TRAILER_BLOCK      ,TRAILER_BLOCK       + _tcslen(TRAILER_BLOCK));
                   parameter[2].ConvertFromASCII_Latin1(TRAILER_POS        ,TRAILER_POS         + _tcslen(TRAILER_POS));

                   if ( !pTransformationDef->updateTrailer( parameter ) )
                   {
                         Error               = pTransformationDef->isError();
                         NativeError         = pTransformationDef->getNativeError();
                         ErrorPos            = pTransformationDef->getErrorPos();
                         ReturnCode          = pTransformationDef->getReturnCode();
                         ErrorText           = pTransformationDef->getErrorText();
                         NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                         pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                         return ReturnCode;
                   }

                   parameter[0].Erase();
                   parameter[1].Erase();
                   parameter[2].Erase();

                   // RESTART
                   if ( !pTransformationDef->updateRestart( parameter[3], parameter[4]) )
                   {
                       Error               = pTransformationDef->isError();
                       NativeError         = pTransformationDef->getNativeError();
                       ErrorPos            = pTransformationDef->getErrorPos();
                       ReturnCode          = pTransformationDef->getReturnCode();
                       ErrorText           = pTransformationDef->getErrorText();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                       return ReturnCode;
                   }
                   // TIMESTAMP END
                   if ( !pTransformationDef->updateEnd( parameter[3], parameter[4]) )
                   {
                       Error               = pTransformationDef->isError();
                       NativeError         = pTransformationDef->getNativeError();
                       ErrorPos            = pTransformationDef->getErrorPos();
                       ReturnCode          = pTransformationDef->getReturnCode();
                       ErrorText           = pTransformationDef->getErrorText();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                       return ReturnCode;
                   }
                   // SYSTEM TIME
                   if ( !pTransformationDef->updatePerformance( parameter[3], parameter[4]) )
                   {
                       Error               = pTransformationDef->isError();
                       NativeError         = pTransformationDef->getNativeError();
                       ErrorPos            = pTransformationDef->getErrorPos();
                       ReturnCode          = pTransformationDef->getReturnCode();
                       ErrorText           = pTransformationDef->getErrorText();
                       NativeErrorAsString = pTransformationDef->getNativeErrorAsString();

                       pMsgInstance->ls98PrintMessage(layTableExtract_els98, errODBCErrors_els98, NativeErrorAsString.StrPtr());

                       return ReturnCode;
                   }
             }//if ( pDatExtrRec->derTabEmpty_ls00 == true )
        }//if (  pDatExtrRec->derTaskSwitched_ls00 == false  )
    }
    return rc;

}//ls27UpdateTransformationPackage

/*
  -----------------------------------------------------------------------------
  function: ls27UnloadTransformationPackage
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls27UnloadTransformationPackage(MetaDataDef       *&pMetaDataDef,
                                tls00_DatExtrCmd  *pDatExtrCmd,
                                tls00_DextractRec *pDatExtrRec,
                                tsp00_Addr         ErrText)
{
    ROUTINE_DBG_MEO00 ("ls27UnloadTransformationPackage");

    RepServMessages *pMsgInstance = RepServMessages::Instance();
    SAPDB_Long       rc = errOK_els00;

    TransformationDef * pTransformationDef = pMetaDataDef->getTransformationModel();
    tls00_DBInfo      * pDBInfo            = pMetaDataDef->getConnection();

    Tools_DynamicUTF8String transformationpackage;
    transformationpackage = "UNLOAD TRANSFORMATIONPACKAGE ";
    transformationpackage += "x'";
    transformationpackage += pDBInfo->dbiPackageid_ls00;
    transformationpackage += "'";

    Tools_DynamicUTF8String  transformation;
    _TCHAR sztransformation[4096];
    sztransformation[0] = 0;
    _TCHAR sztransformationfile[PATH_MXSP00];
    sztransformationfile[0] = 0;

    if ( ptTable_els00  < pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 )
    {
        if ( dtypeFile_ls00 == pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00 )
        {
            sp77sprintf(sztransformationfile, PATH_MXSP00, "%s%0*d", pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00, FILE_EXTENSION_SIZE_LS27, 0);
            sp77sprintf(sztransformation    , 4096, EXTRACT_CONFIGURATION_TABLE_LS00,pDBInfo->dbiPackageid_ls00.StrPtr(), pszDeviceTypeName_ls00[pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00], sztransformationfile );
            transformation.AssignRaw(REINTERPRET_CAST(unsigned char*,sztransformation), STATIC_CAST(tsp00_Uint4, _tcslen(sztransformation)));
            rc = ls11ParseDispatchLoadCommand(transformation, pDBInfo, ErrText);
        }
        if ( dtypeTape_ls00 == pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00 )
        {
            sp77sprintf(sztransformationfile ,  PATH_MXSP00, "%s", pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00 );
            sp77sprintf(sztransformation, 4096, EXTRACT_CONFIGURATION_TABLE_LS00,pDBInfo->dbiPackageid_ls00.StrPtr(), pszDeviceTypeName_ls00[pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00], sztransformationfile );
            transformation.AssignRaw(REINTERPRET_CAST(unsigned char*,sztransformation), STATIC_CAST(tsp00_Uint4, _tcslen(sztransformation)));
            rc = ls11ParseDispatchLoadCommand(transformation, pDBInfo, ErrText);
        }
        if ( errOK_els00 == rc && NULL != pDatExtrCmd->dexPackageFile_ls00.fsFileName_ls00 )
        {
            sztransformation[0] = 0;
            sp77sprintf(sztransformation, 4096, EXTRACT_MAP_TABLE_RECORDS_LS00,pDBInfo->dbiPackageid_ls00.StrPtr(),pDatExtrCmd->dexPackageFile_ls00.fsFileName_ls00, pszCodeTypeName_ls00[pDatExtrCmd->dexPackageFile_ls00.fsFormat_ls00.ffoCodeType_ls00]);
            transformation.AssignRaw(REINTERPRET_CAST(unsigned char*,sztransformation), STATIC_CAST(tsp00_Uint4, _tcslen(sztransformation)));
            rc = ls11ParseDispatchLoadCommand(transformation, pDBInfo, ErrText);

            if ( errOK_els00 == rc )
            {
                sztransformation[0] = 0;
                sp77sprintf(sztransformation, 4096, DEL_MAP_TABLE_ALL_73_LS00, pDBInfo->dbiPackageid_ls00.StrPtr());
                pTransformationDef->executeSQL(sztransformation);
            }
        }
    }
    if ( ptTable_els00  ==  pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 )
    {
        if ( errOK_els00 == rc && pDatExtrCmd->dexInternal_ls00 == SAPDB_FALSE )    
        {
            sztransformation[0] = 0;
            sp77sprintf(sztransformation, 4096, DEL_MAP_TABLE_ALL_73_LS00, pDBInfo->dbiPackageid_ls00.StrPtr());
            pTransformationDef->executeSQL(sztransformation);
        }
    }
    if ( ptTable_els00  < pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 && errOK_els00 == rc )
         pMsgInstance->ls98PrintMessage(layTableExtract_els98, msgCommandExecuting_els98, transformationpackage.StrPtr());

    return STATIC_CAST(tsp00_Int4, rc);

}//ls27UnloadTransformationPackage

/*
  -----------------------------------------------------------------------------
  function:     ls27_InitStreams
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_InitStreams( tls00_DBInfo      *pDBInfo,
                  tls00_DatExtrCmd  *pDatExtrCmd,
                  tls00_DextractRec *pDatExtrRec,
                  tsp00_Addr         ErrText)
{
    ROUTINE_DBG_MEO00 ("ls27_InitStreams");

    tsp00_Int4 rc = errOK_els00;

    // mapping file, datafile, commandfile
    tls00_VFile *pDataFile = NULL;

    // switch tableempty
    if (false == pDatExtrRec->derTabEmpty_ls00  )
    {
        //*
        //*  Record
        //*
        if ( NULL != pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00 )
        {
            // Allocate file structure
            pDatExtrRec->derDataFile_ls00 = new tls00_VFile;
            memset(pDatExtrRec->derDataFile_ls00, 0, sizeof(tls00_VFile));

            pDataFile = pDatExtrRec->derDataFile_ls00;

            //*
            //*     initialize data streams depending on stream type - FILE, PIPE, TAPE
            //*
            if (dtypeFile_ls00 == pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00)
            {
                tsp05_RteFileMode tmpFileMode;
                tmpFileMode.becomes(sp5vf_append);


                // switch part
                if ( (pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 > ptTable_els00)     &&
                    (true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompact_ls00) )
                {
                    // switch restart
                    if (SAPDB_FALSE == pDatExtrCmd->dexRestart_ls00.rstRestart_ls00)
                    {
                        tmpFileMode.becomes(sp5vf_write);
                    }

                    //*
                    //* data file
                    //*
                    pDataFile->vfFileName_ls00 = new _TCHAR[PATH_MXSP00];
                    pDataFile->vfFileName_ls00[0] = 0;

                    sp77sprintf(pDataFile->vfFileName_ls00, PATH_MXSP00, "%s%0*d", pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00,FILE_EXTENSION_SIZE_LS27,1);

                    ls30VFInitialize(*pDataFile, pDataFile->vfFileName_ls00, sp5vf_binary, tmpFileMode);
                }
                else
                {
                    // switch streamformat
                    if ( (  true  == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompress_ls00)  ||
                        ( (true  == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoFormatted_ls00) &&
                        (false == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoBinary_ls00)    )  )
                    {
                        ls30VFInitialize(*pDataFile,
                                        pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00,
                                        sp5vf_text,
                                        sp5vf_write,
                                        pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCodeType_ls00);
                    }
                    else
                    {
                        if ( (false       == pDatExtrCmd->dexCmdFile_ls00.fsAppend_ls00) &&
                            (SAPDB_FALSE == pDatExtrCmd->dexRestart_ls00.rstRestart_ls00) )
                        {
                            tmpFileMode.becomes(sp5vf_write);
                        }
                        ls30VFInitialize(*pDataFile, pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00, sp5vf_binary, tmpFileMode);
                    }
                }   // end else of if ( (pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00 > ptTable_els00) ...

                //*
                //*   streamname
                //*
                if ( NULL != pDataFile->vfFileName_ls00 )
                {
                    _TCHAR* pszDest = strrchr(pDataFile->vfFileName_ls00, SLASH_LS00);
                    if (NULL == pszDest)
                    {
                        pszDest = strrchr(pDataFile->vfFileName_ls00, BACKSLASH_LS00);
                        if (NULL == pszDest)
                            pszDest = pDataFile->vfFileName_ls00;
                    }
                    pszDest = (pszDest == pDataFile->vfFileName_ls00) ? pszDest : pszDest + 1;

                    pDatExtrRec->derDataFile_ls00->vfName_ls00.ConvertFromASCII_Latin1(pszDest,pszDest + _tcslen(pszDest) );
                }

            }// end if  FILE
            else
            {// stream types PIPE and TAPE

                if (dtypeTape_ls00 == pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00)
                {
                    StudioOAL_String  DBNode;
                    StudioOAL_String  DBName;

                    DBNode.ConvertFromASCII_Latin1(pDBInfo->dbiSession_ls00.si_pDBNode_ls00,
                                                pDBInfo->dbiSession_ls00.si_pDBNode_ls00 + _tcslen(pDBInfo->dbiSession_ls00.si_pDBNode_ls00));
                    DBName.ConvertFromASCII_Latin1(pDBInfo->dbiSession_ls00.siDBName_ls00,
                                                pDBInfo->dbiSession_ls00.siDBName_ls00+_tcslen(pDBInfo->dbiSession_ls00.siDBName_ls00));

                    pDataFile->vfFileType_ls00   = dtypeTape_ls00;
                    pDataFile->vfBackupTool_ls00 = pDBInfo->dbiSession_ls00.siBackupTool_ls00;
                    pDataFile->vfNodeName_ls00   = DBNode;
                    pDataFile->vfDBName_ls00     = DBName;
                }
                // switch streamformat
                if ( (  true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompress_ls00)  ||
                     ( (true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoFormatted_ls00) &&
                       (false == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoBinary_ls00)    )  )
                {
                    ls30VFInitialize(*pDataFile,
                                     pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00,
                                     sp5vf_text,
                                     sp5vf_write,
                                     pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCodeType_ls00);
                }
                else
                {
                    ls30VFInitialize(*pDataFile, pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00, sp5vf_binary, sp5vf_write);
                }
            }
        } // RECORD

        //*
        //*  Transformation
        //*
        if ( ( (resUnknown_els00 != pDatExtrCmd->dexRestore_ls00) || 
               (pDatExtrCmd->dexPackageFile_ls00.fsFileName_ls00  != NULL  && 
                pDatExtrCmd->dexPartSpec_ls00.psPartType_ls00     == ptTable_els00) ) &&
             (pDatExtrCmd->dexRestore_ls00 < resDataUpdt_els00) )
        {
            tsp05_RteFileMode tmpFileMode;
            tmpFileMode.becomes(sp5vf_write);
            if (true == pDatExtrCmd->dexCmdFile_ls00.fsAppend_ls00)
            {
                tmpFileMode.becomes(sp5vf_append);
            }
            pDatExtrRec->derCmdFile_ls00 = new tls00_VFile;
            memset(pDatExtrRec->derCmdFile_ls00, 0, sizeof(tls00_VFile));
            if ( NULL == pDatExtrCmd->dexPackageFile_ls00.fsFileName_ls00 )
            {
                ls30VFInitialize(*pDatExtrRec->derCmdFile_ls00, pDatExtrCmd->dexCmdFile_ls00.fsFileName_ls00,
                                 sp5vf_text, tmpFileMode);
            }
            else //if ( NULL != pDatExtrCmd->dexPackageFile_ls00.fsFileName_ls00 )
            {
                ls30VFInitialize(*pDatExtrRec->derCmdFile_ls00, pDatExtrCmd->dexPackageFile_ls00.fsFileName_ls00,
                                 sp5vf_text, tmpFileMode);
            }
            // Set right stream type
            pDatExtrRec->derCmdFile_ls00->vfFileType_ls00 = pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00;

            rc = ls30VFOpen(*pDatExtrRec->derCmdFile_ls00, ErrText);
        }
    }   // end if (false == pDatExtrRec->derTabEmpty_ls00)
    else
    {
        if ( NULL != pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00 &&
             true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompact_ls00 )
        {
            // Allocate file structure (datafile for MetaDataHeader)
            pDatExtrRec->derDataFile_ls00 = new tls00_VFile;
            memset(pDatExtrRec->derDataFile_ls00, 0, sizeof(tls00_VFile));

            pDataFile = pDatExtrRec->derDataFile_ls00;

            //*
            //*     Initialize data file
            //*
            if (true == pDatExtrCmd->dexCmdFile_ls00.fsAppend_ls00)
            {
                ls30VFInitialize(*pDataFile, pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00,
                                 sp5vf_binary, sp5vf_append);
            }
            else
            {
                if ( SAPDB_FALSE == pDatExtrCmd->dexRestart_ls00.rstRestart_ls00 )
                {
                    ls30VFInitialize(*pDataFile, pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00,
                                     sp5vf_binary, sp5vf_write);
                }
                else
                {
                    ls30VFInitialize(*pDataFile, pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00,
                                     sp5vf_binary, sp5vf_append);
                }
                //*
                //*     open data file if any
                //*
                if (errOK_els00 == rc && NULL != pDataFile )
                {
                    rc = ls30VFOpen(*pDataFile, ErrText);
                    pDataFile->vfFileType_ls00 = pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00;
                }
            }
        }   // end if ( true == pDatExtrCmd->dexDatFile_ls00->fsFormat_ls00->ffoCompact_ls00 )

    }   // end else of if (false == pDatExtrRec->derTabEmpty_ls00)

    if ( NULL           != pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00 &&
         dtypeFile_ls00 != pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00)
    {
        pDatExtrRec->derDataFile_ls00->vfName_ls00.ConvertFromASCII_Latin1(pDatExtrRec->derDataFile_ls00->vfFileName_ls00,
                                                                           pDatExtrRec->derDataFile_ls00->vfFileName_ls00 +
                                                                   _tcslen(pDatExtrRec->derDataFile_ls00->vfFileName_ls00) );
    }

    return rc;
}
// ls27_InitStreams()


/*
  -----------------------------------------------------------------------------
  function:     ls27_SwitchStreams
  -----------------------------------------------------------------------------
*/
static tsp00_Int4
ls27_SwitchStreams( tls00_DBInfo      *pDBInfo,
                    tls00_DatExtrCmd  *pDatExtrCmd,
                    tls00_DextractRec *pDatExtrRec,
                    tsp00_Addr        ErrText)
{

    ROUTINE_DBG_MEO00 ("ls27_SwitchStreams");

    RepServMessages *pMsgInstance = RepServMessages::Instance();

    tsp00_Int4 rc = errOK_els00;

    switch (  pDatExtrCmd->dexDatFile_ls00.fsDeviceType_ls00 )
    {
             case dtypePipe_ls00 :
             {
                 rc = errOK_els00;
                 break;
             }
             case dtypeFile_ls00 :
             {
                 break;
             }
             default:
             {
                 rc = errOK_els00;
                 break;
             }
    }

    return rc;
}
//ls27_SwitchStreams

/*
  -----------------------------------------------------------------------------
  function:     ls27_CloseStreams
  -----------------------------------------------------------------------------
*/
static void
ls27_CloseStreams(tls00_DatExtrCmd* pDatExtrCmd, tls00_DextractRec* pDatExtrRec)
{
    ROUTINE_DBG_MEO00 ("ls27_CloseStreams");

    tsp00_Int4 rc = errOK_els00;
    char       pszErrText[256];         // We don't care for errors here

    if (true == pDatExtrRec->derTabEmpty_ls00)
    {
        //*
        //* data file
        //*
        if ( (true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompact_ls00) && 
             (NULL != pDatExtrRec->derDataFile_ls00) )
        {
            ls30VFClose(*pDatExtrRec->derDataFile_ls00, pszErrText);
        }
    }
    else
    {
        //*
        //* data file
        //*
        if (NULL != pDatExtrRec->derDataFile_ls00)
            ls30VFClose(*pDatExtrRec->derDataFile_ls00, pszErrText);

        //*
        //* long file
        //*
        if ( ( 0     != pDatExtrRec->derLongColCnt_ls00) &&
             ( false == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompact_ls00) )
        {
            tls00_Column *pCol = NULL;

            for (int i = 0; i < pDatExtrCmd->dexOutSpec_ls00.mcsCount_ls00; ++i)
            {
                pCol = pDatExtrCmd->dexOutSpec_ls00.mcsColumn_ls00[i];

                // Literals may not be of type LONG
                if (NULL != pCol->colLongInfo.loiFileStruct_ls00)
                    if (false == pCol->colIsLit_ls00)
                        ls30VFClose(*pCol->colLongInfo.loiFileStruct_ls00, pszErrText);
            }   // end for (i; i < pDatExtrCmd->dexColSpec_ls00.mcsCount_ls00; ++i)
        }
    }

    // command file
    if (NULL != pDatExtrRec->derCmdFile_ls00)
        ls30VFClose(*pDatExtrRec->derCmdFile_ls00, pszErrText);
}
// ls27_CloseStreams()


/*
  -----------------------------------------------------------------------------
  function:     ls27_FreeStreams
  -----------------------------------------------------------------------------
*/
static void
ls27_FreeStreams(tls00_DatExtrCmd* pDatExtrCmd, tls00_DextractRec* pDatExtrRec)
{
    ROUTINE_DBG_MEO00 ("ls27_FreeStreams");

    tsp00_Int4 rc = errOK_els00;

    if ( (true == pDatExtrRec->derTabEmpty_ls00) &&
         (true == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompact_ls00) )
    {
        //*
        //* data file
        //*
        if (NULL != pDatExtrRec->derDataFile_ls00)
        {
            delete pDatExtrRec->derDataFile_ls00;
            pDatExtrRec->derDataFile_ls00 = NULL;
        }
    }

    if (false == pDatExtrRec->derTabEmpty_ls00)
    {
        //*
        //* data file
        //*
        if (NULL != pDatExtrRec->derDataFile_ls00)
        {
            delete pDatExtrRec->derDataFile_ls00;
            pDatExtrRec->derDataFile_ls00 = NULL;
        }
        //*
        //* long file
        //*
        if ( ( 0     != pDatExtrRec->derLongColCnt_ls00) &&
             ( false == pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00.ffoCompact_ls00) )
        {
            tls00_Column *pCol = NULL;

            for (int i = 0; i < pDatExtrCmd->dexOutSpec_ls00.mcsCount_ls00; ++i)
            {
                pCol = pDatExtrCmd->dexOutSpec_ls00.mcsColumn_ls00[i];

                // Literals may not be of type LONG
                if (NULL != pCol->colLongInfo.loiFileStruct_ls00)
                {
                    if (false == pCol->colIsLit_ls00)
                    {
                        delete pCol->colLongInfo.loiFileStruct_ls00;
                        pCol->colLongInfo.loiFileStruct_ls00 = NULL;
                    }
                }
            }   // end for (i; i < pDatExtrCmd->dexColSpec_ls00.mcsCount_ls00; ++i)
        }
    }

    // command file
    if (NULL != pDatExtrRec->derCmdFile_ls00)
    {
        delete pDatExtrRec->derCmdFile_ls00;
        pDatExtrRec->derCmdFile_ls00 = NULL;
    }
    return;
}
// ls27_FreeStreams()

/*===========================================================================*
 *     END
 *===========================================================================*/


