/*
  -----------------------------------------------------------------------------

  module: vls10.cpp

  -----------------------------------------------------------------------------

  responsible:  SteffenS and d033893

  special area: SAP DB LOADER

  description:  dispatcher

  version:      7.5.

  -----------------------------------------------------------------------------

  copyright:    (c) 1997-2004 SAP AG-2003

  -----------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 1997-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/
#include "gls00.h"
#include "heo01.h"
#include "ToolsCommon/Tools_DynamicUTF8String.hpp"
#include "hls03.h"
#include "hls11.h"
#include "hls13.h"
#include "hls15.h"
#include "hls20_dataload.h"
#include "hls21.h"
#include "hls26.h"
#include "hls27_dataextract.h"
#include "hls28_TLoad.h"
#include "hls29_dataupdate.h"
#include "hls31_catalogextract.h"
#include "hls32_dbextract.h"
#include "hls98msg.h"
#include "hls99.h"
#include "hls10.h"

/*!
  -----------------------------------------------------------------------------
  Chapter: Local function definition
  -----------------------------------------------------------------------------
 */

static void
ls10_FreeColumnStruct(tls00_MultColSpec* pMultColSpec, tls00_FileFormat*  pFileFormat);
/*!
  -----------------------------------------------------------------------------
  EndChapter: Local function definition
  -----------------------------------------------------------------------------
 */

/*!
  -----------------------------------------------------------------------------
  Chapter: Global function implementation
  -----------------------------------------------------------------------------
 */

/*
  -----------------------------------------------------------------------------
  function:     ls10ExecChangeUser
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls10ExecChangeUser(tls00_DBInfo *DBInfo, tls00_SessionInfo *Session, tsp00_Addr ErrText)
{
    ROUTINE_DBG_MEO00 ("ls10ExecChangeUser");

    RepServMessages*    pMsgObj             = RepServMessages::Instance();  // Error reporting objekt
    tsp00_Int4          rc                  = errOK_els00;
    tls00_SessionInfo*  pOldSession         = NULL;
    tls00_SessionInfo*  pWorkingSession     = &DBInfo->dbiSession_ls00;
    bool                ReconnectAfterError = false;                        // restore old connection in case of error?

    if (NULL != Session->siUserKey_ls00)                    //* User key? Get it.
    {
        char pszTmpText[sizeof(tsp00_ErrText) + 1];

        rc = ls15SQLGetUser(Session, pszTmpText);
        if (errOK_els00 != rc)
        {
            pMsgObj->ls98Msg(laySQLConnect_els98, ErrText, errSqlConnectUserkey_els98, Session->siUserKey_ls00, pszTmpText);
            return rc;
        }
        else    // delete user key info because not longer necessary
        {
            delete [] Session->siUserKey_ls00;
            Session->siUserKey_ls00 = NULL;
        }
    }   // end if (NULL != Session->siUserKey_ls00)

    //*
    //*     Check if user is already connected and we've nothing to do (important for NI connections, too)
    //*
    SAPDB_Bool bUserConnected = SAPDB_FALSE;
    
    if ( (true == DBInfo->dbiSession_ls00.siDBConnection_ls00) &&
         (true == DBInfo->dbiSession_ls00.siSqlStat_ls00) )
    {
        // The check for node name equality is rather simple: both node name strings must be identical;
        // the right TCP/IP address is not considered equal to the hostname itself
        SAPDB_Bool bEqualNodes = SAPDB_FALSE;

        if ( (NULL == DBInfo->dbiSession_ls00.si_pDBNode_ls00) && (NULL == Session->si_pDBNode_ls00) )
        {
            bEqualNodes = SAPDB_TRUE;   // It's the local host in both cases - it' the same
        }
        else
        {
            if ( (NULL != DBInfo->dbiSession_ls00.si_pDBNode_ls00) && (NULL != Session->si_pDBNode_ls00) )
            {
                // Compare case insensitive
                if (0 == _tcsicmp(DBInfo->dbiSession_ls00.si_pDBNode_ls00, Session->si_pDBNode_ls00))
                {
                    bEqualNodes = SAPDB_TRUE;
                }
            }
            else
            {
                // Only one of the node names is specified
                tsp00_NodeIdc  tmpDBHost_c;
                tsp00_ErrText  aErrText;
                aErrText.SetBlank();

                rc = ls03GetHostname(tmpDBHost_c, aErrText);
                if (errOK_els00 == rc)
                {
                    char *p = (NULL == Session->si_pDBNode_ls00) ? DBInfo->dbiSession_ls00.si_pDBNode_ls00 :
                                                                   Session->si_pDBNode_ls00;
                    if (0 == _tcsicmp(tmpDBHost_c.asCharp(), p))
                    {
                        bEqualNodes = SAPDB_TRUE;
                    }
                }
            }
        }   // end else of if ( (NULL == DBInfo->dbiSession_ls00.si_pDBNode_ls00) && (NULL == Session->si_pDBNode_ls00) )

        if (SAPDB_TRUE == bEqualNodes)
        {
            if (0 == _tcsicmp(DBInfo->dbiSession_ls00.siDBName_ls00.asCharp(), Session->siDBName_ls00.asCharp()))
            {
                // User names are case sensitive
                if (0 == _tcscmp(DBInfo->dbiSession_ls00.siUserName_ls00.asCharp(), Session->siUserName_ls00.asCharp()))
                {
                    // this can only be the old password (meaning that the password at DB may have changed
                    // already and the Loader wouldn't have recognized)
                    if (0 == DBInfo->dbiSession_ls00.siUserCPwd_ls00.compare(Session->siUserCPwd_ls00))
                    {
                        bUserConnected = SAPDB_TRUE;
                    }
                }
            }
        }
    }   // end if ( (true == DBInfo->dbiSession_ls00.siDBConnection_ls00) && ...



    if (SAPDB_FALSE == bUserConnected)
    {
        // Reset old encoding - first try to connect is always as ctAscii
        DBInfo->dbiEncoding_ls00 = ctAscii_els00;

        //*
        //*     Save old connection if existing
        //*
        if (errOK_els00 != (rc = ls15CommitUser (DBInfo, ErrText)) )
        {
            // If no connection (errConnectNoServerDB_els98) nothing to save
            if (errConnectNoServerDB_els98 != rc)
            {
                return rc;      // error text is logged by ls15CommitUser
            }
        }
        else
        {
            pOldSession = new tls00_SessionInfo;            // User 'committed': save old session structure
            memcpy(pOldSession, pWorkingSession, sizeof (tls00_SessionInfo));

            // new user may not have db connection and may not be logged on
            Session->siDBConnection_ls00 = false;
            Session->siSqlStat_ls00      = false;

            ReconnectAfterError = true;                     // restore if connection fails
        }

        // Copy new session struct into 'old' DBInfo-struct and overwrite old session struct
        memcpy (pWorkingSession, Session, sizeof (tls00_SessionInfo));

        //*
        //*     New connection
        //* 1. db connection; 2. user logon
        //*
        rc = ls15Connect(DBInfo, ErrText);
        if (errOK_els00 != rc)       // no connection; error is already logged
        {
            if (true == ReconnectAfterError)
            {
                // restore old session structure
                memcpy(pWorkingSession, pOldSession, sizeof (tls00_SessionInfo));

                rc = ls15Connect(DBInfo, ErrText);     // try to connect 'old' user
                if (errOK_els00 == rc)                           // success restoring old connection
                {
                    pMsgObj->ls98Msg( layUserConnect_els98, ErrText, errPreviousUserRestored_els98,
                                                                    pOldSession->siUserName_ls00.asCharp());
                    rc = errPreviousUserRestored_els98;
                }
                // else --> caller logs error
            }
            // else --> caller logs error
        }
        else        // log connected user and server db
        {
            if (true == pWorkingSession->siNolog_ls00)
            {
                pMsgObj->ls98PrintMessage(layUserConnect_els98, msgUserConnectionLog_els98,
                                                                pWorkingSession->siUserName_ls00.asCharp(),
                                                                DBInfo->dbiSession_ls00.siDBName_ls00.asCharp(),
                                                                "NOLOG");
            }
            else
            {
                pMsgObj->ls98PrintMessage(layUserConnect_els98, msgUserConnectionLog_els98,
                                                                pWorkingSession->siUserName_ls00.asCharp(),
                                                                DBInfo->dbiSession_ls00.siDBName_ls00.asCharp(),
                                                                " ");
            }
        }

        // In case up to here everything went ok check for code page loading necessity
        if (errOK_els00 == rc)
        {
            rc = ls15LoadCodePage(DBInfo, ErrText);
        }

        if (NULL != pOldSession)
        {
            if ( (NULL != pOldSession->si_pDBNode_ls00) && (errPreviousUserRestored_els98 != rc) )
            {
                delete pOldSession->si_pDBNode_ls00;
                pOldSession->si_pDBNode_ls00 = NULL;
            }
            delete pOldSession;
        }
    }   // end if (SAPDB_FALSE == bUserConnected)
    return rc;
}
//  ls10ExecChangeUser()

/*
  -----------------------------------------------------------------------------
  function:     ls10ExecSQLStmt
  -----------------------------------------------------------------------------
*/
int ls10ExecSQLStmt(tls00_Command *pTransformationCmd, tls00_DBInfo *pDBInfo)
{
    ROUTINE_DBG_MEO00 ("ls10ExecSQLStmt");

    tsp00_Int4  rc = errOK_els00;

    if (true != pDBInfo->dbiSession_ls00.siSqlStat_ls00)     // connected ?
    {
        RepServMessages::Instance()->ls98Msg(layBasicConnect_els98, pTransformationCmd->cmdRpyDat_ls00,
                                                                    errNoUserConnection_els98,
                                                                    pDBInfo->dbiSession_ls00.siDBName_ls00.asCharp());
        rc = errNoUserConnection_els98;
    }
    else
        rc = ls13ExecExtSQLStmt(pDBInfo,REINTERPRET_CAST(tls00_DBSqlCmd*, pTransformationCmd->cmdScnDat_ls00), pTransformationCmd->cmdRpyDat_ls00);

    return rc;
}
//  ls10ExecSQLStmt()


/*
  -----------------------------------------------------------------------------
  function:     ls10ExecTransformation
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ls10ExecTransformation(tls00_Command *pTransformationCmd, tls00_DBInfo *pDBInfo)
{
    ROUTINE_DBG_MEO00 ("ls10ExecTransformation");

    RepServMessages*  MsgInstance  = RepServMessages::Instance();

    tsp00_Int4  rc = errOK_els00;


    //*
    //*   connected to db
    //*
    if (true != pDBInfo->dbiSession_ls00.siSqlStat_ls00)
    {
        MsgInstance->ls98Msg(layBasicConnect_els98, pTransformationCmd->cmdRpyDat_ls00,
                                                    errNoUserConnection_els98,
                                                    pDBInfo->dbiSession_ls00.siDBName_ls00.asCharp());
        pTransformationCmd->cmdReqErr_ls00 = errNoUserConnection_els98;
        return pTransformationCmd->cmdReqErr_ls00;
    }
    else
    {
        //*
        //* packageid
        //*
        ls10TransformationPackageId( pTransformationCmd, pDBInfo );

        switch (pTransformationCmd->cmdScanRC_ls00)
        {
            case srFctDBPartExtr_els00:
            {
                tls00_DBPartExtrCmd* pDBPartExtrCmd = REINTERPRET_CAST (tls00_DBPartExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00  = ls32ExecDBPartTransformation(pTransformationCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctDBPartLoad_els00:
            {
                tls00_DBPartLoadCmd* pDBPartLoadCmd = REINTERPRET_CAST (tls00_DBPartLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00  = ls32ExecDBPartTransformation(pTransformationCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctDBExtr_els00:
            {
                tls00_DBExtrCmd* pDBExtrCmd = REINTERPRET_CAST (tls00_DBExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00  = ls32ExecDatabaseTransformation(pTransformationCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctDBLoad_els00:
            {
                tls00_DBLoadCmd* pDBLoadCmd = REINTERPRET_CAST (tls00_DBLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00  = ls32ExecDatabaseTransformation(pTransformationCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctCatExtr_els00:   // extract catalog
            {
                tls00_CatExtrCmd* pCatExtrCmd = REINTERPRET_CAST (tls00_CatExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00  = ls31ExecCatalogTransformation(pTransformationCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctCatLoad_els00:   // load catalog
            {
                tls00_CatLoadCmd* pCatLoadCmd = REINTERPRET_CAST(tls00_CatLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00  = ls31ExecCatalogTransformation(pTransformationCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctTabExtr_els00:    // extract table
            {
                tls00_TabExtrCmd* pTabExtrCmd = REINTERPRET_CAST (tls00_TabExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00 = ls26ExecTableExtract  (pTabExtrCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;

            }
            case srFctTabLoad_els00:    // load table
            {
                tls00_TabLoadCmd* pTabLoadCmd = REINTERPRET_CAST(tls00_TabLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00  = ls28ExecTableLoad(pTabLoadCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctFastLoad_els00:   // load fast
            {
                tls00_FloadCmd* pFastLoadCmd = REINTERPRET_CAST(tls00_FloadCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00 = ls21ExecFastload (pFastLoadCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctDatExtr_els00:    // extract data
            {
                tls00_DatExtrCmd* pDatExtrCmd = REINTERPRET_CAST(tls00_DatExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
                pTransformationCmd->cmdReqErr_ls00 = ls27ExecDataExtract (pDatExtrCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctDatLoad_els00:    // load data
            {
                tls00_DatLoadCmd* pDatLoadCmd   = REINTERPRET_CAST( tls00_DatLoadCmd*, pTransformationCmd->cmdScnDat_ls00 );
                pTransformationCmd->cmdReqErr_ls00 = ls20ExecDataload (pDatLoadCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            case srFctDatUpd_els00:     // update data
            {
                tls00_DatUpdCmd* pDatUpdCmd    = REINTERPRET_CAST( tls00_DatUpdCmd* , pTransformationCmd->cmdScnDat_ls00 );
                pTransformationCmd->cmdReqErr_ls00 = ls29ExecDataUpdate ( pDatUpdCmd, pDBInfo, pTransformationCmd->cmdRpyDat_ls00);
                break;
            }
            default:
            {
                MsgInstance->ls98Msg(layBasicConnect_els98, pTransformationCmd->cmdRpyDat_ls00,errNotImplemented_els98,"Unknown Function");
                pTransformationCmd->cmdReqErr_ls00 = errNotImplemented_els98;
                break;
            }
        }//switch (pTransformationCmd->cmdScanRC_ls00)
        ls10FreeTransformation(pTransformationCmd->cmdScanRC_ls00, pTransformationCmd->cmdScnDat_ls00 );
    }

    return pTransformationCmd->cmdReqErr_ls00;
}
//  ls10ExecTransformation()

/*
  -----------------------------------------------------------------------------
  function:     ls10TransformationPackageId
  -----------------------------------------------------------------------------
*/
void
ls10TransformationPackageId(tls00_Command *&pTransformationCmd, tls00_DBInfo *pDBInfo)
{
    ROUTINE_DBG_MEO00 ("ls10TransformationPackageId");

    RepServMessages *MsgInstance  = RepServMessages::Instance();    // Error reporting objekt
    GUIDGenerator   *GuidInstance = GUIDGenerator::Instance();

    GUID_IdString    packageidString;
    GUID_Id          packageid;

    switch (pTransformationCmd->cmdScanRC_ls00)
    {
        case srFctDBPartExtr_els00:
        {
            tls00_DBPartExtrCmd* pDBPartExtrCmd = REINTERPRET_CAST (tls00_DBPartExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
            if ( false == pDBPartExtrCmd->dbpeRestart_ls00.rstRestart_ls00 )
            {
                 GuidInstance->IdGetNext (packageid);
                 GuidInstance->IdAsString(packageid, packageidString);
                 pDBPartExtrCmd->dbpeRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
            }
            pDBInfo->dbiPackageid_ls00 = pDBPartExtrCmd->dbpeRestart_ls00.rstPackageid_ls00;
            break;
        }
        case srFctDBPartLoad_els00:
        {
            tls00_DBPartLoadCmd* pDBPartLoadCmd = REINTERPRET_CAST (tls00_DBPartLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
            if( false == pDBPartLoadCmd->dbplRestart_ls00.rstRestart_ls00  )
            {
                GuidInstance->IdGetNext (packageid);
                GuidInstance->IdAsString(packageid, packageidString);
                pDBPartLoadCmd->dbplRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));

            }
            pDBInfo->dbiPackageid_ls00 = pDBPartLoadCmd->dbplRestart_ls00.rstPackageid_ls00;
            break;
        }
        case srFctDBExtr_els00:
        {
            tls00_DBExtrCmd* pDBExtrCmd = REINTERPRET_CAST (tls00_DBExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
            if ( false == pDBExtrCmd->dbeRestart_ls00.rstRestart_ls00 )
            {
                 GuidInstance->IdGetNext (packageid);
                 GuidInstance->IdAsString(packageid, packageidString);
                 pDBExtrCmd->dbeRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
            }
            pDBInfo->dbiPackageid_ls00 = pDBExtrCmd->dbeRestart_ls00.rstPackageid_ls00;
            break;
        }
        case srFctDBLoad_els00:
        {
            tls00_DBLoadCmd* pDBLoadCmd = REINTERPRET_CAST (tls00_DBLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
            if( false == pDBLoadCmd->dblRestart_ls00.rstRestart_ls00  )
            {
                GuidInstance->IdGetNext (packageid);
                GuidInstance->IdAsString(packageid, packageidString);
                pDBLoadCmd->dblRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));

            }
            pDBInfo->dbiPackageid_ls00 = pDBLoadCmd->dblRestart_ls00.rstPackageid_ls00;
            break;
        }
        case srFctCatExtr_els00:
        {
            tls00_CatExtrCmd* pCatExtrCmd = REINTERPRET_CAST (tls00_CatExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
            if ( true == pCatExtrCmd->cexRestart_ls00.rstRestart_ls00 )
            {
                pDBInfo->dbiPackageid_ls00 = pCatExtrCmd->cexRestart_ls00.rstPackageid_ls00;
            }
            else if( false == pCatExtrCmd->cexInternal_ls00 ) // && pCatExtrCmd->cexPartSpec_ls00.psPartType_ls00 > ptTable_els00 )
            {
                GuidInstance->IdGetNext (packageid) ;
                GuidInstance->IdAsString(packageid, packageidString);
                pCatExtrCmd->cexRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
                pDBInfo->dbiPackageid_ls00 = pCatExtrCmd->cexRestart_ls00.rstPackageid_ls00;
            }
            break;
        }
        case srFctCatLoad_els00:
        {
            tls00_CatLoadCmd *pCatLoadCmd = REINTERPRET_CAST(tls00_CatLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
            if ( true == pCatLoadCmd->cloRestart_ls00.rstRestart_ls00 )
            {
                pDBInfo->dbiPackageid_ls00 = pCatLoadCmd->cloRestart_ls00.rstPackageid_ls00;
            }
            else if( false == pCatLoadCmd->cloInternal_ls00 ) //&& pCatLoadCmd->cloPartSpec_ls00.psPartType_ls00 > ptTable_els00)
            {
                GuidInstance->IdGetNext (packageid);
                GuidInstance->IdAsString(packageid, packageidString);
                pCatLoadCmd->cloRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
                pDBInfo->dbiPackageid_ls00 = pCatLoadCmd->cloRestart_ls00.rstPackageid_ls00;
            }
            break;
        }
        case srFctTabExtr_els00:    // extract table
        {
            tls00_TabExtrCmd *pTabExtrCmd = REINTERPRET_CAST(tls00_TabExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
            if ( true == pTabExtrCmd->texRestart_ls00.rstRestart_ls00 )
            {
                pDBInfo->dbiPackageid_ls00 = pTabExtrCmd->texRestart_ls00.rstPackageid_ls00;
            }
            else if( false == pTabExtrCmd->texInternal_ls00  )
            {
                GuidInstance->IdGetNext (packageid) ;
                GuidInstance->IdAsString(packageid, packageidString);
                pTabExtrCmd->texRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
                pDBInfo->dbiPackageid_ls00 = pTabExtrCmd->texRestart_ls00.rstPackageid_ls00;
            }
            break;
        }
        case srFctTabLoad_els00:    // load table
        {
            tls00_TabLoadCmd *pTabLoadCmd = REINTERPRET_CAST(tls00_TabLoadCmd*, pTransformationCmd->cmdScnDat_ls00);
            if ( true == pTabLoadCmd->tloRestart_ls00.rstRestart_ls00 )
            {
                pDBInfo->dbiPackageid_ls00 = pTabLoadCmd->tloRestart_ls00.rstPackageid_ls00;
            }
            else if( false == pTabLoadCmd->tloInternal_ls00 )
            {
                GuidInstance->IdGetNext (packageid);
                GuidInstance->IdAsString(packageid, packageidString);
                pTabLoadCmd->tloRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
                pDBInfo->dbiPackageid_ls00 = pTabLoadCmd->tloRestart_ls00.rstPackageid_ls00;
            }
            break;
        }
        case srFctTabUnlo_els00:    // unload table
        {
            MsgInstance->ls98Msg(layBasicConnect_els98, pTransformationCmd->cmdRpyDat_ls00,errNotImplemented_els98,"Table unload");
            pTransformationCmd->cmdReqErr_ls00 = errNotImplemented_els98;
            break;
        }
        case srFctFastLoad_els00:   // load fast
        {
            tls00_FloadCmd* pFastLoadCmd = REINTERPRET_CAST(tls00_FloadCmd*, pTransformationCmd->cmdScnDat_ls00);
            break;
        }
        case srFctTabUpd_els00:     // update table
        {
            break;
        }
        case srFctDatExtr_els00:    // extract data
        {
            tls00_DatExtrCmd *pDatExtrCmd = REINTERPRET_CAST(tls00_DatExtrCmd*, pTransformationCmd->cmdScnDat_ls00);
            if ( true == pDatExtrCmd->dexRestart_ls00.rstRestart_ls00 )
            {
                pDBInfo->dbiPackageid_ls00 = pDatExtrCmd->dexRestart_ls00.rstPackageid_ls00;
            }
            else if( false == pDatExtrCmd->dexInternal_ls00  )
            {
                GuidInstance->IdGetNext (packageid) ;
                GuidInstance->IdAsString(packageid, packageidString);
                pDatExtrCmd->dexRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
                pDBInfo->dbiPackageid_ls00 = pDatExtrCmd->dexRestart_ls00.rstPackageid_ls00;
            }
            break;
        }
        case srFctDatLoad_els00:    // load data
        {
            tls00_DatLoadCmd *pDatLoadCmd = REINTERPRET_CAST( tls00_DatLoadCmd*, pTransformationCmd->cmdScnDat_ls00 );
            if ( true == pDatLoadCmd->dloRestart_ls00.rstRestart_ls00 )
            {
                pDBInfo->dbiPackageid_ls00 = pDatLoadCmd->dloRestart_ls00.rstPackageid_ls00;
            }
            else if( false == pDatLoadCmd->dloInternal_ls00 )
            {
                GuidInstance->IdGetNext (packageid);
                GuidInstance->IdAsString(packageid, packageidString);
                pDatLoadCmd->dloRestart_ls00.rstPackageid_ls00.ConvertFromASCII_Latin1(packageidString,packageidString+_tcslen(packageidString));
                pDBInfo->dbiPackageid_ls00 = pDatLoadCmd->dloRestart_ls00.rstPackageid_ls00;
            }
            break;
        }
        case srFctDatUpd_els00:     // update data
        {
            tls00_DatUpdCmd *pDatUpdCmd = REINTERPRET_CAST( tls00_DatUpdCmd*, pTransformationCmd->cmdScnDat_ls00 );
            break;
        }
        default:
        {
            break;
        }
    }   // end switch (pTransformationCmd->cmdScanRC_ls00)
}
//  ls10TransformationPackageId()

/*
  -----------------------------------------------------------------------------
  function:     ls10ExecRemark
  -----------------------------------------------------------------------------
*/
tsp00_Int4 ls10ExecRemark( tls00_Command *pTransformationCmd, tls00_DBInfo *pDBInfo) //!!LogMessage
{
    ROUTINE_DBG_MEO00 ("ls10ExecRemark");

    tsp00_Int4  rc = errOK_els00;

    RepServMessages *pMsgInstance = RepServMessages::Instance();

    tls00_RemarkCmd* pRemarkCmd = (tls00_RemarkCmd*)pTransformationCmd->cmdScnDat_ls00;

    Tools_DynamicUTF8String  remark;

    remark.ConvertFromASCII_Latin1(pRemarkCmd->dscRemark_ls00,
                                   pRemarkCmd->dscRemark_ls00 + pRemarkCmd->dscRemarkLen_ls00  );

    if ( remark.Length() <= 1023 )
         pMsgInstance->ls98PrintMessage (layCatalogExtract_els98, msgRemark_els98 , remark.StrPtr() );
    else
    {
         remark = remark.SubStr(0,1022);
         pMsgInstance->ls98PrintMessage (layCatalogExtract_els98, msgRemark_els98 , remark.StrPtr() );
    }

    ls10FreeTransformation(srFctRemark_els00, pTransformationCmd->cmdScnDat_ls00);

    return rc;
}
//  ls10ExecRemark

/*
  -----------------------------------------------------------------------------
  function:     ls10FreeTransformation
  -----------------------------------------------------------------------------
*/
void
ls10FreeTransformation(tls00_ScanResult Type, char * &ppTransformationCmd)
{
    ROUTINE_DBG_MEO00 ("ls10FreeTransformation");

    int i = 0;

    if ( NULL != ppTransformationCmd )
    {
        switch (Type)
        {
            case srFctDBExtr_els00:
            {
                tls00_DBExtrCmd* pDBExtrCmd = REINTERPRET_CAST( tls00_DBExtrCmd* , ppTransformationCmd );

                if ( NULL != pDBExtrCmd )
                {
                    if (NULL != pDBExtrCmd->dbeCatfile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBExtrCmd->dbeCatfile_ls00.fsFileName_ls00;
                        pDBExtrCmd->dbeCatfile_ls00.fsFileName_ls00 = NULL;
                    }
                    if (NULL != pDBExtrCmd->dbeDatfile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBExtrCmd->dbeDatfile_ls00.fsFileName_ls00;
                        pDBExtrCmd->dbeDatfile_ls00.fsFileName_ls00 = NULL;
                    }
                    if (NULL != pDBExtrCmd->dbeConfigurationFile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBExtrCmd->dbeConfigurationFile_ls00.fsFileName_ls00;
                        pDBExtrCmd->dbeConfigurationFile_ls00.fsFileName_ls00 = NULL;
                    }
                    if (NULL != pDBExtrCmd->dbePackagefile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBExtrCmd->dbePackagefile_ls00.fsFileName_ls00;
                        pDBExtrCmd->dbePackagefile_ls00.fsFileName_ls00 = NULL;
                    }

                    delete pDBExtrCmd;
                    pDBExtrCmd = NULL;
                }
                break;
            }
            case srFctDBLoad_els00:
            {
                tls00_DBLoadCmd * pDBLoadCmd = REINTERPRET_CAST( tls00_DBLoadCmd* , ppTransformationCmd );

                if ( NULL != pDBLoadCmd )
                {
                    if (NULL != pDBLoadCmd->dblCatfile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBLoadCmd->dblCatfile_ls00.fsFileName_ls00;
                        pDBLoadCmd->dblCatfile_ls00.fsFileName_ls00 = NULL;
                    }
                    if (NULL != pDBLoadCmd->dblDatfile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBLoadCmd->dblDatfile_ls00.fsFileName_ls00;
                        pDBLoadCmd->dblDatfile_ls00.fsFileName_ls00 = NULL;
                    }
                    if (NULL != pDBLoadCmd->dblConfigurationFile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBLoadCmd->dblConfigurationFile_ls00.fsFileName_ls00;
                        pDBLoadCmd->dblConfigurationFile_ls00.fsFileName_ls00 = NULL;
                    }
                    if (NULL != pDBLoadCmd->dblPackagefile_ls00.fsFileName_ls00)
                    {
                        delete [] pDBLoadCmd->dblPackagefile_ls00.fsFileName_ls00;
                        pDBLoadCmd->dblPackagefile_ls00.fsFileName_ls00 = NULL;
                    }

                    delete pDBLoadCmd;
                    pDBLoadCmd = NULL;
                }
                break;
            }
            case srFctCatExtr_els00:
            {
                tls00_CatExtrCmd* pCatExtrCmd = REINTERPRET_CAST( tls00_CatExtrCmd* , ppTransformationCmd );

                if ( NULL != pCatExtrCmd )
                {
                    // release full qualified table name
                    if (NULL != pCatExtrCmd->cexPartSpec_ls00.psFullTabName_ls00)
                    {
                        delete [] pCatExtrCmd->cexPartSpec_ls00.psFullTabName_ls00;
                        pCatExtrCmd->cexPartSpec_ls00.psFullTabName_ls00 = NULL;
                    }
                    // release outfile name
                    if (NULL != pCatExtrCmd->cexOutfile_ls00.fsFileName_ls00)
                    {
                        delete [] pCatExtrCmd->cexOutfile_ls00.fsFileName_ls00;
                        pCatExtrCmd->cexOutfile_ls00.fsFileName_ls00 = NULL;
                    }

                    delete pCatExtrCmd;
                    pCatExtrCmd = NULL;
                }
                break;
            }
            case srFctCatLoad_els00:
            {
                tls00_CatLoadCmd* pCatLoadCmd = REINTERPRET_CAST( tls00_CatLoadCmd* , ppTransformationCmd );

                if ( NULL != pCatLoadCmd )
                {
                    // release full qualified table name
                    if (NULL != pCatLoadCmd->cloPartSpec_ls00.psFullTabName_ls00)
                    {
                        delete [] pCatLoadCmd->cloPartSpec_ls00.psFullTabName_ls00;
                        pCatLoadCmd->cloPartSpec_ls00.psFullTabName_ls00 = NULL;
                    }

                    // release infile name
                    if (NULL != pCatLoadCmd->cloInfile_ls00.fsFileName_ls00)
                    {
                        delete [] pCatLoadCmd->cloInfile_ls00.fsFileName_ls00;
                        pCatLoadCmd->cloInfile_ls00.fsFileName_ls00 = NULL;
                    }

                    delete pCatLoadCmd;
                    pCatLoadCmd = NULL;
                }
                break;
            }
            case srFctTabExtr_els00:
            {
                tls00_TabExtrCmd* pTabExtrCmd = REINTERPRET_CAST(tls00_TabExtrCmd*, ppTransformationCmd);

                if ( NULL != pTabExtrCmd )
                {
                    // release full qualified table name
                    if (NULL != pTabExtrCmd->texPartSpec_ls00.psFullTabName_ls00)
                    {
                        delete [] pTabExtrCmd->texPartSpec_ls00.psFullTabName_ls00;
                        pTabExtrCmd->texPartSpec_ls00.psFullTabName_ls00 = NULL;
                    }
                    // release outfile name
                    if (NULL != pTabExtrCmd->texOutfile_ls00.fsFileName_ls00)
                    {
                        delete [] pTabExtrCmd->texOutfile_ls00.fsFileName_ls00;
                        pTabExtrCmd->texOutfile_ls00.fsFileName_ls00 = NULL;
                    }
                    if (NULL != pTabExtrCmd->texFullUserInput_ls00)
                    {
                        delete [] pTabExtrCmd->texFullUserInput_ls00;
                        pTabExtrCmd->texFullUserInput_ls00 = NULL;
                    }

                    delete pTabExtrCmd;
                    pTabExtrCmd = NULL;
                }
                break;
            }
            case srFctTabLoad_els00:
            {
                tls00_TabLoadCmd* pTabLoadCmd = REINTERPRET_CAST(tls00_TabLoadCmd*, ppTransformationCmd);

                if ( NULL != pTabLoadCmd )
                {
                    // release full qualified table name
                    if (NULL != pTabLoadCmd->tloPartSpec_ls00.psFullTabName_ls00)
                    {
                        delete [] pTabLoadCmd->tloPartSpec_ls00.psFullTabName_ls00;
                        pTabLoadCmd->tloPartSpec_ls00.psFullTabName_ls00 = NULL;
                    }

                    // release infile name
                    if (NULL != pTabLoadCmd->tloInfile_ls00.fsFileName_ls00)
                    {
                        delete [] pTabLoadCmd->tloInfile_ls00.fsFileName_ls00;
                        pTabLoadCmd->tloInfile_ls00.fsFileName_ls00 = NULL;
                    }

                    // release report file name
                    if (NULL != pTabLoadCmd->tloPackageFile_ls00.fsFileName_ls00)
                    {
                        delete [] pTabLoadCmd->tloPackageFile_ls00.fsFileName_ls00;
                        pTabLoadCmd->tloPackageFile_ls00.fsFileName_ls00 = NULL;
                    }

                    delete pTabLoadCmd;
                    pTabLoadCmd = NULL;
                }
                break;
            }
            case srFctFastLoad_els00:
            {
                tls00_FloadCmd* pFastLoadCmd = REINTERPRET_CAST( tls00_FloadCmd* , ppTransformationCmd );

                if ( NULL != pFastLoadCmd )
                {
                    if (NULL != pFastLoadCmd->flcTable_ls00.tsFullTabName_ls00)
                    {
                        delete [] pFastLoadCmd->flcTable_ls00.tsFullTabName_ls00;
                        pFastLoadCmd->flcTable_ls00.tsFullTabName_ls00 = NULL;
                    }

                    // better solution would be recursion
                    if (NULL != pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00)
                    {
                        if (NULL != pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosCond_ls00)
                        {
                            delete pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosCond_ls00;
                            pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosCond_ls00 = NULL;
                        }

                        if (NULL != pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosLeft_ls00)
                        {
                            if (NULL != pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00)
                            {
                                delete pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00;
                                pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00 = NULL;
                            }

                            delete pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosLeft_ls00;
                            pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosLeft_ls00 = NULL;
                        }

                        if (NULL != pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosRight_ls00)
                        {
                            if (NULL != pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00)
                            {
                                delete pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00;
                                pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00 = NULL;
                            }

                            delete pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosRight_ls00;
                            pFastLoadCmd->flcTable_ls00.tsCondSpec_ls00->cosRight_ls00 = NULL;
                        }
                    }

                    ls10_FreeColumnStruct(&pFastLoadCmd->flcColSpec_ls00, &pFastLoadCmd->flcInfile_ls00.fsFormat_ls00);

                    // delete member of structure flcInfile_ls00 (type: tls00_FileSpec)
                    tls00_FileSpec *pFileSpec = &pFastLoadCmd->flcInfile_ls00;
                    if (NULL != pFileSpec->fsFileName_ls00)
                    {
                        delete [] pFileSpec->fsFileName_ls00;
                        pFileSpec->fsFileName_ls00 = NULL;
                    }

                    delete pFastLoadCmd;
                    pFastLoadCmd = NULL;
                }   // end if ( NULL != pFastLoadCmd )
                break;
            }
            case srFctDatExtr_els00:
            {
                tls00_DatExtrCmd* pDatExtrCmd = REINTERPRET_CAST(tls00_DatExtrCmd*, ppTransformationCmd);

                if ( NULL != pDatExtrCmd )
                {
                    ls10_FreeColumnStruct(&pDatExtrCmd->dexOutSpec_ls00, &pDatExtrCmd->dexDatFile_ls00.fsFormat_ls00);

                    if (NULL != pDatExtrCmd->dexTable_ls00.tsFullTabName_ls00)
                    {
                        delete [] pDatExtrCmd->dexTable_ls00.tsFullTabName_ls00;
                        pDatExtrCmd->dexTable_ls00.tsFullTabName_ls00 = NULL;
                    }

                    if (NULL != pDatExtrCmd->dexFullUserInput_ls00)
                    {
                        delete [] pDatExtrCmd->dexFullUserInput_ls00;
                        pDatExtrCmd->dexFullUserInput_ls00 = NULL;
                    }

                    if (NULL != pDatExtrCmd->dexSelOrdStr_ls00)
                    {
                        delete [] pDatExtrCmd->dexSelOrdStr_ls00;
                        pDatExtrCmd->dexSelOrdStr_ls00 = NULL;
                    }

                    if (NULL != pDatExtrCmd->dexCmdFile_ls00.fsFileName_ls00)
                    {
                        delete [] pDatExtrCmd->dexCmdFile_ls00.fsFileName_ls00;
                        pDatExtrCmd->dexCmdFile_ls00.fsFileName_ls00 = NULL;
                    }

                    if (NULL != pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00)
                    {
                        delete [] pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00;
                        pDatExtrCmd->dexDatFile_ls00.fsFileName_ls00 = NULL;
                    }

                    if (NULL != pDatExtrCmd->dexLongSpec_ls00)
                    {
                        if (0 != pDatExtrCmd->dexLongSpec_ls00->mfsCount_ls00)
                        {
                            for (i=0; i < pDatExtrCmd->dexLongSpec_ls00->mfsCount_ls00; ++i)
                            {
                                if (NULL != pDatExtrCmd->dexLongSpec_ls00->mfsFiles_ls00[i]->fsFileName_ls00)
                                {
                                    delete [] pDatExtrCmd->dexLongSpec_ls00->mfsFiles_ls00[i]->fsFileName_ls00;
                                    pDatExtrCmd->dexLongSpec_ls00->mfsFiles_ls00[i]->fsFileName_ls00 = NULL;
                                }

                                if (NULL != pDatExtrCmd->dexLongSpec_ls00->mfsFiles_ls00[i]->fsLongColName_ls00)
                                {
                                    delete [] pDatExtrCmd->dexLongSpec_ls00->mfsFiles_ls00[i]->fsLongColName_ls00;
                                    pDatExtrCmd->dexLongSpec_ls00->mfsFiles_ls00[i]->fsLongColName_ls00 = NULL;
                                }

                                delete pDatExtrCmd->dexLongSpec_ls00->mfsFiles_ls00[i];
                            }
                        }

                        delete pDatExtrCmd->dexLongSpec_ls00;
                    }
                    // release full qualified table name
                    if (NULL != pDatExtrCmd->dexPartSpec_ls00.psFullTabName_ls00)
                    {
                        delete [] pDatExtrCmd->dexPartSpec_ls00.psFullTabName_ls00;
                        pDatExtrCmd->dexPartSpec_ls00.psFullTabName_ls00 = NULL;
                    }

                    delete pDatExtrCmd;
                    pDatExtrCmd = NULL;
                }   // end if ( NULL != pDatExtrCmd )
                break;
            }
            case srFctDatLoad_els00:
            {
                tls00_DatLoadCmd *pDatLoadCmd = REINTERPRET_CAST(tls00_DatLoadCmd*, ppTransformationCmd);

                if ( NULL != pDatLoadCmd )
                {
                    if (NULL != pDatLoadCmd->dloTable_ls00.tsFullTabName_ls00)
                    {
                        delete [] pDatLoadCmd->dloTable_ls00.tsFullTabName_ls00;
                        pDatLoadCmd->dloTable_ls00.tsFullTabName_ls00 = NULL;
                    }

                    // better solution would be recursion
                    if (NULL != pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00)
                    {
                        if (NULL != pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosCond_ls00)
                        {
                            delete pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosCond_ls00;
                            pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosCond_ls00 = NULL;
                        }

                        if (NULL != pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosLeft_ls00)
                        {
                            if (NULL != pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00)
                            {
                                delete pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00;
                                pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00 = NULL;
                            }

                            delete pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosLeft_ls00;
                            pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosLeft_ls00 = NULL;
                        }

                        if (NULL != pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosRight_ls00)
                        {
                            if (NULL != pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00)
                            {
                                delete pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00;
                                pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00 = NULL;
                            }

                            delete pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosRight_ls00;
                            pDatLoadCmd->dloTable_ls00.tsCondSpec_ls00->cosRight_ls00 = NULL;
                        }
                    }

                    ls10_FreeColumnStruct(&pDatLoadCmd->dloColSpec_ls00, &pDatLoadCmd->dloInfile_ls00.fsFormat_ls00);

                    // delete member of structure dloInfile_ls00 (type: tls00_FileSpec)
                    tls00_FileSpec *pFileSpec = &pDatLoadCmd->dloInfile_ls00;
                    if (NULL != pFileSpec->fsFileName_ls00)
                    {
                        //!! ToDO
                        //delete [] pFileSpec->fsFileName_ls00;
                        pFileSpec->fsFileName_ls00 = NULL;
                    }

                    delete pDatLoadCmd;
                    pDatLoadCmd = NULL;
                }   // end if ( NULL != pDatLoadCmd )
                break;
            }
            case srFctDatUpd_els00:
            {
                tls00_DatUpdCmd* pDatUpdCmd = REINTERPRET_CAST(tls00_DatUpdCmd*, ppTransformationCmd);

                if ( NULL != pDatUpdCmd )
                {
                    if (NULL != pDatUpdCmd->dupTable_ls00.tsFullTabName_ls00)
                    {
                        delete [] pDatUpdCmd->dupTable_ls00.tsFullTabName_ls00;
                        pDatUpdCmd->dupTable_ls00.tsFullTabName_ls00 = NULL;
                    }
                    // better solution would be recursion
                    if (NULL != pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00)
                    {
                        if (NULL != pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosCond_ls00)
                        {
                            delete pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosCond_ls00;
                            pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosCond_ls00 = NULL;
                        }

                        if (NULL != pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosLeft_ls00)
                        {
                            if (NULL != pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00)
                            {
                                delete pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00;
                                pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosLeft_ls00->cosCond_ls00 = NULL;
                            }

                            delete pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosLeft_ls00;
                            pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosLeft_ls00 = NULL;
                        }

                        if (NULL != pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosRight_ls00)
                        {
                            if (NULL != pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00)
                            {
                                delete pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00;
                                pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosRight_ls00->cosCond_ls00 = NULL;
                            }

                            delete pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosRight_ls00;
                            pDatUpdCmd->dupTable_ls00.tsCondSpec_ls00->cosRight_ls00 = NULL;
                        }
                    }

                    ls10_FreeColumnStruct(&pDatUpdCmd->dupColSpec_ls00, &pDatUpdCmd->dupInfile_ls00.fsFormat_ls00);

                    // delete member of structure dloInfile_ls00 (type: tls00_FileSpec)
                    tls00_FileSpec *pFileSpec = &pDatUpdCmd->dupInfile_ls00;
                    if (NULL != pFileSpec->fsFileName_ls00)
                    {
                        delete [] pFileSpec->fsFileName_ls00;
                        pFileSpec->fsFileName_ls00 = NULL;
                    }

                    delete pDatUpdCmd;
                    pDatUpdCmd = NULL;
                }   // end if ( NULL != pDatUpdCmd )
                break;
            }
            case srFctRemark_els00:
            {
                tls00_RemarkCmd* pRemarkCmd = REINTERPRET_CAST(tls00_RemarkCmd*, ppTransformationCmd);

                if ( NULL != pRemarkCmd )
                {
                    if (NULL != pRemarkCmd->dscRemark_ls00)
                    {
                        delete [] pRemarkCmd->dscRemark_ls00;
                        pRemarkCmd->dscRemark_ls00 = NULL;
                    }

                    delete pRemarkCmd;
                    pRemarkCmd = NULL;
                }   // end if ( NULL != pRemark )
                break;
            }
        }   // end switch (Type)
    }   // end if ( NULL != ppTransformationCmd )
    return;
}
//  ls10FreeTransformation()


/*
  -----------------------------------------------------------------------------
  function:     ls10_FreeColumnStruct
  -----------------------------------------------------------------------------
*/
static void
ls10_FreeColumnStruct(tls00_MultColSpec* pMultColSpec, tls00_FileFormat*  pFileFormat)
{
    if (NULL != pMultColSpec)
    {
        for (tsp00_Int4 i = 0; i < pMultColSpec->mcsCount_ls00; ++i)
        {
            tls00_Column *pCol = pMultColSpec->mcsColumn_ls00[i];

            if ( NULL != pCol )
            {
                // LONGFILE info in tls00_Column structure
                if ( (true  == pCol->colLongInfo.loiIsLong_ls00) &&
                     (false == pFileFormat->ffoCompact_ls00) )
                {
                    if (NULL != pCol->colLongInfo.loiFileName_ls00)
                    {
                        delete [] pCol->colLongInfo.loiFileName_ls00;
                        pCol->colLongInfo.loiFileName_ls00 = NULL;
                    }

                    if (NULL != pCol->colLongInfo.loiFileStruct_ls00)
                    {
                        delete pCol->colLongInfo.loiFileStruct_ls00;
                        pCol->colLongInfo.loiFileStruct_ls00 = NULL;
                    }
                }


                // Assignment values
                if (NULL != pCol->colAssign_ls00.fasAsgnVal_ls00)
                {
                    delete pCol->colAssign_ls00.fasAsgnVal_ls00;
                    pCol->colAssign_ls00.fasAsgnVal_ls00 = NULL;
                }

                if (NULL != pCol->colNullCond_ls00)        // RECURSION REQUIRED
                {
                    if (NULL != pCol->colNullCond_ls00->cosCond_ls00)
                    {
                        if (NULL != pCol->colNullCond_ls00->cosCond_ls00->conCompVal_ls00)
                        {
                            // in case of ascii or utf8 files the null spec pointer points to
                            // null spec in file format and MUST NOT BE DELETED
                            if (pFileFormat->ffoNullSpec_ls00 !=
                                pCol->colNullCond_ls00->cosCond_ls00->conCompVal_ls00)
                            {
                                delete [] pCol->colNullCond_ls00->cosCond_ls00->conCompVal_ls00;
                            }
                            pCol->colNullCond_ls00->cosCond_ls00->conCompVal_ls00 = NULL;
                        }
                        if (NULL != pCol->colNullCond_ls00->cosCond_ls00->conBinCompVal_ls00)
                        {
                            delete [] pCol->colNullCond_ls00->cosCond_ls00->conBinCompVal_ls00;
                            pCol->colNullCond_ls00->cosCond_ls00->conBinCompVal_ls00 = NULL;
                        }
                        pCol->colNullCond_ls00->cosCond_ls00->conValLen_ls00 = 0;


                        delete pCol->colNullCond_ls00->cosCond_ls00;
                        pCol->colNullCond_ls00->cosCond_ls00 = NULL;
                    }
                    delete pCol->colNullCond_ls00;
                    pCol->colNullCond_ls00 = NULL;
                }

                memset(pCol, 0, sizeof(tls00_Column));
                delete pCol;
                pCol = NULL;

            }   // end if ( NULL != pCol )
        }   // end for (tsp00_Int4 i = 0; i < pMultColSpec->mcsCount_ls00; ++i)
    }   // end if (NULL != pMultColSpec)
}
// ls10_FreeColumnStruct()

/*===========================================================================*
 *     END
 *===========================================================================*/

