/*!***************************************************************************

    module      : vkb391.cpp
    responsible : MartinB,UweH
    special area: KB_Logging
    last changed: 2001-05-03
    copyright   : (c) 2000-2004 SAP AG
    description : Interface to new Logging component Log_SaveIterator
                  added WaitListHandling (UH)
                  added RTEConf_Parameter handling (UH)

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

*****************************************************************************/
#include "Logging/Log_Page.hpp"
#include "Logging/Log_SaveIterator.hpp"
#include "Logging/Log_Volume.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_NewDestroy.hpp"
#include "KernelCommon/Kernel_TaskWaitList.hpp"
#include "KernelCommon/Kernel_Version.hpp"
#include "RunTime/Configuration/RTEConf_ParameterAccessKernelInterface.hpp"
#include "RunTime/Configuration/RTEConf_ParameterAccess.hpp"

#include "hkb57.h"   // k57save_restartrecord
#include "hkb57_1.h" // k57restartrec

#include "hkb391.h"
/*---------------------------------------------------------------------------*/
externPascal void kb391_InitLogSaveIterator (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    tsp00_Int2                           maxToBeRead,
    pasbool                              dbIsCold,
    pasbool                              forAutosave,
    pasbool                              completeLog,
    Log_SaveIterator*&                   iterator)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391_InitLogSaveIterator", LogVolume_Trace, 5);

    trans.trError_gg00 = e_ok;

    if (maxToBeRead < 0)
    {
        RTE_Crash( SAPDBErr_Exception(__FILE__, __LINE__,
                      SAPDBERR_ASSERT_STATE_FAILED,
                      "kb391_InitLogSaveIterator:max_to_be_read < 0"));
    }

    SAPDBMem_IRawAllocator& allocator =
              *(reinterpret_cast <SAPDBMem_IRawAllocator*>(trans.trAllocator_gg00));

    Log_Volume           &log = Log_Volume::Instance();
    Log_Volume::SaveInfo  saveInfo;
    
    log.GetSaveLogIterator(  trans.trTaskId_gg00,
                             allocator,
                             1==dbIsCold,
                             1==forAutosave,
                             1==completeLog,
                             maxToBeRead,
                             iterator,
                             saveInfo);

    if ( iterator == NULL )
    {
        switch (saveInfo)
        {
            case Log_Volume::okay:
                    trans.trError_gg00 = e_ok;
                    break;
            case Log_Volume::noLogToSave:
					if ( forAutosave )
						trans.trError_gg00 = e_incomplete_logsegm;
					else
						trans.trError_gg00 = e_no_log_to_save;
                    break;
            case Log_Volume::historyIsLost:
                    trans.trError_gg00 = e_backup_history_lost; 
                    break;
            case Log_Volume::incompleteSegment:
                    trans.trError_gg00 = e_incomplete_logsegm;
                    break;
            default:
                    trans.trError_gg00 = e_ok;
                    break;
        }
    }
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_ReadNextLogpages (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    Log_SaveIterator&                    iterator,
    tkb3_block_ptr                       queue_buffer,
    tsp00_Int2            VAR_VALUE_REF  num_read)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391_ReadNextLogpages", LogVolume_Trace, 5);
    Log_DeviceBlockCount unsigned_numPagesRead;
    
    iterator.GetNextLogPages (trans.trTaskId_gg00, queue_buffer, unsigned_numPagesRead);
    num_read = unsigned_numPagesRead;
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_PrepareFreeLogForPipe (
    Log_SaveIterator&                    iterator,
    tgg00_BasisError      VAR_VALUE_REF  trError)
{
    // PTS 1114791 mb 2002-08-22 new
    Log_Volume &log = Log_Volume::Instance();
    Log_Volume::PipeHandlingResult result;
    result = log.PrepareFreeLogForPipe(iterator.GetStartIOSequence(),
                                       iterator.GetEndIOSequence(),
                                       iterator.GetStartPageOffset(),
                                       iterator.GetEndPageOffset());
    switch (result)
    {
        case Log_Volume::pipeOk:
            trError = e_ok; 
            break;
        case Log_Volume::noFreeLogCalledInbetween:
            trError = e_invalid; // same error as in V7.3
            break;
        default:
            trError = e_log_error;
    }
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_DeleteIterator (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    Log_SaveIterator**                   iterator)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391_DeleteIterator", LogVolume_Trace, 5);
    SAPDBERR_ASSERT_STATE( iterator != NULL );

    // deallocate memory */
    SAPDBMem_IRawAllocator& Allocator =
        *(REINTERPRET_CAST (SAPDBMem_IRawAllocator*,
                            trans.trAllocator_gg00));

    (*iterator)->Delete();
    destroy(*iterator, Allocator);
    *iterator = NULL;
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetStartTime (
    Log_SaveIterator&                    iterator,
    Kernel_Time&                         startTime,
    Kernel_Date&                         startDate)
{
    iterator.GetStartTime(startDate, startTime);
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetEndTime (
    Log_SaveIterator&                    iterator,
    Kernel_Time&                         endTime,
    Kernel_Date&                         endDate)
{
    iterator.GetEndTime(endDate, endTime);
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetStartIOSeq (
    Log_SaveIterator&                    iterator,
    Log_IOSequenceNo&                    startIOSeq)
{
    startIOSeq = iterator.GetStartIOSequence();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetEndIOSeq (
    Log_SaveIterator&                    iterator,
    Log_IOSequenceNo&                    endIOSeq)
{
    endIOSeq = iterator.GetEndIOSequence();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetStartPageOffset (
    Log_SaveIterator&                    iterator,
    Log_RawDeviceOffset&                 startOffset)
{
    startOffset = iterator.GetStartPageOffset();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetLastSavedIOSeq (
    Log_SaveIterator&                    iterator,
    Log_IOSequenceNo&                    endIOSeq)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391_GetLastSavedIOSeq", LogVolume_Trace, 5);
    endIOSeq = iterator.GetEndIOSequence();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetLastSavedPageOffset (
    Log_SaveIterator&                    iterator,
    tsp00_Uint4           VAR_VALUE_REF  endOffset)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391_GetLastSavedPageOffset", LogVolume_Trace, 5);
    endOffset = iterator.GetEndPageOffset();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetNumOfPagesLeftForSegment (
    Log_SaveIterator*                    iterator,
    tsp00_Uint4           VAR_VALUE_REF  numOfUnsavedPages)
{
    if ( 0 == iterator )
        numOfUnsavedPages = 0;
	else
        numOfUnsavedPages = iterator->GetNumPagesLeft();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetNumOfPagesLeftForLogDev (
    Log_SaveIterator*                    iterator,
    tsp00_Uint4           VAR_VALUE_REF  numOfUnsavedPages)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391_GetNumOfPagesLeftForLogDev", LogVolume_Trace, 5);

    Log_Volume &log = Log_Volume::Instance();
    if ( 0 == iterator )
    {
        numOfUnsavedPages = log.GetNumberOfUnsavedPages();
        return;
    }

    Log_DeviceBlockCount autosaveDistance = log.GetAutosaveDistance();

    if ( log.GetNumberOfUnsavedPages() >= autosaveDistance )
        numOfUnsavedPages = log.GetNumberOfUnsavedPages()
                            - autosaveDistance + iterator->GetNumPagesLeft();
    else
        // the last segment to be saved
        numOfUnsavedPages = iterator->GetNumPagesLeft();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391_GetBackupCount (
    Log_SaveIterator&                    iterator,
    tsp00_Uint4           VAR_VALUE_REF  backupCount)
{
    backupCount = iterator.GetLogBackupCount();
}
/*---------------------------------------------------------------------------*/
externPascal tsp00_Uint4 kb391_ReadFirstWriterIOSequenceNo (
    tkb00_Page            VAR_VALUE_REF  page)
{
    Kernel_IPage::PageFrame frame(&page, g01page_size);
    Log_Page log_page(frame);
    return log_page.ReadFirstWriterIOSequenceNo().RawValue();
}
/*---------------------------------------------------------------------------*/
externPascal tsp00_Uint4 kb391_ReadLastWriterIOSequenceNo (
    tkb00_Page            VAR_VALUE_REF  page)
{
    Kernel_IPage::PageFrame frame(&page, g01page_size);
    Log_Page log_page(frame);
    return log_page.ReadLastWriterIOSequenceNo().RawValue();
}
/*---------------------------------------------------------------------------*/
externPascal pasbool kb391_IsCompleteSegment (
    Log_SaveIterator&                    iterator)
{
    return iterator.IsCompleteSegment();
}
//----------------------------------------------------------------------------
Kernel_TaskWaitList kb391WaitListAutosaveRead;
Kernel_TaskWaitList kb391WaitListAutosaveWrite;
Kernel_TaskWaitList kb391WaitListBackupRead;
Kernel_TaskWaitList kb391WaitListBackupWrite;
//----------------------------------------------------------------------------
externPascal void kb391InitializeWaitLists ()
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391InitializeWaitLists", LogVolume_Trace, 5);
	// necessary as long as we have a C++ Main so the compiler generates
    // code to call the constructors of global variables.
    kb391WaitListAutosaveRead  = Kernel_TaskWaitList();
    kb391WaitListAutosaveWrite = Kernel_TaskWaitList();
    kb391WaitListBackupRead    = Kernel_TaskWaitList();
    kb391WaitListBackupWrite   = Kernel_TaskWaitList();
}
//----------------------------------------------------------------------------
static inline Kernel_TaskWaitList& GetWaitList ( bool forRead,
                                                 bool forAutosave )
{
    if ( forRead )
        return forAutosave ? kb391WaitListAutosaveRead : kb391WaitListBackupRead;
	else
        return forAutosave ? kb391WaitListAutosaveWrite : kb391WaitListBackupWrite;
}
/*---------------------------------------------------------------------------*/
externPascal void kb391ResumeAllWaitingTask (
    pasbool                              forAutosave)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391ResumeAllWaitingTask", LogVolume_Trace, 5);
    if ( forAutosave )
    {
        kb391WaitListAutosaveRead.ResumeAll();
        kb391WaitListAutosaveWrite.ResumeAll();
    }
    else
    {
        kb391WaitListBackupRead.ResumeAll();
        kb391WaitListBackupWrite.ResumeAll();
    }
}
/*---------------------------------------------------------------------------*/
externPascal void kb391ResumeFirstWaitingTask (
    pasbool                              forRead,
    pasbool                              forAutosave)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391ResumeFirstWaitingTask", LogVolume_Trace, 5);
    GetWaitList(forRead,forAutosave).ResumeFirst();
}
/*---------------------------------------------------------------------------*/
externPascal void kb391InsertWaitingTaskAtLast (
    tsp00_TaskId                         taskid,
    tgg00_WaitContext     VAR_VALUE_REF  waitcontext,
    pasbool                              forRead,
    pasbool                              forAutosave)
{
    SAPDBTRACE_ROUTINE_DEBUG ("kb391ResumeFirstWaitingTask", LogVolume_Trace, 5);
    GetWaitList(forRead,forAutosave).InsertAsLast (taskid, waitcontext);
}
/*---------------------------------------------------------------------------*/
externPascal void kb391Create_RTEConf_ParameterAccess (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    tsp00_Addr            VAR_VALUE_REF  params)
{
    SAPDBMem_IRawAllocator& allocator    = *(reinterpret_cast<SAPDBMem_IRawAllocator*>(trans.trAllocator_gg00));
    const SAPDB_Char *      dummyDbname  = "";
    const SAPDB_Int4        dummyVersion = 0;

    if ( params != 0 )
        RTE_Crash(SAPDBErr_Exception(__CONTEXT__,SAPDBERR_ASSERT_ARGUMENT_FAILED, "params == 0"));

    RTEConf_Parameter* auxParams = new (allocator)
        RTEConf_Parameter(dummyDbname, dummyVersion, RTECONF_MAXNAMELENGTH, RTECONF_MAXSTRINGLENGTH);

    if ( auxParams == 0 )
        trans.trError_gg00 = e_no_more_memory;

    params = reinterpret_cast<tsp00_Addr>(auxParams);
}
/*---------------------------------------------------------------------------*/
externPascal void kb391AppendTo_RTEConf_ParameterAccess (
    RTEConf_Parameter&                   params,
    tkb3_xparam_page      VAR_VALUE_REF  xpPage)
{
    SAPDB_Byte * data = reinterpret_cast<SAPDB_Byte*>(&(xpPage.xp_data()));
    SAPDBErr_MessageList  error;
    if ( ! params.AddBinaryData(data, xpPage.xp_length(), error) )
    {
        RTE_Message(error);
        return;
    }
}
/*---------------------------------------------------------------------------*/
externPascal tgg00_BasisError kb391CheckParameterForRestore (
    RTEConf_Parameter&                   params)
{
    SAPDBErr_MessageList error;
    SAPDB_Bool           dataRead;

    if ( ! params.InterpretBinaryData(dataRead, error) )
    {
        RTE_Message(error);
        return e_conf_parameter_error;
    }

    // Add the checks here
    
    RTEConf_Parameter::String valueFromBackup;
    RTEConf_Parameter::String valueFromInstance;
    const SAPDB_UTF8 * paramName = UTF8("_UNICODE");

    if ( ! RTEConf_ParameterAccess::Instance()->GetString(paramName, valueFromInstance, error) ) 
    {
        RTE_Message(error);
        return e_conf_parameter_error;
    }

    if ( ! params.GetReference(paramName, valueFromBackup, error) ) 
    {
        RTE_Message(error);
        return e_conf_parameter_error;
    }
    
    if ( valueFromInstance[0] != valueFromBackup[0] )
    {
        RTE_Message(SAPDBErr_Exception(__CONTEXT__,SAPDBERR_ASSERT_STATE_FAILED,"_UNICODE must be equal"));
        return e_wrong_configuration;
    }

    return e_ok;
}
/*---------------------------------------------------------------------------*/
externPascal void kb391Destroy_RTEConf_ParameterAccess (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    tsp00_Addr            VAR_VALUE_REF  params)
{
    SAPDBMem_IRawAllocator& allocator = *(reinterpret_cast<SAPDBMem_IRawAllocator*>(trans.trAllocator_gg00));
    destroy(params, allocator);
    params = 0;
}
/*-----------------------------------------------------------------------------------*/

externPascal pasbool kb391SupressDBIdentCheckForMigration (    // PTS 1116030 mb 2002-06-03 new
    tsp00_Version         VAR_ARRAY_REF  backupVersion)
{
    // Migration: DBIdentifier may not be correct initialized in 7.4.1 or 7.4.2 with build > 7
    Kernel_Version auxBackupVersion (backupVersion);

    if ( auxBackupVersion == Kernel_Version(7,4,1)
         ||
         (auxBackupVersion == Kernel_Version(7,4,2) && auxBackupVersion.BuildNo() > 7) )
    {
        return true;
    }

    return false;
}

/*-----------------------------------------------------------------------------------*/

externPascal void kb391CheckLogTapeAndDataCompatibility (
    tgg00_TransContext    VAR_VALUE_REF  trans,
    const tkb00_Page&                    restartRecord,
    tsp00_Version         VAR_ARRAY_REF  backupVersion,
    tsp00_Line            VAR_ARRAY_REF  dbident,
    const Log_IOSequenceNo&                 firstIOSeq,
    const Log_IOSequenceNo&                 lastIOSeq)
{
    trans.trError_gg00 = e_ok;

    const Log_IOSequenceNo lastSavepointIOSequence = restartRecord.rstLastSavept_kb00().svpIOsequence_kb00;
    
    // PTS 1113550 mb 2002-05-06
    if ( lastSavepointIOSequence.IsInvalid() )
    {
        if ( firstIOSeq == MIN_IOSEQUENCE )
        {
            // for a restart from the Log_IOSequence 0 into an empty DB: Copy DB-Identifier
            k57restartrec->rstDbIdent_kb00() = dbident;
            k57save_restartrecord (trans.trTaskId_gg00);
    
            Log_Volume &log = Log_Volume::Instance();
            log.SetLogDBIdent (trans.trTaskId_gg00,dbident);
        }
        else
        {
            // RESTORE LOG directly after INIT CONFIG without RESTORE DATA
            trans.trError_gg00 = e_incompatible_log;
        }
        return;
    }

    /* PTS 1113550 mb 2002-22-03 */
    if (memcmp(dbident, &(restartRecord.rstDbIdent_kb00()), sizeof(tsp00_Line)))
    {
        if (!kb391SupressDBIdentCheckForMigration(backupVersion))
        {
            trans.trError_gg00 = e_invalid_label;
            return;
        }
    }

    if ( lastSavepointIOSequence.IsValid())
    {
        if (!lastSavepointIOSequence.IsInRange(firstIOSeq, lastIOSeq))
        {
            trans.trError_gg00 = e_incompatible_log;
            return;
        }
    }
}

