/*!
  @file           veo07.c
  @author         JoergM
  @special area   Multithreading Runtine OS Independend
  @brief          Multithread Interface using WIN32 Threads, CThreads or
              PThreads DRAFT 4-10. See DEFINES section below...
  @see            example.html ...

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/

/*
 * _REENTRANT is defined for all UNIX platforms if compiled multithreaded
 * but for WIN32 this is _MT ...
 */
#if ( defined(_REENTRANT) && !defined(_WIN32) ) || ( defined(_MT) && defined(_WIN32) )

#define MOD__ "veo07.c :"

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/
#ifdef _WIN32
#include <windows.h>
#include <limits.h>
#endif

#if defined(SUN)
#include <errno.h>
#endif

#if defined(OSF1) || defined(_IBMR2) || defined(NMP) || defined(LINUX) || defined(SUN)
#include <pthread.h>
#endif

#if defined(HP9)
#if defined(PA11) || defined(PA20W)
#define _POSIX_C_SOURCE 199506L
#include <pthread.h>
#endif /* PA11 || PA20W */
#endif /* HP9 */

#if defined(HP_IA64)
#define _POSIX_C_SOURCE 199506L
#include <pthread.h>
#endif /* HP_IA64 */

#if !defined(_WIN32)
#include <unistd.h>
#endif

#include <signal.h>

#include "geo00_0.h"
#include "geo57.h"
#include "heo07.h"
#include "geo001.h"
#include "geo007_1.h" /* For error messages */
#include "geo007_2.h" /* For error messages */
#if defined(_WIN32) && defined(KERNEL_LZU)
#include "geo50_0.h" /* nocheck */ /* sql57k_pmalloc, sql57k_pfree, vabort */
#endif /*_WIN32 && KERNEL_LZU*/

#if defined(KERNEL_LZU) && !defined(WIN32)
#    include "RunTime/RTE_Coroutine.h" /*nocheck*/
#  ifndef SAPDB_WITH_EXCEPTION_SAVE_COROUTINES
/* #define SAPDB_WITH_EXCEPTION_SAVE_COROUTINES 1 */
#    define SAPDB_WITH_EXCEPTION_SAVE_COROUTINES 0
#  endif

#  if SAPDB_WITH_EXCEPTION_SAVE_COROUTINES > 0
#    define GET_SELF ((teo07_ThreadObj *)eo07_GetSelf())
#  else
#    define GET_SELF ((teo07_ThreadObj *)sqlgettls(self_key_eo07))
#  endif
#else
#  define GET_SELF ((teo07_ThreadObj *)sqlgettls(self_key_eo07))
#endif

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

#if defined(_WIN32)

# define YIELD_CALL Sleep(0)

#else

# undef CTHREADS
# undef PTHREADS
# undef PTHREAD_DRAFT
# undef HAVE_PTHREAD_SCOPE_PROCESS
# undef TIMEOUT_SYSTEM_RC
    /* PTS 1105217 */
# undef USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID
# undef USE_GETSEQUENCE_NP_FOR_THREAD_ID
# undef USE_CLONE_PID_FOR_THREAD_ID

# if defined(SUN)
#  define PTHREADS
#  define PTHREAD_DRAFT 10
#  define TIMEOUT_SYSTEM_RC ETIMEDOUT
#  define USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID 1
#  define YIELD_CALL sched_yield()
#  define HAVE_PTHREAD_SCOPE_PROCESS
# endif

# if defined(OSF1)
#  define PTHREADS
#  define PTHREAD_DRAFT 10
#  define TIMEOUT_SYSTEM_RC ETIMEDOUT
#  define USE_GETSEQUENCE_NP_FOR_THREAD_ID 1
#  define YIELD_CALL sched_yield()
#  define HAVE_PTHREAD_SCOPE_PROCESS
#  define WANT_HIGHEST_ADDRESS_AS_STACK_ADDRESS
# endif

# if defined(_IBMR2)
#  define PTHREADS
#  define PTHREAD_DRAFT 7
#  define TIMEOUT_SYSTEM_RC ETIMEDOUT
#  define USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID 1
#  define YIELD_CALL yield()
#  define HAVE_PTHREAD_SCOPE_PROCESS
#  define WANT_HIGHEST_ADDRESS_AS_STACK_ADDRESS
# endif

# if defined(HP9)
#  if defined(PA11) || defined(PA20W)
#   define PTHREADS
#   define PTHREAD_DRAFT 10
#   define TIMEOUT_SYSTEM_RC ETIMEDOUT
#   define USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID 1
#   define YIELD_CALL sched_yield()
#  else
#   define PTHREADS
#   define PTHREAD_DRAFT 4
#   define TIMEOUT_SYSTEM_RC EAGAIN
#   define USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID 1
#   define YIELD_CALL pthread_yield()
#  endif 
# endif /* HP9 */

# if defined(HP_IA64)
#   define PTHREADS
#   define PTHREAD_DRAFT 10
#   define TIMEOUT_SYSTEM_RC ETIMEDOUT
#   define USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID 1
#   define YIELD_CALL sched_yield()
# endif /* HP_IA64 */

# if defined(NMP)
#  define PTHREADS
#  define PTHREAD_DRAFT 10
#  define TIMEOUT_SYSTEM_RC ETIMEDOUT
#  define USE_CLONE_PID_FOR_THREAD_ID 1
#  define YIELD_CALL sched_yield()
#  ifndef _POSIX_THREAD_ATTR_STACKSIZE
#    define _POSIX_THREAD_ATTR_STACKSIZE
#  endif
# endif  /* NMP */

# if defined(LINUX)
#  define PTHREADS
#  define PTHREAD_DRAFT 10
#  define TIMEOUT_SYSTEM_RC ETIMEDOUT
#  define USE_CLONE_PID_FOR_THREAD_ID 1
#  define YIELD_CALL sched_yield()
# endif  /* LINUX */

#endif /* !_WIN32 */

#if defined(_WIN32)
# define THREAD_TYPENAME_EO07 HANDLE
# define MUTEX_TYPENAME_EO07  CRITICAL_SECTION
#endif
#if defined(CTHREADS)
# define THREAD_TYPENAME_EO07 thread_t
# define MUTEX_TYPENAME_EO07  mutex_t
# define COND_TYPENAME_EO07   cond_t
#endif
#if defined(PTHREADS)
# define THREAD_TYPENAME_EO07 pthread_t
# define MUTEX_TYPENAME_EO07  pthread_mutex_t
# define COND_TYPENAME_EO07   pthread_cond_t
#endif

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/

#ifdef USE_MALLOC
#include <malloc.h>
#undef ALLOC_MEM_EO57
#define ALLOC_MEM_EO57(_pBuf,_s) \
    ( (*_pBuf = malloc(_s)) != NULL ? NO_ERROR_EO001 : ERROR_NOT_ENOUGH_MEMORY_EO001 )
#undef FREE_MEM_EO57
#define FREE_MEM_EO57(_Buf) free(_Buf)
#endif /* USE_MALLOC */

#ifdef DO_ABORT
#  undef DO_ABORT
#endif /* DO_ABORT */

#if defined(_WIN32)
#  define DO_ABORT() ABORT()
#else
#  if defined(KERNEL_LZU)
#    define DO_ABORT() vabort(1) /* WRITE_CORE ... */
#  else
#    define DO_ABORT() sqlabort()
#  endif
#endif

#ifdef PTHREADS
#if PTHREAD_DRAFT < 7
#define PTHREAD_RC(_rc,_x) (_rc) = (_x);if ( (_rc) == -1 ) (_rc) = errno
#define PTHREAD_MUTEX_INIT(_pmutex) pthread_mutex_init(_pmutex, pthread_mutexattr_default)
#define PTHREAD_COND_INIT(_pcond) pthread_cond_init (_pcond, pthread_condattr_default)
#endif
#if PTHREAD_DRAFT > 6
#define PTHREAD_RC(_rc,_x) (_rc) = (_x)
#define PTHREAD_MUTEX_INIT(_pmutex) pthread_mutex_init(_pmutex, NULL)
#define PTHREAD_COND_INIT(_pcond) pthread_cond_init (_pcond, NULL)
#endif
#endif /* PTHREADS */

/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/

/* TYPEDEFS */
typedef struct _teo07_Thread
{
  THREAD_TYPENAME_EO07  thread;
  teo07_ThreadId        thread_id;
  teo07_ThreadSemaphore suspend_sem; /* for suspended start and selfsuspend */
  tsp00_Int4            start_suspended;
#ifdef LINUX
  char *                StackAddr;
  size_t                StackSize;
#  if SAPDB_WITH_EXCEPTION_SAVE_COROUTINES > 0
  int                   threadLocalErrno;
  int                  *pThreadLocalErrno;
  int                   threadLocalHErrno;
  int                  *pThreadLocalHErrno;
  int                   libcSpecificTlsKeys[RTE_MAXIMUM_LIBC_SPECIFIC_KEYS];
#  endif
#endif
#if SAPDB_WITH_EXCEPTION_SAVE_COROUTINES > 0
  void              *** tlsKeys;
#endif
  void *              (*proc)(void *);
  void                 *arg;
} teo07_ThreadObj;

struct _teo07_Mutex
{
  MUTEX_TYPENAME_EO07   mutex;
  teo07_ThreadId        owning_thread;
  tsp00_Int4            lock_count;
};

#if !defined(_WIN32)
typedef struct _teo07_ThreadSemaphore
{
    tsp00_Int4          token;
    tsp00_Int4          waiter;
    MUTEX_TYPENAME_EO07 mutex;
    COND_TYPENAME_EO07  condition;
} teo07_SemObj;
#endif /* !WIN32 */
  

/*===========================================================================*
 *  GLOBAL VARIABLES                                                         *
 *===========================================================================*/

#  if SAPDB_WITH_EXCEPTION_SAVE_COROUTINES > 0
#ifdef LINUX
static int            eo07_mainThreadLocalErrno = 0;
static int            eo07_mainThreadLocalHErrno = 0;
static int            eo07_mainThreadLibcSpecificTlsKeys[RTE_MAXIMUM_LIBC_SPECIFIC_KEYS];
#endif

static void        ***eo07_mainThreadTlsKeys = (void ***)0;
#endif

static int            eo07_EverInitialized = 0;

/*===========================================================================*
 *  LOCAL VARIABLES                                                          *
 *===========================================================================*/

static char *EO07_ERR_NO_MEM           = { ERRMSG_ALLOC_MEMORY };
static char *EO07_ERR_STACKSIZE        = { ERRMSG_STACKSIZE };
static char *EO07_ERR_RESOURCE_LIMIT   = { ERRMSG_RESOURCE_LIMIT };
static char *EO07_ERR_PROCADDRESS_NULL = { ERRMSG_PROCADDRESS_NULL };
static char *EO07_ERR_UNKNOWN_THREADID = { ERRMSG_UNKNOWN_THREADID };
static char *EO07_ERR_INVALID_SIGNAL   = { ERRMSG_INVALID_SIGNAL };
static char *EO07_ERR_DEADLOCK         = { ERRMSG_DEADLOCK };
static char *EO07_ERR_INVALID_KEY      = { ERRMSG_INVALID_TLSKEY };
static char *EO07_ERR_BUSY_KEY         = { ERRMSG_TLSKEY_BUSY };
static char *EO07_ERR_NOTIMPLEMENTED   = { ERRMSG_NOT_YET_IMPLEMENTED };

#ifndef SAPDB_WITH_COROUTINES
static tsp00_Int4 self_key_eo07 = 0;
#endif

#ifdef CTHREADS
 static mutex_t onceMutex_eo07 = {0};
#endif /* CTHREADS */

#ifdef PTHREADS
/*
 * This is used for synchronization of access to nextThreadId_eo77
 */
#ifndef HAS_GETSEQUENCE_NP
 static teo07_ThreadId nextThreadId_eo07 = (teo07_ThreadId)0;
# ifdef PTHREAD_MUTEX_INITIALIZER
   static pthread_mutex_t threadIdMutex_eo07 = PTHREAD_MUTEX_INITIALIZER;
# else
   static pthread_mutex_t threadIdMutex_eo07;
# endif /* PTHREAD_MUTEX_INITIALIZER */
#endif /* HAS_GETSEQUENCE_NP */

#ifdef PTHREAD_MUTEX_INITIALIZER
 static pthread_mutex_t onceMutex_eo07 = PTHREAD_MUTEX_INITIALIZER;
#else
 static pthread_mutex_t onceMutex_eo07;
#endif /* PTHREAD_MUTEX_INITIALIZER */

#endif /* PTHREADS */

/*===========================================================================*
 *  LOCAL FUNCTIONS (PROTOTYPES)                                             *
 *===========================================================================*/

static void eo07_CreateThread( tsp00_Int4 StackSize, 
                               void *StackStartAddress,
                               void *(*proc)(void *), 
                               void *arg, 
                               tsp00_Int4 flags, 
                               teo07_Thread *thread, 
                               tsp00_ErrTextc errtext, 
                               teo07_ThreadErr *ok);

#ifdef _WIN32
static teo07_ThreadErr eo07_GetError(tsp00_ErrTextc errtext);
static DWORD eo07_Win32threadEntry( void *arg );
#endif

#ifdef CTHREADS
static void *eo07_CthreadEntry( void *arg );
#endif /* CTHREADS */

#ifdef PTHREADS
static void *eo07_PthreadEntry( void *arg );
# ifdef USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID
static teo07_ThreadId eo07_GetSequenceNumber();
# endif /* USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID */
#endif /* PTHREADS */

#ifdef LINUX
/* PTS 1105262/1105263 */
static void * eo07_NoheapMalloc(size_t size);
static void   eo07_NoheapFree(void *ptr, size_t size);
static void * eo07_DummyPthreadEntry( void *arg );
/* PTS 1106738 */
/* With RedHat6.2 the mmap() prototype was fixed. This prevented the kernel from */
/* starting. The message given was 'invalid argument'. This Message was given due */
/* to the last argument of mmap() which was defined as 'off_t' and later 'off64_t' */
/* So on RedHat6.2 the last argument was expected to be a 64bit value, therefore */
/* the extra 32bits are collected from the stack. Such an offset is not PAGESIZE */
/* aligned, so it leads to 'invalid argument'... */
#undef __USE_FILE_OFFSET64
#define __USE_LARGEFILE64
#include <sys/mman.h>
#endif /* LINUX */


/*===========================================================================*
 *  GLOBAL FUNCTIONS (CODE)                                                  *
 *===========================================================================*/

void sqlinit_multithreading( teo07_Thread *thread,
                             tsp00_ErrTextc errtext, 
                              teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"sqlinit_multithreading"

    static teo07_ThreadObj *pThreadObj = NULL;

    if ( thread == NULL )
    {
        *ok = THR_NOT_OK_EO07;
        if ( errtext != NULL )
        {
            memcpy(errtext, EO07_ERR_UNKNOWN_THREADID, strlen(EO07_ERR_UNKNOWN_THREADID)+1);
        }
        return;
    }

    if ( pThreadObj == NULL )
    {
        if ( ALLOC_MEM_EO57( (void **)&pThreadObj, sizeof(teo07_ThreadObj) ) != NO_ERROR_EO001 )
        {
            *ok = THR_NOT_OK_EO07;
            if (errtext != NULL)
            {
                strcpy(errtext, EO07_ERR_NO_MEM);
            }
            return;
        }

        *thread = (teo07_Thread)pThreadObj;
        *ok = THR_OK_EO07;
    }
    else
    {
        *thread = (teo07_Thread)pThreadObj;
        *ok = THR_OK_EO07;
        return;
    }

#ifdef _WIN32
    DuplicateHandle(GetCurrentProcess(),
                    GetCurrentThread(),
                    GetCurrentProcess(),
                    & pThreadObj->thread,
                    DUPLICATE_SAME_ACCESS,
                    FALSE,
                    0);
    pThreadObj->thread_id = GetCurrentThreadId();
#endif /* _WIN32 */

#ifdef CTHREADS
    pThreadObj->thread    = pThreadObj->thread_id = thr_self();
    (void)mutex_init(&onceMutex_eo07, USYNC_THREAD, NULL);
#endif /* CTHREADS */

#ifdef PTHREADS
# ifndef HAS_GETSEQUENCE_NP
    nextThreadId_eo07 = (teo07_ThreadId)1;
# endif /* HAS_GETSEQUENCE_NP */

# ifndef PTHREAD_MUTEX_INITIALIZER
#  ifndef HAS_GETSEQUENCE_NP
    (void)PTHREAD_MUTEX_INIT(&threadIdMutex_eo07);
#  endif /* HAS_GETSEQUENCE_NP */
    (void)PTHREAD_MUTEX_INIT(&onceMutex_eo07);
# endif

    pThreadObj->thread    = pthread_self();
# ifdef USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID
    pThreadObj->thread_id = eo07_GetSequenceNumber();
# endif /* USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID */

# ifdef USE_GETSEQUENCE_NP_FOR_THREAD_ID
    pThreadObj->thread_id = (teo07_ThreadId)pthread_getsequence_np(pThreadObj->thread);
# endif

# ifdef USE_CLONE_PID_FOR_THREAD_ID
    /* PTS 1105217 */
    pThreadObj->thread_id = (teo07_ThreadId)getpid();
#endif

#endif /* PTHREADS */

    sqlcreatesem(&(pThreadObj->suspend_sem), 0, errtext, ok);
    if ( *ok != THR_OK_EO07 )
    {
        FREE_MEM_EO57(pThreadObj);
        pThreadObj = NULL;
        *thread = NULL;
        return;
    }
    pThreadObj->start_suspended = 0;
    pThreadObj->proc            = NULL;
    pThreadObj->arg             = 0;

#ifndef SAPDB_WITH_COROUTINES
    sqlcreatetls(&self_key_eo07, errtext, ok);
#else
    eo07_RegisterSelf(pThreadObj, 
#  ifdef LINUX
    pThreadObj->pThreadLocalErrno = &eo07_mainThreadLocalErrno;
    pThreadObj->pThreadLocalHErrno = &eo07_mainThreadLocalHErrno;
    memcpy(pThreadObj->libcSpecificTlsKeys,
           eo07_mainThreadLibcSpecificTlsKeys,
           sizeof(int)*RTE_MAXIMUM_LIBC_SPECIFIC_KEYS);
#  endif
    pThreadObj->tlsKeys = eo07_mainThreadTlsKeys;
#endif

    if ( *ok == THR_OK_EO07 )
    {
        eo07_EverInitialized = 1;
#ifndef SAPDB_WITH_COROUTINES
        sqlsettls(self_key_eo07, pThreadObj, errtext, ok);
#endif
    }
    else
    {
        FREE_MEM_EO57(pThreadObj);
        pThreadObj = NULL;
        *thread = NULL;
    }

} /* sqlinit_multithreading */

/*---------------------------------------------------------------------------*/


void sqlbeginthread(tsp00_Int4 StackSize, 
                    void *(*proc)(void *), 
                    void *arg, 
                    tsp00_Int4 flags, 
                    teo07_Thread *thread, 
                    tsp00_ErrTextc errtext, 
                    teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"sqlbeginthread"
    DBGPAS;

    eo07_CreateThread(StackSize, (void *)0, proc, arg, flags, thread, errtext, ok);
} /* sqlbeginthread */

/*---------------------------------------------------------------------------*/

tsp00_Int4 sqlget_thread_min_stack()
{
#if defined(WIN32)
    return 0;
#else
    static tsp00_Int4 min_thread_stack_size = -1;
    if ( -1 == min_thread_stack_size )
    {
#if defined(_SC_THREAD_STACK_MIN)
        min_thread_stack_size = (SAPDB_Int4)sysconf(_SC_THREAD_STACK_MIN);
#endif
        if ( min_thread_stack_size <= 0 )
        {
            min_thread_stack_size = 0;
#ifdef PTHREADS
#ifdef PTHREAD_STACK_MIN
            min_thread_stack_size = ((tsp00_Int4)PTHREAD_STACK_MIN);
#else
#endif
#endif

#ifdef CTHREADS
#ifdef CTHREAD_STACK_MIN
            min_thread_stack_size = ((tsp00_Int4)thr_min_stack());
#else
#endif
#endif
        }
    }
    return min_thread_stack_size;
#endif
}

/*---------------------------------------------------------------------------*/

void sqlbeginthread_on_own_stack(tsp00_Int4 StackSize, 
                    void *StackStartAddress,
                    void *(*proc)(void *), 
                    void *arg, 
                    tsp00_Int4 flags, 
                    teo07_Thread *thread, 
                    tsp00_ErrTextc errtext, 
                    teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"sqlbeginthread_on_own_stack"

    DBGPAS;

    eo07_CreateThread(StackSize, StackStartAddress, proc, arg, flags, thread, errtext, ok);
} /* sqlbeginthread_on_own_stack */

/*---------------------------------------------------------------------------*/

void sqlendthread(tsp00_Int4 returncode)
{
#undef MF__
#define MF__ MOD__"sqlendthread"

#ifdef _WIN32
  ExitThread(returncode);
#endif

#ifdef CTHREADS
  thr_exit((void *)returncode);
#endif

#ifdef PTHREADS
  pthread_exit((void *)returncode);
#endif
} /* sqlendthread */

/*---------------------------------------------------------------------------*/

void sqlkillthread(teo07_Thread thread, 
                   tsp00_ErrTextc errtext, 
                   teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"sqlkillthread"

    sqlsigthread(thread, SIGTERM, errtext, ok);
} /* sqlkillthread */

/*---------------------------------------------------------------------------*/

void sqlsigthread(teo07_Thread thread, 
                  tsp00_Int4 sig, 
                  tsp00_ErrTextc errtext, 
                  teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"sqlsigthread"

#ifndef _WIN32
    teo001_SystemRc rc;
#endif /* !_WIN32 */

    teo07_ThreadObj *pThreadObj = (teo07_ThreadObj *)thread;

    if ( pThreadObj == NULL )
    {
        *ok = THR_NOT_OK_EO07;
    }

#ifdef _WIN32
    if(!TerminateThread(pThreadObj->thread, sig)) 
    { 
        *ok = eo07_GetError(errtext);
    }
    else
    {
        *ok = THR_OK_EO07;
    }
#else /* UNIX */

#ifdef CTHREADS
    rc = thr_kill(pThreadObj->thread, sig);
#endif /* CTHREADS */

#ifdef PTHREADS
#if PTHREAD_DRAFT > 6
    PTHREAD_RC(rc, pthread_kill(pThreadObj->thread, sig));
#else /* PTHREAD_DRAFT < 7 */
    rc = 0;
    *ok = THR_NOT_IMPLEMENTED_EO07;
    if(errtext != NULL)
    {
        strcpy(errtext, EO07_ERR_NOTIMPLEMENTED);
    }
#endif /* PTHREAD_DRAFT */
#endif /* PTHREADS */

    if (rc != 0) 
    {
        *ok = THR_NOT_OK_EO07;
        if(errtext != NULL)
        {
            switch(rc) 
            {
            case (ESRCH):
                strcpy(errtext, EO07_ERR_UNKNOWN_THREADID);
                break;
                
            case (EINVAL):
                strcpy(errtext, EO07_ERR_INVALID_SIGNAL);
                break;

            default:
                strcpy(errtext, EO07_ERR_NOTIMPLEMENTED);
                break;
            }
        }
    }
    else 
    {
        *ok = THR_OK_EO07;
    }

#endif /* UNIX */
} /* sqlsigthread */

/*---------------------------------------------------------------------------*/

void sqlcancelthread(teo07_Thread thread,
                     tsp00_Int4 exitcode,
                     tsp00_ErrTextc errtext, 
                     teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"sqlcancelthread"

#ifdef CTHREADS

    /* CThreads do not support thr_cancel() */
    *ok = THR_OK_EO07;

#else /* !CTHREADS */

#ifdef PTHREADS
    teo001_SystemRc rc;
#endif /* PTHREADS */

    teo07_ThreadObj *pThreadObj = (teo07_ThreadObj *)thread;

    *ok = THR_OK_EO07;

    if ( (pThreadObj == NULL) || (pThreadObj->thread_id == 0) )
    {
        return;
    }

#ifdef _WIN32
    if ( (pThreadObj->thread != ((HANDLE)0))
      && (pThreadObj->thread != INVALID_HANDLE_VALUE) )
    {
        if ( !TerminateThread ( pThreadObj->thread, exitcode ) )
        {
            *ok = eo07_GetError(errtext);
        }
    }
#endif

#ifdef PTHREADS
    PTHREAD_RC(rc,pthread_cancel(pThreadObj->thread));
    if ( rc != 0
      && rc != ESRCH 
      && rc != EINVAL )
    {
        *ok = THR_NOT_OK_EO07;
        if ( errtext != NULL )
        {
            switch(rc)
            {
            case (EDEADLK):
                strcpy(errtext, EO07_ERR_DEADLOCK);
                break;

            default:
                strcpy(errtext, EO07_ERR_NOTIMPLEMENTED);
                break;
            }
        }
    }
#endif /* PTHREADS */

#endif /* !CTHREADS */
}

/*---------------------------------------------------------------------------*/

void sqlforgetthread (teo07_Thread thread)
{
#undef MF__
#define MF__ MOD__"sqlforgetthread"

    teo07_ThreadObj *pThreadObj = (teo07_ThreadObj *)thread;

    if ( (pThreadObj == NULL) || (pThreadObj->thread_id == 0) )
    {
        return;
    }

#ifdef CTHREADS
    /* no detach possible */
#endif

#ifdef PTHREADS
#if PTHREAD_DRAFT < 7
    (void)pthread_detach (&pThreadObj->thread);
#else
    (void)pthread_detach (pThreadObj->thread);
#endif
#endif /* PTHREADS */

#ifdef _WIN32
    if ( (pThreadObj->thread != ((HANDLE)0))
      && (pThreadObj->thread != INVALID_HANDLE_VALUE) )
    {
        CloseHandle ( pThreadObj->thread );
    }
#endif

    sqldestroysem( pThreadObj->suspend_sem );

    FREE_MEM_EO57(pThreadObj);
} /* sqlforgetthread */

/*---------------------------------------------------------------------------*/

void sqlyieldthread()
{
#undef MF__
#define MF__ MOD__"sqlyieldthread"

    YIELD_CALL;
}

/*---------------------------------------------------------------------------*/

teo07_Thread sqlgetthread()
{
#undef MF__
#define MF__ MOD__"sqlgetthread"
    return (teo07_Thread)GET_SELF;
}

/*---------------------------------------------------------------------------*/

#if SAPDB_WITH_EXCEPTION_SAVE_COROUTINES > 0
#ifdef LINUX
int *sqlErrnoLocation()
{
    if ( !eo07_EverInitialized ) return &eo07_mainThreadLocalErrno;

    return GET_SELF->pThreadLocalErrno;
}

/*---------------------------------------------------------------------------*/

int *sqlHErrnoLocation()
{
    if ( !eo07_EverInitialized ) return &eo07_mainThreadLocalHErrno;

    return GET_SELF->pThreadLocalHErrno;
}

/*---------------------------------------------------------------------------*/

extern int sqlLibc_internal_tsd_set(int key, const void *arg)
{
    if ( key < 0 || key > RTE_MAXIMUM_LIBC_SPECIFIC_KEYS )
    {
        return EINVAL;
    }

    if ( !eo07_EverInitialized )
    {
        eo07_mainThreadLibcSpecificTlsKeys[key] = (void *)arg;
    }
    else
    {
        GET_SELF->libcSpecificTlsKeys[key] = (void *)arg;
    }
}

/*---------------------------------------------------------------------------*/

extern void *sqlLibc_internal_tsd_get(int key)
{
    if ( key < 0 || key > RTE_MAXIMUM_LIBC_SPECIFIC_KEYS )
    {
        return (void *)0;
    }

    if ( !eo07_EverInitialized )
    {
        return eo07_mainThreadLibcSpecificTlsKeys[key];
    }
    else
    {
        return GET_SELF->libcSpecificTlsKeys[key];
    }
}
#endif /* LINUX */

/*---------------------------------------------------------------------------*/

void ****sqlGetTlsArray()
{
    if ( !eo07_EverInitialized ) return &eo07_mainThreadTlsKeys;

    return &(GET_SELF->tlsKeys);
}
#endif /* SAPDB_WITH_COROUTINES */

/*---------------------------------------------------------------------------*/

teo07_ThreadErr sqlgetthreadpriority(  teo07_Thread thread, 
                                       tsp00_Int4 * pPriority )
{
    int rc = 0;
    teo07_ThreadObj *pThreadObj = (teo07_ThreadObj *)thread;

#if defined(_WIN32)
    int priority;
    priority = GetThreadPriority(pThreadObj->thread);
    if ( THREAD_PRIORITY_ERROR_RETURN == priority )
    {
        rc = 1;
    }
    else
    {
        *pPriority = priority;
    }
#endif

#ifdef CTHREADS
    int schedpriority;

    rc = thr_getprio(pThreadObj->thread, &schedpriority);
    if ( rc == 0 )
    {
        *pPriority = schedpriority;
    }
#endif

#ifdef PTHREADS
    int schedpolicy;
    struct sched_param schedparam;

    PTHREAD_RC(rc, pthread_getschedparam(pThreadObj->thread, &schedpolicy, &schedparam));
    if ( rc == 0 )
    {
        *pPriority = schedparam.sched_priority;
    }
#endif

    return ( rc == 0 ? THR_OK_EO07 : THR_NOT_OK_EO07 );
}

/*---------------------------------------------------------------------------*/

teo07_ThreadErr sqlgetmythreadpriority( tsp00_Int4 * pPriority )
{
    int rc = 0;

#if defined(_WIN32)
    HANDLE myThread = GetCurrentThread();
    int priority;

    priority = GetThreadPriority(myThread);
    if ( THREAD_PRIORITY_ERROR_RETURN == priority )
    {
        rc = 1;
    }
    else
    {
        *pPriority = priority;
    }
#endif

#ifdef CTHREADS
    int schedpriority;
    thread_t myThread = thr_self();

    rc = thr_getprio(myThread, &schedpriority);
    if ( rc == 0 )
    {
        *pPriority = schedpriority;
    }
#endif

#ifdef PTHREADS
    int schedpolicy;
    struct sched_param schedparam;
    pthread_t myThread = pthread_self();

    PTHREAD_RC(rc, pthread_getschedparam(myThread, &schedpolicy, &schedparam));
    if ( rc == 0 )
    {
        *pPriority = schedparam.sched_priority;
    }
#endif

    return ( rc == 0 ? THR_OK_EO07 : THR_NOT_OK_EO07 );
}

/*---------------------------------------------------------------------------*/

teo07_ThreadErr sqlsetthreadpriority( teo07_Thread thread, 
                                      tsp00_Int4 priority )
{
    int rc = 0;
    teo07_ThreadObj *pThreadObj = (teo07_ThreadObj *)thread;

#if defined(_WIN32)
    rc = ( 0 != SetThreadPriority(pThreadObj->thread, (int)priority) );
#endif

#ifdef CTHREADS
    rc = thr_setprio(pThreadObj->thread, (int)priority );
#endif

#ifdef PTHREADS
    int schedpolicy;
    struct sched_param schedparam;
    pthread_t myThread = pthread_self();

    PTHREAD_RC(rc, pthread_getschedparam(pThreadObj->thread, &schedpolicy, &schedparam));
    if ( rc == 0 )
    {
        schedparam.sched_priority = priority;
        PTHREAD_RC(rc, pthread_setschedparam(pThreadObj->thread, schedpolicy, &schedparam));
    }
#endif

    return ( rc == 0 ? THR_OK_EO07 : THR_NOT_OK_EO07 );
}

/*---------------------------------------------------------------------------*/

teo07_ThreadErr sqlsetmythreadpriority( tsp00_Int4 priority)
{
    int rc = 0;

#if defined(_WIN32)
    HANDLE myThread = GetCurrentThread();

    rc = ( 0 != SetThreadPriority(myThread, (int)priority) );
#endif

#ifdef CTHREADS
    thread_t myThread = thr_self();

    rc = thr_setprio(myThread, (int)priority );
#endif

#ifdef PTHREADS
    int schedpolicy;
    struct sched_param schedparam;
    pthread_t myThread = pthread_self();

    PTHREAD_RC(rc, pthread_getschedparam(myThread, &schedpolicy, &schedparam));
    if ( rc == 0 )
    {
        schedparam.sched_priority = priority;
        PTHREAD_RC(rc, pthread_setschedparam(myThread, schedpolicy, &schedparam));
    }
#endif

    return ( rc == 0 ? THR_OK_EO07 : THR_NOT_OK_EO07 );
}

/*---------------------------------------------------------------------------*/

teo07_ThreadId sqlgetthreadid()
{
#undef MF__
#define MF__ MOD__"sqlgetthreadid"

#ifdef _WIN32
    return (teo07_ThreadId)GetCurrentThreadId();
#endif /* _WIN32 */

#ifdef CTHREADS
    return (teo07_ThreadId)thr_self();
#endif /* CTHREADS */

#ifdef PTHREADS
# ifdef USE_CLONE_PID_FOR_THREAD_ID
    return (teo07_ThreadId)getpid();
# else
    teo07_ThreadObj *pThreadObj;

    pThreadObj = GET_SELF;
    if ( pThreadObj == NULL )
    {
        union {
            pthread_t      pt;
            teo07_ThreadId id;
        } u;
    
        u.pt = pthread_self();
        return u.id;
    }
    else
    {
      return pThreadObj->thread_id;
    }
# endif /* USE_CLONE_PID_FOR_THREAD_ID */
#endif /* PTHREADS */
}

/*---------------------------------------------------------------------------*/

teo07_ThreadId sqlthreadid(teo07_Thread thread)
{
#undef MF__
#define MF__ MOD__"sqlthreadid"

    return ((teo07_ThreadObj *)thread)->thread_id;
}

/*---------------------------------------------------------------------------*/

void sqlsuspendthread(teo07_Thread thread, 
                      tsp00_ErrTextc errtext, 
                      teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqlsuspendthread"

  if ( (void *)GET_SELF != (void *)thread )
  {
      *ok = THR_NOT_IMPLEMENTED_EO07;
      if(errtext != NULL)
          strcpy(errtext, EO07_ERR_NOTIMPLEMENTED);
  }
  else
  {
      DBG1(( MF__,"suspending thread %d\n", sqlthreadid(thread) ));
      sqlwaitsem( ((teo07_ThreadObj *)thread)->suspend_sem );
      *ok = THR_OK_EO07;
  }
}

/*---------------------------------------------------------------------------*/

tsp00_Bool sqlIsThreadSuspended(teo07_Thread thread)
{
#undef MF__
#define MF__ MOD__"sqlsuspendthread"
    tsp00_Bool isSuspended = false;
#ifdef WIN32
    /* I found no API to find out if a thread is already suspended by waiting on semaphore */
#else
    teo07_SemObj *semObj = (teo07_SemObj*) ( ((teo07_ThreadObj *)thread)->suspend_sem );

#ifdef CTHREADS
    mutex_lock (&semObj->mutex);
    isSuspended = (semObj->waiter > 0);
    mutex_unlock (&semObj->mutex);
#endif /* CTHREADS */

#ifdef PTHREADS
    pthread_mutex_lock (&semObj->mutex);
    isSuspended = (semObj->waiter > 0);
    pthread_mutex_unlock (&semObj->mutex);
#endif /* PTHREADS */

#endif
    return isSuspended;
}

/*---------------------------------------------------------------------------*/

void sqlresumethread(teo07_Thread thread, 
                     tsp00_ErrTextc errtext, 
                     teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqlresumethread"
  DBG1(( MF__,"resuming thread %d\n", sqlthreadid(thread) ));
  sqlsignalsem( ((teo07_ThreadObj *)thread)->suspend_sem );
  *ok = THR_OK_EO07;
}

/*---------------------------------------------------------------------------*/

void sqljointhread(teo07_Thread thread, 
                   tsp00_Int4 *status,
                   tsp00_ErrTextc errtext,
                   teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqljointhread"

    teo07_ThreadObj *pThreadObj;
#ifndef _WIN32
    teo001_SystemRc rc;
#endif

#ifdef CTHREADS
    void *exitStatus;
#endif
#ifdef PTHREADS
    void *exitStatus;
#endif

    pThreadObj = (teo07_ThreadObj *)thread;

    if ( pThreadObj == NULL )
    {
      *ok = THR_NOT_OK_EO07;
      strcpy(errtext, EO07_ERR_UNKNOWN_THREADID);
      return;
    }

    MSGD(( INFO_THREAD_JOIN, pThreadObj->thread_id ));

#ifdef _WIN32
    for(;;) 
    {
		/* PTS 1109286 MaO	*/
		WaitForSingleObject(pThreadObj->thread, INFINITE);

        if(!GetExitCodeThread(pThreadObj->thread, status)) 
        {
            *ok = eo07_GetError(errtext);
            break;
        }
        if (*status == STILL_ACTIVE)
        {
            *ok = eo07_GetError(errtext);
            break;
        }
        else 
        {
            *ok = THR_OK_EO07;
            break;
        }
    }
#else /* UNIX */

#ifdef CTHREADS
    rc=thr_join(pThreadObj->thread, NULL, &exitStatus);
    *status = (tsp00_Int4)(((char *)exitStatus)-((char *)0));
#endif

#ifdef PTHREADS

    /**/
    /* LINUX problem: Deadly to put it here.... This leads to dying */
    /* manager threads due to threads that die in _exit()... */
    /* It seems the manager thread does something with the stack of the */
    /* thread that is deadly... */
    /* Instead a sleep(1) was introduced to reduce CPU consumption */
    /**/
    /* rc=pthread_join(pThreadObj->thread, &exitStatus); */

#ifdef LINUX
    if ( pThreadObj->StackSize != 0 ) /* PTS 1106187 */
    {
    /*
     * Some words.... PTS 1105262/1105263
     * There was a LINUX specific crash without this YIELD_CALL loop. The linuxthread 
     * pthread manager does not synchronize the pthread_join() call with the actions
     * of freeing internal references to the joined thread. Since these references are
     * situated on the pthread stack, freeing this stack results in program crashes depending
     * on the scheduling. Forcing the join to wait until the thread actually was terminating
     * solves part of this problem. Creating another dummy thread solves the synchronization.
     */
    /* PTS 1105678 poll after join not before join... and use kill instead of pthread_kill */
      while ( kill( pThreadObj->thread_id, 0) == 0 ) 
      {
        sleep(1); /* YIELD_CALL; */ /* let him finsh */
      }
    }
#endif

    rc=pthread_join(pThreadObj->thread, &exitStatus);

    *status = (tsp00_Int4)(((char *)exitStatus)-((char *)0));
#endif

    if (rc != 0) 
    {
        *ok = THR_NOT_OK_EO07;
        switch(rc) 
        {
        case (ESRCH):
        case (EINVAL):
            if (rc == EINVAL) 
                *ok = THR_NOT_OK_EO07;
            else
                *ok = THR_NO_DATA_FOUND_EO07;
            if(errtext != NULL)
                strcpy(errtext, EO07_ERR_UNKNOWN_THREADID);
            break;
            
        case (EDEADLK):
        default:
            *ok = THR_NOT_OK_EO07;
            if(errtext != NULL)
                strcpy(errtext, EO07_ERR_DEADLOCK);
            break;
        }
    }
    else 
    {
        *ok = THR_OK_EO07;
    }

#ifdef LINUX
    if ( pThreadObj->StackSize != 0 ) /* PTS 1106187 */
    {
      /* PTS 1105262/1105263/1106187 */
      pthread_t DummyThread;
      pthread_attr_t DummyAttr;

      if ( pthread_attr_init(&DummyAttr) != 0 )
      {
        MSGD(( ERR_CREATING_THREAD, "Dummy", errno ));
        DO_ABORT();
      }

      if ( pthread_attr_setdetachstate(&DummyAttr, PTHREAD_CREATE_DETACHED) != 0 )
      {
        MSGD(( ERR_CREATING_THREAD, "Dummy", errno ));
        DO_ABORT();
      }

      if ( pthread_create( &DummyThread, 
                            NULL, 
                            eo07_DummyPthreadEntry, 
                            (void *)&DummyThread ) != 0 )
      {
        MSGD(( ERR_CREATING_THREAD, "Dummy", errno ));
        DO_ABORT();
      }

      eo07_NoheapFree(pThreadObj->StackAddr, pThreadObj->StackSize);
    }
#endif /* LINUX */


#endif /* UNIX */

#ifdef _WIN32
    if ( (pThreadObj->thread != ((HANDLE)0))
      && (pThreadObj->thread != INVALID_HANDLE_VALUE) )
    {
        CloseHandle ( pThreadObj->thread );
    }
#endif

    sqldestroysem( pThreadObj->suspend_sem );

    memset(pThreadObj, 0, sizeof( teo07_ThreadObj ));
    FREE_MEM_EO57(pThreadObj);

} /* sqljointhread */

/*---------------------------------------------------------------------------*/

int sqlthreadalive( teo07_Thread thread, 
                    tsp00_ErrTextc errtext,
                    teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqlthreadalive"

    teo07_ThreadObj *pThreadObj;
    teo001_SystemRc rc;

    pThreadObj = (teo07_ThreadObj *)thread;
    if ( errtext )
    {
        errtext[0] = 0;
    }
    *ok = THR_OK_EO07;

#ifdef _WIN32
    rc = ((WaitForSingleObject(pThreadObj->thread, 0) == WAIT_TIMEOUT) ? 0 : 1);
#endif /* _WIN32 */

#ifdef CTHREADS
    rc = thr_kill( pThreadObj->thread, 0);
#endif /* CTHREADS */

#ifdef PTHREADS
#if PTHREAD_DRAFT > 6
    PTHREAD_RC(rc,pthread_kill( pThreadObj->thread, 0));
#else
    rc = 0; /* always alive */
#endif
#endif /* PTHREADS */

    return ( rc == 0 );
}

/*---------------------------------------------------------------------------*/

void sqlonce (tsp00_Int4* initialized, 
              teo07_ThreadCallback callback, 
              void* arg)
{
#undef MF__
#define MF__ MOD__"sqlonce"

#ifdef _WIN32
    static LONG lock = 1;

    while(InterlockedExchange(&lock,0) == 0)  
    {
        Sleep(50);
    }
    if (*initialized == 1) 
    {
        lock = 1; /* release the lock just acquired */
        return;
    }

    callback(arg);
    *initialized = 1;
    lock = 1; /* release the lock just acquired */
#endif /* _WIN32 */

#ifdef CTHREADS
    mutex_lock( &onceMutex_eo07 );
    if (*initialized == 1) 
    {
        mutex_unlock( &onceMutex_eo07 );
        return;
    }

    callback(arg);
    *initialized = 1;
    mutex_unlock( &onceMutex_eo07 );
#endif /* CTHREADS */

#ifdef PTHREADS
    pthread_mutex_lock( &onceMutex_eo07 );
    if (*initialized == 1) 
    {
        pthread_mutex_unlock( &onceMutex_eo07 );
        return;
    }

    callback(arg);
    *initialized = 1;
    pthread_mutex_unlock( &onceMutex_eo07 );
#endif /* PTHREADS */
} /* sqlonce */

/*---------------------------------------------------------------------------*/

/*
 Mutex
 */

void sqlcreatemutex(teo07_Mutex *ppMutex)
{
#undef MF__
#define MF__ MOD__"sqlcreatemutex"
    teo07_Mutex pMutex;

    if( ppMutex == NULL )
    {
        MSGD(( ERR_INVALID_PARAMETER, "createMutex" ));
        DO_ABORT();
    }
    
    if ( ALLOC_MEM_EO57( (void **)ppMutex, sizeof(struct _teo07_Mutex) ) != NO_ERROR_EO001 )
    {
      MSGD(( ERR_CANT_ALLOC_HEAP_MEMORY, errno ));
      DO_ABORT();
    }
    pMutex = *ppMutex;
    memset(pMutex, 0, sizeof(struct _teo07_Mutex));
    
#ifdef _WIN32
    InitializeCriticalSection(&(pMutex->mutex));  
#endif
    
#ifdef CTHREADS
    mutex_init(&(pMutex->mutex), USYNC_THREAD, NULL);  
#endif
    
#ifdef PTHREADS
    PTHREAD_MUTEX_INIT(&(pMutex->mutex));
#endif /* PTHREADS */
} /* sqlcreatemutex */

/*---------------------------------------------------------------------------*/

void sqldestroymutex(teo07_Mutex *ppMutex)
{  
#undef MF__
#define MF__ MOD__"sqldestroymutex"

    teo07_Mutex pMutex;

    if( ppMutex == NULL )
    {
        MSGD(( ERR_INVALID_PARAMETER, "destroyMutex" ));
        DO_ABORT();
    }

    pMutex = *ppMutex;

    if ( pMutex != NULL )
    {
#ifdef _WIN32
        DeleteCriticalSection(&(pMutex->mutex));
#endif
        
#ifdef CTHREADS
        mutex_destroy(&(pMutex->mutex));
#endif
        
#ifdef PTHREADS
        pthread_mutex_destroy(&(pMutex->mutex));  
#endif
        
        FREE_MEM_EO57(pMutex);
        *ppMutex = NULL;
    }
} /* sqldestroymutex */

/*---------------------------------------------------------------------------*/

void sqlbeginmutex(teo07_Mutex *ppMutex)
{
#undef MF__
#define MF__ MOD__"sqlbeginmutex"
    teo07_Mutex pMutex;
    
    if( ppMutex == NULL )
    {
        MSGD(( ERR_INVALID_PARAMETER, "beginMutex" ));
        DO_ABORT();
    }

    pMutex = *ppMutex;

    if ( pMutex == NULL ) 
    {
        MSGD(( ERR_INVALID_PARAMETER, "beginMutex Value" ));
        DO_ABORT();
    }
    
    /* this prevents the thread to deadlock itselves */
    if( pMutex->owning_thread != sqlgetthreadid())
    {
#ifdef TRACEX
        printf("beginmutex %#x %ld %ld, %ld\n", 
            pMutex, 
            sqlgetthreadid(),
            pMutex->owning_thread,
            pMutex->lock_count);
#endif

#ifdef _WIN32
        EnterCriticalSection(&(pMutex->mutex));
#endif
        
#ifdef CTHREADS
        mutex_lock(&(pMutex->mutex));
#endif
        
#ifdef PTHREADS
        pthread_mutex_lock(&(pMutex->mutex));
#endif
        pMutex->lock_count = 1;
        pMutex->owning_thread = sqlgetthreadid();
    }
    else 
    {
        pMutex->lock_count++;
    }
} /* sqlbeginmutex */

/*---------------------------------------------------------------------------*/

int sqlissetmutex(teo07_Mutex *ppMutex)
{
    return ( (*ppMutex)->lock_count != 0 );
} /* sqlissetmutex */

/*---------------------------------------------------------------------------*/

tsp00_Int4 sqltrybeginmutex(teo07_Mutex *ppMutex)
{
#undef MF__
#define MF__ MOD__"sqltrybeginmutex"
    teo001_SystemRc rc=0;
    teo07_Mutex pMutex;
    
    if( ppMutex == NULL ) 
    {
        MSGD(( ERR_INVALID_PARAMETER, "trybeginMutex" ));
        DO_ABORT();
    }

    pMutex = *ppMutex;

    if ( pMutex == NULL )
    {
        MSGD(( ERR_INVALID_PARAMETER, "trybeginMutex Value" ));
        DO_ABORT();
    }

    /* this prevents the thread to deadlock itselves */
    /* Windows NT doesn't deadlock sections but it works also */
    if( pMutex->owning_thread != sqlgetthreadid())
    {
#ifdef TRACEX
        printf("beginmutex %#x %ld %ld, %ld\n", 
            pMutex, 
            sqlgetthreadid(),
            pMutex->owning_thread,
            pMutex->lock_count);
#endif

        if( (pMutex->owning_thread) == 0 )
        {
#ifdef _WIN32
        /* TryEnterCriticalSection not yet implemented on Windows 95 */
        /* so uses this to emulate TryEnterCriticalSection */
            EnterCriticalSection(&(pMutex->mutex));
            if (pMutex->owning_thread != 0) 
            {
                LeaveCriticalSection(&(pMutex->mutex));
                rc = 1;
            }
#endif /* _WIN32 */

#ifdef CTHREADS
            rc = (mutex_trylock(&(pMutex->mutex)) != 0);
#endif
        
#ifdef PTHREADS
            rc = (pthread_mutex_trylock(&(pMutex->mutex)) != 0);
#endif
        }

        if ( rc == 0 )
        {
            pMutex->lock_count = 1;
            pMutex->owning_thread = sqlgetthreadid();
        }
    }
    else
    {
        pMutex->lock_count++;
    }
    return(rc);
} /* sqltrybeginmutex */

/*---------------------------------------------------------------------------*/

void sqlendmutex(teo07_Mutex *ppMutex)
{
#undef MF__
#define MF__ MOD__"sqlendmutex"

    teo07_Mutex pMutex;
    
    if ( ppMutex == NULL )
    {
        MSGD(( ERR_INVALID_PARAMETER, "endMutex" ));
        DO_ABORT();
    }
    
    pMutex = *ppMutex;

    if( pMutex == NULL ) 
    {
        MSGD(( ERR_INVALID_PARAMETER, "endMutex Value" ));
        DO_ABORT();
    }
    
    if( pMutex->owning_thread == sqlgetthreadid()
     && pMutex->lock_count > 1 ) 
    {
        pMutex->lock_count--;
    }
    else
    {
#ifdef TRACEX
        printf("end  mutex %#x %ld %ld, %ld\n", 
                pMutex, sqlgetthreadid(), pMutex->owning_thread, pMutex->lock_count);
#endif
        pMutex->lock_count = 0;
        pMutex->owning_thread = 0;
#ifdef _WIN32
        LeaveCriticalSection(&pMutex->mutex);
#endif
            
#ifdef CTHREADS
        mutex_unlock(&pMutex->mutex);
#endif
            
#ifdef PTHREADS
        pthread_mutex_unlock(&pMutex->mutex);
#endif
    }
} /* sqlendmutex */

/*---------------------------------------------------------------------------*/

/*
 Thread Local Storage 
 */

void sqlcreatetls(tsp00_Int4 *phKey, 
                  tsp00_ErrTextc errtext, 
                  teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqlcreatetls"

#ifdef _WIN32
  if ((*phKey = TlsAlloc()) == 0xFFFFFFFF) 
  {
    *ok = eo07_GetError(errtext);
  }
  else
  {
    *ok = THR_OK_EO07;
  }
#endif /* _WIN32 */

#ifdef CTHREADS
  teo001_SystemRc rc;
  *phKey = 0; /* SUN returns 0 if no threadlib is linked with the Applikation */
  rc = thr_keycreate((thread_key_t*)phKey, NULL);
  if (rc == ENOMEM) 
  {
    *ok = THR_NOT_OK_EO07;
    if(errtext != NULL)
      strcpy(errtext, EO07_ERR_NO_MEM);
  }
  else
  {
    *ok = THR_OK_EO07;
  }
#ifdef SUN
  if (*phKey == 0) {
    *ok = THR_NOT_IMPLEMENTED_EO07;
    if(errtext != NULL)
      strcpy(errtext, EO07_ERR_NOTIMPLEMENTED);
  }
#endif /* SUN */
#endif /* CTHREADS */

#ifdef PTHREADS
  teo001_SystemRc rc;
  *phKey = 0; /* SUN returns 0 if no threadlib is linked with the Applikation */
#if PTHREAD_DRAFT < 7
  PTHREAD_RC(rc,pthread_keycreate((pthread_key_t*)phKey, NULL));
#else
  PTHREAD_RC(rc,pthread_key_create((pthread_key_t*)phKey, NULL));
#endif /* PTHREAD_DRAFT < 7 */
  if (rc == ENOMEM) 
  {
    *ok = THR_NOT_OK_EO07;
    if(errtext != NULL)
    {
      strcpy(errtext, EO07_ERR_NO_MEM);
    }
  }
  else
  {
    *ok = THR_OK_EO07;
  }
#endif /* PTHREADS */
} /* sqlcreatetls */

/*---------------------------------------------------------------------------*/

void sqldestroytls(tsp00_Int4 hKey, 
                   tsp00_ErrTextc errtext, 
                   teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqldestroytls"

#ifdef _WIN32
  if(TlsFree((DWORD)hKey) == 0) 
  {
    *ok = eo07_GetError(errtext);
  }
  else
  {
    *ok = THR_OK_EO07;    
  }
#else /* UNIX */

#ifdef CTHREADS
  teo001_SystemRc rc;
#ifdef SUN
  rc = 0;
  /* on sun doesn't supports thr_keydelete */
  *ok = THR_OK_EO07;
#else
  rc = thr_keydelete((thread_key_t)hKey);
#endif
#endif /* CTHREADS */

#ifdef PTHREADS
  teo001_SystemRc rc;
#if PTHREAD_DRAFT > 6
  PTHREAD_RC(rc,pthread_key_delete((pthread_key_t)hKey));
#else
  rc = 0;
#endif
#endif /* PTHREADS */

  if (rc != 0) 
  {
    *ok = THR_NOT_OK_EO07;    
    if(errtext != NULL)
    {
      switch(rc) 
      {
      case (EBUSY):
        strcpy(errtext, EO07_ERR_BUSY_KEY);
        break;

      case (EINVAL):
      default:
        strcpy(errtext, EO07_ERR_INVALID_KEY);
        break;
      }
    }
  }
  else
  {
    *ok = THR_OK_EO07;
  }
#endif /* UNIX */
} /* sqldestroytls */

/*---------------------------------------------------------------------------*/

void sqlsettls(tsp00_Int4 hKey, 
               void*  pObjPtr, 
               tsp00_ErrTextc errtext, 
               teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqlsettls"

#ifdef _WIN32
  if(TlsSetValue((DWORD)hKey, pObjPtr) == 0) 
  {
    *ok = eo07_GetError(errtext);
  }
  else
  {
    *ok = THR_OK_EO07;
  }
#else /* UNIX */

#ifdef CTHREADS
  teo001_SystemRc rc = thr_setspecific((thread_key_t)hKey, pObjPtr);
#endif /* CTHREADS */

#ifdef PTHREADS
  teo001_SystemRc rc;
  PTHREAD_RC(rc,pthread_setspecific((pthread_key_t)hKey, pObjPtr));
#endif /* PTHREADS */

  if (rc != 0) 
  {
    *ok = THR_NOT_OK_EO07;    
    if(errtext != NULL)
    {
      switch(rc) {
      case (ENOMEM):
        strcpy(errtext, EO07_ERR_NO_MEM);
        break;

      case (EINVAL):
      default:
        strcpy(errtext, EO07_ERR_INVALID_KEY);
        break;
      }
    }
  }
  else
  {
    *ok = THR_OK_EO07;    
  }
#endif /* UNIX */
} /* sqlsettls */

/*---------------------------------------------------------------------------*/

void *sqlgettls(tsp00_Int4 hKey)
{
#undef MF__
#define MF__ MOD__"sqlgettls"

#ifdef _WIN32
  return(TlsGetValue((DWORD)hKey));
#endif /* _WIN32 */

#ifdef CTHREADS
  void*  pObjPtr=NULL;
  teo001_SystemRc rc = thr_getspecific((thread_key_t)hKey, &pObjPtr);
  if (rc == EINVAL)
    return(NULL);
  else
    return(pObjPtr);
#endif /* CTHREADS */

#ifdef PTHREADS

#if PTHREAD_DRAFT < 7
  teo001_SystemRc rc;
  void*  pObjPtr=NULL;
  PTHREAD_RC(rc,pthread_getspecific((pthread_key_t)hKey, &pObjPtr));
  if (rc == EINVAL)
    return(NULL);
  else
    return(pObjPtr);
#else
  return pthread_getspecific((pthread_key_t)hKey);
#endif /* PTHREAD_DRAFT > 6 */

#endif /* PTHREADS */
} /* sqlgettls */

/*---------------------------------------------------------------------------*/

void sqlcreatesem (teo07_ThreadSemaphore* sem,
                   tsp00_Int4 initval,
                   tsp00_ErrTextc errtext,
                   teo07_ThreadErr* ok)
{
#undef MF__
#define MF__ MOD__"sqlcreatesem"
#ifdef _WIN32
    *sem = CreateSemaphore (NULL, initval, LONG_MAX, NULL);
    if (*sem == INVALID_HANDLE_VALUE) 
    {
        *ok = eo07_GetError(errtext);
    }
    else
    {
        *ok = THR_OK_EO07;
    }
#else /* UNIX */

    teo07_SemObj *semObj = NULL;

    if ( ALLOC_MEM_EO57( (void **)&semObj, sizeof (teo07_SemObj)) != NO_ERROR_EO001 )
    {
        *ok = THR_NOT_OK_EO07;
        if (errtext != NULL) 
        {
            strcpy(errtext, EO07_ERR_NO_MEM);
        }
        return;
    }
    semObj->token = initval;
    semObj->waiter = 0;
#ifdef CTHREADS
    (void)mutex_init(&semObj->mutex, USYNC_THREAD, NULL);
    (void)cond_init(&semObj->condition, USYNC_THREAD, NULL); 
#endif /* CTHREADS */

#ifdef PTHREADS
    (void)PTHREAD_MUTEX_INIT (&semObj->mutex);
    (void)PTHREAD_COND_INIT (&semObj->condition);
#endif /* PTHREADS */
    *sem = semObj;
    *ok = THR_OK_EO07;
#endif /* UNIX */
} /* sqlcreatesem */

/*---------------------------------------------------------------------------*/

void sqldestroysem (teo07_ThreadSemaphore sem)
{
#undef MF__
#define MF__ MOD__"sqldestroysem"
#ifdef _WIN32
    if ( sem != ((HANDLE)0)
      && sem != INVALID_HANDLE_VALUE)
    {
      CloseHandle (sem);
    }
#else /* UNIX */

    teo07_SemObj *semObj = (teo07_SemObj*) sem;

#ifdef CTHREADS
    (void)mutex_destroy(&semObj->mutex);
    (void)cond_destroy(&semObj->condition);
#endif /* CTHREADS */

#ifdef PTHREADS
    pthread_mutex_destroy (&semObj->mutex);
    pthread_cond_destroy (&semObj->condition);
#endif /* PTHREADS */
    FREE_MEM_EO57(sem);

#endif /* UNIX */
} /* sqldestroysem */

/*---------------------------------------------------------------------------*/

void sqlwaitsem (teo07_ThreadSemaphore sem)
{
#undef MF__
#define MF__ MOD__"sqlwaitsem"
#ifdef _WIN32

    WaitForSingleObject (sem, INFINITE);

#else /* UNIX */

    teo07_SemObj *semObj = (teo07_SemObj*) sem;

#ifdef CTHREADS
    mutex_lock (&semObj->mutex);
    while ( semObj->token < 1 )
    {
      ++semObj->waiter;
      cond_wait (&semObj->condition, &semObj->mutex);
      --semObj->waiter;
    }
    --semObj->token;
    mutex_unlock (&semObj->mutex);
#endif /* CTHREADS */

#ifdef PTHREADS
    pthread_mutex_lock (&semObj->mutex);
    while ( semObj->token < 1 )
    {
      ++semObj->waiter;
      pthread_cond_wait (&semObj->condition, &semObj->mutex);
      --semObj->waiter;
    }
    --semObj->token;
    pthread_mutex_unlock (&semObj->mutex);
#endif /* PTHREADS */

#endif /* UNIX */
} /* sqlwaitsem */

/*---------------------------------------------------------------------------*/

void sqltimedwaitsem (teo07_ThreadSemaphore sem, 
                      tsp00_Int4 seconds, 
                      teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"sqltimedwaitsem"
#ifdef _WIN32

    DWORD result = WaitForSingleObject (sem, seconds*1000);
    if ( result == WAIT_TIMEOUT )
    {
      *ok = THR_TIMEOUT_EO07;
    }
    else if ( result == WAIT_OBJECT_0 )
    {
      *ok = THR_OK_EO07;
    }
    else
    {
      *ok = THR_NOT_OK_EO07;
    }
#else /* UNIX */

    teo07_SemObj *semObj = (teo07_SemObj*) sem;
    tsp00_Bool        blocked;
    tsp00_Int4          result;
    teo001_SystemRc   rc;

#ifdef CTHREADS
    timestruc_t timeout;
    timeout.tv_sec = time (NULL) + seconds;
    timeout.tv_nsec = 0;

    mutex_lock (&semObj->mutex);
    while ( semObj->token < 1 ) 
    {
      ++semObj->waiter;
      rc = cond_timedwait (&semObj->condition, &semObj->mutex, &timeout);
      --semObj->waiter;
      if (rc == TIMEOUT_SYSTEM_RC) 
      {
        break;
      }
    }
    if ( semObj->token >= 1 ) 
    {
      --semObj->token;
      *ok = THR_OK_EO07;
    }
    else 
    {
      *ok = THR_TIMEOUT_EO07;
    }
    mutex_unlock (&semObj->mutex);
#endif /* CTHREADS */

#ifdef PTHREADS
    struct timespec timeout;
    timeout.tv_sec = time (NULL) + seconds;
    timeout.tv_nsec = 0;

    pthread_mutex_lock (&semObj->mutex);
    while ( semObj->token < 1 ) 
    {
      ++semObj->waiter;
      PTHREAD_RC(rc,pthread_cond_timedwait (&semObj->condition, &semObj->mutex, &timeout));
      --semObj->waiter;
      if (rc == TIMEOUT_SYSTEM_RC) 
      {
        break;
      }
    }
    if ( semObj->token >= 1 ) 
    {
      --semObj->token;
      *ok = THR_OK_EO07;
    }
    else 
    {
      *ok = THR_TIMEOUT_EO07;
    }
    pthread_mutex_unlock (&semObj->mutex);
#endif /* PTHREADS */

#endif /* UNIX */
} /* sqltimedwaitsem */

/*---------------------------------------------------------------------------*/

void sqlsignalsem (teo07_ThreadSemaphore sem)
{
#undef MF__
#define MF__ MOD__"sqlsignalsem"
#ifdef _WIN32

    LONG        previous;

    ReleaseSemaphore (sem, 1, &previous);

#else /* UNIX */

    teo07_SemObj *semObj = (teo07_SemObj*) sem;
    tsp00_Int4         token;

#ifdef CTHREADS
    mutex_lock (&semObj->mutex);
    ++semObj->token;
    if ( semObj->waiter )
    {
      cond_signal (&semObj->condition);
    }
    mutex_unlock (&semObj->mutex);
#endif /* CTHREADS */

#ifdef PTHREADS
    pthread_mutex_lock (&semObj->mutex);
    ++semObj->token;
    if ( semObj->waiter )
    {
        pthread_cond_signal (&semObj->condition);
    }
    pthread_mutex_unlock (&semObj->mutex);
#endif /* PTHREADS */

#endif /* UNIX */
} /* sqlsignalsem */

/*===========================================================================*
 *  LOCAL FUNCTIONS (CODE)                                                   *
 *===========================================================================*/

static void eo07_CreateThread( tsp00_Int4 StackSize, 
                               void *StackStartAddress,
                               void *(*proc)(void *), 
                               void *arg, 
                               tsp00_Int4 flags, 
                               teo07_Thread *thread, 
                               tsp00_ErrTextc errtext, 
                               teo07_ThreadErr *ok)
{
#undef MF__
#define MF__ MOD__"eo07_CreateThread"

  teo07_ThreadObj *pThreadObj = NULL;
#ifndef _WIN32
  teo001_SystemRc rc;
#endif /* !_WIN32 */
#ifdef PTHREADS
  pthread_attr_t attr;
#endif /* PTHREADS */
  
  *ok = THR_OK_EO07;
  if ( proc == NULL )
  {
      *ok = THR_NOT_OK_EO07;
      if (errtext != NULL)
          strcpy(errtext, EO07_ERR_PROCADDRESS_NULL);
      return;
  }
  
  if ( ALLOC_MEM_EO57( (void **)&pThreadObj, sizeof(teo07_ThreadObj) ) != NO_ERROR_EO001 )
  {
      *ok = THR_NOT_OK_EO07;
      if (errtext != NULL)
        strcpy(errtext, EO07_ERR_NO_MEM);
      return;
  }

  pThreadObj->proc = proc;
  pThreadObj->arg  = arg;
  pThreadObj->thread_id = 0;

  *thread = (teo07_Thread)pThreadObj;

  pThreadObj->start_suspended = ((flags & THR_CREATE_SUSPENDED_EO07) != 0);

  sqlcreatesem(&(pThreadObj->suspend_sem), 0, errtext, ok);
  if ( THR_OK_EO07 != *ok )
  {
      FREE_MEM_EO57(pThreadObj);
      *thread = NULL;
      return;
  }

#ifdef _WIN32
  pThreadObj->thread = CreateThread( NULL,
                            StackSize,
                            (LPTHREAD_START_ROUTINE)eo07_Win32threadEntry,
                            (void *)pThreadObj,
                            CREATE_SUSPENDED,
                            &pThreadObj->thread_id);
  if (pThreadObj->thread == NULL)
  {
    *ok = eo07_GetError(errtext);
  }
  else
  {
    if ( ResumeThread(pThreadObj->thread) == 0xFFFFFFFF )
    {
      *ok = eo07_GetError(errtext);
    }
  }
#endif /* _WIN32 */

#ifdef CTHREADS

  rc = thr_create(StackStartAddress,
                  StackSize, 
                  eo07_CthreadEntry,
                  (void *)pThreadObj,
                  0, /* create not suspended! Use suspend semaphore instead */
                  &pThreadObj->thread);
  /* done twice: here and in CthreadEntry to make sure scheduler want make a problem */
  /* Callable only after thread exists... */
  pThreadObj->thread_id = pThreadObj->thread;
#ifdef SUN
  if (rc == -1) 
  {
      *ok = THR_NOT_IMPLEMENTED_EO07;
      if (errtext != NULL)
          strcpy(errtext, EO07_ERR_NOTIMPLEMENTED);
      FREE_MEM_EO57(pThreadObj);
      *thread = NULL;
      return;
  }
#endif
  if (rc != 0) 
  {
      *ok = THR_NOT_OK_EO07;
      if (errtext != NULL) 
      {
          switch(rc) 
          {
          case ENOMEM:
              strcpy(errtext, EO07_ERR_NO_MEM);      
              break;
              
          case EINVAL:
              strcpy(errtext, EO07_ERR_STACKSIZE);
              break;
              
          case EAGAIN:
          default:
              strcpy(errtext, EO07_ERR_RESOURCE_LIMIT);      
              break;
          }
      }
  }

  if ( THR_OK_EO07 != *ok )
  {
    sqldestroysem(pThreadObj->suspend_sem);
  }
#endif /* CTHREADS */

#ifdef PTHREADS

#if PTHREAD_DRAFT < 7
  (void)pthread_attr_create(&attr);
#else
  (void)pthread_attr_init(&attr);
#endif

  if ( (flags & THR_CREATE_PROCESS_SCOPE_EO07) != 0 )
  {
#ifdef HAVE_PTHREAD_SCOPE_PROCESS
    (void)pthread_attr_setscope(&attr, PTHREAD_SCOPE_PROCESS);
#else
    (void)pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
#endif
  }
  else
  {
#ifdef PTHREAD_SCOPE_SYSTEM
    (void)pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
#endif
  }

  if ( (flags & THR_CREATE_DETACHED_EO07) == 0 )
  {
#ifdef PTHREAD_CREATE_UNDETACHED
  (void)pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_UNDETACHED);
#endif

#ifdef PTHREAD_CREATE_JOINABLE
  (void)pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
#endif
  }
  else
  {
#ifdef PTHREAD_CREATE_DETACHED
  (void)pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
#endif
  }

#if SAPDB_WITH_EXCEPTION_SAVE_COROUTINES > 0
#  ifdef LINUX
  pThreadObj->threadLocalErrno = 0;
  pThreadObj->pThreadLocalErrno = &pThreadObj->threadLocalErrno;
  pThreadObj->threadLocalHErrno = 0;
  pThreadObj->pThreadLocalHErrno = &pThreadObj->threadLocalHErrno;
  memset(pThreadObj->libcSpecificTlsKeys, 0, sizeof(int)*RTE_MAXIMUM_LIBC_SPECIFIC_KEYS);
#  endif
  pThreadObj->tlsKeys = (void ***)0;
#endif

#ifdef _POSIX_THREAD_ATTR_STACKSIZE
#ifdef LINUX
  pThreadObj->StackAddr = NULL;
  pThreadObj->StackSize = 0;
#endif
  if ( StackSize > 0 )
  {
      if ( StackSize < sqlget_thread_min_stack() )
      {
          StackSize = sqlget_thread_min_stack();
      }
#ifdef LINUX
      /* PTS 1105262/1105263 */
      /*
       * With linuxthreads stack size is variable, but other than in the standard
       * implementation if stacksize differs, stackaddr must be set from outside...
       * StackAddr given must point above the Stack...
       */
      StackSize = ((StackSize + 4095)&(~4095)); /* round up */

      if ( !StackStartAddress )
      {
          pThreadObj->StackAddr = (char *)eo07_NoheapMalloc( StackSize );
      }
      else
      {
          pThreadObj->StackAddr = StackStartAddress;
      }
      pThreadObj->StackSize = StackSize;
      if ( pThreadObj->StackAddr == NULL )
      {
          MSGD(( ERR_CANT_ALLOC_HEAP_MEMORY, errno ));
          DO_ABORT();
      }
      /* PTS 1109531 Workaround due to bug (4MB Limit) in pthread_attr_setstacksize() pthread-0.9.so */
      attr.__stacksize = StackSize;
      attr.__stackaddr = pThreadObj->StackAddr + StackSize;
      attr.__stackaddr_set = 1;
#else
      (void)pthread_attr_setstacksize(&attr, StackSize);
      if ( StackStartAddress )
      {
          int errCode;
#ifdef WANT_HIGHEST_ADDRESS_AS_STACK_ADDRESS
          StackStartAddress = ((char *)StackStartAddress)+StackSize;
#endif
          errCode = pthread_attr_setstackaddr(&attr, StackStartAddress);
          if ( ENOSYS == errCode )
          {
              strcpy(errtext, "POSIX option pthread_attr_setstackaddr not supported");
              FREE_MEM_EO57(pThreadObj);
              *thread = NULL;
              *ok = THR_NOT_OK_EO07;
              return;
          }
      }
#endif /* LINUX */
  }
#endif /* _POSIX_THREAD_ATTR_STACKSIZE */

#ifdef USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID
  pThreadObj->thread_id = eo07_GetSequenceNumber();
#endif /* USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID */

#if PTHREAD_DRAFT < 7
  PTHREAD_RC(rc, pthread_create(&pThreadObj->thread, attr, eo07_PthreadEntry, (void *)pThreadObj));
  (void)pthread_attr_delete(&attr);
#else
  PTHREAD_RC(rc, pthread_create(&pThreadObj->thread, &attr, eo07_PthreadEntry, (void *)pThreadObj));
  (void)pthread_attr_destroy(&attr);
#endif

  if (rc != 0) 
  {
    *ok = THR_NOT_OK_EO07;
    if (errtext != NULL) 
    {
      switch(rc) 
      {
      case ENOMEM:
          strcpy(errtext, EO07_ERR_NO_MEM);      
          break;
          
      case EINVAL:
          strcpy(errtext, EO07_ERR_STACKSIZE);
          break;

      case EAGAIN:
      default:
          strcpy(errtext, EO07_ERR_RESOURCE_LIMIT);      
          break;
      }
    }
  }

  if ( THR_OK_EO07 != *ok )
  {
      sqldestroysem(pThreadObj->suspend_sem);
  }
  else
  {
# ifdef USE_GETSEQUENCE_NP_FOR_THREAD_ID
    /* done twice: here and in PthreadEntry to make sure scheduler want make a problem */
    /* Callable only after thread exists... */
    pThreadObj->thread_id = (teo07_ThreadId)pthread_getsequence_np(pThreadObj->thread);
# endif

# ifdef USE_CLONE_PID_FOR_THREAD_ID
  /* PTS 1105217 */
    volatile teo07_ThreadObj *pTheThreadObj = pThreadObj;
    while ( pTheThreadObj->thread_id == 0 )
    {
      YIELD_CALL;
    }
#endif
  }
#endif /* PTHREADS */

  if ( *ok != THR_OK_EO07 )
  {
#ifdef _WIN32
      if ( (pThreadObj->thread != ((HANDLE)0))
        && (pThreadObj->thread != INVALID_HANDLE_VALUE) )
      {
          CloseHandle ( pThreadObj->thread );
      }
#endif
      FREE_MEM_EO57(pThreadObj);
      *thread = NULL;
  }

} /* eo07_CreateThread */

#ifdef _WIN32
static teo07_ThreadErr eo07_GetError(tsp00_ErrTextc errtext)
{
#undef MF__
#define MF__ MOD__"eo07_GetError"
  LPVOID lpMsgBuf;
  DWORD err1 = GetLastError();
  DWORD len;

  len = FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_ALLOCATE_BUFFER, NULL, err1, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
                       (LPTSTR)&lpMsgBuf, 
                       0, 
                       NULL);
  if (len != 0) 
  {
    strncpy(errtext, lpMsgBuf, sizeof(tsp00_ErrText));
    errtext[sizeof(tsp00_ErrText)] = '\0';
    LocalFree(lpMsgBuf);        
  }
  else 
  {
    DWORD err2;
    UCHAR szTmp[80];
    err2 = GetLastError();
    sprintf(szTmp, __FILE__" FormatMessage failed (%ld)\x0d\x0a GetLastError (%ld)", 
        err2, err1);
    FatalAppExit(0, szTmp);
  }
  return(THR_NOT_OK_EO07);
} /* eo07_GetError */

/*---------------------------------------------------------------------------*/

/*
 * Needed to setup TLS key
 */
static DWORD eo07_Win32threadEntry( void *arg )
{
#undef MF__
#define MF__ MOD__"eo07_Win32threadEntry"
    teo07_ThreadObj *pThreadObj;
    
    pThreadObj = (teo07_ThreadObj *)arg;

    if ( TlsSetValue((DWORD)self_key_eo07, pThreadObj) == 0) 
    {
        return GetLastError();
    }

    MSGD(( INFO_THREAD_START, pThreadObj->thread_id ));

    if ( pThreadObj->start_suspended )
    {
        sqlwaitsem(pThreadObj->suspend_sem);
    }

    return (DWORD)(tsp00_Longuint)(*(pThreadObj->proc))(pThreadObj->arg);
} /* eo07_Win32threadEntry */
#endif /* _WIN32 */

/*---------------------------------------------------------------------------*/

#ifdef CTHREADS
/*
 * Needed to setup TLS key
 */
static void *eo07_CthreadEntry( void *arg )
{
#undef MF__
#define MF__ MOD__"eo07_CthreadEntry"
    teo001_SystemRc rc;
    teo07_ThreadObj *pThreadObj;
    
    pThreadObj = (teo07_ThreadObj *)arg;

    rc = thr_setspecific((thread_key_t)self_key_eo07, pThreadObj);
    if ( rc != 0 )
    {
        return (void *)rc;
    }

      /* Callable only after thread exists... */
    pThreadObj->thread_id = pThreadObj->thread;

    MSGD(( INFO_THREAD_START, pThreadObj->thread_id ));

    if ( pThreadObj->start_suspended )
    {
        sqlwaitsem(pThreadObj->suspend_sem);
    }

    return (*(pThreadObj->proc))(pThreadObj->arg);
} /* eo07_CthreadEntry */
#endif /* CTHREADS */

/*---------------------------------------------------------------------------*/

#ifdef PTHREADS
/*
 * Since pthreads do not support suspend, selfsuspending is done via a mutex
 */
static void *eo07_PthreadEntry( void *arg )
{
#undef MF__
#define MF__ MOD__"eo07_PthreadEntry"
    teo001_SystemRc rc;
    teo07_ThreadObj *pThreadObj;
    
    pThreadObj = (teo07_ThreadObj *)arg;

    PTHREAD_RC(rc,pthread_setspecific((pthread_key_t)self_key_eo07, pThreadObj));
    if ( rc != 0 ) 
    {
        return (void *)rc;
    }

# ifdef USE_GETSEQUENCE_NP_FOR_THREAD_ID
      /* Callable only after thread exists... */
    pThreadObj->thread_id = (teo07_ThreadId)pthread_getsequence_np(pThreadObj->thread);
# endif /* USE_GETSEQUENCE_NP_FOR_THREAD_ID */

# ifdef USE_CLONE_PID_FOR_THREAD_ID
    /* PTS 1105217 */
    pThreadObj->thread_id = (teo07_ThreadId)getpid();
# endif /* USE_CLONE_PID_FOR_THREAD_ID */

    MSGD(( INFO_THREAD_START, pThreadObj->thread_id ));

    if ( pThreadObj->start_suspended )
    {
        sqlwaitsem(pThreadObj->suspend_sem);
    }

    return (*(pThreadObj->proc))(pThreadObj->arg);
} /* eo07_PthreadEntry */

/*---------------------------------------------------------------------------*/

#ifdef USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID

static teo07_ThreadId eo07_GetSequenceNumber()
{
    teo07_ThreadId id;

    (void)pthread_mutex_lock(&threadIdMutex_eo07);
    id = nextThreadId_eo07++;
    (void)pthread_mutex_unlock(&threadIdMutex_eo07);
    return id;
}

#endif /* USE_EO07_GET_SEQUENCE_NUMBER_FOR_THREAD_ID */

#endif /* PTHREADS */

/*---------------------------------------------------------------------------*/

#ifdef LINUX
/* PTS 1105262/1105263 */
static void *
eo07_NoheapMalloc(size_t size)
{
    int fd_dev_zero;
    caddr_t ra;

    if ((fd_dev_zero = open("/dev/zero", O_RDWR, 0)) == -1) {
        perror("opening /dev/zero");
        return (NULL);
    }

    if (size == 0) size = sizeof (int);

    if ((ra = (caddr_t)mmap((caddr_t) 0, size, 
                   (PROT_READ | PROT_WRITE), MAP_PRIVATE,
                   fd_dev_zero, 0)) == (caddr_t)-1) {
        perror("mmap'ing /dev/zero");
        ra = NULL;
    }

    close(fd_dev_zero);
    return((void *)ra);
}

/*---------------------------------------------------------------------------*/

static void
eo07_NoheapFree(void *ptr, size_t size)
{
    if (ptr == NULL) return;

    if (munmap((caddr_t)ptr, size))
        perror("munmap");
/*
 * To test on any program using the already freed memory, use the mprotect() call
 * below as alternative.
 *

  if ( mprotect(ptr, size, PROT_NONE) )
	perror("mprotect");

 */
}

/*---------------------------------------------------------------------------*/

static void *
eo07_DummyPthreadEntry( void *arg )
{
  return arg;
}
#endif /* LINUX */

#endif /* ( defined(_REENTRANT) && !defined(_WIN32) ) || ( defined(_MT) && defined(_WIN32) ) */

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/
