/*!
 * \file    LVC_ProcServerExtender.hpp
 * \author  IvanS
 * \brief   Extender for processing calls from out-of-process procedures in liveCache kernel.
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

#ifndef __LVC_PROCSERVER_EXTENDER
#define __LVC_PROCSERVER_EXTENDER

#include "liveCache/LVC_ProcServerInterface.hpp"
#include "liveCache/LVC_ProcServerExecutor.hpp"
#include "liveCache/LVC_Dispatcher.hpp"
#include "Oms/OMS_DbpBase.hpp"
#include "geo38.h"

#define LC_DBPROC_PROTO
#include "Oms/OMS_DbpGlobals.hpp"

class SAPDBMem_IRawAllocator;


// {CA72FBE0-9775-4b3b-A421-A29D2E27EDCC}
DEFINE_GUID(IID_LVC_ProcServerExtender, 
            0xca72fbe0, 0x9775, 0x4b3b, 0xa4, 0x21, 0xa2, 0x9d, 0x2e, 0x27, 0xed, 0xcc);

/*!
 * \brief Extender for processing calls from out-of-process procedures.
 *
 * This class handles the requests from an out-of-process procedure and
 * processes them locally in kernel. The results are sent back to the
 * out-of-process procedure.
 *
 * It also handles startup of new stored procedures by sending the call
 * to the ProcServer.
 *
 * There is one instance of this class per one ProcServer and one session.
 *
 * \see \ref omsprocserver
 *
 * \ingroup oms procserver
 */
class LVC_ProcServerExtender : public IliveCacheSpec, 
  public LVC_ProcServerInterface, public LVC_ProcServerExecutor
{
public:
  /*!
   * \brief Construct a new instance.
   *
   * \param alloc allocator for memory allocation,
   * \param sink local sink to use.
   */
  LVC_ProcServerExtender(SAPDBMem_IRawAllocator *alloc, IliveCacheSink *sink);

  /*!
   * \brief Process one request packet.
   *
   * This method reads the packet from communication area and processes
   * the command contained in the packet. Then, it sends a reply to the
   * ProcServer.
   */
  void processPacket();

  // Inherited from LVC_ProcServerInterface:
  virtual void getSendArea(void* &addr, size_t &maxSize);
  virtual void getSecondarySendArea(void* &addr, size_t &maxSize);
  virtual void getReceiveArea(void* &addr, size_t &maxSize);
  virtual void sendData(size_t size);
  virtual void receiveData();
  virtual size_t getOffsetForAddress(void *addr);
  virtual void *getAddressForOffset(size_t offs);

#define COM_INTERFACE LVC_ProcServerExtender
#define COM_INTERFACE_IID IID_LVC_ProcServerExtender
//#define OMS_DBPGLOBALS_NOSYSFNC   // own system function handling
//#include "Oms/OMS_DbpGlobals.hpp" // default implementation of some virtual functions

  // system functions that will be forwarded to ProcServer
  STDMETHOD ( sysRequest) ( int __RPC_FAR *lpbuflen, unsigned char __RPC_FAR *lpv );
  STDMETHOD ( sysMonitorInit ) (void);
  STDMETHOD ( sysMonitorProlog ) ( int dispId, unsigned char __RPC_FAR *lpMethodName );
  STDMETHOD ( sysMonitorEpilog ) ( int dispId, int runTime );
  STDMETHOD ( sysSqlSupport ) ( BOOLEAN __RPC_FAR *sql_support)     
  { *sql_support = false; return S_OK; }
  STDMETHOD ( Register ) (void);

private:
  /// Allocator for memory allocation.
  SAPDBMem_IRawAllocator  *m_alloc;

  long      m_cRef;       ///< Reference counter.
  IUnknown  *m_pIUnknown; ///< Reference to the sink.
};

#endif  // __LVC_PROCSERVER_EXTENDER

