/*!
    @file     SharedSQL_Plan.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    AVLTree to store the execution plan for SQLCommands
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SHAREDSQL_PLAN_HPP
#define SHAREDSQL_PLAN_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_IPlan.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Messages.hpp"
#include "SQLManager/Catalog/Catalog_ISharedSQLInterface.hpp"

#include "Container/Container_AVLTree.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_IRawAllocator.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_NewDestroy.hpp"
#include "SAPDBCommon/Tracing/SAPDBTrace_Usage.hpp"

#include "RunTime/Synchronisation/RTESync_RWRegion.hpp"
#include "RunTime/RTE_Message.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/// PlanComparator
typedef Catalog_ISharedSQLInterface SharedSQL_PlanComparator;

/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_PlanNode
    @brief          Node within the SharedSQL_PlanTree
 */
class SharedSQL_PlanNode : public Container_AVLNode<void*, SharedSQL_PlanComparator>
{
public:
    /*!
        @brief  Constructor
        @param  k [in]
    */
    SharedSQL_PlanNode( void *const& k  ) : 
        Container_AVLNode<void*, SharedSQL_PlanComparator>(k) {};

    /*!
        @copydoc        Container_AVLNode
    */
    virtual void CleanUp(SAPDBMem_IRawAllocator& alloc, SharedSQL_PlanComparator& cmp)
    {
        cmp.DestroyPlanObject(alloc, m_Key);
    }
};  // SharedSQL_PlanNode



/*---------------------------------------------------------------------------*/
/*!
    @brief          Container_AVLTree to store SharedSQ_PlanNode 
 */
typedef Container_AVLTree<SharedSQL_PlanNode, void*, SharedSQL_PlanComparator>
        SharedSQL_PlanTree;

/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_Plan
    @brief          Class to handle execution plans
 */
class SharedSQL_Plan : public SharedSQL_IPlan
{
public:
    /*!
        @brief  Constructor
        @param  ID           [in]
        @param  SpinLockPool [in]
        @param  Allocator    [in]
    */
    SharedSQL_Plan( SharedSQL_CommandID ID, RTESync_BaseSpinlockPool& SpinLockPool, SAPDBMem_IRawAllocator& Allocator) :
            mpPlanLock ( RTESync_CreateRWRegion (ID, SpinLockPool, Allocator)),
            mPlanComparator(),
	        mPlanTree(&mPlanComparator, &Allocator),
            mPlanSize(0),
            mParseID()

    {
    }

    /*!
        @brief  Destructor
    */
    ~SharedSQL_Plan()
    {
        if (mpPlanLock)
            RTESync_DestroyRWRegion(mpPlanLock, *(mPlanTree.GetAllocator()));
    }

    /*!
        @brief  Returns the allocator used to store the execution plan
        @return SAPDBMem_IRawAllocator*
    */
    SAPDBMem_IRawAllocator* GetAllocator( void )
    {
        return mPlanTree.GetAllocator();
    }

    /*!
        @brief  Returns wheather or not the plan could be constructed completely. FALSE if the RWRegion to protect the plan could not be created
        @return SAPDB_Bool
    */
    SAPDB_Bool CompletelyConstructed( void )
    {
        return mpPlanLock != NULL;
    }

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual void Drop( bool DelMasterParseID )
    {
        mPlanTree.DeleteAll();
        mPlanSize = 0;
        if (DelMasterParseID)
            mParseID = SharedSQL_ParseID();
    }

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual SAPDB_Bool IsEmpty( void ) const
    {
        return ( mPlanTree.GetSize() == 0);
    }

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual SAPDB_Int4 GetSize( void ) const
    {
        return mPlanSize;
    }

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual const void* Find( void* Key ) const
    {
        if (SharedSQL_PlanNode* N = mPlanTree.Find(Key))
                return (*(N->GetKey()));
        return 0;
    }

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual SAPDB_Bool Insert( SharedSQL_ParseID PID, void *const P, SAPDB_ULong Size )
    {
        if ( mParseID == SharedSQL_ParseID() )
            mParseID = PID;

        if ( PID == mParseID )
        {
            Container_AVLTreeError rc;
            if (mPlanTree.Insert(P, rc) )
            {
                mPlanSize += Size;
                return SAPDB_TRUE;
            }
            else
            {
                switch (rc)
                {
                case AVLTree_DuplicateKey: 
                    RTE_Message(SAPDBErr_MessageList("SharedSQL", __CONTEXT__, SHAREDSQL_ERR_PUTPLAN_DUPKEY));
                    break;
                case AVLTree_OutOfMemory:
                    RTE_Message(SAPDBErr_MessageList("SharedSQL", __CONTEXT__, SHAREDSQL_ERR_PUTPLAN_NOMEM));
                    break;
                }
            }
        }
        else
        {
            RTE_Message(SAPDBErr_MessageList("SharedSQL", __CONTEXT__, SHAREDSQL_ERR_PUTPLAN_INVALIDID));
        }
        return SAPDB_FALSE;
    }

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual SharedSQL_ParseID& GetMasterParseID( void )
    {
        return mParseID;
    }

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual void EnterLock(bool exclusive) 
    { 
        mpPlanLock->enter(exclusive); 
//        SAPDBTrace_Stream stream (&SharedSQL_Trace.GetTracePrinter ()); // DDT
//        stream << "INFO: SharedSQL_Plan::EnterLock" << exclusive << NewLine; // DDT
    };

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual bool TryEnterLock(bool exclusive) 
    { 
        bool r = mpPlanLock->tryEnter(exclusive); 
//        SAPDBTrace_Stream stream (&SharedSQL_Trace.GetTracePrinter ()); // DDT
//        if (r)
//            stream << "INFO: SharedSQL_Plan::TryEnterLock successful" << exclusive << NewLine; // DDT
//        else
//            stream << "INFO: SharedSQL_Plan::TryEnterLock unsuccessful" << exclusive << NewLine; // DDT
        return r;
    };

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual void LeaveLock(bool exclusive) 
    { 
        mpPlanLock->leave(exclusive); 
//        SAPDBTrace_Stream stream (&SharedSQL_Trace.GetTracePrinter ()); // DDT
//        stream << "INFO: SharedSQL_Plan::LeaveLock" << exclusive << NewLine; // DDT
    };

    /*!
        @copydoc        SharedSQL_IPlan
    */
    virtual void SwitchLockToNonExclusive( void )
    {
        mpPlanLock->SwitchToNonExclusiveAccess();
//        SAPDBTrace_Stream stream (&SharedSQL_Trace.GetTracePrinter ()); // DDT
//        stream << "INFO: SharedSQL_Plan::SwitchToNonExclusiveAccess" << NewLine; // DDT
    }

private:
    /// RWRegion to protect the execution plan
    RTESync_IRWRegion                  *mpPlanLock;     
    /// AVL Tree that stores the execution plan
    SharedSQL_PlanTree	   			    mPlanTree;
    /// Size of the plan elements stored in the plan tree
    SAPDB_Int4                          mPlanSize;
    /// ParseID used to create the execution plan
    SharedSQL_ParseID                   mParseID;
    /// Comparator to compare and find plan elements
    SharedSQL_PlanComparator            mPlanComparator;

};

/* ------------------------------------------------------------------------- */

#endif  /* SHAREDSQL_PLAN_HPP */
