/*!
  @file           Catalog_Object.hpp
  @author         ThomasA
  @ingroup        Catalog
  @brief          base class of all catalog objects

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/


#ifndef CATALOG_OBJECT_HPP
#define CATALOG_OBJECT_HPP

#include "SQLManager/Catalog/Catalog_Types.hpp"

class Catalog_TableObject;
class Catalog_MessBlockObject;
class SAPDBMem_IRawAllocator;

/*!
   @brief base class of all objects stored in the Catalog_SessionCache
            
          Catalog_Objects describe objects of the database, for example users, tables, views etc.
          Furhtermore the describe temporary objects created in the course of sql execution,
          for example execution plans.
*/

class Catalog_Object : public Catalog_Object_Pascal
{
#define TYPE_MESSBLOCK      cak_cmessblock         //!< type flag of Catalog_MessBlockObject
#define TYPE_TABLE          cak_ctable             //!< type flag of Catalog_TableObject
#define TYPE_RESULT         cak_cresult            //!< type flag of Catalog_TableObject(temporary)
#define TYPE_RESULTNAME     cak_cresname           //!< type flag of result name object
#define TYPE_SHARED_SQL     cak_esharedSQL_info    //!< type flag of Catalog_SharedSqlObject  
#define TYPE_METHOD         cak_emethod            //!< type flag of Catalog_DBProcObject
#define TYPE_METHODFILEINFO cak_emethodfileinfo    //!< type flag of Catalog_DBProcFileObject
#define TYPE_METHODINFO     cak_emethodinfo        //!< type flag of Catalog_DBProcInfoObject
#define TYPE_TRIGGERPARSEINFO cak_ctriggerparsinfo //< type flag for trigger parse infos
public :
	/*!
	   @brief standard keylength 12
	*/
	enum { DefaultKeyLen = 12 };
   	/*!
	   @brief offset of a standard catalog object with keylength 12
	*/
	enum { DefaultOffset = cak_sysbufferoffset };

    /*!
       @brief defines the key of a Catalog_Object. Keys are used to identify catalog objects. 
    */
	class Catalog_Key : public tgg00_SysInfoKey
	{
	public :
        Catalog_Key(); //!< default constructor

        /*!
           @brief copy constructor for a standard key
           @param standardKey source key to be copied 
        */
        Catalog_Key(const tsp00_C12& standardKey);
        /*!
           @brief puts the entrytype into the key
           @param entryType type flag to be assigned
        */
        inline void SetEntryType(unsigned char entryType); 
        /*!
           @brief puts a surrogate into the key
           @param surrogate, surrogate to be assigned
        */
        inline void SetSurrogate(const SQLMan_Surrogate& surrogate); 
	};

	/*!
       @brief constructor, initializes keylen and record length
       @param keyLen  the length of the key
       @param infoLen the length of the info part of the object
    */
	Catalog_Object (int keyLen, int infoLen);
	/*!
	   @brief gets pointer to information part of the catalog object
    */
	const void* GetInfo(int& infoLen) const;
    /*!
       @brief invalidates the transaction info of the given catalog object.
       @param p memory containing the catalog object
       @param offset catalog object starts at offset in p
    */
    static void InvalidateTransaction(void* p, int offset);
	/*!
	   @brief stores information in a catalog object
    */
	int PutInfo (void* pInfo, int infoLength);
    /*!
       @brief returns the type of the object
    */
    int Type() const;
    /*!
       @brief casts current record to a Catalog_TableObject
    */
    inline operator Catalog_TableObject&()
    {
        return *reinterpret_cast<Catalog_TableObject*>(this);
    }
    /*!
       @brief casts current record to a Catalog_MessBlockObject
    */
    inline operator Catalog_MessBlockObject&()
    {
        return *reinterpret_cast<Catalog_MessBlockObject*>(this);
    }
    /*!
       @brief creates a copy using a given allocator
    */
    void* CreateCopy (SAPDBMem_IRawAllocator& allocator, int offset, SAPDB_UInt4& memoryUsage);
    /*!
       @brief returns the address of the object key
    */
    inline const void* GetKey() const;
    /*!
       @brief returns the length of the object key
    */
    inline int KeyLength() const;
    /*!
       @brief returns the length of the object
    */
    inline int Length() const;
};

inline const void* Catalog_Object::GetKey() const
{
    return &this->syskey().all();
};

inline int Catalog_Object::KeyLength() const
{
    return this->syskey().skeylen;
}

inline int Catalog_Object::Length() const
{
    return this->b_sl();
}

inline void Catalog_Object::Catalog_Key::SetEntryType(unsigned char entryType)
{
    this->sentrytyp()[1] = entryType;
}

inline void Catalog_Object::Catalog_Key::SetSurrogate(const SQLMan_Surrogate& surrogate)
{
    memcpy (&this->stableid()[0], &surrogate[0], sizeof(surrogate));
}

#endif