/*!
  @file           RTESys_AtomicOperation.hpp
  @author         JoergM
  @ingroup        Runtime
  @brief          Atomic operation (C++ Interface) (cmpxchg, atomic_increment, atomic_decrement)

  This header files defines five function for atomic read, compare and exchange and atomic modifications.

  The functions are RTESys_AtomicRead, RTESys_AtomicWrite RTESys_CompareAndExchange, RTESys_AtomicModify
  and RTESys_AtomicOperation.

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/
#ifndef RTESYS_ATOMICOPERATION_HPP
#define RTESYS_ATOMICOPERATION_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "RunTime/System/RTESys_AtomicOperation.h"

//--------------------------------------------------------------------------------
// RTESys_AtomicRead
//--------------------------------------------------------------------------------

#  if !defined(BIT64)
/*!
   @brief Atomic read operation on 8byte aligned signed value

   On 32bit platform reading an aligned 8Byte value is not atomic.

   @param memoryPosition [in] the memory position to read
   @return [SAPDB_Int8] the value at the given position
 */
inline SAPDB_Int8 RTESys_AtomicRead( SAPDB_Int8 & memoryPosition )
{ return RTESys_AtomicReadInt8(memoryPosition); }

/*!
   @brief Atomic read operation on 8byte aligned signed value

   On 32bit platform reading an aligned 8Byte value is not atomic.

   @param memoryPosition [in] the memory position to read
   @return [SAPDB_UInt8] the value at the given position
 */
inline SAPDB_UInt8 RTESys_AtomicRead( SAPDB_UInt8 & memoryPosition )
{ return RTESys_AtomicReadUInt8(memoryPosition); }
#  else

/*!
   @brief Atomic read operation on 8byte aligned signed value
   @param memoryPosition [in] the memory position to read
   @return [SAPDB_Int8] the value at the given position
 */
inline SAPDB_Int8 RTESys_AtomicRead( SAPDB_Int8 & memoryPosition )
{ return memoryPosition; }
/*!
   @brief Atomic read operation on 8byte aligned unsigned value
   @param memoryPosition [in] the memory position to read
   @return [SAPDB_UInt8] the value at the given position
 */
inline SAPDB_UInt8 RTESys_AtomicRead( SAPDB_UInt8 & memoryPosition )
{ return memoryPosition; }

#  endif /* !defined(BIT64) */

/*!
   @brief Atomic read operation on 4byte aligned signed value

   On 32 and 64 bit platforms reading an aligned 4Byte value is atomic.

   @param memoryPosition [in] the memory position to read
   @return [SAPDB_Int8] the value at the given position
 */
inline SAPDB_Int4 RTESys_AtomicRead( SAPDB_Int4 & memoryPosition )
{ return memoryPosition; }

/*!
   @brief Atomic read operation on 4byte aligned unsigned value

   On 32 and 64 bit platforms reading an aligned 4Byte value is atomic.

   @param memoryPosition [in] the memory position to read
   @return [SAPDB_Int8] the value at the given position
 */
inline SAPDB_UInt4 RTESys_AtomicRead( SAPDB_UInt4 & memoryPosition )
{ return memoryPosition; }

/*!
   @brief Atomic read operation on aligned pointer value

   On 32 and 64 bit platforms reading a aligned pointer value is atomic.

   @param memoryPosition [in] the memory position to read
   @return [SAPDB_Int8] the value at the given position
 */
template <class PointerType>
PointerType * RTESys_AtomicRead( PointerType * &   pointerPosition )
{ return pointerPosition; }

//--------------------------------------------------------------------------------
// RTESys_AtomicWrite
//--------------------------------------------------------------------------------

#  if !defined(BIT64)
/*!
   @brief Atomic write operation on 8byte aligned signed value

   On 32bit platform writing an aligned 8Byte value is not atomic.

   @param memoryPosition [in] the memory position to write
   @param newValue [in] the new value for the given position
 */
inline void RTESys_AtomicWrite( SAPDB_Int8 & memoryPosition, SAPDB_Int8 const newValue )
{ RTESys_AtomicWriteInt8(memoryPosition, newValue); }

/*!
   @brief Atomic write operation on 8byte aligned signed value

   On 32bit platform writing an aligned 8Byte value is not atomic.

   @param memoryPosition [in] the memory position to write
   @param newValue [in] the new value for the given position
 */
inline void RTESys_AtomicWrite( SAPDB_UInt8 & memoryPosition, SAPDB_UInt8 const newValue  )
{ RTESys_AtomicWriteUInt8(memoryPosition, newValue); }
#  else

/*!
   @brief Atomic write operation on 8byte aligned signed value
   @param memoryPosition [in] the memory position to write
   @param newValue [in] the new value for the given position
 */
inline void RTESys_AtomicWrite( SAPDB_Int8 & memoryPosition, SAPDB_Int8 const newValue )
{ memoryPosition = newValue; }
/*!
   @brief Atomic write operation on 8byte aligned unsigned value
   @param memoryPosition [in] the memory position to write
   @param newValue [in] the new value for the given position
 */
inline void RTESys_AtomicWrite( SAPDB_UInt8 & memoryPosition, SAPDB_UInt8 const newValue )
{ memoryPosition = newValue; }

#  endif /* !defined(BIT64) */

/*!
   @brief Atomic write operation on 4byte aligned signed value

   On 32 and 64 bit platforms writing an aligned 4Byte value is atomic.

   @param memoryPosition [in] the memory position to write
   @param newValue [in] the new value for the given position
 */
inline void RTESys_AtomicWrite( SAPDB_Int4 & memoryPosition, SAPDB_Int4 const newValue )
{ memoryPosition = newValue; }

/*!
   @brief Atomic write operation on 4byte aligned unsigned value

   On 32 and 64 bit platforms writing an aligned 4Byte value is atomic.

   @param memoryPosition [in] the memory position to write
   @param newValue [in] the new value for the given position
 */
inline void RTESys_AtomicWrite( SAPDB_UInt4 & memoryPosition, SAPDB_UInt4 const newValue )
{ memoryPosition = newValue; }

/*!
   @brief Atomic write operation on aligned pointer value

   On 32 and 64 bit platforms writing a aligned pointer value is atomic.

   @param memoryPosition [in] the memory position to write
   @param newValue [in] the new value for the given position
 */
template <class PointerType>
void RTESys_AtomicWrite( PointerType * &   pointerPosition, PointerType * const newValue )
{ pointerPosition = newValue; }

//--------------------------------------------------------------------------------
// RTESys_CompareAndExchange
//--------------------------------------------------------------------------------

/*! 
   @brief Atomic CompareAndExchange value

   NOTE: since normaly some modification have been made on a structure, these calls IMPLICITLY enforces
   memory barriers, so that all updates are visible.

   If the old value is no longer matching, the function does not modify the memory
   and returns false. Otherwise the new value is set and true is returned.

  @param memoryPosition [in] the memory position to exchange
  @param expectedValue [in] the expected old value
  @param newValue [in] the wanted new value
  @param oldValue [out] the compared old value
  @return [SAPDB_Bool] true if newValue replaced oldValue, false if expectedValue was no longer found */
inline SAPDB_Bool RTESys_CompareAndExchange( SAPDB_Int4 &     memoryPosition,
                                             SAPDB_Int4 const expectedValue,
                                             SAPDB_Int4 const newValue,
                                             SAPDB_Int4 &     oldValue )
{ return RTESys_CmpxchgInt4(memoryPosition, expectedValue, newValue, oldValue); }

/*! @brief Atomic CompareAndExchange value
    @param memoryPosition [in] the memory position to exchange
    @param expectedValue [in] the expected old value
    @param newValue [in] the wanted new value
    @param oldValue [out] the compared old value
    @return [SAPDB_Bool] true if newValue replaced oldValue, false if expectedValue was no longer found */
inline SAPDB_Bool RTESys_CompareAndExchange( SAPDB_Int8 &     memoryPosition,
                                             SAPDB_Int8 const expectedValue,
                                             SAPDB_Int8 const newValue,
                                             SAPDB_Int8 &     oldValue )
{ return RTESys_CmpxchgInt8(memoryPosition, expectedValue, newValue, oldValue); }

/*! @brief Atomic CompareAndExchange value
    @param memoryPosition [in] the memory position to exchange
    @param expectedValue [in] the expected old value
    @param newValue [in] the wanted new value
    @param oldValue [out] the compared old value
    @return [SAPDB_Bool] true if newValue replaced oldValue, false if expectedValue was no longer found */
inline SAPDB_Bool RTESys_CompareAndExchange( SAPDB_UInt4 &     memoryPosition,
                                             SAPDB_UInt4 const expectedValue,
                                             SAPDB_UInt4 const newValue,
                                             SAPDB_UInt4 &     oldValue )
{ return RTESys_CmpxchgUInt4(memoryPosition, expectedValue, newValue, oldValue); }
/*! @brief Atomic CompareAndExchange value
    @param memoryPosition [in] the memory position to exchange
    @param expectedValue [in] the expected old value
    @param newValue [in] the wanted new value
    @param oldValue [out] the compared old value
    @return [SAPDB_Bool] true if newValue replaced oldValue, false if expectedValue was no longer found */
inline SAPDB_Bool RTESys_CompareAndExchange( SAPDB_UInt8 &     memoryPosition,
                                             SAPDB_UInt8 const expectedValue,
                                             SAPDB_UInt8 const newValue,
                                             SAPDB_UInt8 &     oldValue )
{ return RTESys_CmpxchgUInt8(memoryPosition, expectedValue, newValue, oldValue); }

/*! @brief Atomic CompareAndExchange value
    @param pointerPosition [in] the memory position to modify
    @param expectedValue [in] the expected old value
    @param newValue [in] the wanted new value
    @param oldValue [out] the compared old value
    @return [SAPDB_Bool] true if newValue replaced oldValue, false if expectedValue was no longer found */
template <class PointerType>
SAPDB_Bool RTESys_CompareAndExchange( PointerType * &     pointerPosition,
                                      PointerType * const expectedValue,
                                      PointerType * const newValue,
                                      PointerType * &     oldValue )
{
    void **voidPosition = (void **)&pointerPosition;
    void *oldValueFound;
    if ( RTESys_CmpxchgPointer(*voidPosition, expectedValue, newValue, oldValueFound) )
    {
        oldValue = (PointerType *)oldValueFound; 
        return true;
    }
    return false;
}

//--------------------------------------------------------------------------------
// RTESys_AtomicModify
//--------------------------------------------------------------------------------

/*!
   @brief Atomic modification operation

   The value at the given position is modified by adding the delta value atomically.

       memoryPosition = memoryPostion + deltaValue

   This allows to implement AtomicIncrement/AtomicDecrement

   @param memoryPosition [inout] the memory position to exchange
   @param deltaValue [in] the delta to add
   @return resulting value
 */
inline SAPDB_Int4 RTESys_AtomicModify( SAPDB_Int4 &     memoryPosition,
                                 SAPDB_Int4 const deltaValue )
{ return RTESys_AtomicModifyInt4(memoryPosition, deltaValue); }
/*! @brief Atomic modification operation
    @param memoryPosition [in] the memory position to exchange
    @param deltaValue [in] the delta to add
    @return resulting value
 */
inline SAPDB_Int8 RTESys_AtomicModify( SAPDB_Int8 &     memoryPosition,
                                 SAPDB_Int8 const deltaValue )
{ return RTESys_AtomicModifyInt8(memoryPosition, deltaValue); }
/*! @brief Atomic modification operation
    @param memoryPosition [in] the memory position to exchange
    @param deltaValue [in] the delta to add
    @return resulting value
 */
inline SAPDB_UInt4 RTESys_AtomicModify( SAPDB_UInt4 &    memoryPosition,
                                 SAPDB_Int4 const deltaValue )
{ return RTESys_AtomicModifyUInt4(memoryPosition, deltaValue); }
/*! @brief Atomic modification operation
    @param memoryPosition [in] the memory position to exchange
    @param deltaValue [in] the delta to add
    @return resulting value
 */
inline SAPDB_UInt8 RTESys_AtomicModify( SAPDB_UInt8 &    memoryPosition,
                                 SAPDB_Int8 const deltaValue )
{ return RTESys_AtomicModifyUInt8(memoryPosition, deltaValue); }

/*! @brief Atomic modification operation
    @param pointerPosition [in] the memory position to modify
    @param scaling [in] the size of the pointed object
    @param deltaValue [in] the value to add
    @return resulting value
 */
inline void * RTESys_AtomicModify( void *          & pointerPosition,
                                 SAPDB_ULong const scaling,
                                 SAPDB_Long const  deltaValue )
{ return RTESys_AtomicModifyPointer(pointerPosition, scaling, deltaValue); }

/*! @brief Atomic modification operation
    This works as normal pointer arithmetic does
    @param pointerPosition [in] the memory position to modify
    @param deltaValue [in] the value to add
    @return resulting value
 */
template <class PointerType>
void * RTESys_AtomicModify( PointerType * &   pointerPosition,
                            SAPDB_Long const  deltaValue )
{
    void **voidPosition = (void **)&pointerPosition;
    return RTESys_AtomicModifyPointer(*voidPosition, sizeof(PointerType), deltaValue);
}

//--------------------------------------------------------------------------------
// RTESys_AtomicOperation
//--------------------------------------------------------------------------------

/*!
   @brief Atomic operation on given memory position

   This functions only make sure the atomicity of the read/modify/write cycle on the given memory
   position. This does not make the functions supplied thread save... If the memory position is modified
   very frequently, this function will loop, but since atomicread() and cmpxchg() are used the actual
   spinning time will be short.

   The value at the given position is retrieved, the function is applied and the result
   is written back if the function returns true.
   
   The function is getting three arguments, old_value, reference to new_value and supplied context.
   
     AtomicAdd(ValueType oldValue, ValueType & newValue, void *context)
     {
         struct AtomicAddContext_ {
             ValueType deltaValue;
         } *atomicAddContext = (struct AtomicAddContext_ *)context;
         newValue = oldValue + atomicAddContext->deltaValue;
         return true;
     }

     AtomicFactorizeWithLimit(ValueType oldValue, ValueType & newValue, void *context)
     {
         struct AtomicFactorizeWithLimitContext_ {
             ValueType factorValue;
             ValueType limitValue;
             ValueType oldValue;
         } *atomicFactorizeWithLimitContext = (struct AtomicFactorizeWithLimitContext_ *)context;

         atomicFactorizeWithLimitContext->oldValue = oldValue;
         newValue = oldValue * atomicFactorizeWithLimitContext->factorValue;
         return ( newValue < atomicFactorizeWithLimitContext->limitValue );
     }

  The function does the modification in the following way:

  void
  RTESys_AtomicOperationValueType( ValueType & memoryPosition,
                                   SAPDB_Bool (*operationFunction)(ValueType const, ValueType &, void *),
                                   void *operationContext )
  {
    ValueType oldValue = RTESys_AtomicRead(memoryPosition);
    do
    {
       ValueType expectedValue = oldValue;
       ValueType newValue;
       if ( !(*operationFunction)(oldValue, newValue, operationContext) )
           return;
    } while ( RTESys_CompareAndExchange(memoryPosition, expectedValue, newValue, oldValue) );
    return;
  }

   @param memoryPosition [inout] the memory position to modify
   @param operationFunction [in] the operation function
   @param operationContext [in] the operation context
   @return none
 */
inline void RTESys_AtomicOperation( SAPDB_Int4 & memoryPosition,
                                    SAPDB_Bool (*operationFunction)(SAPDB_Int4 const, SAPDB_Int4 &, void *),
                                    void *operationContext )
{ RTESys_AtomicOperationInt4(memoryPosition, (RTESys_AtomicOpInt4)operationFunction, operationContext); }
/*! @brief Atomic operation on given memory position
    @param memoryPosition [in] the memory position to modify
    @param operationFunction [in] the operation function
    @param operationContext [in] the operation context
    @return none */
inline void RTESys_AtomicOperation( SAPDB_Int8 & memoryPosition,
                                    SAPDB_Bool (*operationFunction)(SAPDB_Int8 const, SAPDB_Int8 &, void *),
                                    void *operationContext )
{ RTESys_AtomicOperationInt8(memoryPosition, (RTESys_AtomicOpInt8)operationFunction, operationContext); }
/*! @brief Atomic operation on given memory position
    @param memoryPosition [in] the memory position to modify
    @param operationFunction [in] the operation function
    @param operationContext [in] the operation context
    @return none */
inline void RTESys_AtomicOperation( SAPDB_UInt4 & memoryPosition,
                                    SAPDB_Bool (*operationFunction)(SAPDB_UInt4 const, SAPDB_UInt4 &, void *),
                                    void *operationContext )
{ RTESys_AtomicOperationUInt4(memoryPosition, (RTESys_AtomicOpUInt4)operationFunction, operationContext); }
/*! @brief Atomic operation on given memory position
    @param memoryPosition [in] the memory position to modify
    @param operationFunction [in] the operation function
    @param operationContext [in] the operation context
    @return none */
inline void RTESys_AtomicOperation( SAPDB_UInt8 & memoryPosition,
                                    SAPDB_Bool (*operationFunction)(SAPDB_UInt8 const, SAPDB_UInt8 &, void *),
                                    void *operationContext )
{ RTESys_AtomicOperationUInt8(memoryPosition, (RTESys_AtomicOpUInt8)operationFunction, operationContext); }
/*! @brief Atomic operation on given memory position
    @param pointerPosition [in] the memory position to modify pointer
    @param operationFunction [in] the operation function
    @param operationContext [in] the operation context
    @return none */
inline void RTESys_AtomicOperation( void * & pointerPosition,
                                    SAPDB_Bool (*operationFunction)( void * const, void * &, void *),
                                    void *operationContext )
{ RTESys_AtomicOperationPointer(pointerPosition, (RTESys_AtomicOpPointer)operationFunction, operationContext); }

#endif  /* RTESYS_ATOMICOPERATION_HPP */
