/*!
  @file           RTE_ITaskStackControl.hpp
  @author         JoergM
  @ingroup        Runtime
  @brief          UNIX task stack interface

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/
#ifndef RTE_ITASKSTACKCONTROL_HPP
#define RTE_ITASKSTACKCONTROL_HPP


/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "RunTime/RTE_Types.hpp"
#include "SAPDBCommon/SAPDB_Singleton.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!
    @class  RTE_ITaskStackControl
    @brief  Singleton Interface class to access and create task stacks

     This pure virtual class abstracts from task stack creation details
 */
class RTE_ITaskStackControl: public SAPDB_Singleton
{
public:
    static RTE_ITaskStackControl &Instance();

    /*!
      @brief Once called function to initialize task stack control
      @param numberOfTasks [in] number of tasks
             pFirst [in] pointer to first task control record
             pLast [in] pointer to last task control record
      @return none
     */
    virtual void Initialize( SAPDB_Long numberOfTasks, 
                             struct TASK_TYPE *pFirst,
                             struct TASK_TYPE *pLast) = 0;

    /*!
      @brief Using current stack context return task control block of calling task or 0 if not inside task stack
      @return Task control block pointer of calling task or 0 if not on task stack
    */
    virtual struct TASK_TYPE *GetTaskControl() const = 0;

    /*!
      @brief The tasks may have the stack on stack of UKT thread. In that case the mapping information
      is known at a very late moment. This call defines the interface used in such a case.
      The call allows to associate a large number of task stack with a single call. This is used in a
      user task thread, where lots of identical tasks are added.

      If stack address space is not used for task stack, this call is not implemented.

      @param firstStack [in] address of first stack to map
      @param singleStackSize [in] size of a single stack size in bytes
      @param firstId [in] the first task id associated with the first task stack
      @param numberOfTasksToAdd [in] the number of tasks added with identical task stack size
      @return true if task stack mapping was added, false if not
    */
    virtual SAPDB_Bool AddTaskStackMapping( SAPDB_Byte *firstStack,
                                            SAPDB_ULong singleStackSize,
                                            RTE_TaskId  firstId,
                                            SAPDB_UInt4 numberOfTasksToAdd ) = 0;
}; /*! @endclass RTE_ITaskStackControl */

#endif  /* RTE_ITASKSTACKCONTROL_HPP */
