/*!---------------------------------------------------------------------
  @file           RTE_DBRegister.hpp
  @author         JoergM, RobinW
  @brief          DBM Support: SAPDB Instance and Database Registration and Management Interfaces
  @see            

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
---------------------------------------------------------------------*/




/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "RunTime/RTE_Types.h"
#include "RunTime/MemoryManagement/RTEMem_Allocator.hpp"
#include "Container/Container_List.hpp"
#include "RunTime/RTE_Installation.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

 /*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/

typedef SAPDB_Char RTE_DBKey[256];

class RTE_DBRegister;
class RTE_Installation;
class RTE_ActiveInstallation;
class RTE_Database;
class RTE_ActiveDatabase;


typedef Container_List<RTE_SpeedInfo> RTE_SpeedList;


/*
  Class: RTE_DBRegister
  Description: Register of installations and databases

  Installations are handled as instances of the classes passive RTE_Installation and RTE_ActiveInstallation.
  A passive installation allows to collect information about an installation. An active installation 
  adds the ability to register/unregister databases.
 */
class RTE_DBRegister
{
public:
    static RTE_DBRegister &Instance();

    /*
      Function: RegisterInstallation
      Description: Register a given Installation

      Registration of a given installation means to store VersionInfo and path to DbRoot in the
      register. If another installation with the same DbRoot it is overwritten and a warning is
      reported.
     */
    virtual bool RegisterInstallation( const RTE_Path dbRoot, 
                                       const RTE_Installation::InstVersionInfo &version,
                                       SAPDBErr_MessageList &errList);
    /*
      Function: UnregisterInstallation
      Description: Unregister a given Installation

      The registration identified by DbRoot is removed from register. Only installations, that
      contain no databases are removable.
     */
    virtual bool UnregisterInstallation( const RTE_Path dbRoot,
                                         SAPDBErr_MessageList &errList);

    /*
      Function: GetInstallations
      Description: Get a container filled with the passive installation instance references

      The installations are only suitable to enumerate registered installations and databases.
      They are not usefull to add databases or start/stop them. Use GetActiveInstallation()
      or GetActiveDatabase() for these purposes.
     */
    virtual Container_List<RTE_Installation>* GetInstallations();

    /*
     Function: GetInstallation
     Description: Get an passive installation instance reference using either dbName or dbRoot

     The returned 
     Prototypes: 2
     */
    virtual RTE_Installation * GetInstallationByDBName( const RTE_DBName dbName,
                                                        SAPDBErr_MessageList &errList);
    virtual RTE_Installation * GetInstallationByDBRoot( const RTE_Path   dbRoot,
                                                        SAPDBErr_MessageList &errList);

    /*
     Function: GetActiveInstallation
     Description: Get an active installation instance reference using either dbName or dbRoot

     The returned 
     Prototypes: 2
     */
    virtual RTE_ActiveInstallation * GetActiveInstallationByDBName( const RTE_DBName dbName,
                                                                    SAPDBErr_MessageList &errList);
    virtual RTE_ActiveInstallation * GetActiveInstallationByDBRoot( const RTE_Path   dbRoot,
                                                                    SAPDBErr_MessageList &errList);

    RTE_Database * GetDatabase( const RTE_DBName dbName,
                                SAPDBErr_MessageList &errList);

    RTE_ActiveDatabase * GetActiveDatabase( const RTE_DBName dbName,
                                            SAPDBErr_MessageList &errList);

private:
    static RTE_DBRegister *m_Instance;
    RTE_DBRegister(){};
    RTE_DBRegister(const RTE_DBRegister &) {}
    SAPDB_Bool getInstVersionFromRegistry(
        RTE_Path instroot,
        RTE_Installation::InstVersionInfo &instversion,
        SAPDB_Int1 dbKind,
        SAPDBErr_MessageList &err);
#ifdef _WIN32
    SAPDB_Bool openInstRegistry (
        HKEY &keyHandle);
    void replaceCharsInString (
        SAPDB_Char* string, 
        SAPDB_Char fromChar, 
        SAPDB_Char toChar);
#else
    void parseVersion (
        const char * versionString,
        RTE_Installation::InstVersionInfo & versionInfo);
#endif

}; /* Endclass: RTE_DBRegister */

/*===========================================================================*
 *  STATIC/INLINE FUNCTION PROTOTYPES                                        *
 *===========================================================================*/


/*===========================================================================*
 *  METHODS                                                                  *
 *===========================================================================*/
#include "gsp09.h"

void RTE_FillXErrorFromErrList(tsp9_rte_xerror* xerror,SAPDBErr_MessageList &errList);

void RTE_FillErrListFromXError(SAPDBErr_MessageList &errList,tsp9_rte_xerror* xerror);

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/
