/*!
 * \file    OMS_LibOmsInterface.cpp
 * \author  IvanS, MarkusSi, PeterG
 * \brief   OMS LibOmsInterface.
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/


#include "Oms/OMS_LibOmsInterfaceInstance.hpp"
#include "Oms/OMS_Globals.hpp"
#include "Oms/OMS_ContainerDictionary.hpp"
#include "Oms/OMS_ContainerInfo.hpp"
#include "Oms/OMS_SinkCriticalSection.hpp"
#include "Oms/OMS_ObjectContainer.hpp"
#include "Oms/OMS_VersionDictionary.hpp"
#include "Oms/OMS_ContainerDictionary.hpp"
#include "Oms/OMS_MonitorDirectory.hpp"
#include "Oms/OMS_LockEntryHash.hpp"
#include "Oms/OMS_Session.hpp"
#include "liveCache/LVC_LockRequest.hpp"
#include "hsp77.h"
#include "hsp78_0.h"

/*---------------------------------------------------------------------------*/
/*  external Globals                                                         */
/*---------------------------------------------------------------------------*/

externCpp bool omsInUnicodeMode = false;

const char* VAR_OBJ_CLASS_NAME_CO10 = "VarObjContainer";

/*---------------------------------------------------------------------------*/
/*                              Globals                                      */
/*---------------------------------------------------------------------------*/

static bool                    m_mallocMonitor  = false;  // PTS 1110287

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::AdviseKernelInterface(LVC_KernelInterface& knlInterface, bool isKernel)
{
    OMS_Globals::KernelInterfaceInstance = &knlInterface;
    if (!OMS_Globals::m_globalsInstance) 
      OMS_Globals::InitSingletons();
    OMS_Globals::m_globalsInstance->SetHost(isKernel);    
    omsInUnicodeMode = OMS_Globals::KernelInterfaceInstance->IsUnicodeInstance();

    for (int i=0; i<OMS_VDIR_SIZE; ++i){
      // Get single lock for every hash-array   // PTS 1124533
      LVC_LockRequest lock(LVC_LockRequest::RWLOCK_CREATE, i + OMS_Globals::m_globalsInstance->m_versionDictionary.GetVDirStartCnt());
      short err = knlInterface.LockRequest(lock);
    }

    // PTS 1124533
    OMS_Globals::m_globalsInstance->m_versionDictionary.m_useRWLocks = (knlInterface.UseReaderWriterLocks() ? true : false);
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::InitLibOms(IliveCacheSink* lcSink) 
{
  if (!OMS_Globals::m_globalsInstance) 
    OMS_Globals::InitSingletons();
  struct VirtualObject {
    void* vtptr;
  }; 
  OMS_VarObjInfo info;
  OMS_Globals::m_globalsInstance->m_classDictionary.RegisterClass (lcSink, VAR_OBJ_CLASS_NAME_CO10, OMS_VAR_OBJ_GUID, NULL,
    0, 0, false, // key-description
    sizeof(info) + OmsObjectContainer::headerSize(), (REINTERPRET_CAST(VirtualObject*, &info))->vtptr);
  OMS_Globals::m_globalsInstance->SetOmsVersionThreshold(OMS_Globals::KernelInterfaceInstance->GetOmsVersionThreshold());
  OMS_LockEntryHash::m_instance.SetTimeout(OMS_Globals::KernelInterfaceInstance->GetLockRequestTimeout());
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::AdviseTracer(OMS_TraceInterface* pTraceObj)
{
  if (!OMS_Globals::m_globalsInstance) 
    OMS_Globals::InitSingletons();
  OMS_Globals::m_globalsInstance->SetTracer(pTraceObj);
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::Dump(OMS_DumpInterface& dumpObj)
{
  OMS_Globals::m_globalsInstance->Dump(dumpObj);
}

/*----------------------------------------------------------------------*/

tsp00_Bool OMS_LibOmsInterfaceInstance::GetFirstVersion(IliveCacheSink*     lcSink,
                                                  tsp00_C24&          versionId,
                                                  tsp00_Date&         createDate,
                                                  tsp00_Time&         createTime,
                                                  tsp00_Date&         openDate,
                                                  tsp00_Time&         openTime,
                                                  tgg91_TransNo&      consistentView,
                                                  tsp00_Bool&         isMarked,
                                                  tsp00_Bool&         isOpen,
                                                  tsp00_Bool&         isUnloaded,
                                                  tsp00_8ByteCounter& heapUsage,
                                                  tsp00_Int4&         hashSize,
                                                  tgg00_FileId&       versionTree,
                                                  tgg00_FileId&       versionInvTree,
                                                  tsp00_C512&         versionDesc,
                                                  short&              error)  // PTS 1126697
{
  try {
    // Critical section is used to synchronize the access to the global iterator
    // structure. It is NOT used to synchronize access to the version dictionary.
    lcSink->EnterCriticalSection(RGN_VERSION_DIR);

    // Reset iterator to the first element
    OMS_Globals::m_globalsInstance->m_versionIter.first(OMS_LOCK_EXCLUSIVE);
  } catch (DbpError &e){  // PTS 1126697
    DbpErrorCallbackInterface *pCBInterface = DbpError::dbpGetCallback();  // PTS 1127871
    if (pCBInterface){
      pCBInterface->dbpCaughtError(e);
    }

    OMS_Globals::m_globalsInstance->m_versionIter.stop();
    lcSink->LeaveCriticalSection(RGN_VERSION_DIR);
    error = e.m_errorNo;
    return false;
  }

  return GetNextVersion(lcSink, versionId, createDate, createTime, 
    openDate, openTime, consistentView, isMarked, isOpen, isUnloaded,
    heapUsage, hashSize, versionTree, versionInvTree, versionDesc, error);
}

/*----------------------------------------------------------------------*/

tsp00_Bool OMS_LibOmsInterfaceInstance::GetMonitorInfo(
                                                 IliveCacheSink*       lcSink,
                                                 tsp00_Addr&           handle,
                                                 tsp00_C16&            iid,
                                                 tsp00_Int4&           dispid,
                                                 tgg01_COMMonitorInfo& monInfo)
{
  static OMS_MonitorDirectory::Iterator Iter;
  if (NULL == handle) {
    lcSink->EnterCriticalSection(RGN_MONITOR);
    Iter = OMS_Globals::m_globalsInstance->m_monitorDirectory.begin();
    handle = REINTERPRET_CAST(tsp00_Addr, &Iter);
  }
  if (Iter) {
    Iter.Get(*REINTERPRET_CAST(CLSID*, &iid), dispid, monInfo);
    ++Iter;
    return true;
  }
  else {
    lcSink->LeaveCriticalSection(RGN_MONITOR);
  }
  return false;
}

/*----------------------------------------------------------------------*/

tsp00_Bool OMS_LibOmsInterfaceInstance::GetNextVersion(IliveCacheSink*     lcSink,
                                                 tsp00_C24&          versionId,
                                                 tsp00_Date&         createDate,
                                                 tsp00_Time&         createTime,
                                                 tsp00_Date&         openDate,
                                                 tsp00_Time&         openTime,
                                                 tgg91_TransNo&      consistentView,
                                                 tsp00_Bool&         isMarked,
                                                 tsp00_Bool&         isOpen,
                                                 tsp00_Bool&         isUnloaded,
                                                 tsp00_8ByteCounter& heapUsage,
                                                 tsp00_Int4&         hashSize,
                                                 tgg00_FileId&       versionTree,
                                                 tgg00_FileId&       versionInvTree,
                                                 tsp00_C512&         versionDesc, 
                                                 short&              error)  // PTS 1126697 
{
  try {
  error = e_ok;
  while (true) {
    if (OMS_Globals::m_globalsInstance->m_versionIter) {
      OMS_Context* pContext = OMS_Globals::m_globalsInstance->m_versionIter();
      if (pContext->IsDropped()) {
        // version is marked as dropped, try to drop it now
        //++OMS_Globals::m_globalsInstance->m_versionIter;   // PTS 1126697
        if (!pContext->IsBoundToTrans()) {
          // version no longer bound to transaction, i.e. can be dropped now
          pContext->AssignLcSink(lcSink);  /* PTS 1107209 */
          OMS_Globals::m_globalsInstance->m_versionDictionary.MarkNotUnloadable(lcSink, pContext);
          OMS_Globals::m_globalsInstance->m_versionIter.getNext(true /*dropCurr*/);
          pContext->DeleteSelf();
        }
        else {                                                // PTS 1126697
          ++OMS_Globals::m_globalsInstance->m_versionIter;
        }
      }
      else {
        memset (&versionId[0], ' ', sizeof(versionId));
        memcpy(&versionId[0], &pContext->GetVersionId()[0], sizeof(OmsVersionId)); 
        createDate     = pContext->GetCreateDate();
        createTime     = pContext->GetCreateTime();
        openDate       = pContext->GetOpenDate();
        openTime       = pContext->GetOpenTime();
        consistentView = pContext->m_consistentView;
        isMarked       = pContext->IsMarked();
        isOpen         = pContext->IsVersionOpen();
        isUnloaded     = pContext->IsUnloaded();
        heapUsage      = pContext->HeapUsed();
        hashSize       = pContext->GetHashSize();
        versionTree    = pContext->VersionFile();
        versionInvTree = pContext->VersionInvFile();
        
        // PTS 1117690
        // Initialize version-desciption
        if (omsInUnicodeMode){
          // initialize with blanks in kernel layout (big endian)
          for (int ix = 0; ix < sizeof(versionDesc); ix += 2)
          {
            versionDesc[ix  ] = 0;
            versionDesc[ix+1] = ' ';
          }
        }
        else {
          memset(versionDesc, ' ', sizeof(versionDesc));
        }
        
        if (pContext->GetVersionDesc() != NULL){
          int len = OMS_Globals::WideStrlen(pContext->GetVersionDesc());
          
          if (omsInUnicodeMode){
            // Source and Target in UCS2 => No Convertion needed but output string could be
            // too long
            int len_byte = (len * sizeof(OmsTypeWyde) > sizeof(versionDesc)
                         ? sizeof(versionDesc)
                         : len * sizeof(OmsTypeWyde));

            memcpy(versionDesc, pContext->GetVersionDesc(), len_byte); 
            
            // check if endianness of maschine is the same as the endianness of the kernel
            if (sp77encodingUCS2Native != sp77encodingUCS2){
              // translate to kernel layout
              for (int ix = 0; ix < len_byte; ix += 2){
                // swap
                char aux = versionDesc[ix];
                versionDesc[ix  ] = versionDesc[ix+1];
                versionDesc[ix+1] = aux;
              }
            }
          }
          else {
            tsp00_Uint4 destBytesWritten;
            tsp00_Uint4 srcBytesParsed;
            tsp78ConversionResult error;

            // Convertion from UCS2 to ASCII
            error = sp78convertString(sp77encodingAscii,              // destEndcoding
                                      versionDesc,                    // destBuffer
                                      sizeof(versionDesc),            // destBufferLengthInBytes
                                      &destBytesWritten,              // destBytesWritten
                                      false,                          // addZeroTerminator
                                      sp77encodingUCS2Native,         // srcEncoding
                                      pContext->GetVersionDesc(),     // srcBuffer
                                      len * sizeof(OmsTypeWyde),      // srcBufferLengthInBytes
                                      &srcBytesParsed);               // srcBytesParsed
          }
        }
        
        ++OMS_Globals::m_globalsInstance->m_versionIter;
        return true;
      }
    }
    else {
      lcSink->LeaveCriticalSection(RGN_VERSION_DIR);
      return false;
    }
  }
  } catch (DbpError &e){  // PTS 1126697
    DbpErrorCallbackInterface *pCBInterface = DbpError::dbpGetCallback();  // PTS 1127871
    if (pCBInterface){
      pCBInterface->dbpCaughtError(e);
    }

    OMS_Globals::m_globalsInstance->m_versionIter.stop();
    lcSink->LeaveCriticalSection(RGN_VERSION_DIR);
    error = e.m_errorNo;
    return false;
  }
}

/*----------------------------------------------------------------------*/

bool OMS_LibOmsInterfaceInstance::GetFirstClassInfo(OMS_LibOmsInterfaceClassInfo &info)
{
  OMS_Globals::m_globalsInstance->ContainerIter.reset(&OMS_Globals::m_globalsInstance->m_classDictionary);
  return GetNextClassInfo(info);
}

/*----------------------------------------------------------------------*/

bool OMS_LibOmsInterfaceInstance::GetNextClassInfo(OMS_LibOmsInterfaceClassInfo &info)
{
  if (OMS_Globals::m_globalsInstance->ContainerIter) {
    OMS_ContainerInfo *i = OMS_Globals::m_globalsInstance->ContainerIter();
    info.handle = i->GetContainerHandle();
    info.schema = i->GetSchema();
    info.container = i->m_ContainerNo;
	GUID_TEMP(lguid, i->GetGuid());
    info.guid = lguid;
    info.keyLen = i->GetKeyDesc().GetLen();  
    info.persSize = i->GetObjectSize();
    const OMS_ClassInfo *ci = i->GetClassInfoPtr();
    info.name = ci->GetClassName();
    info.isVarObject = ci->IsVarObject();
    info.isBaseClass = ci->IsBaseClass();
    info.isDerivedClass = ci->IsDerivedClass();
    info.isDropped = i->IsDropped();
    ++(OMS_Globals::m_globalsInstance->ContainerIter);
    return true;
  } else {
    return false;
  }
}

/*----------------------------------------------------------------------*/

bool OMS_LibOmsInterfaceInstance::IsMonitorOn() const 
{
  return m_mallocMonitor;
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::InspectTimeout() // PTS 1110287
{
  OMS_LockEntryHash::m_instance.InspectTimeout();
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::ResetMonitor(IliveCacheSink* lcSink)
{
  OMS_SinkCriticalSection cs(lcSink, RGN_MONITOR);
  cs.Enter();
  OMS_Globals::m_globalsInstance->m_monitorDirectory.Reset();
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::SetDumpError(tsp00_Int4 errorNo)
{
  OMS_Globals::m_globalsInstance->SetDumpError(errorNo);
}

/*----------------------------------------------------------------------*/
// PTS 1110287
bool OMS_LibOmsInterfaceInstance::NextOmsLockObjInfo(void** pVoidLockInfo, tgg01_OmsLockInfo& lockInfo)
{
  return OMS_LockEntryHash::m_instance.NextLockInfo(pVoidLockInfo, lockInfo);
}

/*----------------------------------------------------------------------*/

bool OMS_LibOmsInterfaceInstance::SetTraceLevel(tsp00_KnlIdentifier& lvl, bool enable)
{
  return TraceLevel_co102.Set(lvl, enable);
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::StartStopMonitor(bool doStart) {
  m_mallocMonitor = doStart;
}

/*----------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::UnAdviseKernelInterface()
{
    OMS_Globals::KernelInterfaceInstance = NULL;
}

/*----------------------------------------------------------------------*/

bool OMS_LibOmsInterfaceInstance::VersionUnloaded()
{
  return OMS_Globals::m_globalsInstance->m_versionDictionary.UnloadOldestVersion(OMS_Globals::GetCurrentLcSink());;
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictInsert(
  const OmsVersionId &versionId, 
  const OMS_Context  *pVersionContext,
  tgg00_BasisError   &error)
{
  error = OMS_Globals::m_globalsInstance->m_versionDictionary.InsertVersion(versionId, 
    const_cast<OMS_Context*>(pVersionContext));
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictFind(
  const OmsVersionId  &versionId,
  OMS_Context        **ppVersionContext)
{
 *ppVersionContext = OMS_Globals::m_globalsInstance->m_versionDictionary.FindVersion(versionId);
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictDrop(
  const OmsVersionId &versionId, 
  tgg00_BasisError   &error)
{
  error = OMS_Globals::m_globalsInstance->m_versionDictionary.DropVersion(versionId);
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictShutdown()
{
  OMS_Globals::m_globalsInstance->m_versionDictionary.Shutdown();
}

/*-------------------------------------------------------------------------*/

bool OMS_LibOmsInterfaceInstance::VersionDictUnloadOldestVersion()
{
  IliveCacheSink *pSink = OMS_Globals::m_globalsInstance->GetCurrentLcSink();
  return OMS_Globals::m_globalsInstance->m_versionDictionary.UnloadOldestVersion(pSink);
}


/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictMarkNotUnloadable(
  OMS_Context    *pVersionContext)
{
  IliveCacheSink *pSink = OMS_Globals::m_globalsInstance->GetCurrentLcSink();
  OMS_Globals::m_globalsInstance->m_versionDictionary.MarkNotUnloadable(pSink, pVersionContext);
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictMarkNotUnloadable(
  OMS_Context  *pVersionContext,
  bool          callFromDestructor)
{
  OMS_Globals::m_globalsInstance->m_versionDictionary.MarkUnloadable(pVersionContext, callFromDestructor);
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictCreateIter(
  OMS_LockMode   lockMode, 
  void         **ppIter, 
  OMS_Context  **ppContext)
{
  *ppIter = OMS_Globals::m_sharedMemAllocatorInstance.allocate(sizeof(OMS_VersionDictionary::Iter));
  OMS_VersionDictionary::Iter* pIter = reinterpret_cast<OMS_VersionDictionary::Iter*>(*ppIter);
  
  pIter->first(lockMode);

  if (*pIter){
    *ppContext = (*pIter)();
  }
  else {
    // Iterator is empty
    OMS_Globals::m_sharedMemAllocatorInstance.deallocate(*ppIter);
    *ppIter    = NULL;
    *ppContext = NULL;
  }
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictGetNext(
  void         *p, 
  bool          dropCurr,
  OMS_Context **ppContext)
{
  OMS_VersionDictionary::Iter* pIter = reinterpret_cast<OMS_VersionDictionary::Iter*>(p);

  pIter->getNext(dropCurr);
  if (*pIter){
    *ppContext = (*pIter)();
  }
  else {
    // Iterator is empty
    OMS_Globals::m_sharedMemAllocatorInstance.deallocate(p);
    *ppContext = NULL;
  }
}

/*-------------------------------------------------------------------------*/

void OMS_LibOmsInterfaceInstance::VersionDictDestructIter(
  void  *p) 
{
  if (p == NULL)
    return;

  OMS_VersionDictionary::Iter* pIter = reinterpret_cast<OMS_VersionDictionary::Iter*>(p);
  if (*pIter){
    // Ensure that all locks are freed
    pIter->stop();
  }
    
  // Iterator is empty
  OMS_Globals::m_sharedMemAllocatorInstance.deallocate(pIter);
}

/*-------------------------------------------------------------------------*/
