/*!
  @file           Kernel_IPage.hpp
  @author         DataAccessGroup
  @ingroup        KernelPage
  @brief          Common page handler interface

\if EMIT_LICENCE
  ========== licence begin  GPL
  Copyright (c) 2000-2004 SAP AG

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  ========== licence end
\endif
*/
#ifndef KERNEL_IPAGE_HPP
#define KERNEL_IPAGE_HPP

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDBCommon/Fields/SAPDBFields_Field.hpp"

/*!
@interface      Kernel_IPage
@brief          Interface class for all page handler classes in the kernel.
*/
class Kernel_IPage
{

public:
	/// Memory frame which contains the page
    typedef SAPDBFields_Field        PageFrame;
	/// Pointer to memory block containing the proper page
    typedef PageFrame::ConstPointer  ConstPageFramePtr;
	/// Pointer to memory block containing the proper page
    typedef PageFrame::Pointer       PageFramePtr;
	/// size of memory block on a page in bytes
    typedef SAPDB_UInt2              SizeType;
	/// offset on a page in bytes
    typedef SizeType                 OffsetType;
    /*!
    @brief          Prepares page for input from device
    @return         (SAPDB_Bool) true if check successful

    Performs parity check on the frame contents. 
    Recalculates parity from frame contents and compares the result
    with parity bytes kept in page header and trailer
    Applicable immediately after a page is read from device.
    */
    virtual SAPDB_Bool CheckAfterRead () const = 0;
	/// Returns the pointer to the proper page frame
    virtual PageFramePtr GetFramePtr() = 0;
    /*!
    @brief          Prepares page for output to device
    @return         (SAPDB_Bool) true if check successful

     Reads and check parity information of frame contents
     Updates parity information with actual frame contents
     Applicable immediately before a page is written to device
     */
    virtual SAPDB_Bool PrepareForWrite() = 0;
    /*!
    @brief          Verifies consistency of page contents
    @return         (SAPDB_Bool) true if verify successful
    */
    virtual SAPDB_Bool Verify() const = 0;
};
#endif // KERNEL_IPAGE_HPP
