/*!
 * @file  Join_IndexAccessOperator.cpp
 * @brief implements Join_IndexAccessOperator
 *
 * @author GertG
 * @ingroup Join
 *
 * @par last changed by:
 * <br>
 * $Author: d024980 $ $DateTime: 2004/05/03 16:16:26 $
 *
 * @sa Join_IndexAccessOperator.hpp
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
#include "Join/Join_IndexAccessOperator.hpp"
#include "Join/Join_AccessOperator.hpp"
#include "Join/Join_InvSelectIterator.hpp"
#include "Trace/Trace_Entry.hpp"
#include "SAPDBCommon/SAPDB_Types.hpp"

#include "ggg00.h"
#include "hak07.h"
#include "gkb07.h"
#include "ggg07.h"
#include "hbd01.h"
#include "hbd03.h"
#include "hta01.h"
#include "hta01_3.h"

#include <stdlib.h>

/* ******************** PUBLIC MEMBERS ********************* */
/*!
 * @param acv [in] global context
 * @param buffersize [in] size of table buffer
 */
Join_IndexAccessOperator::Join_IndexAccessOperator(
        tak_all_command_glob& acv,
        const AccessKind   accessKind,
        SAPDB_Byte*        bufferPtr,
        const SAPDB_UInt4& buffersize)
    :
    Join_AccessOperator(acv, bufferPtr, buffersize),
    m_pInvIterator(0)
{
    SAPDBTRACE_METHOD_DEBUG( "Join_IndexAccessOperator::ctor", Join_Trace, 1 );

    if ( 0 != m_acv.a_returncode )
        return;

    if  ( (ParallelAccess == accessKind)
          && Join_InvSelectIterator::ParallelEnabled() )
    {
        // try to create an inv list iterator, which is able to handle
        // the inv list in parallel
        m_pInvIterator = new(m_allocator)
            Join_InvSelectIterator (
                *m_AccessDesc.MBlock().mb_trns(),
                *reinterpret_cast<SQLMan_MessBlock*>(&m_AccessDesc.MBlock()),
                m_AccessDesc.FileIDs().inv_id,
                m_SelFields, m_AccessDesc.MBlock().mb_qual()->mstack_desc());
        // it doesn't matter if iterator could not be created. In this
        // case 'normal' execution via b03select_invrec takes place
    }

    if ( e_ok != m_AccessDesc.MBlock().mb_trns()->trError_gg00 )
        a07_b_put_error(
            acv,
            m_AccessDesc.MBlock().mb_trns()->trError_gg00,
            1 );
}

Join_IndexAccessOperator::~Join_IndexAccessOperator()
{
    SAPDBTRACE_METHOD_DEBUG(
        "Join_IndexAccessOperator::~Join_IndexAccessOperator",
        Join_Trace, 1 );

    destroy( m_pInvIterator, m_allocator );
}

void Join_IndexAccessOperator::Open()
{
    SAPDBERR_ASSERT_STATE( 0 == m_pInvIterator );
    Join_AccessOperator::Open();
}

/*!
 * @param startkeys [in] restrict record stream to records with key greater/equal than startkey
 * @param stopkeys [in] restrict record stream to records with key lower/equal than stopkey
 */
void Join_IndexAccessOperator::Open(
            const Join_TwoKeys& startkeys,
            const Join_TwoKeys& stopkeys )
{
    SAPDBTRACE_METHOD_DEBUG(
        "Join_IndexAccessOperator::Open(startkeys,stopkeys)", Join_Trace, 1 );

    Join_AccessOperator::Open( startkeys, stopkeys );

    if ( m_pInvIterator )
    {
         m_pInvIterator->Open(
             m_Startkeys.listkey,
             m_Startkeys.reckey,
             m_Stopkeys.reckey,
             m_RangeSet );
    }
}

/*!
 * @param record [in/out] pointer to memory should be filled with record
 * @return information code <tt>[e_ok, e_no_next_record]</tt> / error code
 */
void Join_IndexAccessOperator::SelectRecords()
{
    SAPDBTRACE_METHOD_DEBUG(
        "Join_IndexAccessOperator::SelectRecords", Join_Trace, 1 );

    tgg00_LockReqMode _granted_lock;

    SAPDBTRACE_IF(
        Join_Trace, 5,
        t01surrogate( td_always, "sel INVID   ",
                      m_AccessDesc.FileIDs().inv_id.fileTabId_gg00()));
    SAPDBTRACE_IF(
        Join_Trace, 5,
        t01surrogate( td_always, "sel TABID   ",
                      m_AccessDesc.FileIDs().file_id.fileTabId_gg00()));

    m_SelFields.sfp_m_result_cnt() = m_maxRowRequest; // set buffer count

    if ( m_pInvIterator )
    {
        m_AccessDesc.MBlock().mb_trns()->trError_gg00 =
            m_pInvIterator->NextRow(0);

        if (e_ok == m_AccessDesc.MBlock().mb_trns()->trError_gg00)
        {
            m_SelFields.sfp_m_result_cnt() = 1;
        }
        else
        {
            if (e_inv_list_not_found
                == m_AccessDesc.MBlock().mb_trns()->trError_gg00)
            {
                m_AccessDesc.MBlock().mb_trns()->trError_gg00 =
                    e_no_next_record;
            }
            m_SelFields.sfp_m_result_cnt() = 0;
        }
    }
    else
    {
        b03select_invrec(
            *m_AccessDesc.MBlock().mb_trns(),
            m_AccessDesc.FileIDs(),
            m_Actualkeys,
            m_Stopkeys,
            m_Actualkeys.reckey,
            m_RangeSet,
            0 /*recsize*/, 0 /*recptr*/,
            m_SelFields,
            m_AccessDesc.MBlock().mb_qual()->mstack_desc(),
            _granted_lock,
            true );
    }
}

/*!
 *
 */
void Join_IndexAccessOperator::Close()
{
    SAPDBTRACE_METHOD_DEBUG(
        "Join_IndexAccessOperator::Close", Join_Trace, 1 );

    Join_AccessOperator::Close();

    if ( m_pInvIterator )
    {
        m_pInvIterator->Close();
    }
}

/* ******************* PROTECTED MEMBERS ******************* */


/* ******************** PRIVATE MEMBERS ******************** */
/*!
 *
 */
tgg00_BasisError Join_IndexAccessOperator::init()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_IndexAccessOperator::init", Join_Trace, 3 );

    tgg00_BasisError _b_err = Join_AccessOperator::init();

    // look k75_fetch for mm_first, mm_next
    if ( m_Stopkeys.listkey.k()[0] != csp_undef_byte )
        m_RangeSet.addElement( secondary_stop );
    else if ( m_Stopkeys.listkey.len() > 1
              && m_Stopkeys.listkey.k()[1] != csp_undef_byte )
        m_RangeSet.addElement( secondary_stop );

    b01vstate_fileversion( *m_AccessDesc.MBlock().mb_trns(),
                           m_AccessDesc.FileIDs().file_id );
    _b_err = m_AccessDesc.MBlock().mb_trns()->trError_gg00;

    return _b_err;
}
