/*!
 * @file
 * @brief Realizes pascal stub a686_create_join_result().
 *
 *  Realizes pascal/C++ connector and creates join result by means
 * of Join_JoinOperator.
 *
 * @author GertG
 * @ingroup Join
 *
 * @par last changed by:
 * <br>
 * $Author: d024980 $ $DateTime: 2004/05/03 16:16:26 $
 */
/*!
 * @addtogroup SQLManager
*/
/*!
 * @defgroup Join Join Execution
 * @ingroup SQLManager
*/

/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "SAPDB/SAPDBCommon/ErrorsAndMessages/SAPDBErr_Assertions.hpp"
#include "RunTime/RTE_Crash.hpp"
#include "SAPDB/SAPDBCommon/SAPDB_AutoPtr.hpp"
#include "SAPDB/SAPDBCommon/MemoryManagement/SAPDBMem_NewDestroy.hpp"
#include "SAPDB/Join/Join_JoinOperator.hpp"
#include "SAPDB/Join/Join_IResultSetOperator.hpp"
#include "SAPDB/Join/Join_ResultSetInserter.hpp"
#include "SAPDB/Join/Join_ResultSetHashInserter.hpp"
#include "SAPDB/Join/Join_ResultSetAppenderEx.hpp"
#include "SAPDB/Join/Join_ResultSetAppender.hpp"
#include "SAPDB/Join/Join_ResultSetOneResult.hpp"
#include "SAPDB/Join/Join_ResultSetDistinctOneResult.hpp"
#include "SAPDB/Join/Join_ResultSetDistinct.hpp"
#include "SAPDB/Join/Join_ResultSetAggregator.hpp"
#include "SAPDB/Join/Join_ResultSetAggregatorOneGroup.hpp"
#include "gsp03_1.h"
#include "ggg00.h"
#include "ggg07.h"
#include "ggg04.h"
#include "vak001.h"
#include "gak68.h"
#include "hak682.h"
#include "hak686.h"
#include "hta01.h"
#include "heo51.h" // vdebug_break()
#include "hkb720.h"
#include "hkb05.h"
#include "hak07.h"
#include "hgg01_1.h"
#include "hbd01.h"
#include "hsp41.h"
#include "hak680.h"
#include "hak502.h"

//! shortcut to access global allocator
#define ACV_ALLOCPTR(acv) ((SAPDBMem_IRawAllocator*) acv.a_transinf.tri_trans.trAllocator_gg00)

//! \if false
namespace {
    
inline    
Join_IResultSetOperator* 
    CreateResultSetOperator( 
            tak_all_command_glob&   acv,
            tak_dml_info&           dmli,
            const tgg00_StackList * const   rec_desc,
            tgg00_FileId&           res_tree, 
            const SAPDB_Int4&       rescnt, 
            const SAPDB_Int4&       rescnt_setop, 
            const SAPDB_Int4&       maxresult, 
            const pasbool           stamp_record_counter,
            Join_JoinOperator&      top_operator );

inline
void
    GetStrategyInfo( 
            tak_all_command_glob&   acv,
            tgg00_FileId&           result_id,
            SAPDB_Int4&             rescnt,
            SAPDB_Int4&             union_offs,
            pasbool&                stamp_record_counter);

inline
void
    GetResultDescription(
            tak_all_command_glob&   acv,
            tgg00_StackList*&       res_rec_desc);

} // anonymous namespace
//! \endif


//! routine which generates result of a join
/*!
 * @param acv [in] global context
 * @param dmli [in] statement context
 * @param sequence [in] pointer to join table sequence array
 * @param parsk [in] parskey to be used
 * @param jvrec [in] join view record
 * @param use_old_rescnt [in] delete 'CreateFile' flag of result file?
 * @param del_parsinfos [in] delete parse infos?
 * @param b_err [out] information code <tt>[e_ok, e_no_next_record]</tt> / error code
 */
externPascal void a686_create_join_result (
    tak_all_command_glob  VAR_VALUE_REF  acv,
    tak_dml_info          VAR_VALUE_REF  dmli,
    const tak68_one_seq_join*            sequence,
    const tak_parskey     VAR_VALUE_REF  parsk,
    const tak68_joinview_rec  VAR_VALUE_REF  jvrec,
    const pasbool                        use_old_rescnt,
    const pasbool                        del_parsinfos,
    tgg00_BasisError      VAR_VALUE_REF  b_err )
{
    SAPDBTRACE_ROUTINE_DEBUG( "a686_create_join_result", Join_Trace, 1 );
#ifdef SAPDB_SLOW
    SAPDB_ULong _usedbytes1, _maxusedbytes, _ctrlbyted1, _usedbytes2, _ctrlbyted2;
    ACV_ALLOCPTR(acv)->CalcStatistics(_usedbytes1, _maxusedbytes, _ctrlbyted1);
#endif
    b_err = e_ok;

    //get max result
    SAPDB_Int4 _maxresult;
    k720_maxresult_get( acv.a_mblock.mb_data()->mbp_buf(), dmli.d_rowno, _maxresult, b_err );
    if  ( _maxresult <= 1 ) 
    {   
        if (( cgg04_subquery_one_record == dmli.d_rowno || 
             cgg04_at_least_one_record == dmli.d_rowno ) &&
            ! dmli.d_pseudo_ins_select /* insert value = <select>*/ )
            // EXISTS clause, don't warn about exceeding ROWNO record count
            _maxresult = -1;
        else
            _maxresult = 1;
        SAPDBTRACE_WRITELN( Join_Trace, 3, "special maxresult : " << _maxresult );
    }
    if  ( e_ok != b_err ) a07_b_put_error( acv, b_err, 1 );

    //get messblock of last sequenced table into acv
    a682j_one_record( acv, dmli, dmli.d_cntfromtab, 
            (sequence+dmli.d_cntfromtab-1)->jos_source, 
            parsk, jvrec, use_old_rescnt, del_parsinfos );

    tgg00_FileId _result_id;
    SAPDB_Bool _restree_created = false; 
    pasbool _stamp_record_counter = false;
    SAPDB_Int4 _rescnt, _rescnt_start, _rescnt_start_setop;
    tgg00_StackList *_res_rec_desc = 0;

    GetStrategyInfo( acv, _result_id, _rescnt_start, 
                _rescnt_start_setop, _stamp_record_counter );
    GetResultDescription( acv, _res_rec_desc );

    if ( 0 == acv.a_returncode )
    {
        // create result file
        if ( ! dmli.d_single && _result_id.fileHandling_gg00().includes( hsCreateFile_egg00 ) )
        {
            b01tcreate_file( acv.a_transinf.tri_trans, _result_id );
            if ( e_ok != acv.a_transinf.tri_trans.trError_gg00 )
                a07_b_put_error( acv, acv.a_transinf.tri_trans.trError_gg00, 1 );
            else
                _restree_created = true;
        }

        //initialize top join operator
        Join_JoinOperator _TopJoinOp( acv, dmli, parsk, jvrec, 
                use_old_rescnt, del_parsinfos, sequence, 
                dmli.d_cntfromtab );

        if ( 0 == acv.a_returncode )
        {
            tgg00_Rec* _record;
            SAPDB_AutoPtr<Join_IResultSetOperator> ResultInserter( *ACV_ALLOCPTR(acv) );

            ResultInserter.SetPtr( 
                    CreateResultSetOperator(acv, dmli, _res_rec_desc, 
                        _result_id, _rescnt_start, _rescnt_start_setop, 
                        _maxresult, _stamp_record_counter, _TopJoinOp));
            ResultInserter->Open();
            if ( 0 == acv.a_returncode )
            {
                _TopJoinOp.Open();
                while ( e_ok == b_err &&
                      e_ok == ( b_err = _TopJoinOp.Next( _record ) ) && 
                        0 == acv.a_returncode )
                {
                    b_err = ResultInserter->Add( *_record );
                }
                _TopJoinOp.Close();

            }
            ResultInserter->Close();
            _rescnt = ResultInserter->ResultCount();
            SAPDBTRACE_WRITELN( Join_Trace, 3, "ResultCount(): " << _rescnt );
        }
    }
    SAPDBTRACE_WRITELN( Join_Trace, 3, "b_err: " << SAPDBTrace::BasisError(b_err) );

    if ( 0 != _res_rec_desc )
    {
        ACV_ALLOCPTR(acv)->Deallocate( _res_rec_desc );
    }

    if ( 0 == acv.a_returncode && ( e_ok == b_err || e_no_next_record == b_err ))
    {
        if  ( 0 == _rescnt && _restree_created )
        {
            // what about first part of UNION , could we delete this file ?    
            b01destroy_file( acv.a_transinf.tri_trans, _result_id );
            a502empty_result( acv, dmli, _result_id );
        }
        acv.a_mblock.mb_struct().becomes( mbs_result );
        acv.a_mblock.mb_type().becomes( m_return_result );
        acv.a_mblock.mb_qual()->mr_restree()   = _result_id;
        acv.a_mblock.mb_qual()->mr_res_build() = true;
        // set result count into messblock
        tsp00_NumError _res_err;
        s41p4int( acv.a_mblock.mb_qual()->mr_resnum(), 2, _rescnt, _res_err );
        acv.a_mblock.mb_qual()->mr_resnum()[ 0 ] = csp_defined_byte;
        acv.a_mblock.mb_qual()->mr_pagecnt() =
              acv.a_mblock.mb_trns()->trRteCommPtr_gg00->file_record_cnt;
        if  ( 0 == _rescnt || !dmli.d_single )
            acv.a_mblock.mb_data_len()  = 0;
        if ( dmli.d_single && 0 == _rescnt )
            a07_b_put_error( acv, e_no_next_record, 1 );
        if ( dmli.d_pseudo_ins_select /* insert values comes from a <select> clause */
             && acv.a_transinf.tri_trans.trWarning_gg00.includes(warn12_rowno_used) )
            a07_b_put_error( acv, e_too_many_resultsets, 1 );
    }
    else
    {
        kb05ReturnErrorHandling( acv.a_transinf.tri_trans, false, m_select );
        acv.a_mblock.mb_type().becomes( m_return_error );
        acv.a_mblock.mb_struct().becomes( mbs_nil );
        acv.a_mblock.mb_qual_len() = 0;
        acv.a_mblock.mb_data_len() = 0;
        if  ( e_rollback_wanted == b_err)
        {
            acv.a_rollback_info.clear();
            acv.a_rollback_info.addElement( trans_rollback );
            acv.a_rollback_info.addElement( new_trans );
            a07_b_put_error( acv, e_work_rolled_back, 1 );
        }
        else
        {
            a07_b_put_error( acv, b_err, 1 );
        }
        if  ( del_parsinfos )
        {
            // delete parsinfo's that have not been
            // deleted by ak682_tmpbuf_to_mbuf()
            a680rollback_temp_jinfo( acv, dmli, parsk, jvrec.jv_tabid,
                  dmli.d_cntfromtab );
        }
        if  ( _restree_created )
            b01destroy_file( acv.a_transinf.tri_trans, _result_id );
    }
#ifdef SAPDB_SLOW
    ACV_ALLOCPTR(acv)->CalcStatistics(_usedbytes2, _maxusedbytes, _ctrlbyted2);
    if ( ! _usedbytes1 == _usedbytes2 )
        RTE_Crash( SAPDBErr_Exception( __FILE__, __LINE__, SAPDBERR_ASSERT_STATE_FAILED, "(join exec memory leek)"));
#endif
}


namespace {
//! Create operator which adds records to result set
/*!
 * @param acv [in] global context
 * @param dmli [in] statement context
 * @param rec_desc [in] pointer to result record description
 * @param result_id [in] name of result file
 * @param rescnt [in] amount of records exists in result generated by former 'MASS SELECT' command
 * @param rescnt_setop [in] amount of records exits in result generated by former part of a set operation
 * @param maxresult [in] maximal amount of result set
 * @param top_operator [in] join operator deliveres records
 * @return pointer to a Join_IResultSetOperator corresponding desired properties like distinctness and others
 */
inline 
Join_IResultSetOperator* 
    CreateResultSetOperator( 
            tak_all_command_glob&   acv,
            tak_dml_info&           dmli,
            const tgg00_StackList * const   rec_desc,
            tgg00_FileId&           result_id, 
            const SAPDB_Int4&       rescnt, 
            const SAPDB_Int4&       rescnt_setop, 
            const SAPDB_Int4&       maxresult, 
            const pasbool           stamp_record_counter,
            Join_JoinOperator&      top_operator)
{
    SAPDBTRACE_ROUTINE_DEBUG( "CreateResultSetOperator", Join_Trace, 1 );
    SAPDBTRACE_WRITELN( Join_Trace, 3, "keylen: " << top_operator.GetKeyLength() << "\tselinto: " << dmli.d_single << "\tdistinct: " << dmli.d_distinct );
    SAPDBTRACE_WRITELN( Join_Trace, 3, "a_union_kind: " << acv.a_union_kind << "\ta_union_cnt: " << acv.a_union_cnt );
    SAPDBTRACE_WRITELN( Join_Trace, 3, "warning about ROWNO: " << (maxresult < 0) );

    bool _aggr_func = false, _one_result = ( RESCNT_MXGG04 == top_operator.GetKeyLength() ) ;

    if ( rec_desc )
        for ( SAPDB_Int4 _i = 1; _i <= (*rec_desc)[ 0 ].epos() - 1; ++_i )
        {
            if ( st_func == (*rec_desc)[ _i - 1 ].etype() )
                _aggr_func = true;
            else
            {
                if ( st_output == (*rec_desc)[ _i - 1 ].etype() && 
                     op_o_output_later == (*rec_desc)[ _i - 1 ].eop_out() )
                    _one_result = true;
            }
        }
    
    if ( _aggr_func )
    {
        if ( _one_result )
            return new( *ACV_ALLOCPTR(acv) ) 
                Join_ResultSetAggregatorOneGroup( acv, rec_desc, result_id, maxresult, top_operator, dmli.d_single );
        else
            return new( *ACV_ALLOCPTR(acv) ) 
                Join_ResultSetAggregator( acv, rec_desc, result_id, rescnt, maxresult, top_operator );
    }
    else
    {
        if ( dmli.d_single )
        {
            if ( dmli.d_distinct != no_distinct )
                return new( *ACV_ALLOCPTR(acv) ) 
                    Join_ResultSetDistinctOneResult( acv, rec_desc );
            else
                return new( *ACV_ALLOCPTR(acv) ) 
                    Join_ResultSetOneResult( acv, rec_desc );
        }
        else
        {
            if ( dmli.d_distinct != no_distinct )
            {
                // SELECT DISTINCT or distinct set operation
                if ( distinct_without_warning_byte == dmli.d_distinct )
                    a07_b_put_error( acv, e_not_implemented, 1 );
                if ( stamp_record_counter )
                    return new( *ACV_ALLOCPTR(acv) ) 
                        // ROWNUM makes tuple distinct !
                        Join_ResultSetAppenderEx( acv, rec_desc, result_id, rescnt, maxresult );
                else
                    return new( *ACV_ALLOCPTR(acv) ) 
                        Join_ResultSetDistinct( acv, rec_desc, result_id, rescnt, maxresult );
            }
            else
            {
                /*
                 * set operations not implemented with join selects with
                 * new C++ implementation
                 *
                if ( acv.a_union_kind != 0 || acv.a_union_cnt != 0 )
                {
                    // join is part of set operation
                    if ( acv.a_union_kind != 0 )
                    {
                        if ( acv.a_union_kind < cgg04_union_append )
                        {
                            return new( *ACV_ALLOCPTR(acv) ) 
                                Join_ResultSetHashInserter( acv, result_id, rescnt, maxresult );
                        }
                        else
                        {
                            return new( *ACV_ALLOCPTR(acv) ) 
                                Join_ResultSetAppenderEx( acv, result_id, rescnt, maxresult, rescnt_setop );
                        }
                    }
                    else
                    {
                        return new( *ACV_ALLOCPTR(acv) ) 
                            Join_ResultSetAppenderEx( acv, result_id, rescnt, maxresult );
                    }
                }
                else
                */
                {
                    if ( top_operator.GetKeyLength() > RESCNT_MXGG04 )
                    {
                        return new( *ACV_ALLOCPTR(acv) ) 
                            Join_ResultSetInserter( acv, rec_desc, result_id, rescnt, maxresult );
                    }
                    else
                    {
                        return new( *ACV_ALLOCPTR(acv) ) 
                            Join_ResultSetAppender( acv, rec_desc, result_id, rescnt, maxresult );
                    }
                }
            }
        }
    }
}

//! extract some info from strategy info pool
/*!
 * @param acv [in] global context
 * @param result_id [out] name of result file
 * @param rescnt [out] amount of records exists in result generated by former 'MASS SELECT' command
 * @param union_offs [out] amount of records exits in result generated by former part of a set operation
 * @param stamp_record_counter [out] write record counter (ROWNUM) into record
 */
inline
void
    GetStrategyInfo( 
            tak_all_command_glob&   acv,
            tgg00_FileId&           result_id,
            SAPDB_Int4&             rescnt,
            SAPDB_Int4&             union_offs,
            pasbool&                stamp_record_counter)
{
    if ( acv.a_mblock.mb_qual()->mstrat_pos() > 0 && 
         acv.a_mblock.mb_qual()->mstrat_cnt() > 0 )
    {
        tgg07_StrategyInfo *_strategy = REINTERPRET_CAST( tgg07_StrategyInfo*, 
                acv.a_mblock.mb_strat() + ((*acv.a_mblock.mb_st())[acv.a_mblock.mb_qual()->mstrat_pos()-1].epos()) - 1);
        result_id = _strategy->str_result_id;
        rescnt = _strategy->str_foundresults; // results already in result
        union_offs = _strategy->str_union_key_cnt_offs; // results already in result for set operations
        stamp_record_counter = _strategy->str_use_rowno; // write ROWNUM
    } 
    else
    {
        a07_b_put_error( acv, e_unknown_strategy, 1 );
        g01opmsg( sp3p_knldiag, sp3m_error, csp3_ak_msg,
              csp3_n_join, "MISSING STRATEGY INFO   ", 1 );
    }
}

//! extract result descption
/*!
 * @param acv [in] global context
 * @param res_rec_desc [out] pointer to result description
 */
inline
void
    GetResultDescription(
            tak_all_command_glob&   acv,
            tgg00_StackList*&       res_rec_desc)
{
    SAPDB_UInt2 _stackentry_cnt;
    if ( acv.a_mblock.mb_qual()->mresqual_cnt() != 0 && (st_jump_output == (*acv.a_mblock.mb_st())[acv.a_mblock.mb_qual()->mresqual_pos()-1].etype()) )
    {
        _stackentry_cnt = (*acv.a_mblock.mb_st())[acv.a_mblock.mb_qual()->mresqual_pos()-1].epos() - 1;
        if (  res_rec_desc = (tgg00_StackList*) ACV_ALLOCPTR(acv)->Allocate( _stackentry_cnt * sizeof( tgg00_StackEntry )) )
            memcpy( res_rec_desc, (tgg00_StackEntry *)acv.a_mblock.mb_st() + ( acv.a_mblock.mb_qual()->mresqual_pos() - 1 ), 
                    _stackentry_cnt * sizeof( tgg00_StackEntry ) );

    }
    else
    {
        SAPDBERR_ASSERT_STATE( st_jump_output == (*acv.a_mblock.mb_st())[acv.a_mblock.mb_qual()->mqual_pos()-1].etype() );
        _stackentry_cnt = (*acv.a_mblock.mb_st())[acv.a_mblock.mb_qual()->mqual_pos()-1].epos() - 1;
        if (  res_rec_desc = (tgg00_StackList*) ACV_ALLOCPTR(acv)->Allocate( _stackentry_cnt * sizeof( tgg00_StackEntry )) )
            memcpy( res_rec_desc, (tgg00_StackEntry *)acv.a_mblock.mb_st() + ( acv.a_mblock.mb_qual()->mqual_pos() - 1 ), 
                    _stackentry_cnt * sizeof( tgg00_StackEntry ) );
    }
    if ( ! res_rec_desc )
        a07_b_put_error( acv, e_no_more_memory, 1 );
#ifdef SAPDB_SLOW
    else
    {
        SAPDBTRACE_WRITELN( Join_Trace, 1, "result description: " );
        for ( SAPDB_Int4 _i = 1; _i < (*res_rec_desc)[0].epos(); ++_i)
        {
            t01stackentry( td_always, (*res_rec_desc)[_i-1], _i);
        }
    }
#endif
}

} // anonymous namespace

