/*!
 * @file
 * @brief implements Join_AccessDesc
 *
 * @author GertG
 * @ingroup Join
 *
 * @par last changed by:
 * <br>
 * $Author: d024980 $ $DateTime: 2004/05/03 16:16:26 $
 *
 * @sa Join_AccessDesc.hpp
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
#include "SAPDB/Join/Join_AccessDesc.hpp"
#include "SAPDB/Join/Join_Trace.hpp"
#include "gsp00.h"
#include "gsp03_1.h"
#include "hak07.h"
#include "hta01.h"

#include "hta01_3.h"

#include "hgg04.h"
#include "hgg04_1.h"
#include "hbd01.h"
#include "hbd01_1.h"
#include "hkb720.h"
#include "hgg01_1.h"
#include "hak725.h"

/* ******************** PUBLIC MEMBERS ********************* */
/*!
 * @param acv [in] global context
 *
 */
Join_AccessDesc::Join_AccessDesc(tak_all_command_glob& acv):
m_acv( acv ), m_mblock( acv.a_mblock ), m_strategy(0)
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessDesc::ctor", Join_Trace, 1 );
    
    tgg00_BasisError _b_err = e_ok;

    // initialize messblock
    _b_err = init_mblock();
    
    // initialize strategy
    if ( e_ok == _b_err )
    {
        if ( m_mblock.mb_qual()->mstrat_pos() > 0 && 
             m_mblock.mb_qual()->mstrat_cnt() > 0 )
        {
            m_strategy = reinterpret_cast<tgg07_StrategyInfo*>(m_mblock.mb_strat() + strategy_pos() - 1);
            SAPDBTRACE_IF( Join_Trace, 5, 
                    a725output_gg_strat( td_always, "STRATEGY    ", *m_strategy ));
            if ( m_strategy->str_use_subquery || m_mblock.mb_qual()->msubquery() )
            {
                tgg00_Rec *_rec = reinterpret_cast<tgg00_Rec*>( get_alloc().Allocate( BUF_MXSP00 ) );
                if ( _rec ) 
                {
                    k720_test_subquery( *m_mblock.mb_trns(), 
                            *m_mblock.mb_data(), m_mblock.mb_data_size(),
                            m_mblock.mb_qual()->mstack_desc(), *_rec );
                    get_alloc().Deallocate( _rec );
                }
                else
                    _b_err = e_no_more_memory;
            }
        } 
        else
        {
            _b_err = e_unknown_strategy;
            g01opmsg( sp3p_knldiag, sp3m_error, csp3_ak_msg,
                  csp3_n_join, "MISSING STRATEGY INFO   ", 0 );
        }
    }

    // initialize file ids
    if ( e_ok == _b_err )
    {
        m_fileids.file_id = m_mblock.mb_qual()->mtree();
        m_fileids.file_id.fileLeafNodes_gg00() = cgg_nil_leafnodes;

        if ( IsIndexAccess() ) 
        {
            if ( IsJoinInvAccess() )
            {
                g04index_tree_build( m_fileids.file_id, m_fileids.inv_id, 
                        m_strategy->str_join_multfields().sjmf_index_no );
                m_fileids.inv_id.fileRoot_gg00() = 
                    m_strategy->str_join_multfields().sjmf_invroot;
            }
            else
            {
                g04index_tree_build( m_fileids.file_id, m_fileids.inv_id, 
                        m_strategy->str_inv_in_range().siir_indexno );
                m_fileids.inv_id.fileRoot_gg00() = 
                    m_strategy->str_inv_in_range().siir_invroot;
            }
        }
        else
            m_fileids.inv_id = b01niltree_id;
    }

    if ( e_ok != _b_err ) a07_b_put_error( m_acv, _b_err, 1 );
}

/*!
 *
 */
Join_AccessDesc::~Join_AccessDesc()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessDesc::dtor", Join_Trace, 1 );

    SAPDBTRACE_WRITELN( Join_Trace, 8, "deallocate mb_qual() @ " << (void*)m_mblock.mb_qual() );
    if (m_mblock.mb_qual()) get_alloc().Deallocate(m_mblock.mb_qual());
    SAPDBTRACE_WRITELN( Join_Trace, 8, "deallocate mb_st() @ " << (void*)m_mblock.mb_st() );
    if (m_mblock.mb_st()) get_alloc().Deallocate(m_mblock.mb_st());
    SAPDBTRACE_WRITELN( Join_Trace, 8, "deallocate mb_strat() @ " << (void*)m_mblock.mb_strat() );
    if (m_mblock.mb_strat()) get_alloc().Deallocate(m_mblock.mb_strat());
    SAPDBTRACE_WRITELN( Join_Trace, 8, "deallocate mb_data() @ " << (void*)m_mblock.mb_data() );
    if (m_mblock.mb_data()) get_alloc().Deallocate(m_mblock.mb_data());
}

/*! 
 * @param startkey [out] record stream is restricted to records with key greater/equal than startkey
 * @param stopkey [out] record stream is restricted to records with key lower/equal than stopkey
 */
tgg00_BasisError Join_AccessDesc::CreateKeys(
        Join_TwoKeys& startkey,
        Join_TwoKeys& stopkey ) 
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessDesc::CreateKeys", Join_Trace, 1 );
    tgg00_BasisError _b_err = e_ok;
    if (  m_strategy )
    {
        // create primary keys 
        if ( strat_viewkey == m_strategy->str_strategy )
        {
            startkey.reckey.len() = m_mblock.mb_data()->mbp_key().keyLen_gg00();
            startkey.listkey.len() = 0;
            memcpy( startkey.reckey.k(), 
                    m_mblock.mb_data()->mbp_key().keyVal_gg00(), 
                    startkey.reckey.len());
            stopkey = startkey;
        }
        else
        {
            SAPDB_Int2 _in_stack = 0 , _in_cnt = 0;
            g04find_in_stack( m_mblock, 
                    m_strategy->str_key_in_range().skir_keystart,
                    _in_stack, _in_cnt );
            if ( 0 == _in_stack )
            {
                g04find_in_stack( m_mblock, 
                        m_strategy->str_key_in_range().skir_keystop,
                        _in_stack, _in_cnt );
            }
            
            pasbool _use_stopkeys = true; // check for fullkey
            g04limitprimkeys( m_mblock, 
                    m_strategy->str_key_in_range().skir_keystart, 
                    startkey.reckey, 
                    m_strategy->str_key_in_range().skir_keystop,
                    stopkey.reckey, _use_stopkeys, _in_stack, -_in_cnt );
            
            // create index keys 
            if ( IsIndexAccess() )
            {
                if ( m_strategy->str_inv_in_range().siir_startcnt > 0 || 
                     m_strategy->str_inv_in_range().siir_stopcnt > 0 )
                {
                    tgg00_Rec _dummy_buf;
                    g04minv_key( m_mblock, _dummy_buf, 
                            m_strategy->str_inv_in_range(), 
                            startkey.listkey, stopkey.listkey, 0 );
                }
                else
                {
                    startkey.listkey.len() = 0;
                    stopkey.listkey.len() = MAX_KEYLEN_GG00;
                    stopkey.listkey.k() = b01fullkey;
                }
            }
            else
            {
                startkey.listkey.len() = stopkey.listkey.len() = 0;
            }
        }
    }
    else
        _b_err = e_unknown_strategy;
    
#ifdef SAPDB_SLOW
    if ( Join_Trace.TracesLevel(1) )
    {
        t01key( td_always, "start.rec   ", startkey.reckey );
        t01key( td_always, "stop.rec    ", stopkey.reckey );
        if ( IsIndexAccess() )
        {
            t01key( td_always, "start.list  ", startkey.listkey );
            t01key( td_always, "stop.list   ", stopkey.listkey );
        }
    }
#endif
    return _b_err;
}

/* ******************* PROTECTED MEMBERS ******************* */


/* ******************** PRIVATE MEMBERS ******************** */
/*!
 *
 */
tgg00_BasisError Join_AccessDesc::init_mblock()
{
    SAPDBTRACE_METHOD_DEBUG( "Join_AccessDesc::init_mblock", Join_Trace, 3 );
    m_mblock.mb_qual()->msqlmode() = m_acv.a_mblock.mb_qual()->msqlmode();
    m_mblock.mb_qual()->mst_optimize_pos() = 0; // to avoid copying of mb_data
    
    m_mblock.mb_qual() = 0;
    SAPDBTRACE_WRITELN( Join_Trace, 8, "allocate mb_qual: " << m_mblock.mb_qual_len() );
    if ( m_mblock.mb_qual() = reinterpret_cast<tgg00_QualBufPtr>( get_alloc().Allocate( m_mblock.mb_qual_len() )))
    {
        SAPDBTRACE_WRITELN( Join_Trace, 8, "allocate mb_qual @ " << (void*)m_mblock.mb_qual() );
        memcpy(m_mblock.mb_qual(), m_acv.a_mblock.mb_qual(), 
                m_mblock.mb_qual_len());
        m_mblock.mb_qual_size() = m_mblock.mb_qual_len();

        m_mblock.mb_st() = 0;
        m_mblock.mb_st_max() = m_mblock.mb_qual()->mfirst_free()-1;
        m_mblock.mb_st_size() = m_mblock.mb_st_max() * sizeof(tgg00_StackEntry);
        SAPDBTRACE_WRITELN( Join_Trace, 8, "allocate mb_st: " << m_mblock.mb_st_size() );
        if ( m_mblock.mb_st() = reinterpret_cast<tgg00_StackListPtr>( get_alloc().Allocate( m_mblock.mb_st_size() )))
        {
            SAPDBTRACE_WRITELN( Join_Trace, 8, "allocate mb_st @ " << (void*)m_mblock.mb_st() );
            memcpy(m_mblock.mb_st(), m_acv.a_mblock.mb_st(), 
                    m_mblock.mb_st_size());
            m_mblock.mb_qual()->mst_addr() = m_mblock.mb_st();
        }
    }
    m_mblock.mb_qual()->mst_max() = m_mblock.mb_st_max();
    m_mblock.mb_work_st() = m_acv.a_work_st_addr;
    m_mblock.mb_work_st_max() = m_acv.a_work_st_max;
    m_mblock.mb_workbuf() = m_acv.a_work_buf_addr;
    m_mblock.mb_workbuf_size() = m_acv.a_work_buf_size;

    m_mblock.mb_strat() = 0;
    SAPDBTRACE_WRITELN( Join_Trace, 8, "allocate mb_strat: " << m_mblock.mb_strat_len() );
    if ( m_mblock.mb_strat() = reinterpret_cast<tsp00_MoveObjPtr>( get_alloc().Allocate( m_mblock.mb_strat_len() ) ) )
    {
        SAPDBTRACE_WRITELN( Join_Trace, 8, " allocate mb_strat @ " << (void*)m_mblock.mb_strat() );
        memcpy( m_mblock.mb_strat(), m_acv.a_mblock.mb_strat(), 
                m_mblock.mb_strat_len() );
        m_mblock.mb_strat_size() = m_mblock.mb_strat_len();
    }

    // so we could use mst_optimize_pos <> 0 !!!
    m_mblock.mb_data() = 0;
    SAPDBTRACE_WRITELN( Join_Trace, 8, "allocate mb_data: " << m_mblock.mb_data_len() );
    if ( m_mblock.mb_data() = reinterpret_cast<tgg00_DataPartPtr>( get_alloc().Allocate( m_mblock.mb_data_len() ) ) )
    {
        SAPDBTRACE_WRITELN( Join_Trace, 8, "allocate mb_data @ " << (void*)m_mblock.mb_data() );
        memcpy( m_mblock.mb_data(), m_acv.a_mblock.mb_data(), 
                m_mblock.mb_data_len() );
        m_mblock.mb_data_size() = m_mblock.mb_data_len();
    }

#ifdef SAPDB_SLOW
    t01messblock( td_always, "copied mbloc", m_mblock );
#endif
    if ( !(m_mblock.mb_qual() && m_mblock.mb_st() && m_mblock.mb_strat() && m_mblock.mb_data()) )
        return  e_no_more_memory;
    else
        return e_ok;
}
