/*!
  @file           IFR_FetchInfo.cpp
  @author         D039759
  @ingroup        IFR_Fetch
  @brief          Implements fetch info (stores result meta data)
  @see            

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/
#ifdef HPUX
#include <sys/param.h>
#endif
#include "SAPDB/Interfaces/Runtime/IFR_FetchInfo.h"
#include "SAPDB/Interfaces/Runtime/IFR_Statement.h"
#include "SAPDB/Interfaces/Runtime/Packet/IFRPacket_ReplySegment.h"
#include "SAPDB/Interfaces/Runtime/Packet/IFRPacket_PartEnum.h"
#include "SAPDB/Interfaces/Runtime/Packet/IFRPacket_Part.h"

IFR_FetchInfo::IFR_FetchInfo (IFR_Statement& statement,
                              IFRConversion_Converter **info,
		                      IFRUtil_Vector<IFR_String>* colName,
                              IFR_Bool& memory_ok)
:IFRUtil_RuntimeItem(statement),
 m_Statement(statement), 
 m_CursorName(statement.getCursorName(), memory_ok),
 m_FetchParamString ("", IFR_StringEncodingAscii, allocator, memory_ok), 
 m_ConverterLength (0),
 m_ColumnInfo(0),
 m_columnInfoIsReference(false)
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, IFR_FetchInfo);
    if(memory_ok) {
        if(info!=0 && colName!=0) {
            this->setMetaData(info, colName);
        }
    }
}

//----------------------------------------------------------------------
IFR_FetchInfo::~IFR_FetchInfo ()
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, ~IFR_FetchInfo);
    
    if (!m_columnInfoIsReference) {
        for (IFR_Int4 i = 0; i < m_ConverterLength; i++) {
            IFRUtil_Delete(m_ColumnInfo[i], allocator);
        }  
        IFRUtil_DeleteArray(m_ColumnInfo, m_ConverterLength, allocator);
    }
}

IFR_Retcode IFR_FetchInfo::describe () 
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, describe);
    // return if we have already all information
    if (this->m_ColumnInfo != 0)
        DBUG_RETURN(IFR_OK);
    IFR_Bool memory_ok=true;
    IFR_Connection *connection = m_Statement.getConnection();
    IFRPacket_RequestPacket requestPacket(*this);
    IFR_Retcode rc=connection->getRequestPacket (requestPacket);
    if(rc != IFR_OK) {
        error() = connection->error();
        DBUG_RETURN(rc);
    }

    requestPacket.setEncoding(m_Statement.getCommandEncoding());
    
    IFR_String sqlCmd(requestPacket.getEncoding(), allocator);
    sqlCmd.append("DESCRIBE \"", IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    sqlCmd.append(this->m_CursorName, memory_ok);
    sqlCmd.append("\"", IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }

    IFRPacket_RequestSegment segment(requestPacket, IFRPacket_CommandMessageType::Dbs_C);      
    IFRPacket_CommandPart command;
    if (IFR_OK != segment.addPart(command)) {
      this->error().setRuntimeError(IFR_ERR_PACKET_EXHAUSTED);
      DBUG_RETURN(IFR_NOT_OK);
    }
    if (IFR_OK != command.setText(sqlCmd, this->error())) {
      DBUG_RETURN(IFR_NOT_OK);
    }
    segment.closePart();
    segment.close();
    
    IFRPacket_ReplyPacket replyPacket;
    rc = connection->sqlaexecute (requestPacket, replyPacket, false, error());
    if(rc != IFR_OK) {
        DBUG_RETURN(rc);
    } else if(error()) {
        DBUG_RETURN(IFR_NOT_OK);
    }
    IFRPacket_ReplySegment replysegment(replyPacket);
    IFRPacket_PartEnum partEnum (&replysegment);
    
    IFRConversion_Converter **infos = 0;
    IFRUtil_Vector<IFR_String> *columnNames = 0;
    
    while (partEnum.hasMoreElements ()) {
        int knd = partEnum.partKind ();
        if (knd == IFRPacket_PartKind::Columnnames_C) {
            replysegment.parseColumnNames (&columnNames, allocator, memory_ok);
            if(!memory_ok) {
                error().setMemoryAllocationFailed();
                DBUG_RETURN(IFR_NOT_OK);
            }
        } else if (knd == IFRPacket_PartKind::Shortinfo_C) {
            replysegment.parseShortFields (infos, m_Statement);
        } else {
        }
        partEnum.nextElement ();
    }
    this->setMetaData (infos, columnNames);
    DBUG_RETURN(IFR_OK);
}

//----------------------------------------------------------------------
IFR_Retcode
IFR_FetchInfo::setMetaData (IFRConversion_Converter **info,
                            IFRUtil_Vector<IFR_String>* colName)
{
  DBUG_METHOD_ENTER(IFR_FetchInfo, setMetaData);
  
  IFR_String currentName(allocator);
  int currentFieldEnd;

  this->m_RecordSize = 0;
  this->m_ColumnInfo = info;
  IFR_Bool memory_ok=true;

  for (IFR_UInt4 i = 0; i < colName->GetSize (); i++) {
      info[i]->setName ((*colName)[i], memory_ok);
      if(!memory_ok) {
          error().setMemoryAllocationFailed();
          DBUG_RETURN(IFR_NOT_OK);
      }
      info[i]->setIndex(i+1);
      currentFieldEnd = info[i]->getIOLength () + info[i]->getBufpos () - 1;
      this->m_RecordSize = MAX (this->m_RecordSize, currentFieldEnd);
  }
  m_ConverterLength = colName->GetSize ();
  
  DBUG_RETURN(IFR_NOT_OK);
}

//----------------------------------------------------------------------
void 
IFR_FetchInfo::setMetaDataRefFromParseinfo (IFRConversion_Converter **info,
                                            IFR_Int2 columncount) 
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, copyMetaDataFromParseinfo);
    
    int currentFieldEnd;
    this->m_columnInfoIsReference = true;
    this->m_ConverterLength = columncount;
    this->m_RecordSize = 0;
    this->m_ColumnInfo = info;
    
    for (IFR_Int4 i = 0; i < columncount; i++) {
        info[i]->setIndex(i+1);
        currentFieldEnd = info[i]->getIOLength () + info[i]->getBufpos () - 1;
        this->m_RecordSize = MAX (this->m_RecordSize, currentFieldEnd);
    }
}

//----------------------------------------------------------------------
IFR_Retcode IFR_FetchInfo::executeFetchAbsolute (IFR_Int4 position, 
                                                 IFR_Int4 fetchSize, 
                                                 IFRPacket_ReplyPacket& replyPacket)
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, executeFetchAbsolute);
    DBUG_PRINT(position);
    DBUG_PRINT(fetchSize);
    IFR_Bool memory_ok=true;
    IFR_String cmd ("FETCH ABSOLUTE ", IFR_StringEncodingAscii, allocator, memory_ok);
    cmd.expand(128, memory_ok); 
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
    char pos[14];
    sp77sprintf (pos, sizeof(pos), "%d", position);
    
    cmd.append(pos, IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(" \"",IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(m_CursorName, memory_ok);
    cmd.append("\" INTO ", IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(getFetchParamString(memory_ok), memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
    DBUG_RETURN(executeFetchCommand (cmd, fetchSize, replyPacket));
}

IFR_Retcode IFR_FetchInfo::executeFetchRelative (IFR_Int4 position, 
                                                 IFR_Int4 fetchSize, 
                                                 IFRPacket_ReplyPacket& replyPacket)
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, executeFetchRelative);
    IFR_Bool memory_ok=true;
    IFR_String cmd ("FETCH RELATIVE ", IFR_StringEncodingAscii, allocator, memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
    char pos[14];
    sp77sprintf (pos, sizeof(pos),"%d", position);
    cmd.append(pos, IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(" \"", IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(m_CursorName, memory_ok);
    cmd.append("\" INTO ", IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(getFetchParamString(memory_ok), memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
    DBUG_RETURN(executeFetchCommand (cmd, fetchSize, replyPacket));
}

//----------------------------------------------------------------------
IFR_Retcode 
IFR_FetchInfo::executeFetchFirst(IFR_Int4 fetchSize, 
                                 IFRPacket_ReplyPacket& replyPacket)
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, executeFetchFirst);
    IFR_Bool memory_ok=true;
    IFR_String cmd ("FETCH FIRST \"", IFR_StringEncodingAscii, allocator, memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
    cmd.append(this->m_CursorName, memory_ok);
    cmd.append("\" INTO ", IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(getFetchParamString(memory_ok), memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
    DBUG_RETURN(executeFetchCommand (cmd, fetchSize, replyPacket));
}

//----------------------------------------------------------------------
IFR_Retcode 
IFR_FetchInfo::executeFetchLast (IFR_Int4 fetchSize, 
                                 IFRPacket_ReplyPacket& replyPacket)
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, executeFetchLast);
    IFR_Bool memory_ok=true;
    IFR_String cmd ("FETCH LAST \"", IFR_StringEncodingAscii, allocator, memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
    cmd.append(this->m_CursorName, memory_ok);
    cmd.append("\" INTO ", IFR_StringEncodingAscii, IFR_NTS, memory_ok);
    cmd.append(getFetchParamString(memory_ok), memory_ok);
    if(!memory_ok) {
        error().setMemoryAllocationFailed();
        DBUG_RETURN(IFR_NOT_OK);
    }
        
    DBUG_RETURN(executeFetchCommand (cmd, fetchSize, replyPacket));
}

//----------------------------------------------------------------------
const IFR_String& 
IFR_FetchInfo::getFetchParamString(IFR_Bool& memory_ok)
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, getFetchParamString);

    if(!memory_ok) {
        DBUG_RETURN(m_FetchParamString);
    }

    if(m_FetchParamString.getLength()==0) {
        m_FetchParamString.expand(m_ConverterLength*3-2, memory_ok);
        m_FetchParamString.setBuffer("?", 1, IFR_StringEncodingAscii, memory_ok);
        for(int i=1; i<m_ConverterLength;++i) {
            m_FetchParamString.append(", ?", IFR_StringEncodingAscii, 3, memory_ok);
        }
    }
    
    DBUG_RETURN(m_FetchParamString);
}

//----------------------------------------------------------------------
IFR_Retcode 
IFR_FetchInfo::executeFetchCommand (IFR_String& command, 
                                    IFR_Int4 fetchSize, 
                                    IFRPacket_ReplyPacket& replyPacket)
                                 
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, executeFetchCommand);
    DBUG_PRINT(command);
    DBUG_PRINT(fetchSize);
    
    //>>> SQL TRACE
    IFR_SQL_TRACE << endl << "::FETCH " << m_CursorName << " "
                  << currenttime << endl
                  << "SQL COMMAND: " << command << endl
                  << "FETCH SIZE: " << fetchSize << endl;
    //<<< SQL TRACE

    IFR_Retcode rc = IFR_OK;
    IFR_String errText(allocator);

    IFRPacket_RequestPacket request(*this);
    IFR_Bool reset = true;

    IFR_Connection *connection = m_Statement.getConnection();
    rc=connection->getRequestPacket (request);
    if(rc!=IFR_OK) {
        return rc;
    }
    int currentSQLMode = request.switchSqlMode (IFR_INTERNAL);
    IFRPacket_RequestSegment segment (request, sp1m_dbs, reset);
    if(segment.isValid()) {
        IFRPacket_CommandPart commandpart;
        IFRPacket_LongDemandPart longdemandpart;
        if((rc = segment.addPart(commandpart)) != IFR_OK) {
            error().setRuntimeError(IFR_ERR_PACKET_EXHAUSTED);
            request.switchSqlMode(currentSQLMode);
            DBUG_RETURN(rc);
        }
        if (IFR_OK != commandpart.setText(command, error())) {
          request.switchSqlMode(currentSQLMode);
          DBUG_RETURN(IFR_NOT_OK);
        } 
        segment.closePart ();
        if (m_Statement.m_resultset->hasPositionedRead ()) {
          if((rc = segment.addPart(longdemandpart)) != IFR_OK) {
            error().setRuntimeError(IFR_ERR_PACKET_EXHAUSTED);
            request.switchSqlMode(currentSQLMode);
            DBUG_RETURN(rc);
          }
          const IFRUtil_Vector<IFR_Parameter> *paramvec = m_Statement.m_resultset->getParamVector ();
          for (unsigned int i = 0; i < paramvec->GetSize (); i++) {
              if (m_ColumnInfo[i]->isLong ()) {
                  IFR_Parameter param = (*paramvec)[i];
                  const IFR_Length *pos = param.getPosIndicator ();
#if 1
                  if (pos) {
                      longdemandpart.addPosParams (true, 
                                                   *pos, 
                                                   (*paramvec)[i].getBytesLength ());
                  } else {
                      longdemandpart.addPosParams (true, 
                                                   1, 
                                                   (*paramvec)[i].getBytesLength ());
                  }
#else
                  if (pos) {
                      longdemandpart.addPosParams (true, 
                                                   2, 
                                                   50);
                  } else {
                      longdemandpart.addPosParams (true, 
                                                   3, 
                                                   60);
                  }
#endif
              }
          }
          segment.closePart ();
        }
        error ().clear ();
        if (fetchSize > 1) {
            request.setMassCommand();
        } else {
            fetchSize = 1;
        }
        segment.addFetchSize(fetchSize);
        segment.close();
        IFR_Retcode rc_sqlaexecute;
        if((rc_sqlaexecute=connection->sqlaexecute (request, replyPacket, false, error())) != IFR_OK) {
            DBUG_RETURN(rc_sqlaexecute);
        } 
        IFRPacket_ReplySegment replysegment(replyPacket);
        if (replysegment.getSQLError (error(), allocator)) {
            DBUG_RETURN(IFR_NOT_OK);
        }
        request.switchSqlMode(currentSQLMode);
        DBUG_RETURN(IFR_OK);
    }
    request.switchSqlMode(currentSQLMode);
    DBUG_RETURN(IFR_NOT_OK);
}

//----------------------------------------------------------------------
IFRConversion_Converter * 
IFR_FetchInfo::findColInfo (IFR_Int2 columnIndex)
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, findColInfo);
  if (columnIndex > 0 && columnIndex <= this->m_ConverterLength)
    return (this->m_ColumnInfo[columnIndex-1]); 
  else
    return 0;  
}
//----------------------------------------------------------------------
IFR_Int2
IFR_FetchInfo::getColumnCount()
{
    DBUG_METHOD_ENTER(IFR_FetchInfo, getColumnCount);
    DBUG_RETURN(this->m_ConverterLength);
}


