/*!
  @file           IFR_Environment.h
  @author         DL SAP DB INTERFACES
  @ingroup        IFR_Common
  @brief          Global environment
  @see            

\if EMIT_LICENCE



    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end






\endif
*/
#ifndef IFR_ENVIRONMENT_H
#define IFR_ENVIRONMENT_H

#include "Interfaces/Runtime/IFR_Common.h"
#include "Interfaces/Runtime/IFR_String.h"
#include "Interfaces/Runtime/Util/IFRUtil_RuntimeItem.h"

class IFR_Connection;

/**
 * Global swap kind of the operating system.
 * Valid swap kinds are
 * <ul>
 *   <li><code>SwapUnknown</code> - swap not yet determined (environment not initialised)</li>
 *   <li><code>SwapNormal</code>  - H2H1L2L1 (big endian)</li>
 *   <li><code>SwapFull</code>    - L1L2H1H2 (little endian)</li>
 *   <li><code>SwapHalf</code>    - L2L1H2H1</li>
 * </ul>
 */
extern IFR_ClientSwap IFR_EnvironmentSwapKind;

/**
 * The environment is the frame around getting connections and
 * applying general settings in the interface runtime library.
 *
 * An environment object uses a runtime environment implementation
 * to fulfill its requests. 
 */
class IFR_Environment 
    : public IFRUtil_RuntimeItem, 
      public IFR_ITraceController
{
public:
    /**
     * Constructor. 
     * @param runtime The runtime environment on which this 
     *                environment is based. This must not be 0.
     */
    IFR_Environment(SQLDBC_IRuntime* runtime);
    
    /**
     * Destructor.
     */
    ~IFR_Environment();
    
    /**
     * Get the major version.
     */
    IFR_Int4 getMajorVersion() const;

    /**
     * Get the minor version.
     */
    IFR_Int4 getMinorVersion() const;

    /**
     * Gets the runtime version in the format:
     *   - 1 digit  major number,
     *   - 2 digits minor number,
     *   - 2 digits correction level.
     * @return The runtime version.
     */
    static IFR_Int4 getRuntimeVersion();

    /**
     * Gets a new connection.
     * @return A newly created connection.
     */
    IFR_Connection* getConnection();

    /**
     * Gets a new connection.
     * @param customAllocator The allocator to be used for the connection.
     * @return A newly created connection.
     */
    IFR_Connection* getConnection(SAPDBMem_IRawAllocator& allocator);
    
    /**
     * Releases a connection.
     * @param c The connection to be released. 
     */
    void releaseConnection(IFR_Connection *c);

    
    inline IFR_TraceStream* getTraceStream() 
    {
        return m_tracestream;
    }

public:
    // -- trace controller methods
    virtual void write(const char *data, IFR_Int4 length = -1);
    virtual void writeln(const char *data, IFR_Int4 length = -1);
    virtual SQLDBC_IRuntime::TaskTraceContext *traceflags();
    virtual SAPDBMem_IRawAllocator& getAllocator();
private:
    IFR_Environment(const IFR_Environment&);
    IFR_Environment& operator =(const IFR_Environment&);    

    IFR_TraceStream   *m_tracestream;
    IFR_CallStackInfo  m_callstackroot;
};


#endif //IFR_ENVIRONMENT_H
