/*!
  @file           IFR_Environment.cpp
  @author         DL SAP DB INTERFACES
  @ingroup        IFR_Common
  @brief          Global environment
  @see            

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2001-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/
#include "Interfaces/Runtime/IFR_Common.h"
#include "Interfaces/Runtime/IFR_Environment.h"
#include "Interfaces/Runtime/IFR_Connection.h"


#include "SAPDBCommon/SAPDB_ReuseLib.h" 
#include "Interfaces/SQLDBC/SQLDBC_IRuntime.h"
#include "gsp100.h" 

/**
 * The byte swap kind of the platform.
 */
IFR_ClientSwap IFR_EnvironmentSwapKind=SwapUnknown;

//----------------------------------------------------------------------
IFR_Environment::IFR_Environment(SQLDBC_IRuntime* runtime)
:IFRUtil_RuntimeItem(*runtime),
 m_tracestream(0)
{   
    m_tracestream = new (allocator) IFR_TraceStream(this);

    // swapping - order not important 
    if (IFR_EnvironmentSwapKind == SwapUnknown) {
        union {
            tsp00_Int4 asInt;
            char       asChar [4];
        } swapTest;
        swapTest.asInt = 1;
        if (swapTest.asChar [3] == 1) {
            IFR_EnvironmentSwapKind=SwapNormal;
        }
        else if (swapTest.asChar [0] == 1) {
            IFR_EnvironmentSwapKind=SwapFull;
        }
        else {
            IFR_EnvironmentSwapKind=SwapHalf;
        }   
    }

    // we bootstrap the call stack.
    if(traceflags()) {
        m_callstackroot.scope   = "?";
        m_callstackroot.file    = __FILE__;
        m_callstackroot.line    = __LINE__;
        m_callstackroot.level   = 0;
        m_callstackroot.context = traceflags();
        m_callstackroot.stream  = m_tracestream;
        if(m_callstackroot.context->currentEntry == 0) {
            m_callstackroot.context->currentEntry = &m_callstackroot;
        }
    }
    
    DBUG_METHOD_ENTER(IFR_Environment, IFR_Environment);
}

//----------------------------------------------------------------------
IFR_Environment::~IFR_Environment()
{
    m_callstackroot.stream=0;
    m_callstackroot.context=0;
    // DBUG_METHOD_ENTER(IFR_Environment, ~IFR_Environment);
    IFRUtil_Delete(m_tracestream, allocator);
}

//----------------------------------------------------------------------
int IFR_Environment::getMinorVersion() const
{
    DBUG_METHOD_ENTER(IFR_Environment, getMinorVersion);
    DBUG_RETURN(<MINOR_VERSION>);
}

//----------------------------------------------------------------------
int IFR_Environment::getMajorVersion() const
{
    DBUG_METHOD_ENTER(IFR_Environment, getMajorVersion);
    DBUG_RETURN(<MAJOR_VERSION>);
}

//----------------------------------------------------------------------
IFR_Connection* IFR_Environment::getConnection()
{
    DBUG_METHOD_ENTER(IFR_Environment, getConnection);
    DBUG_RETURN(getConnection(allocator));
}

//----------------------------------------------------------------------
IFR_Connection* IFR_Environment::getConnection
  (SAPDBMem_IRawAllocator& customAllocator) 
{
    DBUG_METHOD_ENTER(IFR_Environment, getConnection);
    IFR_Bool memory_ok = true;
    IFR_Connection *result = 
            new (customAllocator) IFR_Connection(*this, customAllocator, memory_ok);
    if(!memory_ok) {
      IFRUtil_Delete(result, customAllocator);
      result = 0;
    }
    DBUG_RETURN(result);
}


//----------------------------------------------------------------------
void IFR_Environment::releaseConnection(IFR_Connection *conn) 
{
    DBUG_METHOD_ENTER(IFR_Environment, releaseConnection);
    IFRUtil_Delete(conn, conn->allocator);
}

//----------------------------------------------------------------------
IFR_Int4
IFR_Environment::getRuntimeVersion()
{    
    // static DBUG_METHOD_ENTER(IFR_Environment, getRuntimeVersion);
    return (<MAJOR_VERSION>*10000+<MINOR_VERSION>*100+<CORRECTION_LEVEL>);
}

//----------------------------------------------------------------------
void 
IFR_Environment::write(const char *data, IFR_Int4 length)
{
    runtime.write(data, length);
}

//----------------------------------------------------------------------
void 
IFR_Environment::writeln(const char *data, IFR_Int4 length)
{
    runtime.writeln(data, length);
}

//----------------------------------------------------------------------
SQLDBC_IRuntime::TaskTraceContext *
IFR_Environment::traceflags()
{
    return runtime.getTaskTraceContext();
}

//----------------------------------------------------------------------
SAPDBMem_IRawAllocator& 
IFR_Environment::getAllocator()
{
    return allocator;
}

