/*! 
  \file    DBMSrv_Reply.hpp
  \author  TiloH
  \ingroup common classes for the DBMServer
  \brief   declaration of a class handling the reply buffer for DBMServer
           commands

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (C) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end

\endif
*/

#if !defined(DBMSrv_Reply_hpp)
#define DBMSrv_Reply_hpp


//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------

#include "gcn00.h"


//-----------------------------------------------------------------------------
// declaration of class DBMSrv_Reply
//-----------------------------------------------------------------------------

//! \brief a class managing the reply of DBMServer commands
class DBMSrv_Reply
{
  public:

    typedef const char * Position;
    /*! constructor

        initialising a reply object without a real reply buffer, before using
        such an object SetTo() must be called */
    DBMSrv_Reply();

    /*! constructor

        initialising a reply object with the needed information
        \param ReplyData   [IN] a pointer to the replay buffer that shall contain the reply
        \param ReplyLength [IN] a pointer to the length of the current reply
        \param ReplyLenMax [IN] the maximal allowed reply length */
    DBMSrv_Reply(char * ReplyData,
                 int  * ReplyLength,
                 int    ReplyLenMax);

    /*! function for changing the managed reply buffer
        \param ReplyData   [IN] a pointer to the new replay buffer that shall contain the reply
        \param ReplyLength [IN] a pointer to the new length of the current reply
        \param ReplyLenMax [IN] the new maximal allowed reply length */
    void setTo(char * ReplyData,
               int  * ReplyLength,
               int    ReplyLenMax);

    /*! function for obtaining the reply buffer

        This function should be avoided whenever possible, because it's the long term goal of this class
        to capsulate all operations on the reply buffers
        \return a pointer to the reply buffer
        \todo eliminate all occurences of this function on long term */
    char * giveData();

    /*! function for obtaining a const pointer the reply buffer

        \return a const pointer to the reply buffer */
    const char * giveData() const;

    /*! function for obtaining a reference to the reply length

        This function should be avoided whenever possible, because it's the long term goal of this class
        to capsulate all operations on the reply buffers
        \return a reference to the variable holding the reply length
        \todo eliminate all occurences of this function on long term */
    int * giveLength();

    /*! function for adapting the length member of the reply object

        This function determines the string length of the current zero terminated
        reply in the buffer and sets the reply length to that value.
        This function must be used whenever the reply buffer is manipulated
        directly with the help of the pointer returned by the Data() function.
        When all operations on the reply buffers are capsulated in this class,
        the AdaptLength() function is no longer needed.
        \todo eliminate all occurences of this function when Data() is no longer used either*/
    void adaptLength();

    /*! function for obtaining the maximal allowed reply length of the current buffer
        \return maximal allowed length of a reply */
    int giveMaximalLength() const;

    /*! function for obtaining the free space of the current buffer
        \return number of unused characters of the reply buffer*/
    int giveAvailableSpace() const;

    /*! function for obtaining the ReturnCode
        \return the current return code of the DBMServer command*/
    //tcn00_Error   ReturnCode() const;

    /*! function for reporting an OK as reply 
        \todo avoid usage of vcn90.cpp
        \todo check maximal allowed reply length (!)*/
    void startWithOK();

    /*! function for reporting an OK as reply if parameter is OK_CN00
        \param rc [IN] current DBMServer return code for the command
        \todo avoid usage of vcn90.cpp
        \todo check maximal allowed reply length (!)
        \todo integrate rc into Reply and drop the argument*/
    void startWithOKIfRCIsOk(tcn00_Error rc);

    /*! function for reporting a tcn00_Error as reply
        \param Error [IN] DBMServer error to report
        \return copy of Error */
    tcn00_Error startWithError(const tcn00_Error Error,
                               int               ExtError          =0,
                               const char *      ExtErrorText      =0,
                               size_t            ExtErrorTextLength=0,
                               int               ExtError2         =0,
                               const char *      ExtError2Text     =0);

    /*! function for reporting a RTEError as reply
        \param Error            [IN] DBMServer error to report
        \param RTEMessage       [IN] RTEMessage to report
        \param RTEMessageLength [IN] length of the RTEMessage
        \param RTEError         [IN] a number of the RTEError
        \return copy of Error
        \todo avoid usage of vcn90.cpp
        \todo check maximal allowed reply length (!) */
    tcn00_Error startWithRTEError(tcn00_Error  Error,
                                  const char * RTEMessage,
                                  int          RTEMessageLength,
                                  int          RTEError);

    /*! function for reporting an SQLError as reply
        \param Error            [IN] DBMServer error to report
        \param SQLMessage       [IN] SQLMessage to report (must contain trailing '\0')
        \param SQLError         [IN] SQL error code
        \return copy of Error
        \todo check maximal allowed reply length (!) */
    tcn00_Error startWithSQLError(tcn00_Error  Error,
                                  const char*  SQLMessage,
                                  const int    SQLError);

    /*! function for reporting a teo200_EventList as reply
        \param EventList [IN] event list to report
        \return first error of the event list
        \todo avoid usage of vcn90.cpp
        \todo check maximal allowed reply length (!) */
    tcn00_Error startWithEventList(const teo200_EventList & EventList);

    /*! function for reporting a SAPDBErr_MessageList as reply
        \param EventList [IN] event list to report
        \return first error of the event list
        \todo check maximal allowed reply length (!) */
    tcn00_Error startWithMessageList(const SAPDBErr_MessageList &MsgList);

    /*! function for writing the first line of a reply
        \param Line [IN] the line to start the reply with */
    void startWithLine(const char * Line);

    /*! function for appending a line break to the reply */
    void appendLine(); 

    /*! function for appending a line to the reply
        \param Line [IN] the line to add to the reply */
    void appendLine(const char * Line);

    /*! function for appending a not necessarily zero
        terminated line into the reply
        \param Line     [IN] the line to add to the reply
        \param LineSize [IN] the size of the line in characters */
    void appendLineN(const char * Line, size_t LineSize);

    /*! function for appending two lines to the reply
        \param Line1 [IN] first line to add to the reply
        \param Line2 [IN] second line to add to the reply */
    void appendLines(const char * Line1, const char * Line2);

    /*! function for appending tree lines to the reply
        \param Line1 [IN] first line to add to the reply
        \param Line2 [IN] second line to add to the reply
        \param Line3 [IN] third line to add to the reply*/
    void appendLines(const char * Line1, const char * Line2, const char * Line3);

    /*! function for appending a not necessarily zero terminated
        string and a line break to reply, so that the resulting
        output is indented and a maximal line size is forcing
        additional line breaks at word boundaries, if need.
        \param LineStart  [IN] start of the string, containing the words
        \param LineLength [IN] length of tha string
        \param Indent     [IN] number of spaces used for indent
        \param WrapAt     [IN] number of characters were a line break should happen
        \param PositionInOutputLine [IN] starting position within the first line of output*/
    void appendIndentedWrappedLineN(const char   * LineStart,
                                    const size_t   LineLength,
                                    const size_t   Indent,
                                    const size_t   WrapAt,
                                    size_t         PositionInOutputLine);

    /*! function for appending a not necessarily zero terminated
        string and a line break to reply, so that the resulting
        output is indented and a maximal line size and a special
        line break string is forcing additional line breaks at
        word boundaries
        \param LineStart       [IN] start of the string, containing the words
        \param LineLength      [IN] length of tha string
        \param Indent          [IN] number of spaces used for indent
        \param WrapAt          [IN] number of characters were a line break should happen
        \param LineBreakString [IN] special string for forcing line breaks
        \param PositionInOutputLine [IN] starting position within the first line of output*/
    void appendIndentedWrappedLineN(const char   * LineStart,
                                    const size_t   LineLength,
                                    const size_t   Indent,
                                    const size_t   WrapAt,
                                    const char   * LineBreakString,
                                    size_t         PositionInOutputLine);

    /*! function for appending a string to the reply
        \param String [IN] string to add to the reply */
    void appendString(const char *String);

    /*! function for appending a not necessarily zero
        terminated string into the reply
        \param String     [IN] string to add to the reply
        \param StringSize [IN] the size of the string in characters */
    void appendStringN(const char *String, size_t StringSize);

    /*! function for appending two strings to the reply
        \param String1 [IN] first string to add to the reply
        \param String2 [IN] second string to add to the reply */
    void appendStrings(const char *String1, const char *String2);

    /*! function for appending three strings to the reply
        \param String1 [IN] first string to add to the reply
        \param String2 [IN] second string to add to the reply
        \param String3 [IN] third string to add to the reply */
    void appendStrings(const char *String1, const char *String2, const char *String3);

    /*! function for appending two strings, followed by a
        line break to the reply
        \param String1 [IN] first string to add to the reply
        \param Line2   [IN] the second string to add to the reply */
    void appendStringAndLine(const char *String1, const char *Line2);

    /*! function for appending two strings (the last one terminated via
        length option), followed by a line break to the reply
        \param String1     [IN] first string to add to the reply
        \param Line2       [IN] the second string to add to the reply
        \param Line2Length [IN] the length of the second string */
    void appendStringAndLineN(const char *String1, const char *Line2, size_t Line2Length);

    /*! function for appending tree strings, followed by a
        line break to the reply
        \param String1 [IN] first string to add to the reply
        \param String2 [IN] second string to add to the reply
        \param Line3   [IN] the third string to add to the reply */
    void appendStringsAndLine(const char *String1, const char *String2, const char *Line3);

    /*! function for appending three strings a not zero terminated string
        another string and a line break to the reply
        \param String1       [IN] first string to add to the reply
        \param String2       [IN] second string to add to the reply
        \param String3       [IN] third string to be added to the reply
        \param String4       [IN] fourth not zero terminated string to be added to the reply
        \param Stirng4Length [IN] the length of the fourth string
        \param Line          [IN] the fifth an last string to be added */
    void appendStringsStringNAndLine(const char   * String1,
                                     const char   * String2,
                                     const char   * String3,
                                     const char   * String4,
                                     const size_t   String4Length,
                                     const char   * Line);

    /*! function for appending a string, at least MinimalWidth
        characters are written, if string has less
        than MinimalWidth characters, spaces are added
        \param String       [IN] string to add to the reply
        \param MinimalWidth [IN] guaranteed number of characters,
                                 that are written to reply*/
    void appendStringWithMinWidth(const char *String, size_t MinimalWidth);

    /*! function for appending a not zero terminated string, at least MinimalWidth
        characters are written, if string has less
        than MinimalWidth characters, spaces are added
        \param String       [IN] string to add to the reply
        \param StringLength [IN] length of string String
        \param MinimalWidth [IN] guaranteed number of characters,that are
                                 written to reply*/
    void appendStringNWithMinWidth(const char *String, size_t StringLength, size_t MinimalWidth);

    /*! a function adding a character to the reply
        \param Character [IN] the character that is to be added */
    void appendChar(const char Character);

    /*! a function for appending a character a number of times
        \param Character       [IN] the character to append
        \param NumberOfAppends [IN] the number of times Character is to ba appended*/
    void appendCharacterRepeatedly(const char Character, size_t NumberOfAppends);

    /*! function returning the current end of the reply buffer
        \return end of current reply buffer */
    Position giveCurrentEnd() const;

    /*! function returning the current end of the reply buffer
        \return end of current reply buffer
        \todo eliminate all occurences of this function on long term */
    char * giveCurrentEnd();

    /*! function terminating the string at Position
        \param the position in the reply buffer, where to end the reply at*/
    void skipAt(Position SkipPosition);

    /*! function for inserting a string into existing reply
        \param InsertPosition [IN] position where the string is inserted
        \param String         [IN] string to add to the reply */
    void insertString(Position InsertPosition, const char * String);

  private:
    char        * replyData;
    int         * replyLength;
    int           maximalReplyLength;

    size_t calculateLengthOfNextWord(const char *WordStart, size_t MaxWordLength, const char * LineBreakString);

    size_t CStringLengthFromPascalString(const char * SourcePascalString,
                                         const int    PascalStringMaximalLength);
};

#endif
