(*
    ||M||  This file is part of HELM, an Hypertextual, Electronic        
    ||A||  Library of Mathematics, developed at the Computer Science     
    ||T||  Department, University of Bologna, Italy.                     
    ||I||                                                                
    ||T||  HELM is free software; you can redistribute it and/or         
    ||A||  modify it under the terms of the GNU General Public License   
    \   /  version 2 or (at your option) any later version.      
     \ /   This software is distributed as is, NO WARRANTY.     
      V_______________________________________________________________ *)

(* $Id: orderings.ml 9869 2009-06-11 22:52:38Z denes $ *)

let debug s = prerr_endline (Lazy.force s) ;; 
let debug _ = ();; 

let monster = 100;;
    
module type Paramod =
  sig
    type t
    type input
    type state
    type szsontology = 
      | Unsatisfiable of (t Terms.bag * int * int list) list
      | GaveUp 
      | Error of string 
      | Timeout of int * t Terms.bag
    type bag = t Terms.bag * int
    val empty_state : state
    val mk_passive : bag -> input * input -> bag * t Terms.unit_clause
    val mk_goal : bag -> input * input -> bag * t Terms.unit_clause
    val forward_infer_step : 
      state ->
      t Terms.unit_clause ->
      int ->
      state
    val paramod :
      useage:bool ->
      max_steps:int ->
      ?timeout:float ->
      bag -> 
      g_passives:t Terms.unit_clause list -> 
      passives:t Terms.unit_clause list -> szsontology
  end

module Paramod (B : Orderings.Blob) = struct
  module Pp = Pp.Pp (B) 
  module FU = FoUnif.Founif(B) 
  module IDX = Index.Index(B) 
  module Sup = Superposition.Superposition(B) 
  module Utils = FoUtils.Utils(B) 
  module Order = B
  module WeightOrderedPassives =
      struct
        type t = B.t Terms.passive_clause
        let compare = Utils.compare_passive_clauses_weight
      end

  module AgeOrderedPassives =
      struct
        type t = B.t Terms.passive_clause
        let compare = Utils.compare_passive_clauses_age
      end
  
  module WeightPassiveSet = Set.Make(WeightOrderedPassives)
  module AgePassiveSet = Set.Make(AgeOrderedPassives)

  type t = B.t
  type input = B.input
  type state = 
      B.t Terms.bag 
      * int 
      * Index.Index(B).active_set 
      * (WeightPassiveSet.t * AgePassiveSet.t) 
      * B.t Terms.unit_clause list 
      * (WeightPassiveSet.t * AgePassiveSet.t)
  type szsontology = 
    | Unsatisfiable of (B.t Terms.bag * int * int list) list
    | GaveUp 
    | Error of string 
    | Timeout of int * B.t Terms.bag
  exception Stop of szsontology
  type bag = B.t Terms.bag * int

  let empty_state = 
    Terms.empty_bag,
    0,
    ([],IDX.DT.empty),
    (WeightPassiveSet.empty,AgePassiveSet.empty),
    [],
    (WeightPassiveSet.empty,AgePassiveSet.empty)
;;

  let add_passive_clause ?(no_weight=false) (passives_w,passives_a) cl =
    let cl = if no_weight then (0,cl)
    else Utils.mk_passive_clause cl in
    WeightPassiveSet.add cl passives_w, AgePassiveSet.add cl passives_a
  ;;

  let add_passive_goal ?(no_weight=false) (passives_w,passives_a) g =
    let g = if no_weight then (0,g)
    else Utils.mk_passive_goal g in
    WeightPassiveSet.add g passives_w, AgePassiveSet.add g passives_a
  ;;

  let remove_passive_clause (passives_w,passives_a) cl =
    let passives_w = WeightPassiveSet.remove cl passives_w in
    let passives_a = AgePassiveSet.remove cl passives_a in
      passives_w,passives_a
  ;;

  let add_passive_clauses ?(no_weight=false)
      (passives_w,passives_a) new_clauses =
    let new_clauses_w,new_clauses_a =
      List.fold_left (add_passive_clause ~no_weight)
      (WeightPassiveSet.empty,AgePassiveSet.empty) new_clauses
    in
      (WeightPassiveSet.union new_clauses_w passives_w,
       AgePassiveSet.union new_clauses_a passives_a)
  ;;

  let add_passive_goals ?(no_weight=false)
      (passives_w,passives_a) new_clauses =
    let new_clauses_w,new_clauses_a =
      List.fold_left (add_passive_goal ~no_weight)
      (WeightPassiveSet.empty,AgePassiveSet.empty) new_clauses
    in
      (WeightPassiveSet.union new_clauses_w passives_w,
       AgePassiveSet.union new_clauses_a passives_a)
  ;;

  let is_passive_set_empty (passives_w,passives_a) =
    if (WeightPassiveSet.is_empty passives_w) then begin
      assert (AgePassiveSet.is_empty passives_a); true
    end else begin
      assert (not (AgePassiveSet.is_empty passives_a)); false
    end
  ;;

  let passive_set_cardinal (passives_w,_) = WeightPassiveSet.cardinal passives_w
  
  let passive_empty_set =
    (WeightPassiveSet.empty,AgePassiveSet.empty)
  ;;

  let pick_min_passive ~use_age (passives_w,passives_a) =
    if use_age then AgePassiveSet.min_elt passives_a
    else WeightPassiveSet.min_elt passives_w
  ;;

  let mk_clause bag maxvar (t,ty) =
    let (proof,ty) = B.saturate t ty in
    let c, maxvar = Utils.mk_unit_clause maxvar ty proof in
    let bag, c = Terms.add_to_bag c bag in
    (bag, maxvar), c
  ;;
  
  let mk_passive (bag,maxvar) = mk_clause bag maxvar;;
  let mk_goal (bag,maxvar) = mk_clause bag maxvar;;

  (* TODO : global age over facts and goals (without comparing weights) *)
  let select ~use_age passives g_passives =
    if is_passive_set_empty passives then begin
      if (is_passive_set_empty g_passives) then
        raise (Stop GaveUp) (* we say we are incomplete *)
      else
       let g_cl = pick_min_passive ~use_age:use_age g_passives in
        (true,g_cl,passives,remove_passive_clause g_passives g_cl)
    end
    else let cl = pick_min_passive ~use_age:use_age passives in
      if is_passive_set_empty g_passives then
        (false,cl,remove_passive_clause passives cl,g_passives)
      else
        let g_cl = pick_min_passive ~use_age:use_age g_passives in
	let (id1,_,_,_),(id2,_,_,_) = snd cl, snd g_cl in
	let cmp = if use_age then id1 <= id2
	else fst cl <= fst g_cl
	in
          if cmp then
            (false,cl,remove_passive_clause passives cl,g_passives)
          else
            (true,g_cl,passives,remove_passive_clause g_passives g_cl)
  ;;

  let backward_infer_step bag maxvar actives passives
                          g_actives g_passives g_current iterno =
    (* superposition left, simplifications on goals *)
      debug (lazy "infer_left step...");
      let bag, maxvar, new_goals = 
        Sup.infer_left bag maxvar g_current actives 
      in
        debug (lazy "Performed infer_left step");
	let bag = Terms.replace_in_bag (g_current,false,iterno) bag in
        bag, maxvar, actives, passives, g_current::g_actives,
    (add_passive_goals g_passives new_goals)
  ;;

  let forward_infer_step (bag,maxvar,actives,passives,g_actives,g_passives)  
      current iterno =
    (* forward step *)
    
    (* e = select P           *
     * e' = demod A e         *
     * A' = demod [e'] A      *
     * A'' = A' + e'          *
     * e'' = fresh e'         *
     * new = supright e'' A'' *
     * new'= demod A'' new    *
     * P' = P + new'          *)
    debug (lazy "Forward infer step...");
    debug (lazy("Number of actives : " ^ (string_of_int (List.length (fst actives)))));
    let bag, maxvar, actives, new_clauses = 
      Sup.infer_right bag maxvar current actives 
    in
      debug (lazy "Demodulating goals with actives...");
      (* keep goals demodulated w.r.t. actives and check if solved *)
      let bag, g_actives = 
        List.fold_left 
          (fun (bag,acc) c -> 
             match 
               Sup.simplify_goal ~no_demod:false maxvar (snd actives) bag acc c
             with
               | None -> bag, acc
               | Some (bag,c1) -> bag,if c==c1 then c::acc else c::c1::acc)
          (bag,[]) g_actives 
      in
      let ctable = IDX.index_unit_clause maxvar IDX.DT.empty current in
      let bag, maxvar, new_goals = 
        List.fold_left 
          (fun (bag,m,acc) g -> 
             let bag, m, ng = Sup.infer_left bag m g ([current],ctable) in
               bag,m,ng@acc) 
          (bag,maxvar,[]) g_actives 
      in
      let bag = Terms.replace_in_bag (current,false,iterno) bag in
	(* prerr_endline (Pp.pp_bag bag); *)
    bag, maxvar, actives,
    add_passive_clauses passives new_clauses, g_actives,
    add_passive_goals g_passives new_goals
  ;;
 
  let rec given_clause ~useage ~noinfer 
    bag maxvar iterno weight_picks max_steps timeout 
    actives passives g_actives g_passives 
  =
    let iterno = iterno + 1 in
    if iterno = max_steps then raise (Stop (Timeout (maxvar,bag)));
    (* timeout check: gettimeofday called only if timeout set *)
    if timeout <> None &&
      (match timeout with
      | None -> assert false
      | Some timeout -> Unix.gettimeofday () > timeout) then
        if noinfer then
          begin
            debug 
              (lazy("Last chance: all is indexed " ^ string_of_float
                (Unix.gettimeofday())));
            let maxgoals = 100 in
            ignore(List.fold_left 
              (fun (acc,i) x -> 
                 if i < maxgoals then
                 ignore(Sup.simplify_goal ~no_demod:true 
                          maxvar (snd actives) bag acc x)
                 else
                   ();
                 x::acc,i+1)
              ([],0) g_actives);
            raise (Stop (Timeout (maxvar,bag)))
          end
        else if false then (* activates last chance strategy *)
          begin
           debug (lazy("Last chance: "^string_of_float (Unix.gettimeofday())));
           given_clause ~useage ~noinfer:true bag maxvar iterno weight_picks max_steps 
             (Some (Unix.gettimeofday () +. 20.))
             actives passives g_actives g_passives;
           raise (Stop (Timeout (maxvar,bag)));
          end
        else raise (Stop (Timeout (maxvar,bag)));

    let use_age = useage && (weight_picks = (iterno / 6 + 1)) in
    let weight_picks = if use_age then 0 else weight_picks+1
    in

    let rec aux_select bag passives g_passives =
      let backward,(weight,current),passives,g_passives =
        select ~use_age passives g_passives
      in
	if use_age && weight > monster then
	  let bag,cl = Terms.add_to_bag current bag in
	    if backward then
	      aux_select bag passives (add_passive_clause g_passives cl)
	    else
	      aux_select bag (add_passive_clause passives cl) g_passives
	else
	  let bag = Terms.replace_in_bag (current,false,iterno) bag in
        if backward then
          let _ = debug (lazy("Selected goal : " ^ Pp.pp_unit_clause current)) in
         match 
           if noinfer then 
             if weight > monster then None else Some (bag,current)
           else 
             Sup.simplify_goal 
               ~no_demod:false maxvar (snd actives) bag g_actives current 
         with
            | None -> aux_select bag passives g_passives
            | Some (bag,g_current) ->
               if noinfer then 
                 let g_actives = g_current :: g_actives in 
                 bag,maxvar,actives,passives,g_actives,g_passives
               else
                 backward_infer_step bag maxvar actives passives
                   g_actives g_passives g_current iterno
        else
          let _ = debug (lazy("Selected fact : " ^ Pp.pp_unit_clause current)) in
	  (*let is_orphan = Sup.orphan_murder bag (fst actives) current in*)
          match 
            if noinfer then 
              if weight > monster then bag,None 
              else  bag, Some (current,actives)
            else if Sup.orphan_murder bag (fst actives) current then
	      let _ = debug (lazy "Orphan murdered") in
              let bag = Terms.replace_in_bag (current,true,iterno) bag in
                bag, None
            else Sup.keep_simplified current actives bag maxvar
          with
        (*match Sup.one_pass_simplification current actives bag maxvar with*)
              | bag,None -> aux_select bag passives g_passives
              | bag,Some (current,actives) ->
(*                    if is_orphan then prerr_endline
                      ("WRONG discarded: " ^ (Pp.pp_unit_clause current));
                  List.iter (fun x ->
                               prerr_endline (Pp.pp_unit_clause x))
                    (fst actives);*)

(*                  List.iter (fun (id,_,_,_) -> let (cl,d) =
                             Terms.M.find id bag in 
                             if d then prerr_endline
                               ("WRONG discarded: " ^ (Pp.pp_unit_clause cl)))
                    (current::fst actives);*)
                  if noinfer then
                    let actives = 
                      current::fst actives,
                      IDX.index_unit_clause maxvar (snd actives) current
                    in
                    bag,maxvar,actives,passives,g_actives,g_passives
                  else
                    forward_infer_step 
		      (bag,maxvar,actives,passives,g_actives,g_passives)
		      current iterno
    in
    

      (*prerr_endline "Active table :"; 
       (List.iter (fun x -> prerr_endline (Pp.pp_unit_clause x))
          (fst actives)); *)

    let bag,maxvar,actives,passives,g_actives,g_passives =      
      aux_select bag passives g_passives
    in
      debug
        (lazy(Printf.sprintf "Number of active goals : %d"
           (List.length g_actives)));
      debug
        (lazy(Printf.sprintf "Number of passive goals : %d"
           (passive_set_cardinal g_passives)));
      debug
        (lazy(Printf.sprintf "Number of actives : %d" (List.length (fst actives))));
      debug
        (lazy(Printf.sprintf "Number of passives : %d"
           (passive_set_cardinal passives)));
      given_clause ~useage ~noinfer
        bag maxvar iterno weight_picks max_steps timeout 
        actives passives g_actives g_passives
  ;;

  let paramod ~useage ~max_steps ?timeout (bag,maxvar) ~g_passives ~passives =
    let _initial_timestamp = Unix.gettimeofday () in
    let passives =
      add_passive_clauses ~no_weight:true passive_empty_set passives
    in
    let g_passives =
      add_passive_goals ~no_weight:true passive_empty_set g_passives
    in
    let g_actives = [] in
    let actives = [], IDX.DT.empty in
    try 
     given_clause ~useage ~noinfer:false
      bag maxvar 0  0 max_steps timeout actives passives g_actives g_passives
    with 
    | Sup.Success (bag, _, (i,_,_,_)) ->
        let l =
          let rec traverse ongoal (accg,acce) i =
            match Terms.get_from_bag i bag with
              | (id,_,_,Terms.Exact _),_,_ ->
                  if ongoal then [i],acce else
                    if (List.mem i acce) then accg,acce else accg,acce@[i]
              | (_,_,_,Terms.Step (_,i1,i2,_,_,_)),_,_ ->
                  if (not ongoal) && (List.mem i acce) then accg,acce
                  else
                    let accg,acce = 
                      traverse false (traverse ongoal (accg,acce) i1) i2
                    in
                      if ongoal then i::accg,acce else accg,i::acce
          in
          let gsteps,esteps = traverse true ([],[]) i in
            (List.rev esteps)@gsteps
        in
	let max_w = List.fold_left (fun acc i ->
		    let (cl,_,_) = Terms.get_from_bag i bag in
		    max acc (Order.compute_unit_clause_weight cl)) 0 l in
	  prerr_endline "Statistics :";
	  prerr_endline ("Max weight : " ^ (string_of_int max_w));
(*	  List.iter (fun id -> let ((_,lit,_,proof as cl),d,it) =
	    Terms.get_from_bag id bag in
	      if d then
		prerr_endline
		(Printf.sprintf "Id : %d, selected at %d, weight %d,disc, by %s"
		   id it (Order.compute_unit_clause_weight cl) 
		   (Pp.pp_proof_step proof))
	      else
	       prerr_endline
		(Printf.sprintf "Id : %d, selected at %d, weight %d by %s"
		   id it (Order.compute_unit_clause_weight cl) 
		   (Pp.pp_proof_step proof))) l;*)
        (* 
        prerr_endline "Proof:"; 
        List.iter (fun x ->
          prerr_endline (Pp.pp_unit_clause (fst(Terms.M.find x bag)))) l;
        *)
        Unsatisfiable [ bag, i, l ]
    | Stop (Unsatisfiable _) -> Error "stop bug solution found!"
    | Stop o -> o
  ;;

end
