(*
    ||M||  This file is part of HELM, an Hypertextual, Electronic        
    ||A||  Library of Mathematics, developed at the Computer Science     
    ||T||  Department, University of Bologna, Italy.                     
    ||I||                                                                
    ||T||  HELM is free software; you can redistribute it and/or         
    ||A||  modify it under the terms of the GNU General Public License   
    \   /  version 2 or (at your option) any later version.      
     \ /   This software is distributed as is, NO WARRANTY.     
      V_______________________________________________________________ *)

(* $Id: nCicRefiner.mli 9227 2008-11-21 16:00:06Z tassi $ *)

module COT : Set.OrderedType with type t = NCic.term = 
  struct
        type t = NCic.term
        let compare = Pervasives.compare
  end

module HintSet = Set.Make(COT)

module DB = 
  Discrimination_tree.Make(NDiscriminationTree.NCicIndexable)(HintSet)

type db = DB.t

let dummy = NCic.Const (NReference.reference_of_string "cic:/dummy_conv.dec");;
let pair t1 t2 = (NCic.Appl [dummy;t1;t2]) ;;

let index_hint hdb context t1 t2 =
  assert (
    (match t1 with
    | NCic.Meta _ | NCic.Appl (NCic.Meta _ :: _) -> false | _ -> true)
    &&
    (match t2 with
    | NCic.Meta _ | NCic.Appl (NCic.Meta _ :: _) -> false | _ -> true)
  );
  let pair' = pair t1 t2 in
  let data = 
    List.fold_left 
     (fun t (n,e) -> 
        match e with
        | NCic.Decl ty -> NCic.Prod (n,ty,t)
        | _ -> assert false) 
      pair' context in
  let src = 
    List.fold_left 
     (fun t (_,e) -> 
        match e with
        | NCic.Decl _ -> 
            NCicSubstitution.subst (NCic.Meta (-1,(0,NCic.Irl 0))) t
        | _ -> assert false) 
     pair' context in
  prerr_endline ("INDEXING: " ^ 
    NCicPp.ppterm ~metasenv:[] ~subst:[] ~context:[] src ^ " |==> " ^
    NCicPp.ppterm ~metasenv:[] ~subst:[] ~context:[] data);
  DB.index hdb src data
;;

let empty_db = DB.empty ;;

let user_db = ref empty_db ;;

let add_user_provided_hint t =
  let u = UriManager.uri_of_string "cic:/foo/bar.con" in
  let c, a, b = 
    let rec aux ctx = function
      | NCic.Appl l ->
          (match List.rev l with
          | b::a::_ -> ctx, a, b
          | _ -> assert false)
      | NCic.Prod (n,s,t) ->
          aux ((n, NCic.Decl s) :: ctx) t
      | _ -> assert false
    in
      aux [] (fst (OCic2NCic.convert_term u t))
  in
  user_db := index_hint !user_db c a b 
;;

let db () = 
  let combine f cmp l1 l2 =
   List.flatten
     (List.map
       (fun u1 -> 
          HExtlib.filter_map 
            (fun u2 -> if cmp u1 u2 then None else Some (f u1 u2)) l2)
       l1)
  in
  let mk_hint (u1,_,_) (u2,_,_) = 
    let l = OCic2NCic.convert_obj u1 
      (fst (CicEnvironment.get_obj CicUniv.oblivion_ugraph u1)) in
    let r = OCic2NCic.convert_obj u2 
      (fst (CicEnvironment.get_obj CicUniv.oblivion_ugraph u2)) in
    match List.hd l,List.hd r with
    | (_,_,_,_,NCic.Constant (_,_,Some l,_,_)), 
      (_,_,_,_,NCic.Constant (_,_,Some r,_,_)) ->
        let rec aux ctx t1 t2 =
          match t1, t2 with
          | NCic.Lambda (n1,s1,b1), NCic.Lambda(_,s2,b2) ->
              if NCicReduction.are_convertible ~subst:[] ~metasenv:[] ctx s1 s2
              then aux ((n1, NCic.Decl s1) :: ctx) b1 b2
              else None
          | b1,b2 -> 
              if NCicReduction.are_convertible ~subst:[] ~metasenv:[] ctx b1 b2 
              then begin
(*
                prerr_endline ("hint: " ^ NCicPp.ppterm ~metasenv:[] ~subst:[]
                  ~context:ctx b1 ^ " === " ^ NCicPp.ppterm ~metasenv:[]
                  ~subst:[] ~context:ctx b2);
*)
                Some (ctx,b1,b2)
              end else None
        in
          aux [] l r
    | _ -> None
  in
  let hints = 
    List.fold_left 
      (fun acc (_,_,l) -> 
          acc @ 
          if List.length l > 1 then 
           combine mk_hint (fun (u1,_,_) (u2,_,_) -> UriManager.eq u1 u2) l l
          else [])
      [] (CoercDb.to_list ())
  in
  List.fold_left 
    (fun db -> function 
     | None -> db 
     | Some (ctx,b1,b2) -> index_hint db ctx b1 b2)
    !user_db hints
;;


let look_for_hint hdb metasenv subst context t1 t2 =
(*
  prerr_endline ("KEY1: "^NCicPp.ppterm ~metasenv ~subst ~context (pair t1 t2));
  prerr_endline ("KEY2: "^NCicPp.ppterm ~metasenv ~subst ~context (pair t2 t1));
  DB.iter hdb
   (fun p ds ->
      prerr_endline ("ENTRY: " ^
      NDiscriminationTree.NCicIndexable.string_of_path p ^ " |--> " ^
      String.concat "|" (List.map (NCicPp.ppterm ~metasenv:[] ~subst:[]
      ~context:[]) (HintSet.elements ds))));
*)
  let candidates1 = DB.retrieve_unifiables hdb (pair t1 t2) in
  let candidates2 = DB.retrieve_unifiables hdb (pair t2 t1) in
  let candidates1 = 
    List.map (fun ty -> 
       true,NCicMetaSubst.saturate ~delta:max_int metasenv subst context ty 0) 
    (HintSet.elements candidates1) 
  in
  let candidates2 = 
    List.map (fun ty -> 
       false,NCicMetaSubst.saturate ~delta:max_int metasenv subst context ty 0) 
    (HintSet.elements candidates2) 
  in
  List.map
   (function 
    | (true,(NCic.Appl [_; t1; t2],metasenv,_)) -> metasenv, t1, t2
    | (false,(NCic.Appl [_; t1; t2],metasenv,_)) -> metasenv, t2, t1
    | _ -> assert false)
   (candidates1 @ candidates2)
;;
