(* Copyright (C) 2006, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://helm.cs.unibo.it/
 *)

type statement_test =
  GrafiteParser.ast_statement GrafiteParser.localized_option -> bool

let get_loc =
  function
  | GrafiteParser.LSome (GrafiteAst.Executable (loc, _))
  | GrafiteParser.LSome (GrafiteAst.Comment (loc, _))
  | GrafiteParser.LNone loc ->
      loc

let grep_statement ?(status = LexiconEngine.initial_status) ?(callback = ignore)
  ~fname ~include_paths test
=
  let content = HExtlib.input_file fname in
  let lexbuf = Ulexing.from_utf8_string content in
  let parse_fun status =
    GrafiteParser.parse_statement lexbuf ~include_paths status in
  let rec exaust acc status = (* extract "interesting" statement locations *)
    let stm =
      try Some (parse_fun status)
      with End_of_file -> None in
    match stm with
    | None -> List.rev acc
    | Some (status, stm) when test stm -> (* "interesting" statement *)
        let loc_begin, loc_end = HExtlib.loc_of_floc (get_loc stm) in
        let raw_statement =
          Netconversion.ustring_sub `Enc_utf8 loc_begin (loc_end - loc_begin)
            content in
        callback raw_statement;
        exaust (raw_statement :: acc) status
    | Some (status, _stm) -> exaust acc status in
  exaust [] status

let ma_extension_test fname = Filename.check_suffix fname ".ma"

let rgrep_statement ?status ?callback ?(fname_test = ma_extension_test)
  ~dirname ~include_paths test
=
  let files = HExtlib.find ~test:fname_test dirname in
  List.flatten
    (List.map
      (fun fname ->
        let callback =
          match callback with
          | None -> None
          | Some f -> Some (fun s -> f (fname, s)) in
        List.map (fun raw -> fname, raw)
          (grep_statement ?status ?callback ~fname ~include_paths test))
      files)

