/* 

Minitime - A mini dockable clock.

Copyright (c) 2003 Matthew Allum

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution.
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <libmb/mb.h>

#ifdef USE_XFT
#include <X11/Xft/Xft.h>
#endif

#ifdef USE_XFT
XftFont  *Fnt    = NULL;
XftDraw  *XftDrw = NULL;
XftColor  Col;
#endif

MBPixbuf *Pixbuf;

void usage()
{

  exit(1);
}

static void
set_fg_col(MBTrayApp *app, char* spec)
{
#ifdef USE_XFT   
  XRenderColor  colortmp;
#endif

  XColor col;
  if (!XParseColor(mb_tray_app_xdisplay(app), 
		   DefaultColormap(mb_tray_app_xdisplay(app), 
				   mb_tray_app_xscreen(app)), 
		   spec, &col))
    {
      fprintf(stderr, "minitime: failed to parse color %s\n", spec);
      return;
    } else {
      XAllocColor(mb_tray_app_xdisplay(app), 
		   DefaultColormap(mb_tray_app_xdisplay(app), 
				   mb_tray_app_xscreen(app)), 
		  &col);
    }
#ifdef USE_XFT   
  colortmp.red   = col.red;
  colortmp.green = col.green;
  colortmp.blue  = col.blue;
  colortmp.alpha = 0xffff; 

  XftColorAllocValue(mb_tray_app_xdisplay(app), 
		     DefaultVisual(mb_tray_app_xdisplay(app), 
				   mb_tray_app_xscreen(app)),
		     DefaultColormap(mb_tray_app_xdisplay(app), 
				     mb_tray_app_xscreen(app)),
		     &colortmp, &Col);
#endif
}


void
paint_callback ( MBTrayApp *app, Drawable drw )
{
#ifdef USE_XFT	 	 
  struct timeval   tv;
  struct timezone  tz;
  struct tm       *localTime = NULL; 
  time_t           actualTime;
  char             timestr[5] = { 0 };
  MBPixbufImage   *img_bg = NULL;

  /* Figure out  the actual time */
  gettimeofday(&tv, &tz);
  actualTime = tv.tv_sec;
  localTime = localtime(&actualTime);

  sprintf(timestr, "%.2d:%.2d", localTime->tm_hour, localTime->tm_min);

  img_bg = mb_tray_app_get_background (app, Pixbuf);


  if (XftDrw) XftDrawDestroy (XftDrw);


  if (mb_tray_app_tray_is_vertical(app))
    {
      /* 
	 - create a new HxW pixmap
	 - rotate background img +90 onto it
	 - render the text to it
	 - call drawable to pixbuf
	 - rotate the new pixbuf -90 
      */
      Pixmap pxm_rot;
      MBPixbufImage *img_bg_rot, *img_txt, *img_txt_rot;

      int font_y = ((mb_tray_app_width(app) - (Fnt->ascent+Fnt->descent))/2);

      img_bg_rot = mb_pixbuf_img_transform (Pixbuf, img_bg,
					    MBPIXBUF_TRANS_ROTATE_90);

      pxm_rot = XCreatePixmap(mb_tray_app_xdisplay(app), 
			      mb_tray_app_xrootwin(app), 
			      mb_pixbuf_img_get_width(img_bg_rot), 
			      mb_pixbuf_img_get_height(img_bg_rot), 
			      DefaultDepth(mb_tray_app_xdisplay(app), 
					   mb_tray_app_xscreen(app)));

      mb_pixbuf_img_render_to_drawable (Pixbuf, img_bg_rot, pxm_rot, 0, 0);

      XftDrw = XftDrawCreate(mb_tray_app_xdisplay(app), pxm_rot,
			     DefaultVisual(mb_tray_app_xdisplay(app), 
					   mb_tray_app_xscreen(app)),
			     DefaultColormap(mb_tray_app_xdisplay(app), 
					     mb_tray_app_xscreen(app)));

      XftDrawString8(XftDrw, &Col, Fnt, 1, font_y + Fnt->ascent,
		     (unsigned char *) timestr, strlen(timestr));

      img_txt = mb_pixbuf_img_new_from_drawable (Pixbuf,
						 pxm_rot,
						 None,
						 0, 0,
						 mb_pixbuf_img_get_width(img_bg_rot), 
						 mb_pixbuf_img_get_height(img_bg_rot)); 

      img_txt_rot = mb_pixbuf_img_transform (Pixbuf, img_txt,
					     MBPIXBUF_TRANS_ROTATE_90);
      
      mb_pixbuf_img_render_to_drawable (Pixbuf, img_txt_rot, drw, 0, 0);

      mb_pixbuf_img_free(Pixbuf, img_bg_rot); 
      mb_pixbuf_img_free(Pixbuf, img_txt);
      mb_pixbuf_img_free(Pixbuf, img_txt_rot);

      XFreePixmap(mb_tray_app_xdisplay(app), pxm_rot);

    } else {
      int font_y = ((mb_tray_app_height(app) - (Fnt->ascent+Fnt->descent))/2);

      mb_pixbuf_img_render_to_drawable (Pixbuf, img_bg, drw, 0, 0);

      XftDrw = XftDrawCreate(mb_tray_app_xdisplay(app), drw,
			     DefaultVisual(mb_tray_app_xdisplay(app), 
					   mb_tray_app_xscreen(app)),
			     DefaultColormap(mb_tray_app_xdisplay(app), 
					     mb_tray_app_xscreen(app)));
      
      XftDrawString8(XftDrw, &Col, Fnt, 1, font_y + Fnt->ascent,
		     (unsigned char *) timestr, strlen(timestr));
    }
  mb_pixbuf_img_free(Pixbuf, img_bg);

#endif
}

void
button_callback (MBTrayApp *app, int x, int y, Bool is_released )
{
  if (is_released)
    {
      time_t now;
      now = time (0);
      mb_tray_app_tray_send_message(app, ctime(&now), 6000);
    }
}

void
resize_callback ( MBTrayApp *app, int w, int h )
{
#ifdef USE_XFT   
  XGlyphInfo extents;
  char *font_name   = NULL;
  int   pt_sizes[]  = { 72, 48, 32, 24, 20, 16, 12, 10, 8, 7, 6, 5, 0 }; 
  int   i           = 0;
  Bool  found       = False;

  font_name = alloca(sizeof(char)*(strlen("Sans Serif:")+15));


  while (pt_sizes[i] && !found)
    {
      if (Fnt) XftFontClose(mb_tray_app_xdisplay(app), Fnt);

      sprintf(font_name, "Sans Serif-%i:bold:", pt_sizes[i]);

      if ((Fnt = XftFontOpenName(mb_tray_app_xdisplay(app), 
				 mb_tray_app_xscreen(app), font_name)) == NULL)
	  { fprintf(stderr, "mbmenu: Cant open font %s\n", font_name); exit(0); }

      if ( (Fnt->ascent + Fnt->descent) 
	   < (mb_tray_app_tray_is_vertical(app) ? w : h))
	{
	  found = True;
	}
      else i++;
    }

  if (!found)
    {
      fprintf(stderr, "mbmenu: Cant find a suitable font!\n"); exit(0);
    }

  XftTextExtents8(mb_tray_app_xdisplay(app), Fnt, 
		  (unsigned char *) "99999", 5, &extents);


  if (mb_tray_app_tray_is_vertical(app))
    {
      mb_tray_app_request_size (app, w, extents.width+2);
    } else {
      mb_tray_app_request_size (app, extents.width+2, h);
    }

#endif
}

void 
theme_change_callback (MBTrayApp *app, char *theme_name )
{
  char *theme_path = mb_util_get_theme_full_path(theme_name);

  if (theme_path)
    {
      MBDotDesktop *theme  = NULL;
      char *theme_desktop_path = alloca(strlen(theme_path)+15);

      sprintf(theme_desktop_path, "%s/theme.desktop", theme_path);

      theme = mb_dotdesktop_new_from_file(theme_desktop_path);
      if (theme)
	{
	  /* Get the PanelFgColor key value if exists */
	  if (mb_dotdesktop_get(theme, "PanelFgColor"))
	    {
	      /* Set out font foreground color and repaint */
	      set_fg_col(app, mb_dotdesktop_get(theme, "PanelFgColor"));
	    }
	  mb_dotdesktop_free(theme);
	}
    }
}

void
timeout_callback ( MBTrayApp *app )
{
  mb_tray_app_repaint (app);
}

int 
main(int argc, char **argv)
{
  MBTrayApp *app = NULL;
  MBPixbufImage *img_icon = NULL;
  struct timeval tv;
  char *icon_path;

#ifndef USE_XFT
  fprintf(stderr, "Minitime needs XFT to work.\nPlease rebuild matchbox with --enable-xft\n"); 
  exit(0);
#endif

  app = mb_tray_app_new ( "Clock",
			  resize_callback,
			  paint_callback,
			  &argc,
			  &argv );  

  if (app == NULL) usage();

  Pixbuf = mb_pixbuf_new(mb_tray_app_xdisplay(app), 
			 mb_tray_app_xscreen(app));

  set_fg_col(app, "#000000");
  
  memset(&tv,0,sizeof(struct timeval));
  tv.tv_sec = 30;

  mb_tray_app_set_timeout_callback (app, timeout_callback, &tv); 

  mb_tray_app_set_button_callback (app, button_callback );

  mb_tray_app_set_theme_change_callback (app, theme_change_callback );

  if ((icon_path = mb_dot_desktop_icon_get_full_path (NULL, 
						     16, 
						      "minitime.png"))
      != NULL)
    {
      if ((img_icon = mb_pixbuf_img_new_from_file(Pixbuf, icon_path)) != NULL)
	mb_tray_app_set_icon(app, Pixbuf, img_icon);
    }

  mb_tray_app_main (app);

  return 1;
}
