/* This file is part of the KDE project

   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef MAILBOXLISTVIEW_H
#define MAILBOXLISTVIEW_H

#include "datalistview.h"
#include "mailodybaselistview.h"

namespace Mailody {

class MessageData;
class MailBoxListView;
class MailBoxListViewItem;
class ToolTip;

/**
 * @class MailBoxListView
 * This holds all the mailboxes, reimplemented to auto open when something
 * is dragged over a closed items. Also it toggles the open state when
 * doubleclicking on an item.
 * @author Tom Albers <tomalbers@kde.nl>
 */
class MailBoxListView : public DataListView
{
    Q_OBJECT

    public:
       /**
        * Constructor
        * @param item the parent
        */
        explicit MailBoxListView( QWidget* item );

        /**
         * accept if we can
         */
        virtual bool acceptDrag(QDropEvent *event) const;

        /**
         * returns a MailboxListViewItem with unread messages.
         */
        MailBoxListViewItem* nextWithUnread();

    protected:
        /**
         * reimplemented so the listview opens automatically when
         * there an items is dragged over an item.
         */
        virtual void contentsDragEnterEvent(QDragEnterEvent *e);

        /**
         * reimplemented so the listview opens automatically when
         * there an items is dragged over an item.
         */
        virtual void contentsDragLeaveEvent(QDragLeaveEvent *e);

        /**
         * reimplemented so the listview opens automatically when
         * there an items is dragged over an item.
         */
        virtual void contentsDropEvent(QDropEvent *e);

        /**
         * reimplemented to catch left-click events in the info column
         */
        virtual void contentsMousePressEvent(QMouseEvent *e);

    private:
        QTimer*                 m_drop;
        ToolTip*                m_mbtip;

    private slots:
        void slotOpen();
        void slotToggleOpen(QListViewItem* lv, const QPoint&, int);
        void slotShowMBDetails(MailBoxListViewItem* item, int c);
        void slotHideMBDetails();

    signals:
        /**
          * emitted when the left mouse button is pressed in the info column.
          * @param item the qlistview item.
          * @param c the column
          */
        void infoLeftMouse(MailBoxListViewItem* item, int c);
};

/**
 * @class MailBoxListViewItem
 * This holds one line of a mailbox
 * @author Tom Albers <tomalbers@kde.nl>
 */
class MailBoxListViewItem : public MailodyBaseListViewItem
{
    public:

        /**
         * Constructor
         * @param item the parent
         * @param visual the visual name in the list
         * @param full the full name
         * @param mailcheck if this mailbox should be checked for new mail
         */
        MailBoxListViewItem( MailBoxListViewItem* item, const QString& visual,
                             const QString& full, const bool mailcheck);

        /**
         * Constructor
         * @param item the parent
         * @param visual the visual name in the list
         * @param full the full name
         * @param mailcheck if this mailbox should be checked for new mail
         */
        MailBoxListViewItem( KListView* item, const QString& visual,
                             const QString& full, const bool mailcheck);

        /**
         * Destructor
         */
        ~MailBoxListViewItem();

        /**
         * returns the full name
         */
        QString fullName() const { return m_full;  }

        /**
         * returns the visual name
         */
        QString visName() const  { return m_visualName;  }

        /**
         * returns if this mailbox has to be checked for new mail
         */
        bool check() const       { return m_check; }

        /**
         * call this to check for new mail
         * @param i true to check for mail, false for not
         */
        void setCheck(bool i)    { m_check = i;    }

        /**
         * call this to set this the unseen count to @p i messages
         */
        void updateUnseen( int i );

        /**
         * returns the amount of unseen messages
         */
        int getUnseen() const    { return m_unseen; }

        /**
         * call this to set the number of messages to @p i messages
         */
        void setMessages( int i ) { m_messages = i; }

        /**
         * gets the number of messages
         */
        int getMessages() const { return m_messages; }

        /**
         * call this to set the total number of messages to @p i messages
         */
        void setTotalMessages( int i) { m_totalMessages = i; }

        /**
         * gets the total number of messages
         */
        int getTotalMessages() const { return m_totalMessages; }

        /**
         * reimplemented to color it blue and show the unseen count
         */
        virtual void paintCell( QPainter * p, const QColorGroup & cg,
                        int column, int width, int align );

    private:
        QString     m_full;
        QString     m_visualName;
        bool        m_check;
        int         m_unseen;
        int         m_messages;
        int         m_totalMessages;
};

}

#endif
