/* This file is part of the KDE project

   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>

   This file is based on digikams albumdb which is
   Copyright 2004 by Renchi Raju <no_working@address.known>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/** @file db.h */

#ifndef DB_H
#define DB_H

#include <qstringlist.h>

namespace Mailody {

class DBPriv;

typedef struct sqlite3 sqleet3; // hehe.
typedef QValueList<int>     IntList;
typedef QValueList<Q_LLONG> LLongList;

/**
 * This class is responsible for the communication
 * with the sqlite database.
 */
class DB
{
public:

    /**
     * Constructor
     */
    DB();

    /**
     * Destructor
     */
    ~DB();

    /**
     * Makes a connection to the database and makes sure all tables
     * are available.
     * @param path The database to open
     */
    void setDBPath(const QString& path);

    /**
     * This will execute a given SQL statement to the database.
     * @param sql The SQL statement
     * @param values This will be filled with the result of the SQL statement
     * @param debug If true, it will output the SQL statement
     * @return It will return if the execution of the statement was succesfull
     */
    bool execSql(const QString& sql, QStringList* const values = 0,
                 const bool debug = false);

    void getResult(const QString& sql, QStringList& values);
    bool isValid() const { return m_valid; }
    void beginTransaction();
    void commitTransaction();

    /**
     * To be used only if you are sure of what you are doing
     * @return the last inserted row in one the db table.
     */
    Q_LLONG lastInsertedRow() const;

    void createID( int uid, const QString& mb  );
    void insertMailBox(const QString& mb);
    void renameMailBox(const QString& oldBox, const QString& newBox);
    QStringList getMailBoxList();
    QStringList getCheckMailBoxList();

    void setTotalMessagesMailbox(const QString& box, int amount);
    void setCheckMailMailbox(const QString& box, int check);
    int getTotalMessagesMailbox(const QString& box);
    int getTotalMessagesMessages(const QString& box);
    int getTotalSeenMessagesMessages(const QString& box);
    void getMinMax(const QString& box, int& min, int& max);

    void storeHeaders(int uid, const QString& mb, const QString &header);
    void storeBody(int uid, const QString& mb, const QString &body);

    void deleteMessagesAndMailBoxes();
    void deleteMessagesAndMailBoxes(const QString& mb);
    void deleteMessages(const QString& mb);
    void deleteMessage( int uid, const QString& mb);
    void expunge(const QString& mb);

    QString getHeader( int uid, const QString& mb);
    QString getBody( int uid, const QString& mb );

    bool hasHeader( int uid, const QString& mb );
    void getCurrentMessages( const QString& mb, QStringList& values );
    void search( const QStringList& keywords, QStringList& values);
    void getCurrentMessageIDs( const QString& mb, QStringList& values );
    bool hasBody( int uid, const QString& mb );

    bool hasFlag( int uid, const QString& mb, const QString& flag );
    QString getFlags( int uid, const QString& mb );
    void setFlags( int uid, const QString& mb, const QString& flag );
    void addFlag( int uid, const QString& mb, const QString flag);
    void addFlag( const QString& mb, const QString flag);
    void removeFlag( int uid, const QString& mb, const QString flag);

    bool hasCert( const QString&, int error );
    void addCert( const QString&, int error );

    void getRecentList(QStringList& values);
    void getTopTenRecentList(QStringList& values);
    void addToRecentList(const QString& email, const QString& name);
    void deleteFromRecentList(const QString& email);

    QString getSetting(const QString& keyword);
    void setSetting(const QString& keyword, const QString& value );

    static DB *dbinstance();

private:

    /**
     * Checks the available tables and creates them if they are not
     * available.
     */
    void initDB();

    /**
     * Escapes text fields. This is needed for all queries to the database
     * which happens with an argument which is a text field. It makes sure
     * a ' is replaced with '', as this is needed for sqlite.
     * @param str String to escape
     * @return The escaped string
     */
    QString escapeString(const QString& str) const;

    sqleet3* m_db;
    bool     m_valid;
    static DB*   m_dbinstance;
    DBPriv*   d;
};

}

#endif
