/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file ObsItemFamily.cc
    \brief Implementation of the Template class ObsItemfamily.
    
    Magics Team - ECMWF 2004
    
    Started: Mon 21-Jun-2004
    
    Changes:
    
*/
#include "ObsItemFamily.h"
#include "iomanip"
#include "Transformation.h"

#include "Flag.h"
#include "Text.h"
#include "PaperPoint.h"

using namespace magics;


map<int, string> ObsWind::origins_;

void  ObsWind::setOrigins() 
{
	if ( !origins_.empty() ) return;
	origins_[0] = "N_0";
	origins_[1] = "N_1";
	origins_[2] = "N_2";
	origins_[3] = "N_3";
	origins_[4] = "N_4";
	origins_[5] = "N_5";
	origins_[6] = "N_6";
	origins_[7] = "N_7";
	origins_[8] = "N_8";
}

void ObsWind::operator()(CustomisedPoint& point, ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;

	//Log::dev()<< "Here we try to plot the wind! " << endl;
	//Log::dev()<< "wind_speed = " <<point["wind_speed"] << ", wind_direction" << point["wind_speed"] << "\n";

	CustomisedPoint::const_iterator speed = point.find("wind_speed");
	if ( speed == point.end() ) return; 
	CustomisedPoint::const_iterator direction = point.find("wind_direction");
	if ( direction == point.end() ) return; 

	FlagItem* flag = new FlagItem();
	flag->setColour(attributes_->getColour());
	flag->length(size_ * 3); // Size to be adjusted later!

	map<int, string>::const_iterator marker = origins_.find(int(point["cloud_amount"]));
	const string origin = ( marker != origins_.end()) ? marker->second : "magics_13";
	Log::dev() << "wind origin--->" << origin << "\n";
	const double size = ( marker != origins_.end()) ? size_ : 0.05;
Log::dev()<< "OBS>>> "<<point["cloud_amount"]<<" > "<<origin<<endl;
	
	flag->setOriginMarker(origin);
	flag->setOriginHeight(size);
	flag->x(0);
	flag->y(0);
	flag->speed(speed->second);
	flag->direction(direction->second);
	
	if (point.latitude() <0 ) flag->setHemisphere(SOUTH);
	symbol.add(flag);
}

void ObsPressure::visit(std::set<string>& tokens)
{
	if ( !attributes_ ) attributes_ = new ObsPressureAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("msl_pressure");
}

void ObsPressure::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("msl_pressure");
	if ( value == point.end() ) return;
	
	TextItem*  object = new TextItem();
	// for the pressure we show the 3 last digits of the pressure in decaPascal.
	object->x(column_);
	object->y(row_);
	MagFont font;
	font.colour(attributes_->getColour());
	font.name("sansserif");
	font.size(size_);
	
	ostringstream os;
	double pressure = fmod(value->second/10, 1000);
	os <<  setw(3) << setfill('0') << pressure;
	object->text(os.str());
	Log::dev() << "Pressure--->" << value->second << " = " << pressure << "\n";
	object->font(font);
	symbol.add(object);
}

void ObsPressureLevel::visit(std::set<string>& tokens)
{
	if ( !attributes_ ) attributes_ = new ObsPressureLevelAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("pressure");
}

void ObsPressureLevel::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("pressure");
	if ( value == point.end() ) return;
	TextItem*  object = new TextItem();
	object->x(column_);
	object->y(row_);
	
	MagFont font("sansserif");
	font.colour(attributes_->getColour());
	
	const double pressure = value->second/100;

	object->text(tostring(pressure));
	Log::dev() << "Pressure--->" << value->second << " = " << pressure << "\n";
	font.size(size_);
	object->font(font);
	symbol.add(object);
}
	
void ObsPressureTendency::visit(std::set<string>& tokens)
{
	if ( !attributes_ ) attributes_ = new ObsPressureTendencyAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("pressure_tendency_amount");
	tokens.insert("pressure_tendency_characteristic");
}
	
void ObsPressureTendency::operator()(CustomisedPoint& point, ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("pressure_tendency_amount");
	if ( value == point.end() ) return; 
	TextItem*  object = new TextItem();
	object->x(column_);
	object->y(row_);
	
	ostringstream os;	
	if(value->second>0) os <<  setw(2) << setfill('0') << value->second * .1;
	else os << "-" << setw(2) << setfill('0') << value->second * -.1;
	
	// The Pressure tendancy is red if negative!
	MagFont font("sansserif");
	font.colour(( value->second < 0 ) ? Colour("red") : attributes_->getColour());

	object->text(os.str());
	font.size(size_);
	object->font(font);
	symbol.add(object);
	value = point.find("pressure_tendency_characteristic");
	if ( value == point.end() ) return; 
	SymbolItem*  tendancy = new SymbolItem();
	tendancy->x(column_+1);
	tendancy->y(row_);	
	
	tendancy->colour(attributes_->getColour());
	ostringstream oss;	
	oss << "a_"  << value->second;
	tendancy->symbol(oss.str());
	Log::dev() << "Pressure tendency--->" << oss.str() << "\n";
	tendancy->height(size_);
	symbol.add(tendancy);
}

	
void ObsDewPoint::visit(std::set<string>& tokens)
{
	if ( !attributes_ ) attributes_ = new ObsDewPointAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("dewpoint");
}

void ObsDewPoint::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("dewpoint");
	if ( value == point.end() ) return; 
	TextItem*  object = new TextItem();
	MagFont font("sansserif");
	
	font.colour(attributes_->getColour());
	font.size(size_);
	object->x(column_);
	object->y(row_);
	
	// The temperature is displayed in Celsius.
	const double tempe = maground(value->second-273.25);
	Log::dev() << "DewPoint--->" << point["dewpoint_2meters"] << " = " << tempe << "\n";
	object->text(tostring(tempe));
	object->font(font);
	//object->setJustification(MCENTRE);	
	
	symbol.add(object);
}

	
void ObsVisibility::visit(std::set<string>& tokens)
{
	if ( !attributes_ ) attributes_ = new ObsVisibilityAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("horizontal_visibility");
}
	
void ObsVisibility::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("horizontal_visibility");
	if ( value == point.end() ) return; 
	TextItem*  object = new TextItem();
	object->x(column_);
	object->y(row_);

	MagFont font("sansserif");
	font.colour(attributes_->getColour());
	
	const double vv = value->second;
	if ( vv <= 5000 ) {
		object->text(tostring(vv/100));
		Log::dev() << "Visibility--->" << vv << " = " << vv/100 << "\n";
	}
	else if (vv <= 30000) {
		object->text(tostring(maground(vv/1000) + 50));
		Log::dev() << "Visibility--->" << vv << " = " << maground(vv/1000) + 50 << "\n";
	}	
	else {
		object->text(">59");
		Log::dev() << "Visibility--->" << vv << " = >59\n";
	}
	font.size(size_);
	object->font(font);
	//object->setJustification(MRIGHT);

	symbol.add(object);
}
	
void ObsPresentWeather::visit(std::set<string>& tokens)
{
	if ( !attributes_ ) attributes_ = new ObsPresentWeatherAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("present_weather");
}

void ObsPresentWeather::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("present_weather");
	if ( value == point.end() ) return; 
	SymbolItem*  object = new SymbolItem();
	object->x(column_);
	object->y(row_);
	
	object->colour(attributes_->getColour());
	//object->setFont("sansserif");
	ostringstream os;
	
	os << "ww_" << setw(2) << setfill('0') << value->second;

	
	object->symbol(os.str());
	Log::dev() << "Present Weather--->" << os.str() << "\n";
	//time->setJustification(MRIGHT);
	object->height(size_);
	symbol.add(object);
}


void ObsTemperature::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const
{
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("temperature");
	if ( value == point.end() )  return;

	TextItem*  object = new TextItem();
	MagFont font("sansserif");
	font.colour(attributes_->getColour());
	font.size(size_);
	object->font(font);
	object->x(column_);
	object->y(row_);
	
	// The temperature is displayed in Celsius.		
	double tempe = maground(value->second-273.25);
	Log::dev() << "Temperature--->" << tempe << "\n";
	object->text(tostring(tempe));
	//object->setJustification(MCENTRE);

	symbol.add(object);
}

void ObsTimePlot::visit(std::set<string>&) 
{
	if ( !attributes_ ) attributes_ = new ObsTimePlotAttributes();
	if (!attributes_->getVisible()) return;
}
	
void ObsTimePlot::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const 
{ 
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("time");
	if ( value == point.end() ) return;
	Log::dev() << "TimePlot for " << value->second << "at[" << column_ << ", " << row_ << "]" << endl;
	TextItem*  time = new TextItem();
	MagFont font("sansserif");
	font.colour(attributes_->getColour());
	font.size(size_);
	time->x(column_);
	time->y(row_);
	time->font(font);
	time->text(tostring(value->second));
	//time->setJustification(MRIGHT);
	
	symbol.add(time);
} 

void ObsHeight::visit(std::set<string>& tokens ) 
{
	if ( !attributes_ ) attributes_ = new ObsHeightAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("geopotential");
}
	
void ObsHeight::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const 
{ 
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("geopotential");
	if ( value == point.end() ) return;
	double geop = maground(value->second/98.1);
	Log::dev() << "geopotential for " << geop << "at[" << column_ << ", " << row_ << "]" << endl;
	TextItem*  height = new TextItem();
	MagFont font("sansserif");
	font.colour(attributes_->getColour());
	font.size(size_);
	height->x(column_);
	height->y(row_);

	height->text(tostring(geop));
	//height->setJustification(MLEFT);
	height->font(font);
	symbol.add(height);
} 


void ObsThickness::visit(std::set<string>& tokens ) 
{
	if ( !attributes_ ) attributes_ = new ObsThicknessAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("thickness");
}


void ObsThickness::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const 
{ 
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("thickness");
	if ( value == point.end() ) return;
	const double thickness = maground(value->second/98.1);
	Log::dev() << "thickness for " << thickness << "at[" << column_ << ", " << row_ << "]" << endl;
	TextItem*  object = new TextItem();
	MagFont font("sansserif");
	font.colour(attributes_->getColour());
	font.size(size_);
	object->x(column_);
	object->y(row_);
	
	object->text(tostring(thickness));
	//object->setJustification(MLEFT);
	object->font(font);
	symbol.add(object);
} 


void ObsIdentifier::visit(std::set<string>&) 
{
	if ( !attributes_ ) attributes_ = new ObsIdentifierAttributes();
	if (!attributes_->getVisible()) return;
}

	
void ObsIdentifier::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const 
{ 
	if (!attributes_->getVisible()) return;
	TextItem*  time = new TextItem();
	Log::dev() << "Identification for " << point.identifier() << "at[" << column_ << ", " << row_ << "]" << endl;
	MagFont font("sansserif");
	font.colour(attributes_->getColour());
	font.size(size_);
	time->x(column_);
	time->y(row_);
	time->text(point.identifier());
	//time->setJustification(MRIGHT);
	time->font(font);
	
	symbol.add(time);
} 

void ObsPastWeather::visit(std::set<string>& tokens) 
{
	if ( !attributes_ ) attributes_ = new ObsPastWeatherAttributes();
	if (!attributes_->getVisible()) return;
	tokens.insert("past_weather_1");
	tokens.insert("past_weather_2");
}

void ObsPastWeather::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const 
{ 
	if (!attributes_->getVisible()) return;
	CustomisedPoint::const_iterator value = point.find("past_weather_1");
	if ( value == point.end() ) 
	{
		value = point.find("past_weather_2");
		if ( value == point.end() ) 
		   return;
	}
	SymbolItem*  object = new SymbolItem();
	object->x(column_);
	object->y(row_);
	object->colour(attributes_->getColour());
	ostringstream os;	
	os << "W_"  << value->second;	
	object->symbol(os.str());
	Log::dev() << "Past Weather--->" << os.str() << "\n";
	object->height(size_);
	symbol.add(object);
} 

void ObsCloud::visit(std::set<string>& tokens) 
{
	if ( !attributes_ ) attributes_ = new ObsCloudAttributes();
	if (!attributes_->getVisible()) return;
	if ( attributes_->getLow() ) tokens.insert("low_cloud");
	if ( attributes_->getMedium() ) tokens.insert("medium_cloud");
	if ( attributes_->getHigh() ) tokens.insert("high_cloud");
}

	
void ObsCloud::operator()(CustomisedPoint& point,  ComplexSymbol& symbol) const 
{ 
	if (!attributes_->getVisible()) return;
	
	CustomisedPoint::const_iterator value = point.find("low_cloud");
	if ( value != point.end() ) {		
		SymbolItem*  cloud = new SymbolItem();
		cloud->x(lowColumn_);
	    cloud->y(lowRow_);
		cloud->colour(attributes_->getLow_colour());
		ostringstream oss;	
		int type = int(value->second - (int (value->second/10) *10));
		oss << "CL_"  << type;
		cloud->symbol(oss.str());
		Log::dev() << "Low Cloud--->" << oss.str() << "\n";
		cloud->height(size_);
		symbol.add(cloud);
	}
	
	
	value = point.find("medium_cloud");
	if ( value != point.end() ) {
		SymbolItem*  cloud = new SymbolItem();
		cloud->x(mediumColumn_);
	    cloud->y(mediumRow_);
		
		cloud->colour(attributes_->getMedium_colour());
		ostringstream oss;	
		int type = int(value->second - (int (value->second/10) *10));
		oss << "CM_"  << type;
		cloud->symbol(oss.str());
		Log::dev() << "Medium Cloud--->" << oss.str() << "\n";
		cloud->height(size_);
		symbol.add(cloud);
	}
	
	value = point.find("high_cloud");
	if ( value != point.end() ) {
		SymbolItem*  cloud = new SymbolItem();
		cloud->x(highColumn_);
		cloud->y(highRow_);
		cloud->colour(attributes_->getHigh_colour());
		ostringstream oss;	
		int type = int(value->second - (int (value->second/10) *10));
		oss << "CH_"  << type;
		cloud->symbol(oss.str());
		Log::dev() << "High Cloud--->" << oss.str() << "\n";
		cloud->height(size_);
		symbol.add(cloud);
	}
}

void ObsCloud::set(const map<string, string>& def)
{   
	lowRow_ = atoi(find(def, "low_row").c_str());
	lowColumn_ = atoi(find(def, "low_column").c_str());

	mediumRow_ = atoi(find(def, "medium_row").c_str());
	mediumColumn_ = atoi(find(def, "medium_column").c_str());

	highRow_ = atoi(find(def, "high_row").c_str());
	highColumn_ = atoi(find(def, "high_column").c_str());
}
