/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file NetcdfGeoMatrixInterpretor.cc
    \brief Implementation of the Template class NetcdfGeoMatrixInterpretor.
    
    Magics Team - ECMWF 2004
    
    Started: Tue 17-Feb-2004
    
    Changes:
    
*/

#include "NetcdfGeoMatrixInterpretor.h"
#include "Factory.h"
#include "Netcdf.h"
#include <limits>

using namespace magics;

NetcdfGeoMatrixInterpretor::NetcdfGeoMatrixInterpretor() 
{}


NetcdfGeoMatrixInterpretor::~NetcdfGeoMatrixInterpretor() 
{}

void NetcdfGeoMatrixInterpretor::interpretAsMatrix(Matrix& matrix)
{
	if ( !matrix.empty() ) return;
    
	matrix.missing(std::numeric_limits<double>::max());
    matrix.missing(100000002004087734272.0);
	Netcdf netcdf(path_);
	// get the data ...
	try
	{
		map<string, string> first, last;
		setDimensions(dimension_, first, last);
       
		netcdf.get(longitude_, matrix.columnsAxis(), first, last);
		netcdf.get(latitude_, matrix.rowsAxis(), first, last);
		
		if  ( magCompare(primary_index_, "latitude") ) {
			// WE reserve the matrix since we are used to lat/lon matrix!
			vector<double> data;
			netcdf.get(field_, data, first, last);
			int columns =  matrix.columnsAxis().size();
			int rows =  matrix.rowsAxis().size();
 
		    for (int lon = 0; lon < columns; lon++)
			     for (int lat = 0; lat < rows; lat++) {
					
			    	matrix.push_back(data[lon + lat*rows]);
			     }
		}
		else 		
			netcdf.get(field_, matrix, first, last);
		
		Log::debug() << "data[" << matrix.size() << ", " << scaling_ << ", " << offset_ << "\n";

		matrix.multiply(scaling_);
		matrix.plus(offset_);
        
		Log::debug() << "data[" << matrix.size() << ", " << scaling_ << ", " << offset_ << "\n";
        
		
        
        matrix.setMapsAxis();
        
		Log::dev() << matrix << "\n";
	}
    
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

/*
 Class information are given to the output-stream.
*/		
void NetcdfGeoMatrixInterpretor::print(ostream& out)  const
{
	out << "NetcdfGeoMatrixInterpretor[";
	NetcdfInterpretor::print(out);
	NetcdfGeoMatrixInterpretorAttributes::print(out);
	out << "]";
}

void NetcdfGeoMatrixInterpretor::interpretAsPoints(PointsList<GeoPoint>& list) 
{
	Netcdf netcdf(path_);
	
	// get the data ...
	try
	{
		vector<double> latitudes;
		vector<double> longitudes;
		vector<double> values;
		map<string, string> first, last;
		setDimensions(dimension_, first, last);
		
		netcdf.get(field_, values, first, last);
		netcdf.get(longitude_, longitudes, first, last);
		netcdf.get(latitude_, latitudes, first, last);
		int val = 0;
		
		for (int  lat  =0 ; lat < latitudes.size(); lat+=latitude_sample_) {
			for ( int lon = 0; lon < longitudes.size(); lon+=longitude_sample_) {
				val = (lat* longitudes.size() + lon);
				if (val >= values.size() ) return;
				if ( values[val] < suppress_below_ ) continue;
				if ( values[val] > suppress_above_ ) continue;
				if ( same(values[val], missing_) ) continue;
				list.push_back(GeoPoint(longitudes[lon],latitudes[lat],values[val]));		
				
				
			}
		}
				
       Log::dev()<< "everything ok" << endl;
	}
    
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

