/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file InputData.cc
    \brief Implementation of the Template class InputData.
    
    Magics Team - ECMWF 2004
    
    Started: Thu 6-May-2004
    
    Changes:
    
*/

#include "InputData.h"





/*!
 Class information are given to the output-stream.
*/		
template <class P>
void InputData<P>::print(ostream& out)  const
{
	out << "InputData[";
    InputDataAttributes::print(out);
	out << "]";
}

template <class P>
void  InputData<P>::dateSetting(vector<string>& dates, vector<double>& values, DateTime& base)
{
	if ( dates.empty() )
		return;
	base = DateTime(dates.front());
	for (vector<string>:: iterator date = dates.begin(); date != dates.end(); ++ date ) {
		DateTime d(*date);
		values.push_back(d-base);
	}
}

template <class P>
void  InputData<P>::numberSetting(vector<double>& from, vector<double>& values)
{
	std::copy(from.begin(), from.end(), back_inserter(values));
}

template <class P>
void  InputData<P>::prepare()
{
    if ( !this->x_values_.empty() ) return;

    if ( magCompare(this->x_type_, "date" ) ) {
    	this->dateSetting(this->date_x_, this->x_values_, baseDateX_);
    }
    else {
    	this->numberSetting(this->x_,  this->x_values_);
    	this->numberSetting(this->longitudes_,  this->x_values_);
    }
    if ( magCompare(this->y_type_, "date" ) ) {
      	this->dateSetting(this->date_y_, this->y_values_, baseDateY_);

     }
     else {
      	this->numberSetting(this->y_,  this->y_values_);
      	this->numberSetting(this->latitudes_,  this->y_values_);
     }
    
    vector<double>::iterator x = this->x_values_.begin();
	vector<double>::iterator y = this->y_values_.begin();
	vector<double>::iterator v = this->values_.begin();
    while ( x != this->x_values_.end() && x != this->x_values_.end() ) {
    	double val = 0;
    	if ( v != this->values_.end() ) {
    		val = *v;
    		++v;
    	}
    	push_back(P(*x, *y, val));
    	++x;
    	++y;

    }
  
}
 
template <class P>
void InputData<P>::customisedPoints(const Transformation& transformation, const std::set<string>&, CustomisedPointsList& out)
{
	prepare();
	vector<double>::const_iterator x = x_values_.begin();
	vector<double>::const_iterator y = y_values_.begin();
	vector<double>::const_iterator xc = this->x_component_.begin();
	vector<double>::const_iterator yc = this->y_component_.begin();
	vector<double>::const_iterator v = this->values_.begin();
	while (  x != x_values_.end() && y != y_values_.end()  ) {
		    	CustomisedPoint* point = new CustomisedPoint();


		    		point->longitude(*x);
		    		point->latitude(*y);
		    		out.push_back(point);


		    	if ( x != x_.end() ) {
		    		if ( same(*x, x_missing_) )
		    			point->missing(true);
		    		(*point)["x"] = *(x++);
		    	}
		    	if ( y != y_.end() ) {
		    		if ( same(*y, y_missing_) )
		    			point->missing(true);
		    		(*point)["y"] = *(y++);
		    	}
		        double speed = 0;
		    	if ( xc != this->x_component_.end() && yc != this->y_component_.end() ) {
		    		speed = sqrt( (*xc * *xc)  +  (*yc * *yc) );
		    		(*point)["x_component"] = *(xc++);
		    		(*point)["y_component"] = *(yc++);
		    	}
		    	if ( v != this->values_.end() ) {
		    		(*point)["colour_component"] = *(v++);
		    	}
		    	else
		    		(*point)["colour_component"] = speed;


		    	
		    }
}
template <class P>
void InputData<P>::customisedPoints(const std::set<string>&, CustomisedPointsList& out)
{
	prepare();
	vector<double>::const_iterator x = x_values_.begin();
	vector<double>::const_iterator y = y_values_.begin();
	vector<double>::const_iterator xc = this->x_component_.begin();
	vector<double>::const_iterator yc = this->y_component_.begin();
	vector<double>::const_iterator v = this->values_.begin();
	while (  x != x_values_.end() || y != y_values_.end()  ) {
		    	CustomisedPoint* point = new CustomisedPoint();


		    		point->longitude(*x);
		    		point->latitude(*y);
		    		out.push_back(point);

		    	if ( x != x_.end() ) (*point)["x"] = *(x++);
		    	if ( y != y_.end() ) (*point)["y"] = *(y++);
		        double speed = 0;
		    	if ( xc != this->x_component_.end() && yc != this->y_component_.end() ) {
		    		speed = sqrt( (*xc * *xc)  +  (*yc * *yc) );
		    		(*point)["x_component"] = *(xc++);
		    		(*point)["y_component"] = *(yc++);
		    	}
		    	if ( v != this->values_.end() ) {
		    		(*point)["colour_component"] = *(v++);
		    	}
		    	else
		    		(*point)["colour_component"] = speed;



		    }
}
template <class P>
MatrixHandler<P>& InputData<P>::matrix()
{
	if  ( !matrix_ ) {
		prepare();
		matrix_ = (*binning_)(*this);
	}

	this->matrixHandlers_.push_back(new MatrixHandler<P>(*matrix_));
	return *(this->matrixHandlers_.back());
}

template <class P>
void InputData<P>::getReady(const Transformation& transformation)
{
	try {
		for ( vector<string>::const_iterator x = date_x_.begin(); x != date_x_.end(); ++x )
			x_.push_back(transformation.x(*x));
		}
	catch (...) {}
	try {
		for ( vector<string>::const_iterator y = date_y_.begin(); y != date_y_.end(); ++y )
			y_.push_back(transformation.y(*y));
	}
	catch (...) {}
	
}
template <class P>
void InputData<P>::visit(Transformation& transformation)
{
	// get the data ...
	try {
		prepare();
		if ( transformation.getAutomaticX() ) {
			double min = INT_MAX;
			double max = -INT_MAX;
			for (vector<double>::iterator x = this->x_values_.begin(); x != this->x_values_.end(); ++x) {
				if ( same(*x, x_missing_) ) continue;
				if ( min > *x ) min = *x;
				if ( max < *x ) max = *x;
			}
			if ( magCompare(this->x_type_, "date" ) ) {

				transformation.setDataMinX(min, this->baseDateX_);
				transformation.setDataMaxX(max, this->baseDateX_);
			}
			else {
				transformation.setMinX(min);
				transformation.setMaxX(max);
			}
		}
		if ( transformation.getAutomaticY() ) {
			double min = INT_MAX;
			double max = -INT_MAX;
			for (vector<double>::iterator y = this->y_values_.begin(); y != this->y_values_.end(); ++y) {
				if ( same(*y, y_missing_) ) continue;
				if ( min > *y ) min = *y;
				if ( max < *y ) max = *y;
			}
			if ( magCompare(this->y_type_, "date" ) ) {
				transformation.setDataMinY(min, this->baseDateY_);
				transformation.setDataMaxY(max, this->baseDateY_);
			}
			else {
				transformation.setMinY(min);
				transformation.setMaxY(max);
			}
		}

	}
	catch ( ... ) {}
}

