/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file GeoRectangularProjection.cc
    \brief Implementation of GeoRectangularProjection.
    \author Meteorological Visualisation Section, ECMWF

    Started: Thu Jan 10 17:24:36 2008

*/

#include <GeoRectangularProjection.h>

#include <Polyline.h>
#include <Text.h>

#include <GeoPoint.h>
#include <SceneVisitor.h>
#include <GridPlotting.h>
#include <LabelPlotting.h>


using namespace magics;

/*!
  \brief Constructor
*/
GeoRectangularProjection::GeoRectangularProjection() : projection_(0)
{
	init();
}

/*!
  \brief Destructor

  \todo do we need here a "delete projection_;" as in MercatorProjection::~MercatorProjection() ?
*/
GeoRectangularProjection::~GeoRectangularProjection() 
{
}

void GeoRectangularProjection::print(ostream& o) const
{
	o << "GeoRectangularProjection[";
	GeoRectangularProjectionAttributes::print(o);
	o << "]"; 
} 

PaperPoint GeoRectangularProjection::operator()(const UserPoint& point)  const
{
	MagLog::dev() << "GeoRectangularProjection::operator()(...) needs implementing." << endl;
	return Transformation::operator()(point);
}

PaperPoint GeoRectangularProjection::operator()(const GeoPoint& point)  const
{
	if ( !projection_ ) 
		return PaperPoint(point.x(), point.y(), point.value(), point.missing(), point.border());
	
	TeCoord2D geo = TeCoord2D(point.longitude()*TeCDR, point.latitude()*TeCDR);
	TeCoord2D xy = projection_->LL2PC(geo);

	return PaperPoint(xy.x(), xy.y(), point.value(), point.missing());
}

PaperPoint GeoRectangularProjection::operator()(const PaperPoint& point)  const
{
	MagLog::dev() << "GeoRectangularProjection::operator()(...) needs implementing." << endl;
	return Transformation::operator()(point);
}

void GeoRectangularProjection::revert(const PaperPoint& xy, GeoPoint& point)  const
{
	if ( !projection_ )
	{
		point = GeoPoint(xy.x(), xy.y());
		return;
	}
	TeCoord2D texy = TeCoord2D(xy.x(), xy.y());
	TeCoord2D geo = projection_->PC2LL(texy);  
	point = GeoPoint(geo.x()*TeCRD, geo.y()*TeCRD);	
}

void GeoRectangularProjection::revert(const PaperPoint& xy, UserPoint& point)  const
{
	MagLog::dev() << "GeoRectangularProjection::revert(...) needs implementing." << endl;
	Transformation::revert(xy, point);
}

void GeoRectangularProjection::revert(const vector<pair<double, double> > & input, vector<pair<double, double> > & output) const
{
	output.reserve(input.size());
	const vector<pair<double, double> >::const_iterator in_end =input.end();
	if ( !projection_ )
	{
		for ( vector<pair<double, double> >::const_iterator pt = input.begin();  pt != in_end; ++pt)
		{
			output.push_back(*pt);
		}
		return;
	}
	assert(projection_);

	for ( vector<pair<double, double> >::const_iterator pt = input.begin();  pt != in_end; ++pt)
	{
		TeCoord2D texy = TeCoord2D(pt->first, pt->second);
		TeCoord2D geo = projection_->PC2LL(texy);
		output.push_back(make_pair(geo.x()*TeCRD, geo.y()*TeCRD));
	}
}

bool GeoRectangularProjection::needShiftedCoastlines()  const
{
	return true;
}

void GeoRectangularProjection::boundingBox(double& xmin, double& ymin, 
			double& xmax, double& ymax)  const
{
	xmin = std::min(min_longitude_, max_longitude_);
	xmax = std::max(min_longitude_, max_longitude_);
	ymin = std::min(min_latitude_, max_latitude_);
	ymax = std::max(min_latitude_, max_latitude_);
	const double tol = 3.;
	xmin = xmin - tol;
	xmax = xmax + tol;
	ymin = (ymin < (-90+tol) ) ? -90. : ymin - tol;
	ymax = (ymax > ( 90-tol) ) ?  90. : ymax + tol;
}

void GeoRectangularProjection::smallestBoundingBox(double& xmin, double& ymin,
			double& xmax, double& ymax)  const
{
	xmin = std::min(min_longitude_, max_longitude_);
	xmax = std::max(min_longitude_, max_longitude_);
	ymin = std::min(min_latitude_, max_latitude_);
	ymax = std::max(min_latitude_, max_latitude_);
}

bool GeoRectangularProjection::verifyDef(const string& def) const
{
	return ( def == "EPSG:4326" || def == "CRS:84"); 
}


void GeoRectangularProjection::aspectRatio(double& w, double& h) 
{
	init();
	Transformation::aspectRatio(w, h);
}


double GeoRectangularProjection::getMinX()  const
{
	 return min_longitude_;
}

double GeoRectangularProjection::getMinY()  const
{
	return min_latitude_;
}

double GeoRectangularProjection::getMaxX()  const
{
	return max_longitude_;
}

double GeoRectangularProjection::getMaxY()  const
{
	return max_latitude_;
}

void GeoRectangularProjection::setMinX(double xx)  
{
	min_longitude_ = xx;
}

void GeoRectangularProjection::setMinY(double yy)  
{
	min_latitude_ = yy;
}

void GeoRectangularProjection::setMaxX(double xx)  
{
	max_longitude_ = xx;
}

void GeoRectangularProjection::setMaxY(double yy)  
{
	max_latitude_ = yy;
}

double GeoRectangularProjection::getMinPCX()  const
{
	return xpcmin_;
}

double GeoRectangularProjection::getMinPCY()  const
{
	return ypcmin_;
}

double GeoRectangularProjection::getMaxPCX()  const
{
	return xpcmax_;
}

double GeoRectangularProjection::getMaxPCY()  const
{
	return ypcmax_;
}

void GeoRectangularProjection::gridLongitudes(const GridPlotting& grid)  const
{
	const vector<double>& longitudes = grid.longitudes();
	const double min = std::max(min_latitude_, -90.);
	const double max = std::min(max_latitude_, 90.);
	const double step = (max - min)/20;
	const vector<double>::const_iterator lon_end =longitudes.end();
	for (vector<double>::const_iterator lon = longitudes.begin(); lon != lon_end; ++lon)
	{
		Polyline poly;
		poly.setAntiAliasing(false);
	
		for (double lat = min; lat <= max+step; lat += step)
		{
			( lat >  max ) ?  
				poly.push_back((*this)(GeoPoint(*lon, max))) :
				poly.push_back((*this)(GeoPoint(*lon,lat)));
		}
		grid.add(poly);
	}
}

void GeoRectangularProjection::gridLatitudes(const GridPlotting& grid)  const
{
	const vector<double>& latitudes = grid.latitudes();
	
	const double step = (max_longitude_ - min_longitude_)/20;
	const vector<double>::const_iterator lat_end = latitudes.end();
	for(vector<double>::const_iterator lat = latitudes.begin(); lat != lat_end; ++lat)
	{
		if ( *lat < -90 ) continue;
		if ( *lat > 90 ) continue;
		Polyline poly;
		poly.setAntiAliasing(false);
		for (double lon = getMinX(); lon <= getMaxX() + step; lon += step)
		{
			poly.push_back((*this)(GeoPoint(lon,*lat)));
		}
		grid.add(poly);
	}
}

/*!
 \brief generates text to mark longitudes at the top
 
 \sa Text
*/
void GeoRectangularProjection::labels(const LabelPlotting& label, TopAxisVisitor&)  const
{
	Text *text;
	const double cy = min_latitude_ + (max_latitude_-min_latitude_)*.2;
	const vector<double>& longitudes = label.longitudes();
	const vector<double>::const_iterator lon_end = longitudes.end();
	for(vector<double>::const_iterator lon = longitudes.begin(); lon != lon_end; ++lon)
	{	
		if ( *lon > min_longitude_ &&  *lon < max_longitude_ )
		{
			GeoPoint point(*lon, cy);
			text = new Text();      
			text->setText(point.writeLongitude());	
			text->setJustification(MCENTRE);
			text->setVerticalAlign(MBOTTOM);
			text->push_back((*this)(point));
			label.add(text);
		}
	}       	
}

/*!
 \brief generates text to mark longitudes at the bottom

 \sa Text
*/
void GeoRectangularProjection::labels(const LabelPlotting& label, BottomAxisVisitor&)  const
{
	Text *text;
	const double cy = min_latitude_ + (max_latitude_-min_latitude_)*.8;
	const vector<double>& longitudes = label.longitudes();
	const vector<double>::const_iterator lon_end = longitudes.end();
	for(vector<double>::const_iterator lon = longitudes.begin(); lon != lon_end; ++lon)
	{	
		if ( *lon > min_longitude_ &&  *lon < max_longitude_ )
		{
			GeoPoint point(*lon, cy);
			text = new Text();      
			text->setText(point.writeLongitude());	
			text->setJustification(MCENTRE);
			text->setVerticalAlign(MTOP);
			text->push_back((*this)(point));
			label.add(text);
		}
	}
}

/*!
 \brief generates text to mark latitudes at the left
  
 \sa Text
*/
void GeoRectangularProjection::labels(const LabelPlotting& label, LeftAxisVisitor&)  const
{
	Text *text;
	const vector<double>& latitudes = label.latitudes();
	const vector<double>::const_iterator lat_end = latitudes.end();
	const double lon = max_longitude_ - ((max_longitude_-min_longitude_)*.1);
	for(vector<double>::const_iterator lat = latitudes.begin(); lat != lat_end; ++lat)
	{	
		if ( *lat > min_latitude_ &&  *lat < max_latitude_ )
		{
			GeoPoint point(lon, *lat);
			text = new Text();      
			text->setText(point.writeLatitude());	
	        text->setJustification(MRIGHT);
			text->setVerticalAlign(MHALF);
			text->push_back((*this)(point));
			label.add(text);
		}
	}
}

void GeoRectangularProjection::labels(const LabelPlotting&, DrawingVisitor&)  const
{
}

/*!
 \brief generates text to mark latitudes at the right
 
 \sa Text
*/
void GeoRectangularProjection::labels(const LabelPlotting& label, RightAxisVisitor&)  const
{
	Text *text;
	const vector<double>& latitudes = label.latitudes();
	const vector<double>::const_iterator lat_end = latitudes.end();
	for(vector<double>::const_iterator lat = latitudes.begin(); lat != lat_end; ++lat)
	{	
		if ( *lat > min_latitude_ &&  *lat < max_latitude_ )
		{
			const double lon = min_longitude_ + ((max_longitude_-min_longitude_)*.1);
			GeoPoint point(lon, *lat);
			text = new Text();
			text->setText(point.writeLatitude());	
	        	text->setJustification(MLEFT);
			text->setVerticalAlign(MHALF); 
			text->push_back((*this)(point));
			label.add(text);
		}
	}
}


void GeoRectangularProjection::init()
{
	// make sure min < max! 

	while (min_longitude_ > max_longitude_) {
		max_longitude_ += 360;
		MagLog::warning() << "lower_left_longitude > upper_right_longitude --> upper_right_longitude is change to " << max_longitude_ << endl;
	}

	if (min_latitude_ > max_latitude_) {
		MagLog::warning() << "lower_left_latitude > upper_right_latitude --> swap" << endl;
		std::swap(min_latitude_, max_latitude_);
	}

	if ( max_longitude_ - min_longitude_  < min_area_ ) {
		max_longitude_ = min_longitude_ + min_area_;
		MagLog::warning() << "The geographical area has been extented to respect the minimal dimension" << endl;
	}

	if ( max_latitude_ -  min_latitude_ < min_area_) {
		max_latitude_ = min_latitude_ + min_area_;
		MagLog::warning() << "The geographical area has been extented to respect the minimal dimension" << endl;
	}

	if (max_latitude_ -  min_latitude_ < 360) {
	// Now 	we try to get the min longitudes in the the ranges -180/+360
		while ( min_longitude_ < -180) {
			min_longitude_ += 360;
			max_longitude_ += 360;
		}
		while ( max_longitude_ > 360 ) {
			min_longitude_ -= 360;
			max_longitude_ -= 360;
		}
	}
	else {
	// Now 	we try to get the min longitudes in the the ranges -360/+720
		while ( min_longitude_ < -360) {
			min_longitude_ += 360;
			max_longitude_ += 360;
		}
		while ( max_longitude_ > 720 ) {
			min_longitude_ -= 360;
			max_longitude_ -= 360;
		}	
	}
	xpcmin_ = min_longitude_;
	ypcmin_ = min_latitude_;
	xpcmax_ = max_longitude_;
	ypcmax_ = max_latitude_;	
}

MercatorProjection::MercatorProjection()
{
}

MercatorProjection::~MercatorProjection()
{
	delete projection_;
}

void MercatorProjection::print(ostream& o) const
{
	o << " mercator[";
	GeoRectangularProjection::print(o);
	o << "]";
} 

void MercatorProjection::init()
{
	if (!projection_) 
		projection_ = new TeMercator(TeDatum(), 0);
	// make sure min < max! 
	if (min_longitude_ > max_longitude_)
	{
		MagLog::warning() << "lower_left_lon > upper_right_lon --> swap" << endl;
		std::swap(min_longitude_, max_longitude_);
	}
	if (min_latitude_ > max_latitude_)
	{
		MagLog::warning() << "lower_left_lat > upper_right_lat --> swap" << endl;
		std::swap(min_latitude_, max_latitude_);
	}

	const double t = 2;
	min_latitude_ = std::max(min_latitude_, -89.);
	max_latitude_ = std::min(max_latitude_, 89.);
	min_longitude_ = std::max(min_longitude_, -180.);
	max_longitude_ = std::min(max_longitude_, 720.);
	
	if ( max_longitude_ - min_longitude_  < t) 
			max_longitude_ = min_longitude_ + t;
	if ( max_latitude_ -  min_latitude_ < t) 
			max_latitude_ = min_latitude_ + t;
	
	GeoPoint   ll(min_longitude_, min_latitude_);
	GeoPoint   ur(max_longitude_, max_latitude_);

	PaperPoint xy;	

	xy = (*this)(ll);	
	xpcmin_ = xy.x();
	ypcmin_ = xy.y();

	xy = (*this)(ur);
	xpcmax_ = xy.x();
	ypcmax_ = xy.y();			
} 

double round( double x)
{
        double prec = 100.;
        return floor(x*prec+0.5)/prec;
}

void GeoRectangularProjection::setNewPCBox(double minx, double miny, double maxx, double maxy)
{
	   PaperPoint p1(minx, miny);
	   PaperPoint p2(maxx, maxy);
	   GeoPoint   ll, ur;
 
	   revert(p1, ll);
	   revert(p2, ur);
 
	   min_longitude_ =ll.x();
	   max_longitude_ = ur.x();
	   min_latitude_ = ll.y();
	   max_latitude_ = ur.y();
}


/*!
  \brief Set the resolution of Coastlines
*/
void GeoRectangularProjection::coastSetting(map<string, string>& setting, double abswidth, double absheight) const
{
	// work out the ratios of geographical to paper lengths
	const double xratio = ( max_longitude_ - min_longitude_ ) / abswidth;
	const double yratio = ( max_latitude_  - min_latitude_ )  / absheight;

	// choose the smallest (smaller ratio means more detail required)
	const double ratio = min(xratio, yratio);

	std::string resol = "110m";
	if ( ratio < 0.8 )  // highest resolution
	{
		resol = "10m";
	}
	else if ( ratio < 3.5)   // medium resolution
	{
		resol = "50m";
	}

	setting["resolution"] = resol;
	setting["lakes"]      = resol + "/" + resol + "_lakes";
	setting["land"]       = resol + "/" + resol + "_land";
	setting["rivers"]     = resol + "/" + resol + "_rivers_lake_centerlines";

	if(!magCompare(resol,"110m"))
	{
		setting["boundaries"] = "50m/50m_admin_0_boundary_lines_land";
		setting["administrative_boundaries"] = "10m/10m_admin_1_states_provinces_shp";
	}
	else{
		setting["boundaries"] = "110m/110m_admin_0_boundary_lines_land";
		setting["administrative_boundaries"] = "110m/110m_admin_1_states_provinces_shp";
	}

	MagLog::info() << "GeoRectangularProjection::coastSetting[" << abswidth << ", " << absheight << "]->" <<  ratio << " resol: "<<resol<< endl;
}

/*
void GeoRectangularProjection::operator()(const vector<GeoPoint>& from, vector<PaperPoint>& to) const
{
	to.reserve(from.size());
	for (vector<GeoPoint>::const_iterator point = from.begin(); point != from.end(); ++point)
	{
		if ( in(*point) )
			to.push_back(PaperPoint((*point).x(), (*point).y()));

		if ( min_longitude_ < -180 ) {
			GeoPoint shift = point->left();		
			if ( in(shift) )
				to.push_back(PaperPoint(shift.x(), shift.y()));
		}
		if ( max_longitude_ > 360 ) {
			GeoPoint shift = point->right();
			if ( in(shift) )
				to.push_back(PaperPoint(shift.x(), shift.y()));
		}
		if ( max_longitude_ > 720 ) {
			GeoPoint shift = point->right().right();
			if ( in(shift) )
					to.push_back(PaperPoint(shift.x(), shift.y()));
		}
	}
}
*/
