# -*- coding: utf-8 -*-
### BEGIN LICENSE
# This file is in the public domain
### END LICENSE

"""Helpers for an Ubuntu application."""

from __future__ import division

__all__ = [
    'make_window',
    ]

import gtk
import os

from functools import wraps

from magicicada.magicicadaconfig import get_data_file


NO_OP = lambda *a, **kw: None


def get_builder(builder_file_name):
    """Return a fully-instantiated gtk.Builder instance from specified ui file.

    :param builder_file_name: The name of the builder file, without extension.
        Assumed to be in the 'ui' directory under the data path.
    """
    # Look for the ui file that describes the user interface.
    ui_filename = get_data_file('ui', builder_file_name)
    if not os.path.exists(ui_filename):
        ui_filename = None

    builder = gtk.Builder()
    builder.set_translation_domain('magicicada')
    builder.add_from_file(ui_filename)
    return builder


def log(logger):
    """Log input/ouput info for 'f' using 'logger'."""

    def decorator(f):
        """The decorator per se."""

        @wraps(f)
        def inner(*args, **kwargs):
            """Wrap 'f', log input args and result using 'logger'."""
            name = f.__name__
            result = None
            logger.debug("Calling '%s' with args '%s' and kwargs '%s'.",
                         name, args, kwargs)
            try:
                result = f(*args, **kwargs)
            except Exception:  # pylint: disable=W0703
                logger.exception('%s failed with exception:', name)
            logger.debug("Returning from '%s' with result '%s'.", name, result)
            return result

        return inner

    return decorator


# from http://code.activestate.com/recipes/
#                       577081-humanized-representation-of-a-number-of-bytes/


def humanize_bytes(numbytes, precision=1):
    """Return a humanized string representation of a number of bytes.

    Assumes `from __future__ import division`.

    >>> humanize_bytes(1)
    '1 byte'
    >>> humanize_bytes(1024)
    '1.0 kB'
    >>> humanize_bytes(1024*123)
    '123.0 kB'
    >>> humanize_bytes(1024*12342)
    '12.1 MB'
    >>> humanize_bytes(1024*12342,2)
    '12.05 MB'
    >>> humanize_bytes(1024*1234,2)
    '1.21 MB'
    >>> humanize_bytes(1024*1234*1111,2)
    '1.31 GB'
    >>> humanize_bytes(1024*1234*1111,1)
    '1.3 GB'
    """
    abbrevs = (
        (1<<50L, 'PB'),
        (1<<40L, 'TB'),
        (1<<30L, 'GB'),
        (1<<20L, 'MB'),
        (1<<10L, 'kB'),
        (1, 'bytes'))

    if numbytes == 1:
        return '1 byte'
    for factor, suffix in abbrevs:
        if numbytes >= factor:
            break
    # pylint: disable=W0631
    return '%.*f %s' % (precision, numbytes / factor, suffix)
