/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: log.c,v 1.40.2.8 2003/09/16 19:40:28 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza
 * Last audited version: 1.7
 * Notes:
 *
 ***************************************************************************/

#include <zorp/log.h>
#include <zorp/thread.h>
#include <zorp/error.h>

#include <stdarg.h>
#include <stdio.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/types.h>
#ifdef HAVE_SYS_TIME_H
  #include <sys/time.h>
#endif
#include <time.h>

#ifdef G_OS_WIN32
#  include <windef.h>
#  include <winbase.h>
#else
#  include <sys/socket.h>
#  include <sys/un.h>
#endif

#undef G_LOG_DOMAIN
#define G_LOG_DOMAIN "Zorp"

#define SYSLOG_SOCKET "/dev/log"

GHashTable *class_hash = NULL;
GStaticRWLock log_lock = G_STATIC_RW_LOCK_INIT;
unsigned int verbose_level = 3;
unsigned int log_tags = 0;
gchar *logspec = NULL;
gchar *syslog_tag = NULL;
unsigned int syslog_fd = -1;
gboolean stderr_syslog = FALSE;

gchar fake_session_id[256] = "noname/nosession";

static GMainContext *context = NULL;

#ifndef G_OS_WIN32
/*
 * This is a private reimplementation of syslog() as that one had a
 * mysterious bug in it and my bug reports were ignored.
 */

/**
 * z_open_syslog:
 * @tag: program name used in log messages
 *
 * Analogous to openlog(), open the syslog() connection to local syslogd. 
 *
 * Returns: whether
 **/
gboolean
z_open_syslog(gchar *tag)
{
  struct sockaddr_un s_un;
  
  syslog_tag = tag;
  syslog_fd = socket(AF_UNIX, SOCK_STREAM, 0);
  
  if (syslog_fd == -1)
    {
      return FALSE;
    }
  
  s_un.sun_family = AF_UNIX;
  strcpy(s_un.sun_path, SYSLOG_SOCKET);
  if (connect(syslog_fd, (struct sockaddr *) &s_un, sizeof(s_un)) == -1)
    {
      close(syslog_fd);
      syslog_fd = socket(AF_UNIX, SOCK_DGRAM, 0);
      if (connect(syslog_fd, (struct sockaddr *) &s_un, sizeof(s_un)) == -1)
        {
          close(syslog_fd);
          syslog_fd = -1;
          return FALSE;
        }
    }
  return TRUE;
}

/**
 * z_close_syslog:
 * @syslog_fd: syslog connection to close
 *
 * Close the connection to the local syslogd. The syslog connection is
 * specified by the @syslog_fd argument to avoid races.
 * 
 * Returns: whether the operation was succesful
 **/
gboolean
z_close_syslog(int syslog_fd)
{
  if (syslog_fd != -1)
    {
      close(syslog_fd);
      return TRUE;
    }
  return FALSE;
}

/**
 * z_send_syslog:
 * @pri: syslog priority 
 * @msg: syslog message
 *
 * Send the specified message to syslog.
 **/
gboolean
z_send_syslog(gint pri, const gchar *msg)
{
  gchar buf[1024];
  gchar timestamp[32];
  time_t now;
  struct tm t;
  gint len, rc = 0, attempt = 0;
  int sfd = syslog_fd;
  static GStaticMutex lock = G_STATIC_MUTEX_INIT;
  
  now = time(NULL);
  localtime_r(&now, &t);
  
  strftime(timestamp, sizeof(timestamp), "%h %e %H:%M:%S", &t);
  
  g_snprintf(buf, sizeof(buf), "<%d>%s %s[%d]: %s\n", pri, timestamp, syslog_tag, (int) getpid(), msg);
  len = strlen(buf) + 1;
  do
    {
      attempt++;
      if (sfd != -1)
        rc = write(sfd, buf, len);
      if (sfd == -1 || (rc == -1 && errno != EINTR && errno != EAGAIN))
        {
          g_static_mutex_lock(&lock);
          if (sfd == syslog_fd)
            {
              z_open_syslog(syslog_tag);
              z_close_syslog(sfd);
            }
            
          sfd = syslog_fd;
          g_static_mutex_unlock(&lock);
        }
    }    
  while (rc == -1 && attempt <= 1);
  return TRUE;
}
#else

gboolean
z_close_syslog(int syslog_fd)
{
  return TRUE;
}

#endif

/**
 * z_log_enabled:
 * @class: log message class
 * @level: log message level
 *
 * Checks if a message with a given class/level combination would actually
 * be written to the log. It can be used prior to constructing complex log
 * messages to decide whether the messages need to be constucted at all.
 * All results are cached, thus the second invocation will not parse the 
 * log specifications again.
 *
 * Returns: TRUE if the log would be written, FALSE otherwise
 **/
gboolean
z_log_enabled(gchar *class, int level)
{
  gint verbose;

  g_static_rw_lock_reader_lock(&log_lock);
  if (!class_hash)
    {
      g_static_rw_lock_reader_unlock(&log_lock);
      return TRUE;
    }
  verbose = GPOINTER_TO_UINT(g_hash_table_lookup(class_hash, class));
  g_static_rw_lock_reader_unlock(&log_lock);
  if (!verbose)
    {
      verbose = z_log_register_class(class);
      if (verbose >= 0)
        {
          g_static_rw_lock_writer_lock(&log_lock);
          g_hash_table_insert(class_hash, class, GUINT_TO_POINTER(verbose + 1));
          g_static_rw_lock_writer_unlock(&log_lock);
        }
      else
        {
          g_log(G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Invalid syntax in logspec line, class=%s", class);
          /* hack to let this message out */
#ifdef G_OS_WIN32
          Sleep(1000);
#else
          sleep(1);
#endif
          exit(1);
        }
    }
  else
    verbose--;
  
  return (level <= verbose);
}

/**
 * z_logv:
 * @class: log message class
 * @level: log message verbosity level
 * @format: log message format specified in printf form
 * @ap: format arguments va_list
 *
 * This function sends a message formatted as printf format string and
 * arguments to the syslog. The associated class/level pair is checked
 * whether the message really needs to be written.
 **/
void
z_logv(gchar *class, int level, gchar *format, va_list ap)
{
  int saved_errno = errno;
  
  if (z_log_enabled(class, level))
    {
      if (log_tags)
        {
          gchar msgbuf[2048];
          g_vsnprintf(msgbuf, sizeof(msgbuf), format, ap);

#if ZORPLIB_ENABLE_TRACE
          g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%p -> %s(%d): %s", g_thread_self(), class, level, msgbuf);
#else
          g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s(%d): %s", class, level, msgbuf);
#endif
        }
      else
        {
          g_logv(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, format, ap);
        }
    }
  errno = saved_errno;
}

/**
 * z_llog:
 * @class: log message class
 * @level: log message verbosity level
 * @format: log message format specified in printf form
 *
 * This message is the same as z_logv() but format string and arguments
 * are specified directly.
 **/
void
z_llog(gchar *class, int level, gchar *format, ...)
{
  va_list l;

  va_start(l, format);
  z_logv(class, level, format, l);
  va_end(l);
}

#ifdef G_OS_WIN32

void z_log(gchar* session_id, gchar* class, int level, gchar* format, ...)
{
  va_list l;
  gchar msgbuf[2048];

  va_start(l, format);

  g_vsnprintf(msgbuf, sizeof(msgbuf), format, l);

  g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%p -> %s(%d): %s", g_thread_self(), class, level, msgbuf);
/*
  if (session_id == NULL || *((char *) session_id) == 0)
    z_llog(class, level, "(%s %s): ", format, z_log_session_id(session_id) , l);
  else
    z_llog(class, level, "(%s %s): ", format, (char *) session_id , l);
*/
  va_end(l);
}
/*
void z_proxy_log(ZProxy *self, gchar* class, int level, char* format, ...)
{
  va_list l;
  va_start(l, format);

  z_log(((ZProxy *)self)->session_id, class, level, format,  l)

  va_end(l);
}
*/
#endif

#ifndef G_OS_WIN32
/**
 * z_log_func:
 * @log_domain: GLIB log domain
 * @log_flags: GLIB log flags
 * @message: message
 * @user_data: not used 
 *
 * This function is registered as a GLib log handler and sends all GLIB
 * messages to messages to syslog. Zorp itself does not use GLIB logging
 * it calls z_log() directly.
 **/
static void
z_log_func(const gchar *log_domain,
	   GLogLevelFlags log_flags,
	   const gchar *message,
	   gpointer user_data)
{
  int pri = LOG_INFO;
  if (log_flags & G_LOG_LEVEL_DEBUG)
    pri = LOG_DEBUG;
  else if (log_flags & G_LOG_LEVEL_WARNING)
    pri = LOG_WARNING;
  else if (log_flags & G_LOG_LEVEL_ERROR)
    pri = LOG_ERR;

  z_send_syslog(pri | ZORP_SYSLOG_FACILITY, message);
}

#else

/*+ log handler function to send Win32 debug message +*/
static void
z_log_win32_debugmsg(const gchar *log_domain,
                 GLogLevelFlags  log_flags,
                    const gchar *message,
                       gpointer  user_data)
{
  OutputDebugString(message);
  OutputDebugString("\n");
}

#endif

/**
 * z_fetch_stderr:
 * @channel: the read end of the STDERR pipe
 * @condition: the I/O condition triggering this callback
 * @arg: not used
 *
 * This function is registered as a read callback of the STDERR pipe to 
 * fetch messages sent to stderr. It fetches messages line-by-line and
 * uses z_log() to send messages to log.
 *
 * Returns: TRUE to indicate further reading is needed, FALSE otherwise
 **/
gboolean
z_fetch_stderr(GIOChannel *channel, GIOCondition condition, gpointer arg)
{
  gchar *line = NULL;
  GIOStatus status = G_IO_STATUS_NORMAL;
  GError *err = NULL;

  status = g_io_channel_read_line(channel, &line, NULL, NULL, &err);
  
  switch (status)
    {
    case G_IO_STATUS_NORMAL:
      z_log(NULL, CORE_STDERR, 3, "%s", line);
      break;
    case G_IO_STATUS_AGAIN:
      break;
    case G_IO_STATUS_EOF:
      /*LOG
        This message arrive when a program using Zorp library
        closing his stderr.
       */
      z_log(NULL, CORE_STDERR, 4, "The program close its stderr. No further stderr logging will be performed.");
      return FALSE;
    default:
      /*LOG
        This message is attempt when zorplib cannot read from
        program stderr stream.
       */
      z_log(NULL, CORE_STDERR, 3, "Cannot read from stderr; result='%s'", (err != NULL) ? ((err)->message) : ("Unknown error"));
      return FALSE;
    }
  g_free(line);
  return TRUE;
}

/**
 * z_log_source_new:
 * @fd: read side of the stderr pipe
 *
 * Creates the source watching the stderr pipe.
 **/
void
z_log_source_new(gint fd)
{
  GIOChannel *channel;
  GSource *source;
  
  channel = g_io_channel_unix_new(fd);
  g_io_channel_set_encoding(channel, NULL, NULL);
  g_io_channel_set_flags(channel, G_IO_FLAG_NONBLOCK, NULL);
  source = g_io_create_watch(channel, G_IO_IN);
  g_source_set_callback(source, (GSourceFunc) z_fetch_stderr, NULL, NULL);
  g_source_attach(source, context);
  g_source_unref(source);
}

/**
 * z_log_glob_match:
 * @glob: 
 * @class:
 *
 *
 **/
static gboolean
z_log_glob_match(const gchar *glob, const gchar *class)
{
  gchar *p1, *p2;
  gint len1, len2;

  p1 = strchr(glob, '.');
  p2 = strchr(class, '.');

  while (p1 && p2)
    {
      len1 = p1 - glob;
      len2 = p2 - class;
      if (((len1 != 1) || (memcmp(glob, "*", 1) != 0)) &&
          ((len1 != len2) || memcmp(glob, class, len1) != 0))
        return FALSE;
      glob = p1 + 1;
      class = p2 + 1;

      p1 = strchr(glob, '.');
      p2 = strchr(class, '.');
    }
  if (p1)
    len1 = p1 - glob;
  else
    len1 = strlen(glob);
  if (p2)
    len2 = p2 - class;
  else
    len2 = strlen(class);
  if (((len1 != 1) || (memcmp(glob, "*", 1) != 0)) &&
      ((len1 != len2) || memcmp(glob, class, len1) != 0))
    return FALSE;
  glob += len1;
  class += len2;
  if (strlen(glob) > strlen(class))
    return FALSE;
  return TRUE;
}

/**
 * z_log_register_class:
 * @class: message class 
 **/
int
z_log_register_class(gchar *class)
{
  const gchar *src = logspec;
  gint level = verbose_level, new_level;
  
  while (*src)
    {
      const gchar *glob, *num;
      gchar *colon, *end;

      while (*src == ',' || *src == ' ')
        src++;

      glob = src;
      while (isalnum((guchar) (*src)) || *src == '.' || *src == '*')
        src++;

      if (*src != ':')
        {
          /* invalid log spec */
          return -1;
        }
      colon = (gchar *) src;
      src++;
      num = src;

      *colon = 0;
      new_level = strtoul(num, &end, 10);
      if (z_log_glob_match(glob, class))
        level = new_level;
      *colon = ':';
      src = end;
      while (*src && *src != ',')
        src++;
    }
  return level;
}

/**
 * z_log_clear_hash:
 *
 * Clear the log-spec hash. It is called after changing the verbosity level.
 * FIXME: this contains a race condition.
 **/
void
z_log_clear_hash()
{
  g_static_rw_lock_writer_lock(&log_lock);
  g_hash_table_destroy(class_hash);
  class_hash = g_hash_table_new(g_str_hash, g_str_equal);
  g_static_rw_lock_writer_unlock(&log_lock);
}

/**
 * z_log_session_id:
 * @session_id: default session_id
 *
 * This helper function is used by the z_log() macro to get the current
 * session id. If the argument is NULL the session_id assigned to the
 * current thread is used, otherwise the value of the argument is returned.
 **/
const gchar *
z_log_session_id(const gchar *session_id)
{
  if (session_id == NULL || session_id[0] == 0)
    {
      ZThread *thread = z_thread_self();
      if (thread == NULL)
        return fake_session_id;
      else
        return thread->name;
    }
  return session_id;
}

/**
 * z_log_run:
 * @user_data: thread data pointer, assumed to point to the stderr fd
 *
 * STDERR reading thread function.
 **/
void *
z_log_run(gpointer user_data)
{
  GMainContext *c, *old_context;
  gint *fd = (gint *)user_data;
  
  old_context = context = g_main_context_new();
  g_main_context_acquire(context);
  z_log_source_new(*fd);

  do
    {
      c = context;
  
      if (c)
        g_main_context_iteration(c, TRUE);
    }
  while (c);

  g_main_context_release(old_context);
  g_main_context_unref(old_context);
  return NULL;
}

/**
 * z_log-init:
 * @ls: log specification
 * @syslog_name: the program name to appear in syslogs
 * @flags: log flags (ZLF_* macros)
 *
 * Initialize the logging subsystem according to the options
 * in specified in the @flags parameter.
 **/
void
z_log_init(const gchar *ls, const gchar *syslog_name, guint flags)
{
  logspec = (gchar *) (ls ? ls : "");
  log_tags = !!(flags & ZLF_TAGS);
  class_hash = g_hash_table_new(g_str_hash, g_str_equal);

  if (flags & ZLF_SYSLOG)
    {
  
#ifndef G_OS_WIN32

      static int grab[2];
      
      z_open_syslog((char *)syslog_name);
      g_log_set_handler(G_LOG_DOMAIN, 0xff, z_log_func, NULL);
      
      if (flags & ZLF_STDERR)
        {
          if (pipe(grab) < 0)
            {
              /*LOG
                This message is appear when cannot create
                pipe. The possible couse is zorp weaok on fd.
               */
              z_log(NULL, CORE_ERROR, 3, "Error creating stderr-syslog pipe;");
              return;
            }
          stderr_syslog = TRUE;
          dup2(grab[1], 1);
          dup2(grab[1], 2);
          if (grab[1] != 2 && grab[1] != 1)
            close(grab[1]);
          
          if (~flags & ZLF_THREAD)
            {
              context = g_main_context_default();
              if (!g_main_context_acquire(context))
                {
                  context = g_main_context_new();
                  g_main_context_acquire(context);
                }
              g_main_context_ref(context);
              z_log_source_new(grab[0]);
            }
          else
            z_thread_new("Log thread", z_log_run, &grab[0]);
        }

#else

      g_log_set_handler(G_LOG_DOMAIN, 0xff, z_log_win32_debugmsg, NULL);

#endif
    }
}

/**
 * z_log_destroy:
 *
 * Deinitialize the logging subsystem.
 **/
void
z_log_destroy(void)
{
  GMainContext *c;
#ifndef G_OS_WIN32  
  if (stderr_syslog)
    {
      close(1);
      close(2);
    }
#endif
  g_static_rw_lock_writer_lock(&log_lock);
  g_hash_table_destroy(class_hash);
  class_hash = NULL;
  g_static_rw_lock_writer_unlock(&log_lock);
  z_close_syslog(syslog_fd);
  
  c = context;
  context = NULL;
  g_main_context_wakeup(c);
}
