#ifndef TAGCOLL_DERIVEDTAGS_H
#define TAGCOLL_DERIVEDTAGS_H

/*
 * Model a collection of derived tags and a TagcollFilter to add them to a
 * collection
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma interface

#include <tagcoll/TagcollFilter.h>
#include <tagcoll/ParserBase.h>
#include <tagcoll/tagexpr/Tagexpr.h>

#include <map>
#include <string>

#include <stdio.h>

namespace Tagcoll
{

// List of changes to apply to tag names with a tagged collection is being
// parsed
class DerivedTagList
{
protected:
	typedef std::map<std::string, Tagexpr*> expressions_t;
	expressions_t expressions;
	
public:
	virtual ~DerivedTagList() throw ();

	// Add a new derived tag to the list
	void add(const std::string& tag, const Tagexpr* expr) throw ();

	// Read the derived tag expressions from the given file
	void parse(ParserInput& in) throw (ParserException);

	// Return the tags in `tags', plus all the derived tags that matched `tags'
	OpSet<std::string> getDerived(const OpSet<std::string>& tags) const throw ();

	// Return the tags in `tags', plus all the derived tags that matched `tags'
	OpSet<std::string> addDerived(const OpSet<std::string>& tags) const throw () { return tags + getDerived(tags); }
};

template <class ITEM>
class ApplyDerivedTags : public TagcollFilter<ITEM, std::string>
{
protected:
	const DerivedTagList& dtags;
	
public:
	ApplyDerivedTags(const DerivedTagList& dtags) throw ()
		: dtags(dtags) {}

	virtual void consume(const ITEM& item) throw ()
	{
		this->consumer->consume(item, dtags.addDerived(OpSet<std::string>()));
	}

	virtual void consume(const ITEM& item, const OpSet<std::string>& tags) throw ()
	{
		this->consumer->consume(item, dtags.addDerived(tags));
	}
};

template <class ITEM>
class CompressDerivedTags : public TagcollFilter<ITEM, std::string>
{
protected:
	DerivedTagList& dtags;
	
public:
	CompressDerivedTags(DerivedTagList& dtags) throw ()
		: dtags(dtags) {}

	virtual void consume(const ITEM& item) throw ()
	{
		this->consumer->consume(item);
	}

	virtual void consume(const ITEM& item, const OpSet<std::string>& tags) throw ()
	{
		OpSet<std::string> derived = dtags.getDerived(tags);
		this->consumer->consume(item, tags - derived);
	}
};

};

// vim:set ts=4 sw=4:
#endif
