/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim:expandtab:autoindent:tabstop=4:shiftwidth=4:filetype=c:cindent:textwidth=0:
 *
 * Copyright (C) 2005 Dell Inc.
 *  by Michael Brown <Michael_E_Brown@dell.com>
 * Licensed under the Open Software License version 2.1
 *
 * Alternatively, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 */

// compat header should always be first header if including system headers
#define LIBSMBIOS_SOURCE
#include "smbios/compat.h"

#include <sstream>
#include <iomanip>

#include "SmbiosXmlImpl_xerces.h"
#include "StdSmbiosXml.h"
#include "../common/FactoryImpl2.h"
#include "../smbios/SmbiosImpl.h"
#include "../common/XmlUtils.h"

// message.h should be included last.
#include "smbios/message.h"

XERCES_CPP_NAMESPACE_USE;
using namespace std;
using namespace smbiosLowlevel;
using namespace xmlutils;

namespace smbios
{
    //
    // FACTORY
    //

    class SmbiosXmlFactoryImpl: public SmbiosFactoryImpl
    {
    public:
        virtual ISmbiosTable *makeNew();
        SmbiosXmlFactoryImpl() : SmbiosFactoryImpl() {};
        virtual ~SmbiosXmlFactoryImpl() throw () {};
    };

    SmbiosFactory *SmbiosXmlFactory::getFactory()
    {
        // reinterpret_cast<...>(0) to ensure template parameter is correct
        // this is a workaround for VC6 which cannot use explicit member template
        // funciton initialization.
        return SmbiosFactoryImpl::getFactory(reinterpret_cast<SmbiosXmlFactoryImpl *>(0));
    }

    ISmbiosTable *SmbiosXmlFactoryImpl::makeNew()
    {
        // stupid, ugly hack to supress (C4800) warning on msvc++
        bool strict = getParameterNum("strictValidation") ? 1 : 0;

        SmbiosTableXml *table = 0;

        std::vector<SmbiosStrategy *> strategies;

        if (mode == AutoDetectMode)
        {
            strategies.push_back( new SmbiosMemoryStrategy(getParameterNum("offset")) );
#ifdef LIBSMBIOS_PLATFORM_WIN32
            strategies.push_back( new SmbiosWinGetFirmwareTableStrategy() );
            strategies.push_back( new SmbiosWinWMIStrategy() );
#endif
        }
        else if (mode == UnitTestMode)
        {
            strategies.push_back( new SmbiosMemoryStrategy(getParameterNum("offset")) );
        }
        else
        {
        throw NotImplementedImpl(_("Unknown smbios factory mode requested"));
        }


        table = new SmbiosTableXml( 
                strategies,
                strict 
            );
        table->setXmlFilePath( getParameterString("xmlFile") );
        table->initializeWorkaround();
        return table;
    }


    // if user give us a file with smbios xml information, use that
    // if there are any problems parsing the doc, or if they do not give us a
    // filename, use the built-in xml stuff.
    DOMDocument *getSmbiosXmlDoc( DOMBuilder *parser, std::string &xmlFile )
    {
        DOMDocument *doc = 0;

        try
        {
            doc = parser->parseURI( xmlFile.c_str() );
        }
        // catch std::exception and below. Anything else is likely very
        // serious and should pass up raw.
        catch( const std::exception & )
        {
            cerr << "Error during Xerces-c parsing.\n";
            cerr << "Falling back to builtin XML." << endl;
        }

        // fall back to built-in xml file on any kind of error
        if( doc == 0 )
        {
            try
            {
                // create SAX input source
                size_t fileLen = strlen(stdXml);
                unsigned int *lenPtr = reinterpret_cast<unsigned int *>(&fileLen);
                MemBufInputSource* memBufIs = new MemBufInputSource(
                                                  reinterpret_cast<const XMLByte*>(stdXml),
                                                  *lenPtr,
                                                  "standard_xml",
                                                  false
                                              );

                // wrap it with a DOM input source. DOM wrapper adopts the
                // SAX, no need to delete.
                DOMInputSource* Is = new Wrapper4InputSource( memBufIs );

                doc = parser->parse( *Is );

                delete Is;
            }
            // catch std::exception and below. Anything else is likely very
            // serious and should pass up raw.
            catch( const std::exception & )
            {
                cerr << "Error during Xerces-c parsing builin XML.\n";
                cerr << "This is really bad and probably not recoverable." << endl;
                throw ParseExceptionImpl("problem parsing xml file.");
            }

        }

        return doc;
    }

    void validateSmbiosXmlDoc( DOMDocument *doc )
    {
        if( doc )
        {
            DOMElement *root = doc->getDocumentElement();

            // This is pretty much the only checking we do
            // to ensure that the XML is valid.
            // maybe add some kind of schema validation in the future.
            string name = safeXMLChToString( root->getNodeName() );
            if ( ! (name == "STRUCTUREDEFS") )
                throw ParseExceptionImpl("problem parsing xml file. root doc name not STRUCTUREDEFS.");
        }
    }

    unsigned int parseLengthStr(string size)
    {
       if (size == "BYTE")
           return 1;
       else if (size == "WORD")
           return 2;
       else if (size == "DWORD")
           return 4;
       else if (size == "QWORD")
           return 8;

       return strtol(size.c_str(), NULL, 0);
       //throw ParseExceptionImpl("Error parsing length information xml file. Invalid value." );
       //return 0;
    }

    void verifyElementAttr( DOMElement *element, const string elementName, const string value )
    {
        string xmlValue = safeGetAttribute( element, elementName );
        if( value != xmlValue )
            throw ParseExceptionImpl("could not verify element attribute.");
    }

    // sneaky... :-)
    void verifyElementAttr( DOMElement *element, const string elementName, unsigned int size )
    {
        string xmlValue = safeGetAttribute( element, elementName );
        if( size != parseLengthStr(xmlValue) )
            throw ParseExceptionImpl("could not verify element attribute was correct size.");
    }

    int getTypeForString( DOMDocument *doc, const string searchForDesc )
    {
        // find element with this description
        DOMElement *elem = findElement( doc->getDocumentElement(), "STRUCTURE", "description", searchForDesc );

        // return the type as an INT.
        return strtol( safeGetAttribute( elem, "type" ).c_str(), 0, 0);
    }

    const string getStringForType(const DOMDocument *doc, const int searchForType )
    {
        // find matching element
        DOMElement *elem = 0;
        try
        {
            elem = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", searchForType);
        }
        catch(const NotFound &)
        {
            elem = findElement( doc->getDocumentElement(), "STRUCTURE", "type", "unknown");
        }

        // extract the description
        return safeGetAttribute( elem, "description");
    }

    //
    // MEMBER FUNCTIONS
    //

    // CONSTRUCTORS
    //
    // REGULAR CONSTRUCTOR
    SmbiosTableXml::SmbiosTableXml()
            : SmbiosTable(), xmlFile(""), parser(0), doc(0), xmlInitialized(false)
    {
        setXmlFilePath(xmlFile);
    }

    SmbiosTableXml::SmbiosTableXml(std::vector<SmbiosStrategy *> initStrategyList, bool strictValidation)
            : SmbiosTable(initStrategyList, strictValidation), xmlFile(""), parser(0), doc(0), xmlInitialized(false)
    {
        setXmlFilePath(xmlFile);
    }

    // DESTRUCTOR
    SmbiosTableXml::~SmbiosTableXml()
    {
        if(parser)
        {
            parser->resetDocumentPool();
            parser->release();
            parser = 0;
            doc = 0;
        }
        if( xmlInitialized )
            XMLPlatformUtils::Terminate();
    }

    ISmbiosItem &SmbiosTableXml::makeItem(const void *header) const
    {
        // not exception safe yet.
        const smbios_structure_header *structure =
            reinterpret_cast<const smbios_structure_header *>(header);
        SmbiosItemXml *item = new SmbiosItemXml( structure );
        item->setXmlFilePath( xmlFile, doc );
        if( ! initializing )
        {
            dynamic_cast<SmbiosItem*>(item)->fixup( workaround.get() );
        }
        return *item;
    }


    // good exception guarantee.
    // either we allocate new stuff, the new stuff validates, and we
    // set ourselves up with the new stuff, or we keep whatever we
    // used to have and raise the exception.
    void SmbiosTableXml::setXmlFilePath( std::string newFile )
    {
        try
        {
            // Initialize XML DOM subsystem
            if( ! xmlInitialized )
                XMLPlatformUtils::Initialize();

            xmlInitialized = true;

            DOMBuilder *newParser = getParser();
            DOMDocument *newdoc = getSmbiosXmlDoc( newParser, newFile );
            validateSmbiosXmlDoc( newdoc );
            // if we get to this point, that means the
            // new doc exists and is valid.

            // clean up any old messes we may have laying around.
            // release old parser if one is allocated.
            DOMBuilder *oldParser = parser;

            parser = newParser;
            xmlFile = newFile;
            doc = newdoc;

            if( oldParser )
            {
                oldParser->resetDocumentPool();
                oldParser->release();
            }
        }
        catch(const XMLException &toCatch)
        {
            cerr << "Error during Xerces-c Initialization.\n"
            << "  Exception message:"
            << toCatch.getMessage() << endl;
            throw ParseExceptionImpl("xerces initialization failed.");
        }
    }

    const DOMDocument *SmbiosTableXml::getXmlDoc() const
    {
        return doc;
    }

    int SmbiosTableXml::getTypeForString( const string searchForDesc ) const
    {
        return smbios::getTypeForString( doc, searchForDesc );
    }

    // only used by unit test code.
    const string SmbiosTableXml::getStringForType( const int searchForType ) const
    {
        return smbios::getStringForType( doc, searchForType );
    }

    // we were passed a string. convert to a number by looking up the
    // type for this string in the XML File
    // forward to base class operator[]
    SmbiosTable::iterator SmbiosTableXml::operator[] (const string &searchFor)
    {
        int type = getTypeForString( searchFor );
        return SmbiosTable::iterator (this, type);
    }

    // we were passed a string. convert to a number by looking up the
    // type for this string in the XML File
    // forward to base class operator[]
    SmbiosTable::const_iterator SmbiosTableXml::operator[](const string &searchFor) const
    {
        // this == const SmbiosTable();
        int type = getTypeForString( searchFor );
        return SmbiosTable::const_iterator (this, type);
    }


    //
    // SmbiosItemXml members
    //
    void SmbiosItemXml::setXmlFilePath( const std::string newFile,  XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *newDoc )
    {
        xmlFile = newFile;
        doc = newDoc;
    }

    u8 SmbiosItemXml::getU8( const string fieldName ) const
    {
        DOMElement *element = 0;

        // get the element corresponding to the STRUCTURE user specified
        DOMElement *Structure = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", getType() );
        element = findElement( Structure, "FIELD", "name", fieldName );

        // Is this the correct length?
        verifyElementAttr( element, "length", 1 );

        // call parent method to get actual data. :-)
        return SmbiosItem::getU8( getNumberFromXmlAttr(element, "offset", 0) );
    }

    u16 SmbiosItemXml::getU16( const string fieldName ) const
    {
        DOMElement *element = 0;

        // get the element corresponding to the STRUCTURE user specified
        DOMElement *Structure = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", getType() );
        element = findElement( Structure, "FIELD", "name", fieldName );

        // Is this the correct length?
        verifyElementAttr( element, "length", 2 );

        // call parent method to get actual data. :-)
        return SmbiosItem::getU16( getNumberFromXmlAttr(element, "offset", 0) );
    }

    u32 SmbiosItemXml::getU32( const string fieldName ) const
    {
        DOMElement *element = 0;

        // get the element corresponding to the STRUCTURE user specified
        DOMElement *Structure = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", getType() );
        element = findElement( Structure, "FIELD", "name", fieldName );

        // Is this the correct length?
        verifyElementAttr( element, "length", 4 );

        // call parent method to get actual data. :-)
        return SmbiosItem::getU32( getNumberFromXmlAttr(element, "offset", 0) );
    }

    u64 SmbiosItemXml::getU64( const string fieldName ) const
    {
        DOMElement *element = 0;

        // get the element corresponding to the STRUCTURE user specified
        DOMElement *Structure = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", getType() );
        element = findElement( Structure, "FIELD", "name", fieldName );

        // Is this the correct length?
        verifyElementAttr( element, "length", 8 );

        // call parent method to get actual data. :-)
        return SmbiosItem::getU64( getNumberFromXmlAttr(element, "offset", 0) );
    }

    u32 SmbiosItemXml::getBitfield( const string field, const string bitField) const
    {
        DOMElement *bitElement = 0;
        DOMElement *fieldElement = 0;

        try
        {
            DOMElement *Structure = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", getType() );
            fieldElement = findElement( Structure, "FIELD", "name", field );
            bitElement = findElement( fieldElement, "BITS", "name", bitField );
        }
        catch (const NotFound & )
        {
            throw ParseExceptionImpl("could not fine bitfield name in xml file.");
        }

        // Is this the correct length?
        string length = safeGetAttribute( fieldElement, "length" );
        unsigned int lengthVal = 0;
        lengthVal = parseLengthStr(length);

        // call parent method to get actual data. :-)
        return SmbiosItem::getBitfield(
                   getNumberFromXmlAttr(fieldElement, "offset", 0),
                   lengthVal,
                   getNumberFromXmlAttr(bitElement, "lsb", 0),
                   getNumberFromXmlAttr(bitElement, "msb", 0)
               );
    }

    const char* SmbiosItemXml::getString( const string fieldName ) const
    {
        DOMElement *element = 0;

        // get the element corresponding to the STRUCTURE user specified
        DOMElement *Structure = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", getType() );
        element = findElement( Structure, "FIELD", "name", fieldName );

        // Is this the correct length?
        verifyElementAttr( element, "length", 1 );

        // Is this an actual string?
        verifyElementAttr( element, "usage", "STRING" );

        // call parent method to get actual data. :-)
        return SmbiosItem::getString( getNumberFromXmlAttr(element, "offset", 0) );
    }

    bool isBitSet(const ISmbiosItem *itemPtr, unsigned int offset, unsigned int bitToTest)
    {
        bool retval = false;

        unsigned int byte = bitToTest / 8;
        u8 fieldValue = itemPtr->getU8( offset + byte );
        if (fieldValue & (1 << (bitToTest%8)))
            retval = true;

        return retval;
    }


    void printStructureField( std::ostream &cout, const DOMNode *node, const ISmbiosItem &item )
    {
        std::ios::fmtflags old_opts = cout.flags ();
        try
        {
            unsigned int length = parseLengthStr(safeGetAttribute( node, "length" ));
            string strOffset = safeGetAttribute( node, "offset" );
            unsigned int offset = strtol( strOffset.c_str(), 0, 0 );

            string usage = safeGetAttribute( node, "usage" );
            if (usage == "STRING")
            {
                try
                {
                    cout << item.getString( offset );
                }
                catch(const StringUnavailable &)
                {
                }
            }
            else
            {
                cout << hex << "0x";
                for(unsigned int i=0;i<length; i++)
                {
                    cout << setfill('0') << setw(2) << 
                        static_cast<int>(item.getU8(offset + length - i - 1));
                }
            }
        }
        catch( const std::exception & )
        {
            cout.flags (old_opts);
            throw;
        }
        cout.flags (old_opts);
    }

    std::ostream &SmbiosItemXml::streamify( std::ostream &cout ) const
    {
        // If we don't have a ref to XML file, we cannot find this info
        if( ! doc )
            return SmbiosItem::streamify(cout);

        if (header == 0)
        {
            // TODO: this should be an exception...
            cout << "operator << on an uninitialized SmbiosItem!";
            return cout;
        }

        std::ios::fmtflags old_opts = cout.flags ();

        DOMElement *Structure = 0;
        cout << "DMI BLOCK: " << flush;
        try
        {
            cout << smbios::getStringForType( doc, getType() ) << endl;;
            Structure = findElementWithNumericAttr( doc->getDocumentElement(), "STRUCTURE", "type", getType());
        }
        catch ( const NotFound & )
        {
            Structure = findElement(doc->getDocumentElement(), "STRUCTURE", "type", "unknown");
        }

        XMLCh *tagName = X("FIELD"); // NEED TO 'release' !!!
        DOMNodeList *fieldList = Structure->getElementsByTagName(tagName);
        XMLString::release(&tagName);

        if( !fieldList )
            return cout;

        int length = fieldList->getLength();
        for( int index = 0; index < length; ++index )
        {
            DOMNode *node = fieldList->item( index );
            if( node->getNodeType() == DOMNode::ELEMENT_NODE )
            {
                ostringstream tmpBuf("");
                tmpBuf << "\t";
                tmpBuf << safeGetAttribute( node, "name" );
                tmpBuf << ": \t" ;
                try
                {
                    printStructureField( tmpBuf, node, *this );
                }
                catch(const exception &)
                {
                    continue;
                }
                cout << tmpBuf.str() << endl;
            }
        }

        cout.flags (old_opts);
        return cout;
    }

    std::ostream &SmbiosTableXml::streamify(ostream & cout) const
    {
        cout << "\nSMBIOS table " << endl;
        cout << "\tversion    : ";
        cout << static_cast<int>(table_header.major_ver) << ".";
        cout << static_cast<int>(table_header.minor_ver) << endl;
        cout << hex ;
        cout << "\taddress    : " << table_header.table_address << endl;
        cout << dec;
        cout << "\tlength     : " << table_header.table_length << endl;
        cout << "\tnum structs: " << table_header.table_num_structs << endl;
        cout << endl;

        SmbiosTable::const_iterator position = begin();
        while (position != end())
        {
             cout << *position << endl;
            ++position;
        }
        return cout;
    }


    /*********************************
      XML OUTPUT FUNCTIONS
      *******************************/

    std::ostream &toXmlString(const ISmbiosTable &table, ostream & cout)
    {
        UNREFERENCED_PARAMETER(table);
        cout << "XML output not yet supported in std lib." << endl;
        return cout;
    }

}
