# $DUH: Session.pm,v 1.45 2002/12/16 20:56:04 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 SYNOPSIS

PMilter::Session - per-connection milter context

=head1 DESCRIPTION

A PMilter::Session is the context object passed to milter callback functions
as the first argument, typically named "$ctx" for convenience.  This manpage
explains publicly accessible operations on $ctx.

A special global package variable, C<$PMilter::Session::DebugLevel>, can be
set to one of the following (preferably via "local" in the same stack frame
used to call C<PMilter::Server::main()>) to provide LOG_DEBUG messages to
the logger.  Each level includes all messages from the next lower level.

=over 4

=item 1

Shows one line per command indicating the return value of any callback that
returns other than SMFIS_CONTINUE.

=item 2

Shows one line for each of the following:  connect, helo, envfrom, envrcpt,
abort, quit.  ("quit" indicates that the MTA actually sent a SMFIC_QUIT
packet; the "close" callback is always called regardless.)

=item 3

Shows one line per message header, and the "eoh" command.

=item 4

Shows one line for each macro symbol defined.

=item 5

Shows one line for each body packet sent (with only length printed), and the
"eom" command.

=back

=head1 METHODS

=over 4

=cut

package PMilter::Session;
use base Exporter;

use strict;
use warnings;

use Carp qw(cluck);
use PMilter;
use PMilter::Callbacks qw(:all);
use PMilter::Server qw(:log);
use Socket;
use Tie::RefHash;
use UNIVERSAL;

*VERSION = *PMilter::VERSION;

# Exported constants

use constant SMFIA_UNKNOWN	=> 'U';
use constant SMFIA_UNIX		=> 'L';
use constant SMFIA_INET		=> '4';
use constant SMFIA_INET6	=> '6';

our @EXPORT_OK = qw(
	SMFIA_UNKNOWN
	SMFIA_UNIX
	SMFIA_INET
	SMFIA_INET6
);
our %EXPORT_TAGS = ( all => \@EXPORT_OK );

our $DebugLevel = 0;

# Private constants

use constant SMFIC_ABORT	=> 'A';
use constant SMFIC_BODY		=> 'B';
use constant SMFIC_CONNECT	=> 'C';
use constant SMFIC_MACRO	=> 'D';
use constant SMFIC_BODYEOB	=> 'E';
use constant SMFIC_HELO		=> 'H';
use constant SMFIC_HEADER	=> 'L';
use constant SMFIC_MAIL		=> 'M';
use constant SMFIC_EOH		=> 'N';
use constant SMFIC_OPTNEG	=> 'O';
use constant SMFIC_RCPT		=> 'R';
use constant SMFIC_QUIT		=> 'Q';

use constant SMFIR_ADDRCPT	=> '+';
use constant SMFIR_DELRCPT	=> '-';
use constant SMFIR_ACCEPT	=> 'a';
use constant SMFIR_REPLBODY	=> 'b';
use constant SMFIR_CONTINUE	=> 'c';
use constant SMFIR_DISCARD	=> 'd';
use constant SMFIR_ADDHEADER	=> 'h';
use constant SMFIR_CHGHEADER	=> 'm';
use constant SMFIR_PROGRESS	=> 'p';
use constant SMFIR_REJECT	=> 'r';
use constant SMFIR_TEMPFAIL	=> 't';
use constant SMFIR_REPLYCODE	=> 'y';

use constant SMFIP_NOCONNECT	=> 0x01;
use constant SMFIP_NOHELO	=> 0x02;
use constant SMFIP_NOMAIL	=> 0x04;
use constant SMFIP_NORCPT	=> 0x08;
use constant SMFIP_NOBODY	=> 0x10;
use constant SMFIP_NOHDRS	=> 0x20;
use constant SMFIP_NOEOH	=> 0x40;
use constant SMFIP_NONE		=> 0x7F;

no strict 'refs';
my %replynames = map { &{$_} => $_ } qw(
	SMFIR_ADDRCPT
	SMFIR_DELRCPT
	SMFIR_ACCEPT
	SMFIR_REPLBODY
	SMFIR_CONTINUE
	SMFIR_DISCARD
	SMFIR_ADDHEADER
	SMFIR_CHGHEADER
	SMFIR_PROGRESS
	SMFIR_REJECT
	SMFIR_TEMPFAIL
	SMFIR_REPLYCODE
);
use strict 'refs';

=pod

=item $ctx->getpriv

Returns the private data object for this milter instance, set by
$ctx->setpriv() (see below).  Returns undef if setpriv has never been called
by this milter instance.

This is fully compatible with Sendmail::Milter.

=cut

sub getpriv {
	my $this = shift;

	$this->{priv};
}

=pod

=item $ctx->getsymval(NAME)

Retrieves the macro symbol named NAME from the macros available thus far
from the MTA.  This typically consists of a one-letter macro name, or a
multi-letter macro name enclosed in {curly braces}.  If the requested macro
was not defined by the MTA ny the time getsymval is called, returns undef.

Some common macros include the following.  (Since milter is a protocol first
implemented in the Sendmail MTA, the macro names are the same as those in
Sendmail itself.)

=over 2

=item $ctx->getsymval('_')

The remote host name and address, in standard SMTP "name [address]" form.

=item $ctx->getsymval('i')

The MTA's queue ID for the current message.

=item $ctx->getsymval('j')

The MTA's idea of local host name.

=item $ctx->getsymval('{if_addr}')

The local address of the network interface upon which the connection was
received.

=item $ctx->getsymval('{if_name}')

The local hostname of the network interface upon which the connection was
received.

=item $ctx->getsymval('{mail_addr}')

The MAIL FROM: sender's address, canonicalized and angle bracket stripped.
(This is typically not the same value as the second argument to the
"envfrom" callback.)  Will be defined to the empty string '' if the client
issued a MAIL FROM:<> null return path command.

=item $ctx->getsymval('{rcpt_addr}')

The RCPT TO: recipient's address, canonicalized and angle bracket stripped.
(This is typically not the same value as the second argument to the
"envrcpt" callback.)

=back

Not all macros may be available at all times, of course.  Some macros are
only available after a specific phase is reached, and some macros may only
be available from certain MTA implementations.  Care should be taken to
check for undef returns in order to cover these cases.

This is fully compatible with Sendmail::Milter, although PMilter also allows
access to macros defined earlier in the connection (for instance, accessing
macro 'j' in the envfrom callback).

=cut

sub getsymval {
	my $this = shift;
	my $key = shift;

	my $value = undef;

	foreach my $cmd (SMFIC_RCPT, SMFIC_MAIL, SMFIC_HELO, SMFIC_CONNECT) {
		$value = $this->{macros}{$cmd}{$key};
		last if defined($value);
	}

	$value;
}

=pod

=item $ctx->setpriv(DATA)

This is the place to store milter-private data that is sensitive to the
current SMTP client connection.  Only one value can be stored, so typically
an arrayref or hashref is initialized in the "connect" callback and set with
$ctx->setpriv.

This value can be retrieved on subsequent callback runs with $ctx->getpriv.

=cut

sub setpriv {
	my $this = shift;
	$this->{priv} = shift;

	1;
}

=pod

=item $ctx->setreply(RCODE, XCODE, MESSAGE)

Set an extended SMTP status reply (before returning SMFIS_REJECT or
SMFIS_TEMPFAIL).  RCODE should be a short (4xx or 5xx) numeric reply code,
XCODE should be a long ('4.x.x' or '5.x.x') ESMTP reply code, and MESSAGE is
the full text of the message to send.  Example:

        $ctx->setreply(451, '4.7.0', 'Cannot authenticate you right now');
        return SMFIS_TEMPFAIL;

Note that after setting a reply with this method, the SMTP result code comes
from RCODE, not the difference between SMFIS_REJECT or SMFIS_TEMPFAIL.  
However, for consistency, callbacks that set a 4xx response code should use
SMFIS_TEMPFAIL, and those that set a 5xx code should return SMFIS_REJECT.

Returns a true value on success, undef on failure.  In the case of failure,
typically only caused by bad parameters, a generic message will still be
sent based on the SMFIS_* return code.

=cut

sub setreply {
	my $this = shift;
	my $rcode = shift || return undef;
	my $xcode = shift || return undef;
	my $message = shift || return undef;

	return undef if ($rcode !~ /^[45]\d\d$/ || $xcode !~ /^[45]\.\d\.\d$/ || substr($rcode, 0, 1) ne substr($xcode, 0, 1));

	$this->{reply} = "$rcode $xcode $message";
}

=pod

=item $ctx->addrcpt(ADDRESS)

Add address ADDRESS to the list of recipients for this mail.  In PMilter,
can be called as early as the "envfrom" callback.

Returns a true value on success, undef on failure.

This is also available in Sendmail::Milter, but the "real" Sendmail::Milter
implementation allows calling of this method only from the "eom" callback.

=cut

sub addrcpt {
	my $this = shift;
	my $rcpt = shift || return undef;

	return undef unless ($this->{envfrom} && ($this->{cbref}{FLAGS} & SMFIF_ADDRCPT));
	push(@{$this->{eom_actions}}, [SMFIR_ADDRCPT, "$rcpt\0"]);
	1;
}

=pod

=item $ctx->addheader(HEADER, VALUE)

Add header HEADER with value VALUE to this mail.  Does not change any
existing headers with the same name.  In PMilter, can be called as early as
the "envfrom" callback.

Returns a true value on success, undef on failure.

This is also available in Sendmail::Milter, but the "real" Sendmail::Milter
implementation allows calling of this method only from the "eom" callback.

=cut

sub addheader {
	my $this = shift;
	my $header = shift || return undef;
	my $value = shift || return undef;

	return undef unless ($this->{envfrom} && ($this->{cbref}{FLAGS} & SMFIF_ADDHDRS));
	push(@{$this->{eom_actions}}, [SMFIR_ADDHEADER, "$header\0$value\0"]);
	1;
}

=pod

=item $ctx->chgheader(HEADER, INDEX, VALUE)

Change the INDEX'th header of name HEADER to the value VALUE.

Returns a true value on success, undef on failure.

This is also available in Sendmail::Milter, but the "real" Sendmail::Milter
implementation allows calling of this method only from the "eom" callback.

=cut

sub chgheader {
	my $this = shift;
	my $header = shift || return undef;
	my $num = shift || 0;
	my $value = shift || '';

	return undef unless ($this->{envfrom} && ($this->{cbref}{FLAGS} & SMFIF_CHGHDRS));
	push(@{$this->{eom_actions}}, [SMFIR_CHGHEADER, pack('N', $num)."$header\0$value\0"]);
	1;
}

=pod

=item $ctx->delrcpt(ADDRESS)

Remove address ADDRESS from the list of recipients for this mail.  The
ADDRESS argument must match a prior argument to the "envrcpt" callback
exactly (case sensitive, and including angle brackets if present).  In
PMilter, can be called as early as the "envfrom" callback.

Returns a true value on success, undef on failure.  A success return does
not necessarily indicate that the recipient was successfully removed, but
rather that the command was queued for processing.

This is also available in Sendmail::Milter, but the "real" Sendmail::Milter
implementation allows calling of this method only from the "eom" callback.

=cut

sub delrcpt {
	my $this = shift;
	my $rcpt = shift || return undef;

	return undef unless ($this->{envfrom} && ($this->{cbref}{FLAGS} & SMFIF_DELRCPT));
	push(@{$this->{eom_actions}}, [SMFIR_DELRCPT, "$rcpt\0"]);
	1;
}

=pod

=item $ctx->replacebody(BUFFER)

Replace the message body with the data in BUFFER (a scalar).  This method
may be called multiple times, each call appending to the replacement buffer.  
End-of-line should be represented by CR-LF ("\r\n").

Returns a true value on success, undef on failure.

This is also available in Sendmail::Milter, but the "real" Sendmail::Milter
implementation allows calling of this method only from the "eom" callback.

(A future version of PMilter will provide a better way to perform this
operation.)

=cut

sub replacebody {
	my $this = shift;
	my $chunk = shift;

	return undef unless defined($chunk);
	return undef unless ($this->{envfrom} && ($this->{cbref}{FLAGS} & SMFIF_CHGBODY));
	push(@{$this->{body_chunks}}, \$chunk); # encoding handled in main loop
	1;
}

=pod

=head1 EXTENSION METHODS

The following methods are extensions to the L<Sendmail::Milter> interface,
and are only available in PMilter.

=over 4

=item $ctx->getaddress

In a scalar context, returns the IPv4 address of the connecting host.  If the
connecting host did not use IPv4 or connected via a loopback connection,
returns undef.

In a list context, returns a one or three element list consisting of address
family, (numeric) port, and (ASCII) address.  The address family is a string
that may equal (eq, not ==) one of:

=over 2

=item SMFIA_INET

IPv4 address, where the address is a dotted quad;

=item SMFIA_INET6

IPv6 address, where the address is a parseable IPv6 hex address;

=item SMFIA_UNIX

Local socket ("UNIX socket"), where the address is a pathname (if available);

=item SMFIA_UNKNOWN

Unknown connection method, where the port and address are not supplied.

=back

These four constants are exported by PMilter::Session individually, or via
the export tag ':all'.

=cut

sub getaddress {
	my $this = shift;

	(wantarray ? @{$this->{address}} :
		($this->{address}[0] eq SMFIA_INET ? $this->{address}[2] : undef));
}

=pod

=item $ctx->getglobal(KEY[, DEFAULT])

Retrieves a value by key KEY set previously by $ctx->setglobal(NAME,
VALUE).  These values are global to this context object, thus allowing
communication between individual objects in a PMilter::Callbacks tree.

If KEY has not yet been defined or is set to undef, first sets the value to
DEFAULT before returning.

KEY may be a string, or a Perl reference.  (Tie::RefHash is used for
the underlying implementation.)

Returns undef if the value is not set.

=cut

sub getglobal {
	my $this = shift;
	my $key = shift;
	my $default = shift;
	my $value = $this->{globals}{$key};

	$value = $this->{globals}{$key} = $default unless defined($value);

	$value;
}

=pod

=item $ctx->gethost

Returns the SMTP client's hostname (the same value as the second argument to
the connect callback).  This is typically a properly reverse-DNS resolved
hostname, though it may contain special values--such as the fixed string
"localhost" for loopback connections not involving a client socket.

=cut

sub gethost {
	my $this = shift;

	$this->{host};
}

=pod

=item $ctx->qlog([NAME => VALUE[, ...]])

Calls C<PMilter::Server::printlog(LOG_NOTICE, ...)> with a prepended MTA
queue ID (getsymval('i'), or NOQUEUE if not set), then appends NAME=VALUE
pairs separated by comma and space.  The VALUEs are not quoted.  If a VALUE
is undef, its NAME is not printed.

The following standard pairs are printed first, followed by the
caller-supplied pairs (if any):

    callback=<name of callback in progress>
    arg1=<first arg to callback, for all but 'body'>
    from=<first arg to 'envfrom' callback, if callback!='envfrom'>
    relay=$ctx->getsymval('_')

=cut

sub qlog {
	my $this = shift;

	my $qval = $this->getsymval('i') || 'NOQUEUE';
	my @logvalues = ();
	my @logentries = (
		callback => ($this->{callback} && $this->{callback}[0]),

		arg1 => ($this->{callback} &&
			 ($this->{callback}[0] ne 'body') &&
			 $this->{callback}[1]),

		from => ($this->{envfrom} &&
			 ($this->{callback}[0] ne 'envfrom') &&
			 $this->{envfrom}[0]),

		relay => $this->getsymval('_'),

		@_
	);

	while (@logentries > 0) {
		my $name = shift @logentries;
		my $value = shift @logentries;

		next unless defined($value);
		push(@logvalues, "$name=$value");
	}

	printlog(LOG_NOTICE, "$qval: ".join(', ', @logvalues));
}

=item $ctx->reject(RESULT[, NAME => VALUE[, ...]])

Rejects the callback in progress with the given RESULT.  This can be one of
SMFIS_REJECT or SMFIS_TEMPFAIL (causing use of a MTA-default reject
message), or a textual reject message.

A textual MESSAGE must start with a three-digit 4xx or 5xx SMTP status code
and a space.  If it does not, an error will be logged and SMFIS_TEMPFAIL
will be returned.  The message may, optionally, include an extended SMTP
response code after the 4xx/5xx code also separated by a space.

Calls C<$ctx->qlog()> with the NAME => VALUE pairs, if any, followed by a
pair "status=REJECT/TEMPFAIL", and "reply=MESSAGE" if a textual message is
supplied.

Finally, returns SMFIS_REJECT or SMFIS_TEMPFAIL depending on RESULT, so this
method can be called from within a "return" statement to reduce code
clutter.

=cut

sub reject {
	my $this = shift;
	my $message = shift;

	my $rc = SMFIS_TEMPFAIL;

	# eq avoids non-numeric warnings
	if ($message eq SMFIS_REJECT || $message eq SMFIS_TEMPFAIL) {
		$rc = $message;
	} else {
		if ($message !~ /^([45])\d\d /) {
			printlog(LOG_ERR, "Message does not start with SMTP code: $message");
			return $rc;
		}

		if ($1 eq '5') { # faster than using in regex; compiles regexes only once
			$rc = SMFIS_REJECT;
			$message =~ s/^(...)/$1 5.7.0/ if ($message !~ /^... 5\.\d\.\d/);
		} else {
			$message =~ s/^(...)/$1 4.7.0/ if ($message !~ /^... 4\.\d\.\d/);
		}

		$this->{reply} = $message;
	}

	push(@_, undef) if (@_ & 1);

	$this->{replylog} = [ @_,
		status => ($rc == SMFIS_REJECT ? 'REJECT' : 'TEMPFAIL'),
		reply => $this->{reply}
	];

	$rc;
}

=pod

=item $ctx->setglobal(KEY, VALUE)

Sets a key KEY to VALUE, for later retrieval by $ctx->getglobal(NAME).  
These values are global to this context object, thus allowing communication
between individual objects in a PMilter::Callbacks tree.

KEY may be a string, or a Perl reference.  (Tie::RefHash is used for
the underlying implementation.)

=cut

sub setglobal {
	my $this = shift;
	my $key = shift;
	my $value = shift;

	$this->{globals}{$key} = $value;
}

=pod

=back

=cut

# Recursive callback invocation -- not for the faint-hearted.
sub callback {
	my $this = shift;
	my $what = shift;

	return SMFIS_CONTINUE unless $this->{cbref}{$what};

	$this->{callback} = [ $what, @_ ]; # for qlog()
	&{$this->{cbref}{$what}}($this, @_);
}

# Private internal calls

sub call_hooks {
	my $this = shift;
	my $what = shift;

	my $rc = $this->{rc};

	if ($rc eq SMFIR_CONTINUE) {
		my $cbrc = $this->callback($what, @_);

		# translate to response codes
		if ($cbrc == SMFIS_ACCEPT || $cbrc == SMFIS_BREAK) {
			$rc = SMFIR_ACCEPT;
		} elsif ($cbrc == SMFIS_DISCARD) {
			$rc = SMFIR_DISCARD;
		} elsif ($cbrc == SMFIS_REJECT) {
			$this->qlog(@{$this->{replylog}}) if $this->{replylog};
			if (defined($this->{reply})) {
				$rc = SMFIR_REPLYCODE;
			} else {
				$rc = SMFIR_REJECT;
			}
		} elsif ($cbrc == SMFIS_TEMPFAIL) {
			$this->qlog(@{$this->{replylog}}) if $this->{replylog};
			if (defined($this->{reply})) {
				$rc = SMFIR_REPLYCODE;
			} else {
				$rc = SMFIR_TEMPFAIL;
			}
		}
	}

	if ($DebugLevel >= 1 && $rc ne SMFIR_CONTINUE) {
		my $rcname = $replynames{$rc} || "UNKNOWN VALUE: $rc";
		$rcname .= " $this->{reply}" if ($rc eq SMFIR_REPLYCODE);
		printlog(LOG_DEBUG, ">$what< $rcname");
	}

	if ($what eq 'abort' || $what eq 'close') {
		# These two operations shouldn't ever do a reject.

		$rc = SMFIR_CONTINUE;
	} elsif ($what eq 'eom') {
		# Send all pending modification actions.

		if ($this->{eom_actions}) {
			foreach my $action (@{$this->{eom_actions}}) {
				$this->write_packet(@$action);
			}
		}

		if ($this->{body_chunks}) {
			my $len = 0;
			my $socket = $this->{socket};

			foreach my $chunkref (@{$this->{body_chunks}}) {
				$len += length($$chunkref);
			}

			$len = pack('N', ($len + 1));
			$socket->syswrite($len);
			$socket->syswrite(SMFIR_REPLBODY);

			foreach my $chunkref (@{$this->{body_chunks}}) {
				$socket->syswrite($$chunkref);
			}
		}
	} elsif ($what ne 'envrcpt' || $rc eq SMFIR_ACCEPT || $rc eq SMFIR_DISCARD) {
		# envrcpt is Special; rejections affect only the current recipient,
		# and message processing continues.  All else should be global.
		#
		# Sendmail bug(?): SMFIR_ACCEPT and SMFIR_DISCARD on RCPT TO: still
		# affects the entire message, so let them pass.

		$this->{rc} = $rc;

		# If not in message context, keep the result around and reuse it.

		unless ($this->{envfrom}) {
			$this->{defrc} = $rc;
			$this->{defreply} = $this->{reply};
			$this->{defreplylog} = $this->{replylog};
		}
	}

	if ($rc eq SMFIR_ACCEPT && ($this->{body_chunks} || $this->{eom_actions})) {
		# If we're supposed to blind-accept but there are actions pending,
		# continue instead and wait for EOM (which will send the actions).

		$rc = SMFIR_CONTINUE;
	}

	if ($what ne 'abort' && $what ne 'close') {
		if ($rc eq SMFIR_REPLYCODE) {
			$this->write_packet($rc, $this->{reply}."\0");
		} else {
			$this->write_packet($rc);
		}
	}
}

sub read_block {
	my $this = shift;
	my $bufref = shift;
	my $len = shift;

	my $socket = $this->{socket};
	my $sofar = 0;

	$$bufref = '';

	while ($len > $sofar) {
		my $read = $socket->sysread($$bufref, $len - $sofar, $sofar);

		return undef if ($read <= 0); # if EOF

		$sofar += $read;
	}

	1;
}

# Exit message context and initialiaze per-message data.
sub reset_message {
	my $this = shift;

	delete @{$this}{qw(body_chunks eom_actions envfrom)};

	$this->{rc} = $this->{defrc};
	$this->{reply} = $this->{defreply};
	$this->{replylog} = $this->{defreplylog};
}

sub write_packet {
	my $this = shift;
	my $code = shift;
	my $out = shift;

	$out = '' unless defined($out);

	my $len = pack('N', length($out) + 1);
	my $socket = $this->{socket};

	$socket->syswrite($len);
	$socket->syswrite($code);
	$socket->syswrite($out);
}

# Constructor and main loop

sub new {
	my $this = bless {}, shift;
	my $socket = $this->{socket} = shift;
	my $cbref = $this->{cbref} = shift;

	unless (UNIVERSAL::isa($socket, 'IO::Socket')) {
		cluck 'arg not an IO::Socket';
		return undef;
	}

	# Determine required protocol; include any that are needed.
	# We always need CONNECT to get hostname and address.
	# We always need MAIL FROM: to determine start-of-message.

	$this->{protocol} = SMFIP_NONE & ~(SMFIP_NOCONNECT|SMFIP_NOMAIL);
	$this->{protocol} &= ~SMFIP_NOHELO if $cbref->{helo};
	$this->{protocol} &= ~SMFIP_NORCPT if $cbref->{envrcpt};
	$this->{protocol} &= ~SMFIP_NOBODY if $cbref->{body};
	$this->{protocol} &= ~SMFIP_NOHDRS if $cbref->{header};
	$this->{protocol} &= ~SMFIP_NOEOH if $cbref->{eoh};

	$this;
}

sub main {
	my $this = shift;

	my $buf = '';
	my $gotquit = 0;
	my $socket = $this->{socket} || return undef;

	my $split_buf = sub {
		$buf =~ s/\0$//; # remove trailing NUL
		return [ split(/\0/, $buf) ];
	};

	$socket->autoflush(1);
	$this->{defrc} = SMFIR_CONTINUE;
	$this->{globals} = {};
	tie %{$this->{globals}}, 'Tie::RefHash';

	$this->reset_message;

	while (1) {
		$this->read_block(\$buf, 4) || last;
		my $len = unpack('N', $buf);

		if ($len <= 0 || $len > 131072) {
			printlog(LOG_ERR, "Protocol: bad packet length $len");
			last;
		}

		# save the overhead of stripping the first byte from $buf
		$this->read_block(\$buf, 1) || last;
		my $cmd = $buf;

		# mark this command's macros as final (will be nuked next round);
		# also remove SMFIC_RCPT macros if this isn't a SMFIC_RCPT cmd
		$this->{macros}{$cmd}{_final} = 1;
		delete $this->{macros}{&SMFIC_RCPT} if ($cmd ne SMFIC_MACRO && $cmd ne SMFIC_RCPT);

		# get actual data
		$this->read_block(\$buf, $len - 1) || last;

		if ($cmd eq SMFIC_ABORT) {
			$this->reset_message;

			printlog(LOG_DEBUG, '<abort>') if ($DebugLevel >= 2);

			$this->call_hooks('abort');

			# no response
		} elsif ($cmd eq SMFIC_BODY) {
			printlog(LOG_DEBUG, '<body> (length '.length($buf).')') if ($DebugLevel >= 5);

			$this->call_hooks('body', $buf, length($buf));
		} elsif ($cmd eq SMFIC_CONNECT) {
			# Perl RE doesn't like matching multiple \0 instances.
			# To avoid problems, we slice the string to the first null,
			# then use unpack for the rest.

			unless ($buf =~ s/^([^\0]*)\0(.)//) {
				printlog(LOG_ERR, 'Protocol: invalid connect info');
				# XXX should LOG_DEBUG a hexdump here
				last;
			}

			$this->{host} = $1;
			my $af = $2;
			my ($port, $addr) = unpack('nZ*', $buf);

			if (defined($addr)) {
				$this->{address} = [ $af, $port, $addr ];
			} else {
				$this->{address} = [ $af ];
			}

			my $pack = ($af eq SMFIA_INET ?
				pack_sockaddr_in($port, inet_aton($addr)) :
				pack_sockaddr_in(0, INADDR_ANY));

			printlog(LOG_DEBUG, '<connect> '.join(' ', $this->{host}, '<sockaddr>', @{$this->{address}})) if ($DebugLevel >= 2);

			$this->call_hooks('connect', $this->{host}, $pack, @{$this->{address}});
		} elsif ($cmd eq SMFIC_MACRO) {
			unless ($buf =~ s/^(.)//) {
				printlog(LOG_ERR, 'Protocol: empty macro packet');
				last;
			}

			my $code = $1;
			my $marray = &$split_buf;

			# odd number of entries: give last empty value
			push(@$marray, '') if ((@$marray & 1) != 0);

			my %macros = @$marray;
			delete $this->{macros}{$code} if $this->{macros}{$code}{_final};

			while (my ($name, $value) = each(%macros)) {
				$this->{macros}{$code}{$name} = $value;
				printlog(LOG_DEBUG, "<macro $code> $name | $value") if ($DebugLevel >= 4);
			}

			# no response
		} elsif ($cmd eq SMFIC_BODYEOB) {
			printlog(LOG_DEBUG, '<eom>') if ($DebugLevel >= 5);

			$this->call_hooks('eom');
			$this->reset_message;
		} elsif ($cmd eq SMFIC_HELO) {
			my $helo = &$split_buf;

			unless (@$helo == 1) {
				printlog(LOG_ERR, 'Protocol: bad HELO packet');
				last;
			}

			printlog(LOG_DEBUG, "<helo> @$helo") if ($DebugLevel >= 2);

			$this->call_hooks('helo', @$helo);
		} elsif ($cmd eq SMFIC_HEADER) {
			my $header = &$split_buf;

			# empty value: ensure an empty string
			push(@$header, '') if (@$header == 1);

			printlog(LOG_DEBUG, '<header> '.join(' | ', @$header)) if ($DebugLevel >= 3);

			$this->call_hooks('header', @$header);
		} elsif ($cmd eq SMFIC_MAIL) {
			my $envfrom = &$split_buf;

			unless (@$envfrom >= 1) {
				printlog(LOG_ERR, 'Protocol: bad MAIL packet');
				last;
			}

			# Now in a message context; we can do message editing.
			$this->reset_message;
			$this->{envfrom} = $envfrom;

			printlog(LOG_DEBUG, '<envfrom> '.join(' | ', @$envfrom)) if ($DebugLevel >= 2);

			$this->call_hooks('envfrom', @$envfrom);
		} elsif ($cmd eq SMFIC_EOH) {
			printlog(LOG_DEBUG, '<eoh>') if ($DebugLevel >= 3);

			$this->call_hooks('eoh');
		} elsif ($cmd eq SMFIC_OPTNEG) {
			if (length($buf) != 12) {
				printlog(LOG_ERR, 'Protocol: SMFIC_OPTNEG has wrong size');
				last;
			}

			my ($ver, $actions, $protocol) = unpack('NNN', $buf);

			if ($ver != 2) {
				printlog(LOG_ERR, "Protocol: Bad milter protocol version $ver");
				last;
			}

			$this->write_packet(SMFIC_OPTNEG, pack('NNN', 2,
				$this->{cbref}{FLAGS} & $actions,
				$this->{protocol} & $protocol));
		} elsif ($cmd eq SMFIC_RCPT) {
			my $envrcpt = &$split_buf;

			unless (@$envrcpt >= 1) {
				printlog(LOG_ERR, 'Protocol: bad RCPT packet');
				last;
			}

			printlog(LOG_DEBUG, '<envrcpt> '.join(' | ', @$envrcpt)) if ($DebugLevel >= 2);

			$this->call_hooks('envrcpt', @$envrcpt);
		} elsif ($cmd eq SMFIC_QUIT) {
			printlog(LOG_DEBUG, '<quit>') if ($DebugLevel >= 2);

			$gotquit = 1;
			last;
			# that's all, folks!
		} else {
			printlog(LOG_ERR, "Protocol: Unknown milter packet type $cmd");
			last;
		}
	}

	$this->reset_message;
	$this->call_hooks('close');
	$this->write_packet(SMFIR_TEMPFAIL) unless $gotquit;

	%$this = (); # release *all* state information
	$socket->close;

	$gotquit;
}

1;

__END__

=back

=head1 SEE ALSO

L<PMilter::Callbacks> for the callback flags and callback method
descriptions

L<Sendmail::Milter> for notes on the Sendmail::Milter compatibility
interface (PMilter::Session is PMilter's implementation of the "ctx" context
object passed to all callbacks)
