# $DUH: Server.pm,v 1.28 2002/12/13 23:13:09 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::Server - Perl binding of Sendmail Milter protocol

=head1 SYNOPSIS

    use PMilter::Server;
    use PMilter::Callbacks qw(:all);
    PMilter::Server::setconn(DESC);
    PMilter::Server::register(NAME, { CALLBACKS }[, FLAGS]);
    PMilter::Server::main();

=head1 DESCRIPTION

PMilter::Server is the top-level entry point to the PMilter protocol
implementation.  Currently, all methods are "static", that is, they should
be called as "PMilter::Server::method" or imported as regular subroutines.

Nothing is exported by PMilter::Server by default.

Each of these methods and the LOG_* constants may be imported into the
caller's namespace individually, or all of them with the import tag ':all'.  

=head1 METHODS

=over

=cut

package PMilter::Server;
use base Exporter;

use strict;
use warnings;

use Carp qw(cluck confess);
use Errno;
use IO::Select;
use IO::Socket;
use PMilter;
use POSIX;
use Socket;
use UNIVERSAL;

*VERSION = *PMilter::VERSION;

# Exported names

my @loglevels = (undef, qw(alert crit err warning notice info debug));
my @export_log = qw(printlog);

for my $i (1 .. $#loglevels) {
	no strict 'refs';
	my $name = 'LOG_'.uc($loglevels[$i]);
	*{"PMilter::Server::$name"} = sub () { $i };
	push(@export_log, $name);
}

our @EXPORT_OK = (
	qw(
		register
		setconn
		setlogger
		setloglevel
		setsyslog
		main
	),
	@export_log
);

our %EXPORT_TAGS = (
	all => \@EXPORT_OK,
	log => \@export_log
);

# Local state

my $cbref;
my $nchildren = 0;
my $socket;

my $loglevel = 5; # LOG_WARNING
my $logger = sub {
	my $name = shift;
	my $msg = "$name: ".(shift);
	my $sev = shift;

	if ($sev >= &LOG_ERR) {
		print STDERR "$msg\n";
	} else {
		print "$msg\n";
	}
};

# Frontend functions

=pod

=item printlog(SEVERITY, MESSAGE)

If SEVERITY is greater than or equal to the level set by C<setloglevel()>,
logs a message to the current PMilter system logger set by C<setlogger()>.

The SEVERITY can be one of the following, with their recommended usage.  
Priority is listed in descending order (highest priority at top).  
Importing the tag ':log' will import printlog() and all of these constants:

=over 2

=item LOG_ALERT

Emergency messages of a "nothing is working" variety.  The message may be
written to many places.  Not for most people.

=item LOG_CRIT

Messages about problems accessing necessary local system resources, such as
inability to contact a database or open a socket.

=item LOG_ERR

Errors in processing data.  Milter protocol violations are reported with
this level.

=item LOG_WARNING

Warnings that may need eventual administrator intervention.  For example, a
message about a temporary failure accessing a DNSBL should use this.

=item LOG_NOTICE

Special messages that may deserve attention by the administrator.  This is
the level used by default by milter modules when calling $ctx->reject().

=item LOG_INFO

Informational messages, such as startup actions useful for log summaries.

=item LOG_DEBUG

Debugging messages.  The PMilter::Server protocol engine uses this level to
log the entire progress of a connection.

=back

=cut

# flag to note if printlog() is on the stack
our $in_logger = 0;

sub printlog ($$) {
	my $sev = shift;
	my $msg = shift;
	local $in_logger = 1; # prevent infinite recursion

	if ($sev < 0 || $sev > @loglevels) {
		$msg = "[INVALID SEVERITY $sev] $msg";
		$sev = &LOG_ERR;
	}

	&$logger(($cbref && $cbref->{NAME}) || $0, $msg, $sev) if ($loglevel >= $sev);
}

=pod

=item register(NAME, CALLBACKS[, FLAGS])

Sets up the top-level milter object for callbacks.  

NAME is the name of the milter.  For compatibility with L<Sendmail::Milter>,
this should be the same name as passed to auto_getconn() or auto_setconn(),
but PMilter does not enforce this.

CALLBACKS is a hash reference containing one or more callback subroutines.  
This is described in further detail in the documentation for
L<PMilter::Callbacks>.  If a callback is not named in this hashref, the
caller's package will be searched for

FLAGS, if specified, is a bitmask of message modification actions (see
L<PMilter::Callbacks>) that are requested by the callback object for use
during message processing.  If the corresponding bit is not set, an action
will not be allowed during message processing.

NOTE: register() must be called successfully exactly once.  To register
multiple chained milters, create a L<PMilter::Callbacks> container object
and add milters to that container; then call register(undef, $mycallbacks)

Returns a true value on success, undef on failure (for compatibility with
Sendmail::Milter::register).  However, future versions of PMilter::Server
may use C<die> instead.

The above is fully compatible with L<Sendmail::Milter>.  PMilter extensions:

* The NAME argument is completely ignored.  It may be undef.

* Auto-search logic of the caller's package may be configured by setting the
key "PACKAGE" in the CALLBACKS hashref.  The value of this key is used as
the package to search for default callbacks, and can be set to a single
colon (':') to disable auto-search.  If not set, the package of the caller
to register() will be used.

* The CALLBACKS hashref may be undef.  In this case, only the auto-search
logic will be used.

* FLAGS may be specified inside the CALLBACKS hashref under the key "FLAGS".  
If set in the hashref, the argument to register() is ignored.

=cut

sub register (;$$$) {
	confess 'callbacks already registered' if defined($cbref);

	my $name = shift;
	confess 'passed ref as name argument to register()' if ref($name);

	my $callbacks = shift;
	confess 'callbacks not hash ref' if ($callbacks && !UNIVERSAL::isa($callbacks, 'HASH'));

	my $cbcopy = $callbacks ? { %$callbacks } : {};

	$cbcopy->{FLAGS} ||= shift;
	$cbcopy->{PACKAGE} ||= caller;

	eval {
		require PMilter::Callbacks;
		$cbref = new PMilter::Callbacks($cbcopy);
		$cbref->{NAME} ||= $name; # used by loggers
	};

	printlog(&LOG_ALERT, $@) if $@;
	defined($cbref);
}

=pod

=item setconn(DESC)

Sets up the server socket with connection descriptor DESC.  This is
identical to the descriptor syntax used by the "X" milter configuration
lines in sendmail.cf (if using Sendmail).  In this version, this should be
one of the following:

=over 2

=item local:PATH

A local ("UNIX") socket on the filesystem, named PATH.  This has some smarts
that will auto-delete the pathname if it seems that the milter is not
currently running (but this currently contains a race condition that may not
be fixable; at worst, there could be two pmilters running with one never
receiving connections).

=item inet:PORT[@HOST]

An IPv4 socket, bound to address HOST (default INADDR_ANY), on port PORT.  
It is not recommended to open milter engines to the world, so the @HOST part
should be specified.

=back

Returns a true value on success, undef on failure (for compatibility with
Sendmail::Milter::setconn).  However, future versions of PMilter::Server may
use C<die> instead.

This is fully compatible with Sendmail::Milter.

=cut

sub setconn ($) {
	my $conn = shift;

	unless ($conn =~ /^([^:]+):([^:@]+)(@([^:@]+))?$/) {
		cluck "socket $conn: unspecified protocol";
		return undef;
	}

	confess 'socket already created' if defined($socket);

	if ($1 eq 'local' || $1 eq 'unix') {
		my $path = $2;
		my $addr = sockaddr_un($path);

		unless ($path =~ m,^/,,) {
			cluck "socket $conn: path not absolute";
			return undef;
		}

		if (-e $path && ! -S $path) { # exists, not a socket
			$! = Errno::EEXIST;
		} else {
			$socket = IO::Socket::UNIX->new(Type => SOCK_STREAM);
		}

		# Some systems require you to unlink an orphaned inode.
		# There's a race condition here, but it's unfortunately
		# simply not fixable.  Using an END{} block doesn't
		# always work, and that's too wonky with fork() anyway.

		if (defined($socket) && !$socket->bind($addr)) {
			if ($socket->connect($addr)) {
				close $socket;
				undef $socket;
				$! = Errno::EADDRINUSE;
			} else {
				unlink $path; # race condition
				$socket->bind($addr) || undef $socket;
			}
		}
	} elsif ($1 eq 'inet' || $1 eq 'inet6') {
		if ($1 eq 'inet6') {
			confess 'inet6 not supported in this version';
		}

		$socket = IO::Socket::INET->new(
			Proto => 'tcp',
			ReuseAddr => 1,
			LocalPort => $2,
			LocalAddr => $3
		);
	} else {
		cluck "socket $conn: unknown protocol";
		return undef;
	}

	if (defined($socket)) {
		$socket->listen(5) || undef $socket;
	}

	cluck "socket $conn: $!" unless defined($socket);
	1;
}

=pod

=item setlogger BLOCK

Sets the PMilter system logger.  The BLOCK should be a Perl code block, or a
subroutine reference created with C<\&> (for instance, to set logging to sub
foo, do C<setlogger(\&foo)>; similarly C<setlogger(\&$foo)> for sub
reference C<$foo>).

This subroutine is passed three arguments:  the name of the top-level
callback, or $0 if the top-level callback is undefined; the message to log
(as passed to C<printlog()>); and the severity.  Note that the last two
arguments are B<reversed> from those passed to C<printlog()>, as many
subroutines will not care about the severity.

Note that if no logger has been set, each log message is simply printed to
the standard output (or STDERR if severity >= LOG_ERR), followed by a
newline.

=cut

sub setlogger (&) {
	$logger = shift;
}

=pod

=item setloglevel(LEVEL)

Sets the minimum log level to LEVEL.  Messages below this level are not
logged.  This level defaults to LOG_WARNING.

=cut

sub setloglevel ($) {
	my $sev = shift;

	if ($sev < 1 || $sev > @loglevels) {
		printlog(&LOG_ERR, "Invalid severity $sev passed to setloglevel()");
	} else {
		$loglevel = $sev;
	}
}

=pod

=item setsyslog([FACILITY[, FLAGS]])

Sets the PMilter system logger to syslog (see L<Sys::Syslog>), using the
named FACILITY (defaults to 'mail').  If C<Sys:Syslog> is not available,
this call will cause a C<die>.

FLAGS are additional flags to pass to Sys::Syslog::openlog(), as a string
with values separated by commas; 'pid,cons' is popular here.  If not
defined, 'pid' is the default.  (To have no options, use the empty string
''.)

=cut

sub setsyslog (;$$) {
	require Sys::Syslog;

	my $fac = shift || 'mail';
	my $flags = shift;

	$flags = 'pid' unless defined($flags); # allow null string, though

	my $oldname = '';

	eval {
		# Change default to local socket, if available.
		Sys::Syslog::setlogsock('unix');
	};

	$logger = sub {
		my $name = shift;
		my $msg = shift;
		my $sev = shift;

		if ($name ne $oldname) {
			Sys::Syslog::closelog() if $oldname;
			Sys::Syslog::openlog($name, $flags, $fac);
		}

		Sys::Syslog::syslog($loglevels[$sev], $msg);
	};
}

=pod

=item main([MAXCHILDREN[, MAXREQ]])

This is the last method called in the main block of a milter program.  If
successful, this call never returns; the protocol engine is launched and
begins accepting connections.

MAXCHILDREN (default unlimited) specifies the maximum number of connections
that may be serviced simultaneously.  If a connection arrives with the
number of active connections above this limit, the connection will be
immediately dropped, defaulting to the milter-is-missing action specified in
the MTA configuration.

MAXREQ (default 1) is the maximum number of requests that a child may
service before being recycled.  (Currently, PMilter fixes this at 1.)

This is fully compatible with Sendmail::Milter.

In the PMilter implementation, this will fork() a new child process per
connection.  In the future, there may be an additional option or autodetect
logic to use Perl multithreading where available.

Also, while in the context of C<PMilter::Server::main()>, all Perl C<warn>
and C<die> operations in the global context (and things that call them, such
as C<Carp>), will have their output routed through C<printlog()> rather than
to STDERR.

=cut

sub main (;$$) {
	local $SIG{__WARN__} = sub {
		foreach my $line (split(/\n/, $_[0])) {
			printlog(&LOG_WARNING, '*WARN* '.$line);
		}
	};

	local $SIG{__DIE__} = sub {
		die @_ if $^S || $in_logger; # don't do this in an eval

		foreach my $line (split(/\n/, $_[0])) {
			printlog(&LOG_ERR, '*DIE* '.$line);
		}

		exit 1; # nothing to stderr
	};

	confess 'callbacks not registered' unless defined($cbref);

	my $maxchildren = shift;
	my $maxrequests = shift; # not yet implemented

	# Decrement child count on child exit.
	sub sigchld {
		my $pid;

		while (($pid = waitpid(-1, WNOHANG)) > 0) {
			$nchildren--;
		}

		$SIG{CHLD} = \&sigchld;
	}
	local $SIG{CHLD} = \&sigchld;

	local $SIG{INFO} = sub {
		print STDERR "Number of active children: $nchildren\n";
	};

	while (1) {
		my $client = $socket->accept || die "$0: accept: $!\n";

		# If the load's too high, fail and go back to top of loop.
		if ($maxchildren) {
			my $cnchildren = $nchildren; # make constant

			if ($cnchildren >= $maxchildren) {
				printlog(&LOG_WARNING, "load too high: children $cnchildren >= max $maxchildren");
				$client->autoflush(1);
				$client->print(pack('N/a*', 't')); # SMFIR_TEMPFAIL
				$client->close;
				next;
			}
		}

		my $pid = fork;

		if ($pid < 0) {
			die "$0: fork: $!\n";
		} elsif ($pid) {
			$nchildren++;
			$client->close;
		} else {
			$socket->close;
			undef $socket;
			undef $@;

			$SIG{PIPE} = 'IGNORE'; # so close_callback will be reached

			require PMilter::Session;
			my $session = PMilter::Session->new($client, $cbref) || exit 0;

			$session->main;
			exit 0;
		}
	}

	undef;
}

1;

__END__

=back

=head1 SECURITY CONSIDERATIONS

=over 4

=item Running as root

Running Perl as root is dangerous.  Running PMilter as root may well be
system-suicide at this point.  So don't do that.

More specifically, though, it is possible to run a PMilter frontend as root,
in order to gain access to network resources (such as a filesystem socket in
/var/run), and then drop privileges before accepting connections.  To do
this, insert drop-privileges code between calls to setconn/auto_setconn and
main:

    PMilter::Server::setconn(Sendmail::Milter::auto_getconn('pmilter'));
    $> = 65534; # drop root privs
    PMilter::Server::main();

The semantics of properly dropping root privileges in Perl are,
unfortunately, somewhat OS-specific, so this process is not described in
detail here.

=back

=head1 SEE ALSO

L<PMilter::Callbacks> for the callback flags and callback method
descriptions

L<PMilter::Session> on how to use the callback context object

L<Sendmail::Milter> for notes on the Sendmail::Milter compatibility
interface
