# $DUH: CIDR.pm,v 1.1 2002/12/16 22:25:23 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::Match::CIDR - package for matching IPv4 addresses to CIDR patterns

=head1 SYNOPSIS

    use PMilter::Match::CIDR;

    my $matcher = PMilter::Match::CIDR->new('1.2.0.0/16', '3.4.5.16/28');

    $matcher->match('1.2.3.4');

=head1 DESCRIPTION

Provides a fast CIDR pattern matcher.

=head1 SUBROUTINES

The following subs can be imported by name, or all with tag ':all'.

=over 4

=cut

package PMilter::Match::CIDR;
use base Exporter;

use strict;
use warnings;

use Carp;
use PMilter;

*VERSION = *PMilter::VERSION;

our @EXPORT_OK = qw(
	match_valid
);

our %EXPORT_TAGS = ( all => \@EXPORT_OK );

=pod

=item match_valid(ADDRESS)

Returns a true value if the ADDRESS is a valid IP address and does not match
one of the following private-use or invalid address ranges per RFC3330;
undef otherwise.  Typically, this is used to ensure that an IP address is
both valid and used only by public Internet connections.

    0.0.0.0/8       (invalid)
    10.0.0.0/8      (private use)
    14.0.0.0/8      (X.121 nodes)
    127.0.0.0/8     (loopback)
    169.254.0.0/16  (link-local)
    172.16.0.0/12   (private use)
    192.0.2.0/24    (test network)
    192.168.0.0/16  (private use)
    198.18.0.0/15   (test network)
    224.0.0.0/4     (class-D multicast)
    240.0.0.0/4     (class-E future use)

as well as the unallocated /8's (as of 16-Dec-2002): 1, 2, 5, 14, 23, 27,
31, 36, 37, 39, 41, 42, 46, 49, 50, 58-60, 70-79, 83-126.

=cut

our $badmatcher;

sub match_valid {
	$badmatcher = PMilter::Match::CIDR->new qw(
		0/7 2/8 5/8 10/8 14/8 23/8 27/8 31/8 36/7
		39/8 41/8 42/8 46/8 49/8 50/8 58/7 60/8
		70/7 72/5 83/8 84/6 88/5 96/3 224/3
		169.254/16 172.16/12 192.0.2/24 192.168/16 198.18/15
	) unless ($badmatcher);

	my $addr = shift;

	return undef if ($addr !~ /^\d+\.\d+\.\d+\.\d+/);

	map {
		s/^0+(?=\d)//;
		return undef if ($_ < 0 || $_ > 255);
	} split(/\./, $addr);

	!$badmatcher->match($addr) || undef;
}

# The algorithm used here is 32->0 bit descent matching.  The object
# is an arrayref that optionally holds hashrefs in elements 0 .. 32,
# representing the number of bits masked (32 - /NN size).
# These hashrefs use, as keys, text integer representations of the CIDR
# patterns, shifted right by the number of bits masked.

=pod

=back

=head1 METHODS

These are object methods that facilitate creation of custom matchers.

=over 4

=item new(CIDR[, CIDR ...])

Compiles a list of CIDR patterns into an object, which can be called with
"match".  The CIDR patterns must have one or more octets and a size; thus
C<10/8>, C<192.168/16>, and <12.34.56.78/31> are all valid CIDR
specifications.

=cut

sub new {
	my $class = shift || croak 'invoked new as non-class';
	my $this = bless [], $class;

	foreach my $cidr (@_) {
		croak "invalid CIDR: $cidr" unless ($cidr =~ m,^([\d\.]+)(?:/0*(\d+))?$,);

		my @addr = split(/\./, $1);
		my $mask = (defined $2 ? $2 : 32);

		croak "invalid CIDR: $cidr"
			unless ((@addr <= 4) && $mask >= 0 && $mask <= 32);

		# Validate, and treat opening 0s as decimal (not octal).
		@addr = map {
			s/^0+(?=\d)//;
			croak "invalid CIDR: $cidr" unless ($_ >= 0 && $_ <= 255);
			$_
		} @addr;

		my $iaddr = unpack('N', pack('C4', (@addr, 0, 0, 0, 0)));
		$mask = 32 - $mask; # invert

		carp "CIDR has bits in masked portion: $cidr"
			if ($iaddr & ($mask == 32 ? ~0 : ((1 << $mask) - 1)));

		# hashes are autovivified
		$this->[$mask]{$mask == 32 ? 0 : ($iaddr >> $mask)} = 1;
	}

	$this;
}

=pod

=item match(ADDRESS)

Match a dotted-quad IP address against this object.  Returns a true value if
it matched, undef otherwise.  If the ADDRESS passed in is not valid, returns
undef (does not warn or produce error).

=cut

sub match {
	my $this = shift || croak 'bad object';
	my @addr = split(/\./, shift);

	return undef unless (@addr == 4);

	# Validate, and treat opening 0s as decimal (not octal).
	@addr = map {
		s/^0+(?=\d)//;
		return undef unless ($_ >= 0 && $_ <= 255);
		$_
	} @addr;

	my $iaddr = unpack('N', pack('C4', @addr));

	for my $mask (0 .. 32) {
		return 1 if ($this->[$mask] && $this->[$mask]{$iaddr});
		$iaddr >>= 1;
	}

	return undef;
}

1;

__END__

=back
