# $DUH: DNSBL.pm,v 1.14 2002/12/16 21:44:38 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::DNSBL - basic DNSBL list checker

=head1 SYNOPSIS

    my $dnsbl = PMilter::DNSBL->new('my.dnsbl.com', [ '127.0.0.2' ]);

    $dnsbl->addreturns('127.0.0.2'[, ...]);

    my $result = $dnsbl->query('78.56.34.12');

=head1 DESCRIPTION

PMilter::DNSBL is a simple checker for DNSBL-based queries on the domain
name given to PMilter::DNSBL at C<new> time.

=head1 METHODS

=over 4

=cut

package PMilter::DNSBL;

use strict;
use warnings;

use Carp;
use Net::hostent;
use Socket;

=pod

=item new(DOMAIN[, RETURNS[, MESSAGE[, IGNTEMP]]])

DOMAIN is the domain name to search.

If RETURNS is supplied, it may be a value or an array reference of values.  
A successful lookup must have an address matching one of these supplied
values to be considered a match.  RETURNS may be undef, in which case a
successful lookup returning B<any> value will succeed.

MESSAGE is an extended message to return in the case of a successful lookup;
if not supplied, the returned message will be undef.

If IGNTEMP is set to a true value, then temporary failures will be ignored
and returned as if they were no-result lookups.

=cut

sub new {
	my $class = shift;
	my $this = bless {}, $class;

	$this->{domain} = shift || confess 'no domain name supplied';

	my $returns = shift;
	if ($returns) {
		$returns = [ $returns ] unless ref($returns);

		foreach my $return (@$returns) {
			$this->{returns}{$return} = 1;
		}
	}

	$this->{message} = shift;
	$this->{igntemp} = shift;

	return $this;
}

=pod

=item query(ENTRY)

This is the user-visible front end to a DNSBL object.

Queries the DNSBL for the presence of the given ENTRY.  Note that this must
be a valid fragment of a domain name, and will be tacked onto the DNSBL
domain verbatim (IP addresses will B<NOT> be reversed).

Returns undef if the lookup succeeded and returned an empty list ("unknown
host").  Otherwise returns an arrayref of three elements:  an arrayref of
the ASCII address values found (or undef if the lookup failed), the
containing domain, and the extended message provided to the constructor
(undef if not set).

=cut

sub query {
	my $this = shift;
	my $entry = shift;

	my $rv = gethostbyname($entry.'.'.$this->{domain});

	# h_errno 1 == HOST_NOT_FOUND
	return undef if (!$rv && ($? == 1 || $this->{igntemp}));

	return [ undef, $this->{domain}, $this->{message} ] unless $rv;

	return [
		[ map { inet_ntoa($_) } @{$rv->addr_list} ],
		$this->{domain},
		$this->{message}
	] unless keys %{$this->{returns}};

	return undef unless ($rv->addrtype == AF_INET);

	foreach my $addr (@{$rv->addr_list}) {
		return [ [ inet_ntoa($addr) ], $this->{domain}, $this->{message} ]
			if $this->{returns}{inet_ntoa($addr)};
	}

	undef;
}

=pod

=item setdefault(MESSAGE)

Sets the default lookup placed in element [2] of the arrayref returned by
C<query()>.  Returns the current object, so this method call can be chained
with "new".

=cut

sub setdefault {
	my $this = shift;
	$this->{default} = shift;

	$this;
}

1;

__END__

=back

=head1 SEE ALSO

L<PMilter::DNSBL::TXT> for a version which fetches the TXT record for the
extended message

L<PMilter::DNSBL::List> for a DNSBL sequential-list container
