/* Copyright (C) 2005 Red Hat, Inc. */

#ifndef _SEMANAGE_DATABASE_H_
#define _SEMANAGE_DATABASE_H_

#ifndef DBASE_RECORD_DEFINED
typedef void* record_t;
typedef void* record_key_t;
#define DBASE_RECORD_DEFINED
#endif

#ifndef DBASE_DEFINED
typedef void* dbase_t;
#define DBASE_DEFINED
#endif

#include <stddef.h>

/* Circular dependency */
struct semanage_handle;

/* RECORD interface - method table */
typedef struct record_table {

	/* Create a record */
	int (*create) (
		struct semanage_handle* handle,
		record_t** rec);

	/* Extract key from record */
	int (*key_extract) (
		struct semanage_handle* handle,
		record_t* rec, 
		record_key_t** key);

	/* Free record key */
	void (*key_free) (
		record_key_t* key);

	/* Return 0 if record can be matched against key, and 1 otherwise */
	int (*compare) (
		record_t* rec, 
		record_key_t* key);

	/* Deep-copy clone of this record */
	int (*clone) (
		struct semanage_handle* handle,
		record_t* rec, 
		record_t** new_rec);

	/* Deallocate record resources. Must sucessfully handle NULL. */
	void (*free) (
		record_t* rec);

} record_table_t;

/* DBASE interface - method table */
typedef struct dbase_table {

	/* --------------- Database Functionality ----------- */

	/* Add the specified record to
	 * the database if it is not present,
	 * or fail if it already exists */
	int (*add) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		record_key_t* key,
		record_t* data);

	/* Add the specified record to the  
	 * database if it not present. 
	 * If it's present, replace it */
	int (*modify) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		record_key_t* key,
		record_t* data);

	/* Modify the specified record in the database
	 * if it is present. Fail if it does not yet exist */
	int (*set) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		record_key_t* key,
		record_t* data);

	/* Delete a record */
	int (*del) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		record_key_t* key);

	/* Retrieve a record */
	int (*query) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		record_key_t* key,
		record_t** response);

	/* Check if a record exists */
	int (*exists) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		record_key_t* key,
		int* response);

	/* Count the number of records */
	int (*count) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		unsigned int* response);

	/* Execute the specified handler over 
	 * the records of this database. The handler
	 * can signal a successful exit by returning 1,
	 * an error exit by returning -1, and continue by
	 * returning 0 */
	int (*iterate) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		int (*fn) (
			record_t* record,
			void* varg),
		void* fn_arg);

	/* Construct a list of all records in this database */
	int (*list) (
		struct semanage_handle* handle,
		dbase_t* dbase,
		record_t*** records,
		size_t* count);

	/* ---------- Cache/Transaction Management ---------- */

	/* Cache the database (if supported).
 	 * Invoke this function before looking at
	 * any data, since the database may require 
	 * cached content */
	int (*cache) (
		struct semanage_handle* handle,
		dbase_t* dbase);
	
	/* Forgets all changes that haven't been written
	 * to the database backend */
	void (*drop_cache) (
		dbase_t* dbase);

	/* Checks if there are pending changes */
	int (*is_modified) (
		dbase_t* dbase);

	/* Writes the database changes to its backend */
	int (*flush) (
		struct semanage_handle* handle,
		dbase_t* dbase);

	/* ------------- Polymorphism ----------------------- */


	/* Retrieves the record table for this database,
	 * which specifies how to perform basic operations
	 * on each record. */
	record_table_t* (*get_rtable) (
		dbase_t* dbase);

} dbase_table_t;

typedef struct dbase_config {

	/* Database state */
	dbase_t* dbase;

	/* Database methods */
	dbase_table_t* dtable;

} dbase_config_t;

extern int dbase_add (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	record_key_t* key,
	record_t* data);

extern int dbase_modify (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	record_key_t* key,
	record_t* data);

extern int dbase_set (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	record_key_t* key,
	record_t* data);

extern int dbase_del (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	record_key_t* key);

extern int dbase_query (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	record_key_t* key,
	record_t** response);

extern int dbase_exists (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	record_key_t* key,
	int* response);

extern int dbase_count (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	unsigned int* response);

extern int dbase_iterate(
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	int (*fn) (record_t* record, void* fn_arg),
	void* fn_arg);

extern int dbase_list (
	struct semanage_handle* handle,
	dbase_config_t* dconfig,
	record_t*** records,
	size_t* count);

#endif
